;;; ducpel-glyphs.el --- Default ducpel glyphs  -*- lexical-binding: t -*-

;; Copyright © 2014, 2026 Alex Kost

;; Author: Alex Kost <alezost@gmail.com>
;; Created: 29 Mar 2014

;; This program is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation, either version 3 of the License, or
;; (at your option) any later version.
;;
;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.
;;
;; You should have received a copy of the GNU General Public License
;; along with this program.  If not, see <http://www.gnu.org/licenses/>.

;;; Commentary:

;; Default theme of glyphs for the ducpel game.

;;; Code:

(require 'ducpel)


;;; Cell characters

(defvar ducpel-glyphs-empty-char ?\s)
(defvar ducpel-glyphs-wall-char ?#)
(defvar ducpel-glyphs-impassable-char ?@)
(defvar ducpel-glyphs-floor-chars ".ET◄►▲▼=║╬")
(defvar ducpel-glyphs-box-chars   "□et<>^v–|+")
(defvar ducpel-glyphs-man-chars
  (make-string (length ducpel-glyphs-floor-chars) ?○))
(defvar ducpel-glyphs-active-man-chars
  (make-string (length ducpel-glyphs-floor-chars) ?●))

(defun ducpel-glyphs-get-char-by-plist (&rest plist)
  "Return a character by cell plist.
Characters are used in a ducpel gamegrid if `ducpel-use-glyphs' is nil."
  (let ((type (plist-get plist :type)))
    (cond
     ((eql type ducpel-empty)      ducpel-glyphs-empty-char)
     ((eql type ducpel-wall)       ducpel-glyphs-wall-char)
     ((eql type ducpel-impassable) ducpel-glyphs-impassable-char)
     ((eql type ducpel-floor)
      (aref ducpel-glyphs-floor-chars (plist-get plist :floor)))
     ((eql type ducpel-man)
      (aref ducpel-glyphs-man-chars (plist-get plist :floor)))
     ((eql type ducpel-active-man)
      (aref ducpel-glyphs-active-man-chars (plist-get plist :floor)))
     ((eql type ducpel-box)
      (aref ducpel-glyphs-box-chars (plist-get plist :box))))))


;;; Image colors

(defvar ducpel-glyphs-colors
  '(
    ;; general
    (?#  . "None")                      ; background (empty cell)
    (?@  . "#000000")                   ; floor/box borders
    (?$  . "#3E403D")                   ; lines in teleport cell

    ;; floor
    (?\( . "#9B9E60")                   ; background
    (?~  . "#037502")                   ; arrow

    ;; box
    (?\) . "#B4B878")                   ; background
    (?/  . "#6B6D47")                   ; side
    (?-  . "#088508")                   ; arrow
    )
  "Alist of characters and colors used in default glyphs (XPM images).")

(defvar ducpel-glyphs-man-colors
  '((?_  . "#9C9E9B"))
  "Alist of characters and colors specific for the `man' glyphs.")

(defvar ducpel-glyphs-active-man-colors
  '((?_  . "#E5F500"))
  "Alist of characters and colors specific for the `active-man' glyphs.")

(defvar ducpel-glyphs-wall-colors
  '((?+  . "#938577")
    (?=  . "#756A5F")
    (?,  . "#B8A794"))
  "Alist of characters and colors specific for the `wall' glyph.")

(defvar ducpel-glyphs-impassable-colors
  '((?+  . "#777F93")
    (?=  . "#5F6575")
    (?,  . "#949EB8"))
  "Alist of characters and colors specific for the `impassable' glyph.")

(defvar ducpel-color-type-alist
  (list
   (cons ducpel-man        'ducpel-glyphs-man-colors)
   (cons ducpel-active-man 'ducpel-glyphs-active-man-colors)
   (cons ducpel-wall       'ducpel-glyphs-wall-colors)
   (cons ducpel-impassable 'ducpel-glyphs-impassable-colors))
  "Alist of possible color variables for the cell types.
Car of each assoc is a cell type.  Cdr is a variable with specific colors.")

(defun ducpel-glyphs-get-colors (chars &optional type)
  "Return color alist for CHARS.

CHARS is a list of characters.
TYPE is a type of a ducpel cell.

Returning value is alist of all characters from CHARS and their
color values from a variable defined by
TYPE (`ducpel-glyphs-...-colors') or from
`ducpel-glyphs-colors'."
  (let* ((color-var (cdr (assoc type ducpel-color-type-alist)))
         (colors (symbol-value color-var)))
    (mapcar
     (lambda (char)
       (let ((color
              (or (and colors
                       (cdr (assq char colors)))
                  (cdr (assq char ducpel-glyphs-colors))
                  (progn (message "No color for the character '%c'. Using pink."
                                  char)
                         "#ff1493"))))
         (cons char color)))
     chars)))


;;; XPM data

;;; Empty cell

(defvar ducpel-glyphs-empty-xpm-data "\
################################
################################
################################
################################
################################
################################
################################
################################
################################
################################
################################
################################
################################
################################
################################
################################
################################
################################
################################
################################
################################
################################
################################
################################
################################
################################
################################
################################
################################
################################
################################
################################")

;;; Wall

(defvar ducpel-glyphs-wall-xpm-data "\
@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
@@@++++++++++++++++++++++++++@@@
@+@@++++++++++++++++++++++++@@=@
@++@@++++++++++++++++++++++@@==@
@+++@@++++++++++++++++++++@@===@
@++++@@++++++++++++++++++@@====@
@+++++@@@@@@@@@@@@@@@@@@@@=====@
@+++++@,,,,,,,,,,,,,,,,,,@=====@
@+++++@,,,,,,,,,,,,,,,,,,@=====@
@+++++@,,,,,,,,,,,,,,,,,,@=====@
@+++++@,,,,,,,,,,,,,,,,,,@=====@
@+++++@,,,,,,,,,,,,,,,,,,@=====@
@+++++@,,,,,,,,,,,,,,,,,,@=====@
@+++++@,,,,,,,,,,,,,,,,,,@=====@
@+++++@,,,,,,,,,,,,,,,,,,@=====@
@+++++@,,,,,,,,,,,,,,,,,,@=====@
@+++++@,,,,,,,,,,,,,,,,,,@=====@
@+++++@,,,,,,,,,,,,,,,,,,@=====@
@+++++@,,,,,,,,,,,,,,,,,,@=====@
@+++++@,,,,,,,,,,,,,,,,,,@=====@
@+++++@,,,,,,,,,,,,,,,,,,@=====@
@+++++@,,,,,,,,,,,,,,,,,,@=====@
@+++++@,,,,,,,,,,,,,,,,,,@=====@
@+++++@,,,,,,,,,,,,,,,,,,@=====@
@+++++@,,,,,,,,,,,,,,,,,,@=====@
@+++++@@@@@@@@@@@@@@@@@@@@=====@
@++++@@==================@@====@
@+++@@====================@@===@
@++@@======================@@==@
@+@@========================@@=@
@@@==========================@@@
@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@")

;;; Floors

(defvar ducpel-glyphs-floor-simple-xpm-data "\
################################
################################
################################
################################
####@@@@@@@@@@@@@@@@@@@@@@@@@@@@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@@@@@@@@@@@@@@@@@@@@@@@@@@@@")

(defvar ducpel-glyphs-floor-exit-xpm-data "\
################################
################################
################################
################################
####@@@@@@@@@@@@@@@@@@@@@@@@@@@@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@(((((((((@@@@@@@@(((((((((@
####@((((((((@@@@@@@@@@((((((((@
####@(((((((@@@@@@@@@@@@(((((((@
####@((((((@@@@@@@@@@@@@@((((((@
####@((((((@@@@@@@@@@@@@@((((((@
####@((((((@@@@@@@@@@@@@@((((((@
####@((((((@@@@@@@@@@@@@@((((((@
####@((((((@@@@@@@@@@@@@@((((((@
####@((((((@@@@@@@@@@@@@@((((((@
####@((((((@@@@@@@@@@@@@@((((((@
####@((((((@@@@@@@@@@@@@@((((((@
####@((((((@@@@@@@@@@@@@@((((((@
####@((((((@@@@@@@@@@@@@@((((((@
####@((((((@@@@@@@@@@@@@@((((((@
####@((((((@@@@@@@@@@@@@@((((((@
####@((((((@@@@@@@@@@@@@@((((((@
####@((((((@@@@@@@@@@@@@@((((((@
####@((((((@@@@@@@@@@@@@@((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@@@@@@@@@@@@@@@@@@@@@@@@@@@@")

(defvar ducpel-glyphs-floor-teleport-xpm-data "\
################################
################################
################################
################################
####@@@@@@@@@@@@@@@@@@@@@@@@@@@@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((($$$$$$$$$$$$$$$$$$$$(((@
####@((($(((((((((((((((((($(((@
####@((($(((((((((((((((((($(((@
####@((($(((((((((((((((((($(((@
####@((($((($$$$$$$$$$$$((($(((@
####@((($((($(((((((((($((($(((@
####@((($((($(((((((((($((($(((@
####@((($((($(((((((((($((($(((@
####@((($((($((($$$$((($((($(((@
####@((($((($((($(($((($((($(((@
####@((($((($((($(($((($((($(((@
####@((($((($((($$$$((($((($(((@
####@((($((($(((((((((($((($(((@
####@((($((($(((((((((($((($(((@
####@((($((($(((((((((($((($(((@
####@((($((($$$$$$$$$$$$((($(((@
####@((($(((((((((((((((((($(((@
####@((($(((((((((((((((((($(((@
####@((($(((((((((((((((((($(((@
####@((($$$$$$$$$$$$$$$$$$$$(((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@@@@@@@@@@@@@@@@@@@@@@@@@@@@")

(defvar ducpel-glyphs-floor-left-xpm-data "\
################################
################################
################################
################################
####@@@@@@@@@@@@@@@@@@@@@@@@@@@@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@(((((~~(((((((((((((((((((@
####@((((~~~(((((((((((((((((((@
####@((~~~~~~~~~~~~((((((((((((@
####@((~~~~~~~~~~~~((((((((((((@
####@((((~~~(((((((((((((((((((@
####@(((((~~(((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@@@@@@@@@@@@@@@@@@@@@@@@@@@@")

(defvar ducpel-glyphs-floor-right-xpm-data "\
################################
################################
################################
################################
####@@@@@@@@@@@@@@@@@@@@@@@@@@@@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@(((((((((((((((((((~~(((((@
####@(((((((((((((((((((~~~((((@
####@((((((((((((~~~~~~~~~~~~((@
####@((((((((((((~~~~~~~~~~~~((@
####@(((((((((((((((((((~~~((((@
####@(((((((((((((((((((~~(((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@@@@@@@@@@@@@@@@@@@@@@@@@@@@")

(defvar ducpel-glyphs-floor-up-xpm-data "\
################################
################################
################################
################################
####@@@@@@@@@@@@@@@@@@@@@@@@@@@@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((~~((((((((((((@
####@((((((((((((~~((((((((((((@
####@(((((((((((~~~~(((((((((((@
####@((((((((((~~~~~~((((((((((@
####@((((((((((~~~~~~((((((((((@
####@((((((((((((~~((((((((((((@
####@((((((((((((~~((((((((((((@
####@((((((((((((~~((((((((((((@
####@((((((((((((~~((((((((((((@
####@((((((((((((~~((((((((((((@
####@((((((((((((~~((((((((((((@
####@((((((((((((~~((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@@@@@@@@@@@@@@@@@@@@@@@@@@@@")

(defvar ducpel-glyphs-floor-down-xpm-data "\
################################
################################
################################
################################
####@@@@@@@@@@@@@@@@@@@@@@@@@@@@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((~~((((((((((((@
####@((((((((((((~~((((((((((((@
####@((((((((((((~~((((((((((((@
####@((((((((((((~~((((((((((((@
####@((((((((((((~~((((((((((((@
####@((((((((((((~~((((((((((((@
####@((((((((((((~~((((((((((((@
####@((((((((((~~~~~~((((((((((@
####@((((((((((~~~~~~((((((((((@
####@(((((((((((~~~~(((((((((((@
####@((((((((((((~~((((((((((((@
####@((((((((((((~~((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@@@@@@@@@@@@@@@@@@@@@@@@@@@@")

(defvar ducpel-glyphs-floor-horizontal-xpm-data "\
################################
################################
################################
################################
####@@@@@@@@@@@@@@@@@@@@@@@@@@@@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@(((((~~((((((((((((~~(((((@
####@((((~~~((((((((((((~~~((((@
####@((~~~~~~~~~~~~~~~~~~~~~~((@
####@((~~~~~~~~~~~~~~~~~~~~~~((@
####@((((~~~((((((((((((~~~((((@
####@(((((~~((((((((((((~~(((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@@@@@@@@@@@@@@@@@@@@@@@@@@@@")

(defvar ducpel-glyphs-floor-vertical-xpm-data "\
################################
################################
################################
################################
####@@@@@@@@@@@@@@@@@@@@@@@@@@@@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((~~((((((((((((@
####@((((((((((((~~((((((((((((@
####@(((((((((((~~~~(((((((((((@
####@((((((((((~~~~~~((((((((((@
####@((((((((((~~~~~~((((((((((@
####@((((((((((((~~((((((((((((@
####@((((((((((((~~((((((((((((@
####@((((((((((((~~((((((((((((@
####@((((((((((((~~((((((((((((@
####@((((((((((((~~((((((((((((@
####@((((((((((((~~((((((((((((@
####@((((((((((((~~((((((((((((@
####@((((((((((((~~((((((((((((@
####@((((((((((((~~((((((((((((@
####@((((((((((((~~((((((((((((@
####@((((((((((((~~((((((((((((@
####@((((((((((((~~((((((((((((@
####@((((((((((~~~~~~((((((((((@
####@((((((((((~~~~~~((((((((((@
####@(((((((((((~~~~(((((((((((@
####@((((((((((((~~((((((((((((@
####@((((((((((((~~((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@@@@@@@@@@@@@@@@@@@@@@@@@@@@")

(defvar ducpel-glyphs-floor-multi-xpm-data "\
################################
################################
################################
################################
####@@@@@@@@@@@@@@@@@@@@@@@@@@@@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((~~((((((((((((@
####@((((((((((((~~((((((((((((@
####@(((((((((((~~~~(((((((((((@
####@((((((((((~~~~~~((((((((((@
####@((((((((((~~~~~~((((((((((@
####@((((((((((((~~((((((((((((@
####@((((((((((((~~((((((((((((@
####@((((((((((((~~((((((((((((@
####@(((((~~(((((~~(((((~~(((((@
####@((((~~~(((((~~(((((~~~((((@
####@((~~~~~~~~~~~~~~~~~~~~~~((@
####@((~~~~~~~~~~~~~~~~~~~~~~((@
####@((((~~~(((((~~(((((~~~((((@
####@(((((~~(((((~~(((((~~(((((@
####@((((((((((((~~((((((((((((@
####@((((((((((((~~((((((((((((@
####@((((((((((((~~((((((((((((@
####@((((((((((~~~~~~((((((((((@
####@((((((((((~~~~~~((((((((((@
####@(((((((((((~~~~(((((((((((@
####@((((((((((((~~((((((((((((@
####@((((((((((((~~((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@@@@@@@@@@@@@@@@@@@@@@@@@@@@")

;;; Men

(defvar ducpel-glyphs-man-simple-xpm-data "\
################################
################################
################################
################################
####@@@@@@@@@@@@@@@@@@@@@@@@@@@@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((@@@@@@((((((((((@
####@(((((((((@@____@@(((((((((@
####@((((((((@@______@@((((((((@
####@((((((((@________@((((((((@
####@((((((((@________@((((((((@
####@((((((((@________@((((((((@
####@((((((((@________@((((((((@
####@((((((((@@______@@((((((((@
####@(((((((((@@____@@(((((((((@
####@((((((((((@@@@@@((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@@@@@@@@@@@@@@@@@@@@@@@@@@@@")

(defvar ducpel-glyphs-man-exit-xpm-data "\
################################
################################
################################
################################
####@@@@@@@@@@@@@@@@@@@@@@@@@@@@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@(((((((((@@@@@@@@(((((((((@
####@((((((((@@@@@@@@@@((((((((@
####@(((((((@@@@@@@@@@@@(((((((@
####@((((((@@@@@@@@@@@@@@((((((@
####@((((((@@@@@@@@@@@@@@((((((@
####@((((((@@@@@____@@@@@((((((@
####@((((((@@@@______@@@@((((((@
####@((((((@@@________@@@((((((@
####@((((((@@@________@@@((((((@
####@((((((@@@________@@@((((((@
####@((((((@@@________@@@((((((@
####@((((((@@@@______@@@@((((((@
####@((((((@@@@@____@@@@@((((((@
####@((((((@@@@@@@@@@@@@@((((((@
####@((((((@@@@@@@@@@@@@@((((((@
####@((((((@@@@@@@@@@@@@@((((((@
####@((((((@@@@@@@@@@@@@@((((((@
####@((((((@@@@@@@@@@@@@@((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@@@@@@@@@@@@@@@@@@@@@@@@@@@@")

(defvar ducpel-glyphs-man-teleport-xpm-data "\
################################
################################
################################
################################
####@@@@@@@@@@@@@@@@@@@@@@@@@@@@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((($$$$$$$$$$$$$$$$$$$$(((@
####@((($(((((((((((((((((($(((@
####@((($(((((((((((((((((($(((@
####@((($(((((((((((((((((($(((@
####@((($((($$$$$$$$$$$$((($(((@
####@((($((($((@@@@@@(($((($(((@
####@((($((($(@@____@@($((($(((@
####@((($((($@@______@@$((($(((@
####@((($((($@________@$((($(((@
####@((($((($@________@$((($(((@
####@((($((($@________@$((($(((@
####@((($((($@________@$((($(((@
####@((($((($@@______@@$((($(((@
####@((($((($(@@____@@($((($(((@
####@((($((($((@@@@@@(($((($(((@
####@((($((($$$$$$$$$$$$((($(((@
####@((($(((((((((((((((((($(((@
####@((($(((((((((((((((((($(((@
####@((($(((((((((((((((((($(((@
####@((($$$$$$$$$$$$$$$$$$$$(((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@@@@@@@@@@@@@@@@@@@@@@@@@@@@")

(defvar ducpel-glyphs-man-left-xpm-data "\
################################
################################
################################
################################
####@@@@@@@@@@@@@@@@@@@@@@@@@@@@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((@@@@@@((((((((((@
####@(((((((((@@____@@(((((((((@
####@(((((~~(@@______@@((((((((@
####@((((~~~(@________@((((((((@
####@((~~~~~~@________@((((((((@
####@((~~~~~~@________@((((((((@
####@((((~~~(@________@((((((((@
####@(((((~~(@@______@@((((((((@
####@(((((((((@@____@@(((((((((@
####@((((((((((@@@@@@((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@@@@@@@@@@@@@@@@@@@@@@@@@@@@")

(defvar ducpel-glyphs-man-right-xpm-data "\
################################
################################
################################
################################
####@@@@@@@@@@@@@@@@@@@@@@@@@@@@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((@@@@@@((((((((((@
####@(((((((((@@____@@(((((((((@
####@((((((((@@______@@(~~(((((@
####@((((((((@________@(~~~((((@
####@((((((((@________@~~~~~~((@
####@((((((((@________@~~~~~~((@
####@((((((((@________@(~~~((((@
####@((((((((@@______@@(~~(((((@
####@(((((((((@@____@@(((((((((@
####@((((((((((@@@@@@((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@@@@@@@@@@@@@@@@@@@@@@@@@@@@")

(defvar ducpel-glyphs-man-up-xpm-data "\
################################
################################
################################
################################
####@@@@@@@@@@@@@@@@@@@@@@@@@@@@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((~~((((((((((((@
####@((((((((((((~~((((((((((((@
####@(((((((((((~~~~(((((((((((@
####@((((((((((~~~~~~((((((((((@
####@((((((((((~~~~~~((((((((((@
####@((((((((((((~~((((((((((((@
####@((((((((((@@@@@@((((((((((@
####@(((((((((@@____@@(((((((((@
####@((((((((@@______@@((((((((@
####@((((((((@________@((((((((@
####@((((((((@________@((((((((@
####@((((((((@________@((((((((@
####@((((((((@________@((((((((@
####@((((((((@@______@@((((((((@
####@(((((((((@@____@@(((((((((@
####@((((((((((@@@@@@((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@@@@@@@@@@@@@@@@@@@@@@@@@@@@")

(defvar ducpel-glyphs-man-down-xpm-data "\
################################
################################
################################
################################
####@@@@@@@@@@@@@@@@@@@@@@@@@@@@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((@@@@@@((((((((((@
####@(((((((((@@____@@(((((((((@
####@((((((((@@______@@((((((((@
####@((((((((@________@((((((((@
####@((((((((@________@((((((((@
####@((((((((@________@((((((((@
####@((((((((@________@((((((((@
####@((((((((@@______@@((((((((@
####@(((((((((@@____@@(((((((((@
####@((((((((((@@@@@@((((((((((@
####@((((((((((((~~((((((((((((@
####@((((((((((~~~~~~((((((((((@
####@((((((((((~~~~~~((((((((((@
####@(((((((((((~~~~(((((((((((@
####@((((((((((((~~((((((((((((@
####@((((((((((((~~((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@@@@@@@@@@@@@@@@@@@@@@@@@@@@")

(defvar ducpel-glyphs-man-horizontal-xpm-data "\
################################
################################
################################
################################
####@@@@@@@@@@@@@@@@@@@@@@@@@@@@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((@@@@@@((((((((((@
####@(((((((((@@____@@(((((((((@
####@(((((~~(@@______@@(~~(((((@
####@((((~~~(@________@(~~~((((@
####@((~~~~~~@________@~~~~~~((@
####@((~~~~~~@________@~~~~~~((@
####@((((~~~(@________@(~~~((((@
####@(((((~~(@@______@@(~~(((((@
####@(((((((((@@____@@(((((((((@
####@((((((((((@@@@@@((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@@@@@@@@@@@@@@@@@@@@@@@@@@@@")

(defvar ducpel-glyphs-man-vertical-xpm-data "\
################################
################################
################################
################################
####@@@@@@@@@@@@@@@@@@@@@@@@@@@@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((~~((((((((((((@
####@((((((((((((~~((((((((((((@
####@(((((((((((~~~~(((((((((((@
####@((((((((((~~~~~~((((((((((@
####@((((((((((~~~~~~((((((((((@
####@((((((((((((~~((((((((((((@
####@((((((((((@@@@@@((((((((((@
####@(((((((((@@____@@(((((((((@
####@((((((((@@______@@((((((((@
####@((((((((@________@((((((((@
####@((((((((@________@((((((((@
####@((((((((@________@((((((((@
####@((((((((@________@((((((((@
####@((((((((@@______@@((((((((@
####@(((((((((@@____@@(((((((((@
####@((((((((((@@@@@@((((((((((@
####@((((((((((((~~((((((((((((@
####@((((((((((~~~~~~((((((((((@
####@((((((((((~~~~~~((((((((((@
####@(((((((((((~~~~(((((((((((@
####@((((((((((((~~((((((((((((@
####@((((((((((((~~((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@@@@@@@@@@@@@@@@@@@@@@@@@@@@")

(defvar ducpel-glyphs-man-multi-xpm-data "\
################################
################################
################################
################################
####@@@@@@@@@@@@@@@@@@@@@@@@@@@@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((~~((((((((((((@
####@((((((((((((~~((((((((((((@
####@(((((((((((~~~~(((((((((((@
####@((((((((((~~~~~~((((((((((@
####@((((((((((~~~~~~((((((((((@
####@((((((((((((~~((((((((((((@
####@((((((((((@@@@@@((((((((((@
####@(((((((((@@____@@(((((((((@
####@(((((~~(@@______@@(~~(((((@
####@((((~~~(@________@(~~~((((@
####@((~~~~~~@________@~~~~~~((@
####@((~~~~~~@________@~~~~~~((@
####@((((~~~(@________@(~~~((((@
####@(((((~~(@@______@@(~~(((((@
####@(((((((((@@____@@(((((((((@
####@((((((((((@@@@@@((((((((((@
####@((((((((((((~~((((((((((((@
####@((((((((((~~~~~~((((((((((@
####@((((((((((~~~~~~((((((((((@
####@(((((((((((~~~~(((((((((((@
####@((((((((((((~~((((((((((((@
####@((((((((((((~~((((((((((((@
####@((((((((((((((((((((((((((@
####@((((((((((((((((((((((((((@
####@@@@@@@@@@@@@@@@@@@@@@@@@@@@")

;;; Boxes

(defvar ducpel-glyphs-box-simple-xpm-data "\
@@@@@@@@@@@@@@@@@@@@@@@@@@@@@###
@))))))))))))))))))))))))))@@@##
@))))))))))))))))))))))))))@/@@#
@))))))))))))))))))))))))))@//@@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@@@@@@@@@@@@@@@@@@@@@@@@@@@@@//@
@@//////////////////////////@@/@
#@@//////////////////////////@@@
##@@//////////////////////////@@
###@@@@@@@@@@@@@@@@@@@@@@@@@@@@@")

(defvar ducpel-glyphs-box-exit-xpm-data "\
@@@@@@@@@@@@@@@@@@@@@@@@@@@@@###
@))))))))))))))))))))))))))@@@##
@))))))))))))))))))))))))))@/@@#
@))))))))))))))))))))))))))@//@@
@))))))))))))))))))))))))))@///@
@)))))))))@@@@@@@@)))))))))@///@
@))))))))@@@@@@@@@@))))))))@///@
@)))))))@@@@@@@@@@@@)))))))@///@
@))))))@@@@@@@@@@@@@@))))))@///@
@))))))@@@@@@@@@@@@@@))))))@///@
@))))))@@@@@@@@@@@@@@))))))@///@
@))))))@@@@@@@@@@@@@@))))))@///@
@))))))@@@@@@@@@@@@@@))))))@///@
@))))))@@@@@@@@@@@@@@))))))@///@
@))))))@@@@@@@@@@@@@@))))))@///@
@))))))@@@@@@@@@@@@@@))))))@///@
@))))))@@@@@@@@@@@@@@))))))@///@
@))))))@@@@@@@@@@@@@@))))))@///@
@))))))@@@@@@@@@@@@@@))))))@///@
@))))))@@@@@@@@@@@@@@))))))@///@
@))))))@@@@@@@@@@@@@@))))))@///@
@))))))@@@@@@@@@@@@@@))))))@///@
@))))))@@@@@@@@@@@@@@))))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@@@@@@@@@@@@@@@@@@@@@@@@@@@@@//@
@@//////////////////////////@@/@
#@@//////////////////////////@@@
##@@//////////////////////////@@
###@@@@@@@@@@@@@@@@@@@@@@@@@@@@@")

(defvar ducpel-glyphs-box-teleport-xpm-data "\
@@@@@@@@@@@@@@@@@@@@@@@@@@@@@###
@))))))))))))))))))))))))))@@@##
@))))))))))))))))))))))))))@/@@#
@))))))))))))))))))))))))))@//@@
@)))$$$$$$$$$$$$$$$$$$$$)))@///@
@)))$))))))))))))))))))$)))@///@
@)))$))))))))))))))))))$)))@///@
@)))$))))))))))))))))))$)))@///@
@)))$)))$$$$$$$$$$$$)))$)))@///@
@)))$)))$))))))))))$)))$)))@///@
@)))$)))$))))))))))$)))$)))@///@
@)))$)))$))))))))))$)))$)))@///@
@)))$)))$)))$$$$)))$)))$)))@///@
@)))$)))$)))$))$)))$)))$)))@///@
@)))$)))$)))$))$)))$)))$)))@///@
@)))$)))$)))$$$$)))$)))$)))@///@
@)))$)))$))))))))))$)))$)))@///@
@)))$)))$))))))))))$)))$)))@///@
@)))$)))$))))))))))$)))$)))@///@
@)))$)))$$$$$$$$$$$$)))$)))@///@
@)))$))))))))))))))))))$)))@///@
@)))$))))))))))))))))))$)))@///@
@)))$))))))))))))))))))$)))@///@
@)))$$$$$$$$$$$$$$$$$$$$)))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@@@@@@@@@@@@@@@@@@@@@@@@@@@@@//@
@@//////////////////////////@@/@
#@@//////////////////////////@@@
##@@//////////////////////////@@
###@@@@@@@@@@@@@@@@@@@@@@@@@@@@@")

(defvar ducpel-glyphs-box-left-xpm-data "\
@@@@@@@@@@@@@@@@@@@@@@@@@@@@@###
@))))))))))))))))))))))))))@@@##
@))))))))))))))))))))))))))@/@@#
@))))))))))))))))))))))))))@//@@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@)))))--)))))))))))))))))))@///@
@))))---)))))))))))))))))))@///@
@))------------))))))))))))@///@
@))------------))))))))))))@///@
@))))---)))))))))))))))))))@///@
@)))))--)))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@@@@@@@@@@@@@@@@@@@@@@@@@@@@@//@
@@//////////////////////////@@/@
#@@//////////////////////////@@@
##@@//////////////////////////@@
###@@@@@@@@@@@@@@@@@@@@@@@@@@@@@")

(defvar ducpel-glyphs-box-right-xpm-data "\
@@@@@@@@@@@@@@@@@@@@@@@@@@@@@###
@))))))))))))))))))))))))))@@@##
@))))))))))))))))))))))))))@/@@#
@))))))))))))))))))))))))))@//@@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@)))))))))))))))))))--)))))@///@
@)))))))))))))))))))---))))@///@
@))))))))))))------------))@///@
@))))))))))))------------))@///@
@)))))))))))))))))))---))))@///@
@)))))))))))))))))))--)))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@@@@@@@@@@@@@@@@@@@@@@@@@@@@@//@
@@//////////////////////////@@/@
#@@//////////////////////////@@@
##@@//////////////////////////@@
###@@@@@@@@@@@@@@@@@@@@@@@@@@@@@")

(defvar ducpel-glyphs-box-up-xpm-data "\
@@@@@@@@@@@@@@@@@@@@@@@@@@@@@###
@))))))))))))))))))))))))))@@@##
@))))))))))))))))))))))))))@/@@#
@))))))))))))--))))))))))))@//@@
@))))))))))))--))))))))))))@///@
@)))))))))))----)))))))))))@///@
@))))))))))------))))))))))@///@
@))))))))))------))))))))))@///@
@))))))))))))--))))))))))))@///@
@))))))))))))--))))))))))))@///@
@))))))))))))--))))))))))))@///@
@))))))))))))--))))))))))))@///@
@))))))))))))--))))))))))))@///@
@))))))))))))--))))))))))))@///@
@))))))))))))--))))))))))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@@@@@@@@@@@@@@@@@@@@@@@@@@@@@//@
@@//////////////////////////@@/@
#@@//////////////////////////@@@
##@@//////////////////////////@@
###@@@@@@@@@@@@@@@@@@@@@@@@@@@@@")

(defvar ducpel-glyphs-box-down-xpm-data "\
@@@@@@@@@@@@@@@@@@@@@@@@@@@@@###
@))))))))))))))))))))))))))@@@##
@))))))))))))))))))))))))))@/@@#
@))))))))))))))))))))))))))@//@@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))--))))))))))))@///@
@))))))))))))--))))))))))))@///@
@))))))))))))--))))))))))))@///@
@))))))))))))--))))))))))))@///@
@))))))))))))--))))))))))))@///@
@))))))))))))--))))))))))))@///@
@))))))))))))--))))))))))))@///@
@))))))))))------))))))))))@///@
@))))))))))------))))))))))@///@
@)))))))))))----)))))))))))@///@
@))))))))))))--))))))))))))@///@
@))))))))))))--))))))))))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@@@@@@@@@@@@@@@@@@@@@@@@@@@@@//@
@@//////////////////////////@@/@
#@@//////////////////////////@@@
##@@//////////////////////////@@
###@@@@@@@@@@@@@@@@@@@@@@@@@@@@@")

(defvar ducpel-glyphs-box-horizontal-xpm-data "\
@@@@@@@@@@@@@@@@@@@@@@@@@@@@@###
@))))))))))))))))))))))))))@@@##
@))))))))))))))))))))))))))@/@@#
@))))))))))))))))))))))))))@//@@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@)))))--))))))))))))--)))))@///@
@))))---))))))))))))---))))@///@
@))----------------------))@///@
@))----------------------))@///@
@))))---))))))))))))---))))@///@
@)))))--))))))))))))--)))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@@@@@@@@@@@@@@@@@@@@@@@@@@@@@//@
@@//////////////////////////@@/@
#@@//////////////////////////@@@
##@@//////////////////////////@@
###@@@@@@@@@@@@@@@@@@@@@@@@@@@@@")

(defvar ducpel-glyphs-box-vertical-xpm-data "\
@@@@@@@@@@@@@@@@@@@@@@@@@@@@@###
@))))))))))))))))))))))))))@@@##
@))))))))))))))))))))))))))@/@@#
@))))))))))))--))))))))))))@//@@
@))))))))))))--))))))))))))@///@
@)))))))))))----)))))))))))@///@
@))))))))))------))))))))))@///@
@))))))))))------))))))))))@///@
@))))))))))))--))))))))))))@///@
@))))))))))))--))))))))))))@///@
@))))))))))))--))))))))))))@///@
@))))))))))))--))))))))))))@///@
@))))))))))))--))))))))))))@///@
@))))))))))))--))))))))))))@///@
@))))))))))))--))))))))))))@///@
@))))))))))))--))))))))))))@///@
@))))))))))))--))))))))))))@///@
@))))))))))))--))))))))))))@///@
@))))))))))))--))))))))))))@///@
@))))))))))))--))))))))))))@///@
@))))))))))------))))))))))@///@
@))))))))))------))))))))))@///@
@)))))))))))----)))))))))))@///@
@))))))))))))--))))))))))))@///@
@))))))))))))--))))))))))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@@@@@@@@@@@@@@@@@@@@@@@@@@@@@//@
@@//////////////////////////@@/@
#@@//////////////////////////@@@
##@@//////////////////////////@@
###@@@@@@@@@@@@@@@@@@@@@@@@@@@@@")

(defvar ducpel-glyphs-box-multi-xpm-data "\
@@@@@@@@@@@@@@@@@@@@@@@@@@@@@###
@))))))))))))))))))))))))))@@@##
@))))))))))))))))))))))))))@/@@#
@))))))))))))--))))))))))))@//@@
@))))))))))))--))))))))))))@///@
@)))))))))))----)))))))))))@///@
@))))))))))------))))))))))@///@
@))))))))))------))))))))))@///@
@))))))))))))--))))))))))))@///@
@))))))))))))--))))))))))))@///@
@))))))))))))--))))))))))))@///@
@)))))--)))))--)))))--)))))@///@
@))))---)))))--)))))---))))@///@
@))----------------------))@///@
@))----------------------))@///@
@))))---)))))--)))))---))))@///@
@)))))--)))))--)))))--)))))@///@
@))))))))))))--))))))))))))@///@
@))))))))))))--))))))))))))@///@
@))))))))))))--))))))))))))@///@
@))))))))))------))))))))))@///@
@))))))))))------))))))))))@///@
@)))))))))))----)))))))))))@///@
@))))))))))))--))))))))))))@///@
@))))))))))))--))))))))))))@///@
@))))))))))))))))))))))))))@///@
@))))))))))))))))))))))))))@///@
@@@@@@@@@@@@@@@@@@@@@@@@@@@@@//@
@@//////////////////////////@@/@
#@@//////////////////////////@@@
##@@//////////////////////////@@
###@@@@@@@@@@@@@@@@@@@@@@@@@@@@@")


;;; XPM specifications

;; Examples to test the images:
;;
;; (insert-image (find-image (ducpel-glyphs-get-image-spec-by-plist :type ducpel-impassable)))
;; (insert-image (find-image (ducpel-glyphs-get-image-spec-by-plist :type ducpel-box :box ducpel-horizontal)))

(defun ducpel-glyphs-get-image-data-by-plist (&rest plist)
  "Return image data by cell PLIST."
  (let ((type (plist-get plist :type)))
    (cond
     ((eql type ducpel-empty)      ducpel-glyphs-empty-xpm-data)
     ((eql type ducpel-wall)       ducpel-glyphs-wall-xpm-data)
     ((eql type ducpel-impassable) ducpel-glyphs-wall-xpm-data)
     ((eql type ducpel-floor)
      (let ((floor (plist-get plist :floor)))
        (cond
         ((eql floor ducpel-simple)     ducpel-glyphs-floor-simple-xpm-data)
         ((eql floor ducpel-exit)       ducpel-glyphs-floor-exit-xpm-data)
         ((eql floor ducpel-teleport)   ducpel-glyphs-floor-teleport-xpm-data)
         ((eql floor ducpel-left)       ducpel-glyphs-floor-left-xpm-data)
         ((eql floor ducpel-right)      ducpel-glyphs-floor-right-xpm-data)
         ((eql floor ducpel-up)         ducpel-glyphs-floor-up-xpm-data)
         ((eql floor ducpel-down)       ducpel-glyphs-floor-down-xpm-data)
         ((eql floor ducpel-horizontal) ducpel-glyphs-floor-horizontal-xpm-data)
         ((eql floor ducpel-vertical)   ducpel-glyphs-floor-vertical-xpm-data)
         ((eql floor ducpel-multi)      ducpel-glyphs-floor-multi-xpm-data))))
     ((or (eql type ducpel-man)
          (eql type ducpel-active-man))
      (let ((floor (plist-get plist :floor)))
        (cond
         ((eql floor ducpel-simple)     ducpel-glyphs-man-simple-xpm-data)
         ((eql floor ducpel-exit)       ducpel-glyphs-man-exit-xpm-data)
         ((eql floor ducpel-teleport)   ducpel-glyphs-man-teleport-xpm-data)
         ((eql floor ducpel-left)       ducpel-glyphs-man-left-xpm-data)
         ((eql floor ducpel-right)      ducpel-glyphs-man-right-xpm-data)
         ((eql floor ducpel-up)         ducpel-glyphs-man-up-xpm-data)
         ((eql floor ducpel-down)       ducpel-glyphs-man-down-xpm-data)
         ((eql floor ducpel-horizontal) ducpel-glyphs-man-horizontal-xpm-data)
         ((eql floor ducpel-vertical)   ducpel-glyphs-man-vertical-xpm-data)
         ((eql floor ducpel-multi)      ducpel-glyphs-man-multi-xpm-data))))
     ((eql type ducpel-box)
      (let ((box (plist-get plist :box)))
        (cond
         ((eql box ducpel-simple)     ducpel-glyphs-box-simple-xpm-data)
         ((eql box ducpel-exit)       ducpel-glyphs-box-exit-xpm-data)
         ((eql box ducpel-teleport)   ducpel-glyphs-box-teleport-xpm-data)
         ((eql box ducpel-left)       ducpel-glyphs-box-left-xpm-data)
         ((eql box ducpel-right)      ducpel-glyphs-box-right-xpm-data)
         ((eql box ducpel-up)         ducpel-glyphs-box-up-xpm-data)
         ((eql box ducpel-down)       ducpel-glyphs-box-down-xpm-data)
         ((eql box ducpel-horizontal) ducpel-glyphs-box-horizontal-xpm-data)
         ((eql box ducpel-vertical)   ducpel-glyphs-box-vertical-xpm-data)
         ((eql box ducpel-multi)      ducpel-glyphs-box-multi-xpm-data)))))))

(defun ducpel-glyphs-get-image-params (data)
  "Return a list of image parameters.

DATA should be a string with image data in XPM2 format without
headers, with one character per pixel, and with a constant
width (all lines should have the same amount of characters).

Returning value is a list (WIDTH HEIGHT UNIQUE-CHARS).
WIDTH x HEIGHT is a size of the image.
UNIQUE-CHARS is a list of all characters the DATA contains."
  (let* ((lines (split-string data "\n" t))
         (height (length lines))
         width unique-chars)
    (mapc (lambda (line)
            (let ((len (length line)))
              (or width (setq width len))
              (dotimes (i len)
                (let ((char (aref line i)))
                  (or (memq char unique-chars)
                      (push char unique-chars))))))
          lines)
    (list width height unique-chars)))

(defun ducpel-glyphs-get-image-spec (data colors width height)
  "Return a list of image specifications suitable for `find-image'.
For the meaning of DATA, see `ducpel-glyphs-get-image-params'.
COLORS is alist of characters used in DATA and color values.
WIDTH x HEIGHT is a size of the image."
  (let* ((vals   (format "\"%d %d %d 1\",\n"
                         width height (length colors)))
         (colors (mapconcat (lambda (assoc)
                              (format "\"%c c %s\",\n"
                                      (car assoc) (cdr assoc)))
                            colors
                            ""))
         (data   (mapconcat (lambda (line)
                              (concat "\"" line "\""))
                            (split-string data "\n")
                            ",\n"))
         (spec   (concat "/* XPM */\nstatic char *image[] = {\n"
                         vals colors data "};")))
    `((:type xpm :ascent center :data ,spec))))

(defun ducpel-glyphs-get-image-spec-by-plist (&rest plist)
  "Return image specifications of a ducpel glyph by cell PLIST.
Returning value is defined by applying
`ducpel-glyphs-get-image-spec' to the image data from
`ducpel-glyphs-...-xpm-data'."
  (let ((data (apply #'ducpel-glyphs-get-image-data-by-plist plist)))
    (cl-multiple-value-bind (width height unique-chars)
        (ducpel-glyphs-get-image-params data)
      (ducpel-glyphs-get-image-spec
       data
       (ducpel-glyphs-get-colors unique-chars
                                 (plist-get plist :type))
       width height))))

(defun ducpel-glyphs-get-gamegrid-spec (image-spec char)
  "Return a proper gamegrid specification by IMAGE-SPEC and CHAR.

Returning value is a list intended to be an element of
`gamegrid-display-options' array.  The list has a form:

  (GLYPH-SPEC FACE-SPEC COLOR-SPEC)."
  ;; Currently we use images if `gamegrid-display-mode' is `glyph',
  ;; characters otherwise (see `gamegrid-display-type') and we do not
  ;; bother about face and color specifications (see
  ;; `gamegrid-initialize-display').
  `(((glyph ,image-spec)
     (t ,char))
    nil nil))

(defun ducpel-glyphs-get-gamegrid-spec-by-plist (&rest plist)
  "Return a proper gamegrid specification by cell PLIST."
  (ducpel-glyphs-get-gamegrid-spec
   (apply #'ducpel-glyphs-get-image-spec-by-plist plist)
   (apply #'ducpel-glyphs-get-char-by-plist plist)))

(defun ducpel-glyphs-default ()
  "Return alist of default glyph specifications.
Car of each assoc is a unique cell property list.
Cdr is a specification of a gamegrid glyph."
  (let (alist)
    (cl-flet ((push-spec (&rest plist)
                (push (cons plist
                            (apply #'ducpel-glyphs-get-gamegrid-spec-by-plist
                                   plist))
                      alist)))
      (dolist (type ducpel-cell-types)
        (cond
         ((memql type (list ducpel-empty ducpel-wall ducpel-impassable))
          (push-spec :type type))
         ((memql type (list ducpel-floor ducpel-man ducpel-active-man))
          (dolist (floor ducpel-floor-types)
            (push-spec :type type :floor floor)))
         ((eql type ducpel-box)
          (dolist (box ducpel-floor-types)
            (dolist (floor ducpel-floor-types)
              (push-spec :type type :floor floor :box box))))
         (t (message "Warning: Unknown ducpel cell type %s." type)))))
    alist))

(provide 'ducpel-glyphs)

;;; ducpel-glyphs.el ends here
