;;; ecb-face.el --- all face-options of ECB

;; Copyright (C) 2000 - 2005 Jesper Nordenberg,
;;                           Klaus Berndl,
;;                           Kevin A. Burton,
;;                           Free Software Foundation, Inc.

;; Author: Klaus Berndl <klaus.berndl@sdm.de>
;;         Kevin A. Burton <burton@openprivacy.org>
;; Maintainer: Klaus Berndl <klaus.berndl@sdm.de>
;; Keywords: browser, code, programming, tools
;; Created: 2001

;; This program is free software; you can redistribute it and/or modify it under
;; the terms of the GNU General Public License as published by the Free Software
;; Foundation; either version 2, or (at your option) any later version.

;; This program is distributed in the hope that it will be useful, but WITHOUT
;; ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
;; FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
;; details.

;; You should have received a copy of the GNU General Public License along with
;; GNU Emacs; see the file COPYING.  If not, write to the Free Software
;; Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

;; $Id: ecb-face.el,v 1.26 2009/06/04 08:38:15 berndl Exp $

;;; Commentary:

;; This file contains all options with type 'face and all face-definitions of
;; ECB.

;;; History
;;
;; For the ChangeLog of this file see the CVS-repository. For a complete
;; history of the ECB-package see the file NEWS.

;;; Code

(eval-when-compile
  (require 'silentcomp))

(silentcomp-defun set-face-parent)
(silentcomp-defun make-face-bold)
(silentcomp-defun make-face)
(silentcomp-defun set-face-foreground)

(defgroup ecb-face-options nil
  "Settings for all faces used in ECB."
  :group 'ecb
  :prefix "ecb-")

(defgroup ecb-faces nil
  "Definitions of all ECB-faces"
  :group 'ecb-face-options
  :group 'faces
  :prefix "ecb-")

(defmacro ecb-face-default (&optional height bold-p italic-p
                                      inherit
                                      fg-light-col fg-dark-col
                                      bg-light-col bg-dark-col
                                      fg-rest bg-rest
                                      reverse-video-p)
  "Macro for setting default values for an ECB face.
The parameters are set for the following display-types:
- ((class color) (background light)): HEIGHT, BOLD-P, ITALIC-P, INHERIT
                                      FG-LIGHT-COL, BG-LIGHT-COL
- ((class color) (background dark)): HEIGHT, BOLD-P, ITALIC-P, INHERIT
                                     FG-DARK-COL, BG-DARK-COL
- t: HEIGHT, BOLD-P, ITALIC-P, INHERIT, FG-REST, BG-REST, REVERSE-VIDEO."
  `(list (list '((class color) (background light))
               (append (if ,height (list :height ,height))
                       (if ,bold-p (list :weight 'bold))
                       (if ,italic-p (list :slant 'italic))
                       (if ,inherit (list :inherit ,inherit))
                       (if ,fg-light-col (list :foreground ,fg-light-col))
                       (if ,bg-light-col (list :background ,bg-light-col))))
         (list '((class color) (background dark))
               (append (if ,height (list :height ,height))
                       (if ,bold-p (list :weight 'bold))
                       (if ,italic-p (list :slant 'italic))
                       (if ,inherit (list :inherit ,inherit))
                       (if ,fg-dark-col (list :foreground ,fg-dark-col))
                       (if ,bg-dark-col (list :background ,bg-dark-col))))
         (list 't (append (if ,height (list :height ,height))
                          (if ,bold-p (list :weight 'bold))
                          (if ,italic-p (list :slant 'italic))
                          (if ,inherit (list :inherit ,inherit))
                          (if ,fg-rest (list :foreground ,fg-rest))
                          (if ,bg-rest (list :foreground ,bg-rest))
                          (if ,reverse-video-p (list :reverse-video t))))))

(defface ecb-default-general-face (ecb-face-default 1.0)
  "*Basic face for all ECB tree-buffers.
It's recommended to define here the font-family, the font-size, the basic
color etc.
In GNU Emacs all faces used in the ECB tree-buffers inherit from this face.
Therefore the default attributes like font etc. of a face used in a
tree-buffer can be very easily changed with face 'ecb-default-general-face'."
  :group 'ecb-faces)

(defface ecb-tree-guide-line-face (ecb-face-default 1.0 nil nil
                                                    'ecb-default-general-face
                                                    "grey" "gray")
  "*Face for the guide-lines in the tree-buffers."
  :group 'ecb-faces)

(defcustom ecb-tree-guide-line-face 'ecb-tree-guide-line-face
  "*Face for the guide-lines in the tree-buffers."
  :type 'face
  :group 'ecb-face-options)

(defface ecb-directories-general-face (ecb-face-default 1.0 nil nil
                                                        'ecb-default-general-face)
  "*Basic face for the ECB directories buffer.
It's recommended to define here the font-family, the font-size, the basic
color etc."
  :group 'ecb-faces)

(defcustom ecb-directories-general-face 'ecb-default-general-face
  "*Basic face for the ECB directories buffer.
This defines the basic face the whole directory buffer should displayed with.
If the face 'ecb-default-general-face' is used then the display of all
ECB-tree-buffers can be changed by modifying only the face
'ecb-default-general-face'.

Changes take first effect after finishing and reactivating ECB!"
  :group 'ecb-face-options
  :group 'ecb-directories
  :type '(radio (const :tag "Use ecb-default-general-face"
                       :value ecb-default-general-face)
                (face :tag "Special face"
                      :value ecb-directories-general-face)))

(defface ecb-sources-general-face (ecb-face-default 1.0 nil nil
                                                    'ecb-default-general-face)
  "*Basic face for the ECB sources buffer.
It's recommended to define here the font-family, the font-size, the basic
color etc."
  :group 'ecb-faces)

(defcustom ecb-sources-general-face 'ecb-default-general-face
  "*Basic face for the ECB sources buffer.
This defines the basic face the whole sources buffer should displayed with. If
the face 'ecb-default-general-face' is used then the display of all
ECB-tree-buffers can be changed by modifying only the face
'ecb-default-general-face'.

Changes take first effect after finishing and reactivating ECB!"
  :group 'ecb-face-options
  :group 'ecb-sources
  :type 'face
  :type '(radio (const :tag "Use ecb-default-general-face"
                       :value ecb-default-general-face)
                (face :tag "Special face"
                      :value ecb-sources-general-face)))

(defface ecb-methods-general-face (ecb-face-default 1.0 nil nil
                                                    'ecb-default-general-face)
  "*Basic face for the ECB methods buffer.
It's recommended to define here the font-family, the font-size, the basic
color etc."
  :group 'ecb-faces)

(defcustom ecb-methods-general-face 'ecb-default-general-face
  "*Basic face for the ECB methods buffer.
This defines the basic face the whole methods buffer should displayed with. If
the face 'ecb-default-general-face' is used then the display of all
ECB-tree-buffers can be changed by modifying only the face
'ecb-default-general-face'.

Changes take first effect after finishing and reactivating ECB!"
  :group 'ecb-face-options
  :group 'ecb-methods
  :type '(radio (const :tag "Use ecb-default-general-face"
                       :value ecb-default-general-face)
                (face :tag "Special face"
                      :value ecb-methods-general-face)))

(defface ecb-history-general-face (ecb-face-default 1.0 nil nil
                                                    'ecb-default-general-face)
  "*Basic face for the ECB history buffer.
It's recommended to define here the font-family, the font-size, the basic
color etc."
  :group 'ecb-faces)

(defcustom ecb-history-general-face 'ecb-default-general-face
  "*Basic face for the ECB history buffer.
This defines the basic face the whole history buffer should displayed with. If
the face 'ecb-default-general-face' is used then the display of all
ECB-tree-buffers can be changed by modifying only the face
'ecb-default-general-face'.

Changes take first effect after finishing and reactivating ECB!"
  :group 'ecb-face-options
  :group 'ecb-history
  :type '(radio (const :tag "Use ecb-default-general-face"
                       :value ecb-default-general-face)
                (face :tag "Special face"
                      :value ecb-history-general-face)))

(defface ecb-history-bucket-node-face (ecb-face-default nil nil nil
                                                        'ecb-bucket-node-face)
  "*Face used for displaying a bucket-node in the ECB-history-buffer.

This face inherits from the face 'ecb-bucket-node-face'."
  :group 'ecb-faces)

(defcustom ecb-history-bucket-node-face 'ecb-history-bucket-node-face
  "*Basic face for displaying a bucket-node in the ECB-history-buffer.
This defines the basic face for the bucket-nodes used to bucketize the
history-entries as defined with the option `ecb-history-make-buckets'.

Changes take first effect after finishing and reactivating ECB!"
  :group 'ecb-face-options
  :group 'ecb-history
  :type '(radio (const :tag "Use ecb-bucket-node-face"
                       :value ecb-bucket-node-face)
                (face :tag "Special face"
                      :value ecb-history-bucket-node-face)))

(defface ecb-history-bucket-node-dir-soure-path-face (ecb-face-default nil nil nil
                                                                       'ecb-history-bucket-node-face
                                                                       ;; DarkCyan would be good too
                                                                       "DarkMagenta"
                                                                       "DarkMagenta")
  "*Face used for displaying the source-path part in a directory-bucket.
This face inherits from the face 'ecb-history-bucket-node-face'."
  :group 'ecb-faces)

(defcustom ecb-history-bucket-node-dir-soure-path-face 'ecb-history-bucket-node-dir-soure-path-face
  "*Basic face for displaying the source-path part in a directory bucket.
This defines the face for the source-path part in a bucket-node of the
history-buffer if `ecb-history-make-buckets' is set to
'directory-with-source-path. For all other values of
`ecb-history-make-buckets' this option takes no effect.

Changes take first effect after finishing and reactivating ECB!"
  :group 'ecb-face-options
  :group 'ecb-history
  :type '(radio (const :tag "Use ecb-history-bucket-node-face"
                       :value ecb-history-bucket-node-face)
                (face :tag "Special face"
                      :value ecb-history-bucket-node-dir-soure-path-face)))

(defface ecb-history-indirect-buffer-face (ecb-face-default nil nil t
                                                            'ecb-history-general-face)
  "*Define a face for displaying indirect buffers in the history buffer.

This face inherits from the face 'ecb-history-general-face'."
  :group 'ecb-faces)

(defcustom ecb-history-indirect-buffer-face 'ecb-history-indirect-buffer-face
  "*Face for indirect buffers in the history buffer."
  :group 'ecb-history
  :group 'ecb-face-options
  :type '(radio (const :tag "Use ecb-history-general-face"
                       :value ecb-history-general-face)
                (face :tag "Special face"
                      :value ecb-history-indirect-buffer-face)))

(defface ecb-history-dead-buffer-face (ecb-face-default nil nil nil
                                                        'ecb-history-general-face
                                                        "gray60"
                                                        "gray60"
                                                        nil nil
                                                        nil "gray60")
  "*Define a face for history entries pointing to dead buffers.

This face inherits from the face 'ecb-history-general-face'."
  :group 'ecb-faces)

(defcustom ecb-history-dead-buffer-face 'ecb-history-dead-buffer-face
  "*Face for history entries pointing to dead buffers."
  :group 'ecb-history
  :group 'ecb-face-options
  :type '(radio (const :tag "Use ecb-history-general-face"
                       :value ecb-history-general-face)
                (face :tag "Special face"
                      :value ecb-history-dead-buffer-face)))


(defface ecb-default-highlight-face (ecb-face-default nil nil nil
                                                      nil ;'ecb-default-general-face
                                                      "yellow" nil
                                                      "cornflower blue" "magenta"
                                                      nil nil t)
  "*Define basic face for highlighting the selected node in a tree-buffer.
In GNU Emacs all highlighting faces in the ECB tree-buffers inherit from
this face. Therefore the default attributes like font etc. of a face used in a
tree-buffer for highlighting the current tag can be very easily changed with
face 'ecb-default-highlight-face'.
"
  :group 'ecb-faces)

(defface ecb-directory-face (ecb-face-default nil nil nil
                                              'ecb-default-highlight-face
                                              "yellow" nil
                                              "cornflower blue" "magenta"
                                              nil nil t)
  "*Define face used for highlighting current dir in directories buffer."
  :group 'ecb-faces)

(defcustom ecb-directory-face 'ecb-default-highlight-face
  "*Face used for highlighting current directory in the directories-buffer.
If the face 'ecb-default-highlight-face' is used then the display of all
ECB-tree-buffers can be changed by modifying only the face
'ecb-default-highlight-face'.

Changes take first effect after finishing and reactivating ECB!"
  :group 'ecb-face-options
  :group 'ecb-directories
  :type '(radio (const :tag "Use ecb-default-highlight-face"
                       :value ecb-default-highlight-face)
                (face :tag "Special face"
                      :value ecb-directory-face)))

(defface ecb-source-face (ecb-face-default nil nil nil
                                           'ecb-default-highlight-face
                                           "yellow" nil
                                           "cornflower blue" "magenta"
                                           nil nil t)
  "*Define face used for highlighting current source in the sources buffer."
  :group 'ecb-faces)

(defcustom ecb-source-face 'ecb-default-highlight-face
  "*Face used for highlighting current source in the sources buffer.
If the face 'ecb-default-highlight-face' is used then the display of all
ECB-tree-buffers can be changed by modifying only the face
'ecb-default-highlight-face'.

Changes take first effect after finishing and reactivating ECB!"
  :group 'ecb-face-options
  :group 'ecb-sources
  :type '(radio (const :tag "Use ecb-default-highlight-face"
                       :value ecb-default-highlight-face)
                (face :tag "Special face"
                      :value ecb-sources-face)))

(defface ecb-method-face (ecb-face-default nil nil nil
                                           'ecb-default-highlight-face
                                           "yellow" nil
                                           "cornflower blue" "magenta"
                                           nil nil t)
  "*Define face used for highlighting current tag in the methods buffer."
  :group 'ecb-faces)

(defcustom ecb-method-face 'ecb-default-highlight-face
  "*Face used for highlighting current tag in the methods buffer.
If the face 'ecb-default-highlight-face' is used then the display of all
ECB-tree-buffers can be changed by modifying only the face
'ecb-default-highlight-face'.

Changes take first effect after finishing and reactivating ECB!"
  :group 'ecb-face-options
  :group 'ecb-methods
  :type '(radio (const :tag "Use ecb-default-highlight-face"
                       :value ecb-default-highlight-face)
                (face :tag "Special face"
                      :value ecb-method-face)))

(defface ecb-method-non-semantic-face (ecb-face-default nil nil nil
                                                        'ecb-methods-general-face
                                                        "brown" "brown")
  "*Define face used for displaying tags of non-semantic-sources."
  :group 'ecb-faces)

(defcustom ecb-method-non-semantic-face 'speedbar-tag-face
  "*Face used for for displaying tags of non-semantic-sources.
Default is the face used by speedbar for tags.

Changes take first effect after finishing and reactivating ECB!"
  :group 'ecb-face-options
  :group 'ecb-methods
  :type '(radio (const :tag "Use ecb-default-highlight-face"
                       :value ecb-default-highlight-face)
                (face :tag "Special face"
                      :value ecb-method-face)))

(defface ecb-history-face (ecb-face-default nil nil nil
                                            'ecb-default-highlight-face
                                            "yellow" nil
                                            "cornflower blue" "magenta"
                                            nil nil t)
  "*Define face used for highlighting current entry in the history buffer."
  :group 'ecb-faces)

(defcustom ecb-history-face 'ecb-default-highlight-face
  "*Face used for highlighting current entry in the history buffer.
If the face 'ecb-default-highlight-face' is used then the display of all
ECB-tree-buffers can be changed by modifying only the face
'ecb-default-highlight-face'.

Changes take first effect after finishing and reactivating ECB!"
  :group 'ecb-face-options
  :group 'ecb-history
  :type '(radio (const :tag "Use ecb-default-highlight-face"
                       :value ecb-default-highlight-face)
                (face :tag "Special face"
                      :value ecb-history-face)))

(defface ecb-analyse-face (ecb-face-default nil nil nil
                                            'ecb-default-highlight-face
                                            "yellow" nil
                                            "cornflower blue" "magenta"
                                            nil nil t)
  "*Define face used for highlighting current entry in the analyse buffer."
  :group 'ecb-faces)

(defcustom ecb-analyse-face 'ecb-default-highlight-face
  "*Face used for highlighting current entry in the analyse buffer.
If the face 'ecb-default-highlight-face' is used then the display of all
ECB-tree-buffers can be changed by modifying only the face
'ecb-default-highlight-face'.

Changes take first effect after finishing and reactivating ECB!"
  :group 'ecb-face-options
  :group 'ecb-analyse
  :type '(radio (const :tag "Use ecb-default-highlight-face"
                       :value ecb-default-highlight-face)
                (face :tag "Special face"
                      :value ecb-analyse-face)))

(defface ecb-analyse-general-face (ecb-face-default 1.0 nil nil
                                                    'ecb-default-general-face)
  "*Basic face for the ECB analyse buffer.
It's recommended to define here the font-family, the font-size, the basic
color etc."
  :group 'ecb-faces)

(defcustom ecb-analyse-general-face 'ecb-default-general-face
  "*Basic face for the ECB analyse buffer.
This defines the basic face the whole history buffer should displayed with. If
the face 'ecb-default-general-face' is used then the display of all
ECB-tree-buffers can be changed by modifying only the face
'ecb-default-general-face'.

Changes take first effect after finishing and reactivating ECB!"
  :group 'ecb-face-options
  :group 'ecb-analyse
  :type '(radio (const :tag "Use ecb-default-general-face"
                       :value ecb-default-general-face)
                (face :tag "Special face"
                      :value ecb-analyse-general-face)))

(defface ecb-analyse-bucket-element-face (ecb-face-default nil nil nil
                                                           'ecb-analyse-general-face
                                                           "brown")
  "*Face used for displaying elements of buckets in the ECB-analyse-buffer.

This face inherits from the face 'ecb-default-general-face'."
  :group 'ecb-faces)

(defcustom ecb-analyse-bucket-element-face 'ecb-analyse-bucket-element-face
  "*Basic face for displaying elements of buckets in the ECB-analyse-buffer.
This defines the basic face for the elements of category-buckets like Context,
Prefix, Completions etc. in the ECB-analyse-buffer.

Changes take first effect after finishing and reactivating ECB!"
  :group 'ecb-face-options
  :group 'ecb-analyse
  :type '(radio (const :tag "Use ecb-default-general-face"
                       :value ecb-default-general-face)
                (face :tag "Special face"
                      :value ecb-analyse-bucket-element-face)))

(defface ecb-analyse-bucket-node-face (ecb-face-default nil t nil
                                                        'ecb-bucket-node-face)
  "*Face used for displaying a bucket-node in the ECB-analyse-buffer.

This face inherits from the face 'ecb-bucket-node-face'."
  :group 'ecb-faces)

(defcustom ecb-analyse-bucket-node-face 'ecb-analyse-bucket-node-face
  "*Basic face for displaying a bucket-node in the ECB-analyse-buffer.
This defines the basic face for the bucket-nodes like Context, Prefix,
Completions etc. in the ECB-analyse-buffer.

Changes take first effect after finishing and reactivating ECB!"
  :group 'ecb-face-options
  :group 'ecb-analyse
  :type '(radio (const :tag "Use ecb-bucket-node-face"
                       :value ecb-bucket-node-face)
                (face :tag "Special face"
                      :value ecb-analyse-bucket-node-face)))

(defface ecb-symboldef-symbol-face (ecb-face-default nil t nil
                                                     'ecb-default-general-face)
  "*Define face used for the symbol itself in the symboldef-buffer.
"
  :group 'ecb-faces)

(defcustom ecb-symboldef-symbol-face 'use-font-lock-face
  "*Which face should be used for the symbol itself in the symboldef-buffer.

There are two choices: Either a face or the special symbol 'use-font-lock-face
whereas the latter one means that ECB uses that face font-lock uses for
fontifying the symbol \(e.g. font-lock-function-name-face for a function
symbol or font-lock-variable-name-face for a variable symbol)."
  :group 'ecb-face-options
  :group 'ecb-symboldef
  :type '(radio (const :tag "No special face" :value nil)
                (const :tag "Use font-lock face"
                       :value use-font-lock-face)
                (face :tag "Use face"
                      :value ecb-symboldef-symbol-face)))

(defface ecb-symboldef-prototype-face (ecb-face-default nil t nil
                                                        'ecb-default-general-face)
  "*Define face used for the prototype of symbol in the symboldef-buffer.
"
  :group 'ecb-faces)

(defcustom ecb-symboldef-prototype-face nil
  "*Which face should be used for the symbol prototype in the symboldef-buffer.

The prototype can be the function prototype in case of a function or method or
- in case of a variable - the type of the variable \(in case of a typed
language)."
  :group 'ecb-face-options
  :group 'ecb-symboldef
  :type '(radio (const :tag "No special face" :value nil)
                (const :tag "Use ecb-symboldef-prototype-face"
                       :value ecb-symboldef-prototype-face)
                (face :tag "Special face")))

(defface ecb-tag-header-face (ecb-face-default nil nil nil nil nil nil
                                               "SeaGreen1" "SeaGreen1"
                                               nil nil t)
  "*Define face used for highlighting the tag header.
The tag header is the first line of the tag which is highlighted after
jumping to it by clicking onto a node in the methods buffer."
  :group 'ecb-faces)

(defcustom ecb-tag-header-face 'ecb-tag-header-face
  "*Face used for highlighting the tag header.
The tag header is the first line of the tag which is highlighted after
jumping to it by clicking onto a node in the methods buffer."
  :group 'ecb-face-options
  :group 'ecb-methods
  :type 'face)

(defface ecb-source-in-directories-buffer-face (ecb-face-default nil nil nil
                                                                 'ecb-directories-general-face
                                                                 "medium blue"
                                                                 "LightBlue1"
                                                                 nil nil
                                                                 nil "gray")
  "*Define a face for displaying sources in the directories buffer."
  :group 'ecb-faces)

(defcustom ecb-source-in-directories-buffer-face 'ecb-source-in-directories-buffer-face
  "*Face for source files in the directories buffer."
  :group 'ecb-directories
  :group 'ecb-face-options
  :type 'face)

(defface ecb-source-read-only-face (ecb-face-default nil nil t
                                                     'ecb-default-general-face)
  "*Define a face for displaying read-only sources."
  :group 'ecb-faces)

(defcustom ecb-source-read-only-face 'ecb-source-read-only-face
  "*Face for read-only sources."
  :group 'ecb-sources
  :group 'ecb-directories
  :group 'ecb-face-options
  :type 'face)

(defface ecb-directory-not-accessible-face (ecb-face-default nil nil nil
                                                             'ecb-directories-general-face
                                                             "gray60"
                                                             "gray60"
                                                             nil nil
                                                             nil "gray60")
  "*Define a face for displaying not accessible dirs in the directories buffer."
  :group 'ecb-faces)

(defcustom ecb-directory-not-accessible-face 'ecb-directory-not-accessible-face
  "*Face for not accessible dirs in the directories buffer."
  :group 'ecb-directories
  :group 'ecb-face-options
  :type 'face)

(defface ecb-type-tag-class-face (ecb-face-default nil t)
  "*Define face used with option `ecb-type-tag-display'."
  :group 'ecb-faces)

(defface ecb-type-tag-interface-face (ecb-face-default nil t)
  "*Define face used with option `ecb-type-tag-display'."
  :group 'ecb-faces)

(defface ecb-type-tag-struct-face (ecb-face-default nil t)
  "*Define face used with option `ecb-type-tag-display'."
  :group 'ecb-faces)

(defface ecb-type-tag-typedef-face (ecb-face-default nil t)
  "*Define face used with option `ecb-type-tag-display'."
  :group 'ecb-faces)

(defface ecb-type-tag-union-face (ecb-face-default nil t)
  "*Define face used with option `ecb-type-tag-display'."
  :group 'ecb-faces)

(defface ecb-type-tag-enum-face (ecb-face-default nil t)
  "*Define face used with option `ecb-type-tag-display'."
  :group 'ecb-faces)

(defface ecb-type-tag-group-face (ecb-face-default nil t nil nil
                                                   "dim gray" "dim gray")
  "*Define face used with option `ecb-type-tag-display'."
  :group 'ecb-faces)

(defface ecb-bucket-node-face (ecb-face-default nil t nil
                                                'ecb-default-general-face)
  "*Face used for displaying bucket-nodes in the ECB-buffers.
See also `ecb-bucket-node-display'.

This face inherits from the face 'ecb-default-general-face'."
  :group 'ecb-faces)

;; - mode-line faces-------------------------------------------

(defface ecb-mode-line-win-nr-face (ecb-face-default nil t)
  "*Define face for the window-number in the mode-line.
See `ecb-mode-line-display-window-number'."
  :group 'ecb-faces)

(defcustom ecb-mode-line-win-nr-face 'ecb-mode-line-win-nr-face
  "*Face used for the window-number in the mode-line.
See `ecb-mode-line-display-window-number'."
  :group 'ecb-mode-line
  :group 'ecb-face-options
  :type 'face)

(defface ecb-mode-line-prefix-face (ecb-face-default nil nil nil nil
                                                     "forestgreen"
                                                     "forestgreen")
  "*Define face for the prefix in the mode-line.
See `ecb-mode-line-prefixes'."
  :group 'ecb-faces)

(defcustom ecb-mode-line-prefix-face 'ecb-mode-line-prefix-face
  "*Face used for the prefix in the mode-line.
See `ecb-mode-line-prefixes'."
  :group 'ecb-mode-line
  :group 'ecb-face-options
  :type 'face)

(defface ecb-mode-line-data-face (ecb-face-default)
  "*Define face for the data in the mode-line.
See `ecb-mode-line-data'."
  :group 'ecb-faces)

(defcustom ecb-mode-line-data-face 'ecb-mode-line-data-face
  "*Face used for the data in the mode-line.
See `ecb-mode-line-data'."
  :group 'ecb-mode-line
  :group 'ecb-face-options
  :type 'face)

(silentcomp-provide 'ecb-face)

;;; ecb-face.el ends here
