;;; helm-emoji-data.el --- Data for helm-emoji  -*- lexical-binding: t -*-

;; Copyright (C) 2026 Timm Lichte

;; Author: Timm Lichte <timm.lichte@uni-tuebingen.de>
;; URL: https://codeberg.org/timmli/helm-emoji
;; Last modified: 2026-02-18 Wed 12:35:38

;; This program is free software: you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation, either version 3 of the License, or
;; (at your option) any later version.

;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with this program.  If not, see <https://www.gnu.org/licenses/>.

;;; Commentary:

;; Additional data for helm-emoji.

;;; Code:

(require 'cl-lib)
(require 'json)

(defun helm-emoji-data-abbrev-p (str)
  "Return non-nil if STR begins and ends with a colon."
  (and (> (length str) 2)
       (string= (substring str 0 1) ":")
       (string= (substring str -1) ":")))

(defun helm-emoji-data--download ()
  "Create alist of (SYMBOL . ABBREVIATION) based on the Discord Emoji Map."
  (let ((url "https://emzi0767.mzgit.io/discord-emoji/discordEmojiMap-canary.min.json"))
    (with-current-buffer (url-retrieve-synchronously url t)
      (goto-char (point-min))
      (re-search-forward "\n\n" nil t)  ; Skip to the HTTP response header
      (let* ((json-data (json-read-from-string (buffer-substring (point) (point-max))))
             ;; Use append to convert vector into list
             (defs (append (alist-get 'emojiDefinitions json-data) nil)))
        (cl-loop
         for emoji in defs
         for char = (alist-get 'surrogates emoji)
         for names-with-colons =
         (let ((strings (append (alist-get 'namesWithColons emoji) nil)))
           ;; Only keep strings that are not contained in other strings
           (cl-loop
            for str in strings
            unless (cl-some
                    (lambda (other)
                      (and (not (string= str other))
                           (when (and (helm-emoji-data-abbrev-p str)
                                      (helm-emoji-data-abbrev-p other))
                             (string-match-p (regexp-quote (replace-regexp-in-string "^:+\\|:+$" "" str))
                                             (regexp-quote (replace-regexp-in-string "^:+\\|:+$" "" other))))))
                    strings)
            collect str))
         for names-with-colons-string = (string-join names-with-colons " ")
         when (and char names-with-colons-string)
         collect `(,char . ,names-with-colons-string))))))

(defun helm-emoji-data--update ()
  "Update the function definition of `helm-emoji-data--return'."
  (let* ((function-string
          (format "(defun helm-emoji-data--return ()\n\"%s\"\n'(%s))"
                  "Return an alist with emoji abbrevations."
                  (string-join
                   (cl-loop
                    for emoji in (helm-emoji-data--download)
                    collect (prin1-to-string emoji))
                   "\n")))
         (start (progn (search-forward "(defun helm-emoji-data--return" )
                       (match-beginning 0)))
         (end (progn (goto-char start)
                     (forward-sexp)
                     (point))))
    (delete-region start end)
    (goto-char start)
    (insert function-string)))

;; Execute (helm-emoji-data--update) here for updating `helm-emoji-data--return'.
(defun helm-emoji-data--return ()
  "Return an alist with emoji abbrevations."
  '(("😀" . ":grinning_face:")
    ("😃" . ":smiley:")
    ("😄" . ":smile: :D :-D =D =-D")
    ("😁" . ":grin:")
    ("😆" . ":laughing: :satisfied: x-) X-)")
    ("🥹" . ":face_holding_back_tears:")
    ("😅" . ":sweat_smile: ,:) ,:-) ,=) ,=-)")
    ("😂" . ":joy: :'D :'-D :,D :,-D ='D ='-D =,D =,-D")
    ("🤣" . ":rofl: :rolling_on_the_floor_laughing:")
    ("🥲" . ":smiling_face_with_tear: :') :'-) :,) :,-) =') ='-) =,) =,-)")
    ("\342\230\272\357\270\217" . ":relaxed: :smiling_face:")
    ("😊" . ":blush: :\") :-\") =\") =-\")")
    ("😇" . ":innocent: o:) O:) o:-) O:-) 0:) 0:-) o=) O=) o=-) O=-) 0=) 0=-)")
    ("🙂" . ":slight_smile: :slightly_smiling_face: :) :-) =) =-)")
    ("🙃" . ":upside_down_face:")
    ("😉" . ":winking_face: ;) ;-)")
    ("😌" . ":relieved_face:")
    ("😍" . ":heart_eyes:")
    ("🥰" . ":smiling_face_with_3_hearts:")
    ("😘" . ":kissing_heart:")
    ("😗" . ":kissing_face: :* :-* =* =-*")
    ("😙" . ":kissing_smiling_eyes:")
    ("😚" . ":kissing_closed_eyes:")
    ("😋" . ":yum:")
    ("😛" . ":stuck_out_tongue: :P :-P =P =-P")
    ("😝" . ":stuck_out_tongue_closed_eyes:")
    ("😜" . ":stuck_out_tongue_winking_eye:")
    ("🤪" . ":zany_face:")
    ("🤨" . ":face_with_raised_eyebrow:")
    ("🧐" . ":face_with_monocle:")
    ("🤓" . ":nerd_face:")
    ("😎" . ":sunglasses: 8-) B-)")
    ("🥸" . ":disguised_face:")
    ("🤩" . ":star_struck:")
    ("🥳" . ":partying_face:")
    ("😏" . ":smirking_face:")
    ("😒" . ":unamused_face: :s :-S :z :-Z :$ :-$ =s =-S =z =-Z =$ =-$")
    ("😞" . ":disappointed:")
    ("😔" . ":pensive_face:")
    ("😟" . ":worried_face:")
    ("😕" . ":confused_face: :-\\ :-/ =-\\ =-/")
    ("🙁" . ":slight_frown: :slightly_frowning_face: :( :-( =( =-(")
    ("\342\230\271\357\270\217" . ":frowning2: :white_frowning_face:")
    ("😣" . ":persevere:")
    ("😖" . ":confounded:")
    ("😫" . ":tired_face:")
    ("😩" . ":weary_face:")
    ("🥺" . ":pleading_face:")
    ("😢" . ":crying_face: :'( :'-( :,( :,-( ='( ='-( =,( =,-(")
    ("😭" . ":sob: :,'( :,'-( ;( ;-( =,'( =,'-(")
    ("😤" . ":triumph:")
    ("😠" . ":angry_face: >:( >:-( >=( >=-(")
    ("😡" . ":rage: :pouting_face: :@ :-@ =@ =-@")
    ("🤬" . ":face_with_symbols_over_mouth:")
    ("🤯" . ":exploding_head:")
    ("😳" . ":flushed_face:")
    ("🥵" . ":hot_face:")
    ("🥶" . ":cold_face:")
    ("😶\342\200\215🌫\357\270\217" . ":face_in_clouds:")
    ("😱" . ":scream:")
    ("😨" . ":fearful_face:")
    ("😰" . ":cold_sweat:")
    ("😥" . ":disappointed_relieved:")
    ("😓" . ":sweat: ,:( ,:-( ,=( ,=-(")
    ("🤗" . ":hugging_face:")
    ("🤔" . ":thinking_face:")
    ("🫣" . ":face_with_peeking_eye:")
    ("🤭" . ":face_with_hand_over_mouth:")
    ("🫢" . ":face_with_open_eyes_and_hand_over_mouth:")
    ("🫡" . ":saluting_face:")
    ("🤫" . ":shushing_face:")
    ("🫠" . ":melting_face:")
    ("🤥" . ":lying_face: :liar:")
    ("😶" . ":no_mouth:")
    ("🫥" . ":dotted_line_face:")
    ("😐" . ":neutral_face: :| :-| =| =-|")
    ("🫤" . ":face_with_diagonal_mouth:")
    ("😑" . ":expressionless:")
    ("🫨" . ":shaking_face:")
    ("🙂\342\200\215\342\206\224\357\270\217" . ":head_shaking_horizontally:")
    ("🙂\342\200\215\342\206\225\357\270\217" . ":head_shaking_vertically:")
    ("😬" . ":grimacing:")
    ("🙄" . ":face_with_rolling_eyes:")
    ("😯" . ":hushed_face:")
    ("😦" . ":frowning:")
    ("😧" . ":anguished:")
    ("😮" . ":open_mouth: :o :-o :O :-O =o =-o =O =-O")
    ("😲" . ":astonished:")
    ("🥱" . ":yawning_face:")
    ("😴" . ":sleeping_face:")
    ("🤤" . ":drooling_face:")
    ("😪" . ":sleepy_face:")
    ("😮\342\200\215💨" . ":face_exhaling:")
    ("😵" . ":dizzy_face:")
    ("😵\342\200\215💫" . ":face_with_spiral_eyes:")
    ("🤐" . ":zipper_mouth_face:")
    ("🥴" . ":woozy_face:")
    ("🤢" . ":nauseated_face: :sick:")
    ("🤮" . ":face_vomiting:")
    ("🤧" . ":sneezing_face: :sneeze:")
    ("😷" . ":mask:")
    ("🤒" . ":thermometer_face: :face_with_thermometer:")
    ("🤕" . ":face_with_head_bandage:")
    ("🤑" . ":money_mouth_face:")
    ("🤠" . ":face_with_cowboy_hat:")
    ("😈" . ":smiling_imp: ]:) ]:-) ]=) ]=-)")
    ("👿" . ":imp: ]:( ]:-( ]=( ]=-(")
    ("👹" . ":japanese_ogre:")
    ("👺" . ":japanese_goblin:")
    ("🤡" . ":clown_face:")
    ("💩" . ":poop: :shit: :hankey: :pile_of_poo:")
    ("👻" . ":ghost:")
    ("💀" . ":skull: :skeleton:")
    ("\342\230\240\357\270\217" . ":skull_crossbones: :skull_and_crossbones:")
    ("👽" . ":alien:")
    ("👾" . ":space_invader: :alien_monster:")
    ("🤖" . ":robot_face:")
    ("🎃" . ":jack_o_lantern:")
    ("😺" . ":smiley_cat: :grinning_cat:")
    ("😸" . ":smile_cat:")
    ("😹" . ":joy_cat:")
    ("😻" . ":heart_eyes_cat:")
    ("😼" . ":smirk_cat:")
    ("😽" . ":kissing_cat:")
    ("🙀" . ":scream_cat: :weary_cat:")
    ("😿" . ":crying_cat_face:")
    ("😾" . ":pouting_cat:")
    ("🤝" . ":handshake: :shaking_hands:")
    ("🤝🏻" . ":handshake_tone1: :handshake_light_skin_tone:")
    ("🫱🏻\342\200\215🫲🏼" . ":handshake_tone1_tone2: :handshake_light_skin_tone_medium_light_skin_tone:")
    ("🫱🏻\342\200\215🫲🏽" . ":handshake_tone1_tone3: :handshake_light_skin_tone_medium_skin_tone:")
    ("🫱🏻\342\200\215🫲🏾" . ":handshake_tone1_tone4: :handshake_light_skin_tone_medium_dark_skin_tone:")
    ("🫱🏻\342\200\215🫲🏿" . ":handshake_tone1_tone5: :handshake_light_skin_tone_dark_skin_tone:")
    ("🫱🏼\342\200\215🫲🏻" . ":handshake_tone2_tone1: :handshake_medium_light_skin_tone_light_skin_tone:")
    ("🤝🏼" . ":handshake_tone2: :handshake_medium_light_skin_tone:")
    ("🫱🏼\342\200\215🫲🏽" . ":handshake_tone2_tone3: :handshake_medium_light_skin_tone_medium_skin_tone:")
    ("🫱🏼\342\200\215🫲🏾" . ":handshake_tone2_tone4: :handshake_medium_light_skin_tone_medium_dark_skin_tone:")
    ("🫱🏼\342\200\215🫲🏿" . ":handshake_tone2_tone5: :handshake_medium_light_skin_tone_dark_skin_tone:")
    ("🫱🏽\342\200\215🫲🏻" . ":handshake_tone3_tone1: :handshake_medium_skin_tone_light_skin_tone:")
    ("🫱🏽\342\200\215🫲🏼" . ":handshake_tone3_tone2: :handshake_medium_skin_tone_medium_light_skin_tone:")
    ("🤝🏽" . ":handshake_tone3: :handshake_medium_skin_tone:")
    ("🫱🏽\342\200\215🫲🏾" . ":handshake_tone3_tone4: :handshake_medium_skin_tone_medium_dark_skin_tone:")
    ("🫱🏽\342\200\215🫲🏿" . ":handshake_tone3_tone5: :handshake_medium_skin_tone_dark_skin_tone:")
    ("🫱🏾\342\200\215🫲🏻" . ":handshake_tone4_tone1: :handshake_medium_dark_skin_tone_light_skin_tone:")
    ("🫱🏾\342\200\215🫲🏼" . ":handshake_tone4_tone2: :handshake_medium_dark_skin_tone_medium_light_skin_tone:")
    ("🫱🏾\342\200\215🫲🏽" . ":handshake_tone4_tone3: :handshake_medium_dark_skin_tone_medium_skin_tone:")
    ("🤝🏾" . ":handshake_tone4: :handshake_medium_dark_skin_tone:")
    ("🫱🏾\342\200\215🫲🏿" . ":handshake_tone4_tone5: :handshake_medium_dark_skin_tone_dark_skin_tone:")
    ("🫱🏿\342\200\215🫲🏻" . ":handshake_tone5_tone1: :handshake_dark_skin_tone_light_skin_tone:")
    ("🫱🏿\342\200\215🫲🏼" . ":handshake_tone5_tone2: :handshake_dark_skin_tone_medium_light_skin_tone:")
    ("🫱🏿\342\200\215🫲🏽" . ":handshake_tone5_tone3: :handshake_dark_skin_tone_medium_skin_tone:")
    ("🫱🏿\342\200\215🫲🏾" . ":handshake_tone5_tone4: :handshake_dark_skin_tone_medium_dark_skin_tone:")
    ("🤝🏿" . ":handshake_tone5: :handshake_dark_skin_tone:")
    ("🫶" . ":heart_hands:")
    ("🫶🏻" . ":heart_hands_tone1: :heart_hands_light_skin_tone: :heart_hands::skin-tone-1:")
    ("🫶🏼" . ":heart_hands_tone2: :heart_hands_medium_light_skin_tone: :heart_hands::skin-tone-2:")
    ("🫶🏽" . ":heart_hands_tone3: :heart_hands_medium_skin_tone: :heart_hands::skin-tone-3:")
    ("🫶🏾" . ":heart_hands_tone4: :heart_hands_medium_dark_skin_tone: :heart_hands::skin-tone-4:")
    ("🫶🏿" . ":heart_hands_tone5: :heart_hands_dark_skin_tone: :heart_hands::skin-tone-5:")
    ("🤲" . ":palms_up_together:")
    ("🤲🏻" . ":palms_up_together_tone1: :palms_up_together_light_skin_tone: :palms_up_together::skin-tone-1:")
    ("🤲🏼" . ":palms_up_together_tone2: :palms_up_together_medium_light_skin_tone: :palms_up_together::skin-tone-2:")
    ("🤲🏽" . ":palms_up_together_tone3: :palms_up_together_medium_skin_tone: :palms_up_together::skin-tone-3:")
    ("🤲🏾" . ":palms_up_together_tone4: :palms_up_together_medium_dark_skin_tone: :palms_up_together::skin-tone-4:")
    ("🤲🏿" . ":palms_up_together_tone5: :palms_up_together_dark_skin_tone: :palms_up_together::skin-tone-5:")
    ("👐" . ":open_hands:")
    ("👐🏻" . ":open_hands_tone1: :open_hands::skin-tone-1:")
    ("👐🏼" . ":open_hands_tone2: :open_hands::skin-tone-2:")
    ("👐🏽" . ":open_hands_tone3: :open_hands::skin-tone-3:")
    ("👐🏾" . ":open_hands_tone4: :open_hands::skin-tone-4:")
    ("👐🏿" . ":open_hands_tone5: :open_hands::skin-tone-5:")
    ("🙌" . ":raised_hands: :raising_hands:")
    ("🙌🏻" . ":raised_hands_tone1: :raised_hands::skin-tone-1: :raising_hands::skin-tone-1:")
    ("🙌🏼" . ":raised_hands_tone2: :raised_hands::skin-tone-2: :raising_hands::skin-tone-2:")
    ("🙌🏽" . ":raised_hands_tone3: :raised_hands::skin-tone-3: :raising_hands::skin-tone-3:")
    ("🙌🏾" . ":raised_hands_tone4: :raised_hands::skin-tone-4: :raising_hands::skin-tone-4:")
    ("🙌🏿" . ":raised_hands_tone5: :raised_hands::skin-tone-5: :raising_hands::skin-tone-5:")
    ("👏" . ":clap:")
    ("👏🏻" . ":clap_tone1: :clap::skin-tone-1:")
    ("👏🏼" . ":clap_tone2: :clap::skin-tone-2:")
    ("👏🏽" . ":clap_tone3: :clap::skin-tone-3:")
    ("👏🏾" . ":clap_tone4: :clap::skin-tone-4:")
    ("👏🏿" . ":clap_tone5: :clap::skin-tone-5:")
    ("👍" . ":thumbsup: :+1: :thumbup: :thumbs_up:")
    ("👍🏻" . ":thumbsup_tone1: :+1_tone1: :thumbup_tone1: :thumbsup::skin-tone-1: :+1::skin-tone-1: :thumbup::skin-tone-1: :thumbs_up::skin-tone-1:")
    ("👍🏼" . ":thumbsup_tone2: :+1_tone2: :thumbup_tone2: :thumbsup::skin-tone-2: :+1::skin-tone-2: :thumbup::skin-tone-2: :thumbs_up::skin-tone-2:")
    ("👍🏽" . ":thumbsup_tone3: :+1_tone3: :thumbup_tone3: :thumbsup::skin-tone-3: :+1::skin-tone-3: :thumbup::skin-tone-3: :thumbs_up::skin-tone-3:")
    ("👍🏾" . ":thumbsup_tone4: :+1_tone4: :thumbup_tone4: :thumbsup::skin-tone-4: :+1::skin-tone-4: :thumbup::skin-tone-4: :thumbs_up::skin-tone-4:")
    ("👍🏿" . ":thumbsup_tone5: :+1_tone5: :thumbup_tone5: :thumbsup::skin-tone-5: :+1::skin-tone-5: :thumbup::skin-tone-5: :thumbs_up::skin-tone-5:")
    ("👎" . ":thumbsdown: :-1: :thumbdown: :thumbs_down:")
    ("👎🏻" . ":thumbsdown_tone1: :_1_tone1: :thumbdown_tone1: :thumbsdown::skin-tone-1: :-1::skin-tone-1: :thumbdown::skin-tone-1: :thumbs_down::skin-tone-1:")
    ("👎🏼" . ":thumbsdown_tone2: :_1_tone2: :thumbdown_tone2: :thumbsdown::skin-tone-2: :-1::skin-tone-2: :thumbdown::skin-tone-2: :thumbs_down::skin-tone-2:")
    ("👎🏽" . ":thumbsdown_tone3: :_1_tone3: :thumbdown_tone3: :thumbsdown::skin-tone-3: :-1::skin-tone-3: :thumbdown::skin-tone-3: :thumbs_down::skin-tone-3:")
    ("👎🏾" . ":thumbsdown_tone4: :_1_tone4: :thumbdown_tone4: :thumbsdown::skin-tone-4: :-1::skin-tone-4: :thumbdown::skin-tone-4: :thumbs_down::skin-tone-4:")
    ("👎🏿" . ":thumbsdown_tone5: :_1_tone5: :thumbdown_tone5: :thumbsdown::skin-tone-5: :-1::skin-tone-5: :thumbdown::skin-tone-5: :thumbs_down::skin-tone-5:")
    ("👊" . ":punch: :oncoming_fist:")
    ("👊🏻" . ":punch_tone1: :punch::skin-tone-1: :oncoming_fist::skin-tone-1:")
    ("👊🏼" . ":punch_tone2: :punch::skin-tone-2: :oncoming_fist::skin-tone-2:")
    ("👊🏽" . ":punch_tone3: :punch::skin-tone-3: :oncoming_fist::skin-tone-3:")
    ("👊🏾" . ":punch_tone4: :punch::skin-tone-4: :oncoming_fist::skin-tone-4:")
    ("👊🏿" . ":punch_tone5: :punch::skin-tone-5: :oncoming_fist::skin-tone-5:")
    ("\342\234\212" . ":raised_fist:")
    ("\342\234\212🏻" . ":fist_tone1: :raised_fist::skin-tone-1:")
    ("\342\234\212🏼" . ":fist_tone2: :raised_fist::skin-tone-2:")
    ("\342\234\212🏽" . ":fist_tone3: :raised_fist::skin-tone-3:")
    ("\342\234\212🏾" . ":fist_tone4: :raised_fist::skin-tone-4:")
    ("\342\234\212🏿" . ":fist_tone5: :raised_fist::skin-tone-5:")
    ("🤛" . ":left_facing_fist: :left_fist:")
    ("🤛🏻" . ":left_facing_fist_tone1: :left_fist_tone1: :left_facing_fist::skin-tone-1: :left_fist::skin-tone-1:")
    ("🤛🏼" . ":left_facing_fist_tone2: :left_fist_tone2: :left_facing_fist::skin-tone-2: :left_fist::skin-tone-2:")
    ("🤛🏽" . ":left_facing_fist_tone3: :left_fist_tone3: :left_facing_fist::skin-tone-3: :left_fist::skin-tone-3:")
    ("🤛🏾" . ":left_facing_fist_tone4: :left_fist_tone4: :left_facing_fist::skin-tone-4: :left_fist::skin-tone-4:")
    ("🤛🏿" . ":left_facing_fist_tone5: :left_fist_tone5: :left_facing_fist::skin-tone-5: :left_fist::skin-tone-5:")
    ("🤜" . ":right_facing_fist: :right_fist:")
    ("🤜🏻" . ":right_facing_fist_tone1: :right_fist_tone1: :right_facing_fist::skin-tone-1: :right_fist::skin-tone-1:")
    ("🤜🏼" . ":right_facing_fist_tone2: :right_fist_tone2: :right_facing_fist::skin-tone-2: :right_fist::skin-tone-2:")
    ("🤜🏽" . ":right_facing_fist_tone3: :right_fist_tone3: :right_facing_fist::skin-tone-3: :right_fist::skin-tone-3:")
    ("🤜🏾" . ":right_facing_fist_tone4: :right_fist_tone4: :right_facing_fist::skin-tone-4: :right_fist::skin-tone-4:")
    ("🤜🏿" . ":right_facing_fist_tone5: :right_fist_tone5: :right_facing_fist::skin-tone-5: :right_fist::skin-tone-5:")
    ("🫷" . ":leftwards_pushing_hand:")
    ("🫷🏻" . ":leftwards_pushing_hand_tone1: :leftwards_pushing_hand_light_skin_tone: :leftwards_pushing_hand::skin-tone-1:")
    ("🫷🏼" . ":leftwards_pushing_hand_tone2: :leftwards_pushing_hand_medium_light_skin_tone: :leftwards_pushing_hand::skin-tone-2:")
    ("🫷🏽" . ":leftwards_pushing_hand_tone3: :leftwards_pushing_hand_medium_skin_tone: :leftwards_pushing_hand::skin-tone-3:")
    ("🫷🏾" . ":leftwards_pushing_hand_tone4: :leftwards_pushing_hand_medium_dark_skin_tone: :leftwards_pushing_hand::skin-tone-4:")
    ("🫷🏿" . ":leftwards_pushing_hand_tone5: :leftwards_pushing_hand_dark_skin_tone: :leftwards_pushing_hand::skin-tone-5:")
    ("🫸" . ":rightwards_pushing_hand:")
    ("🫸🏻" . ":rightwards_pushing_hand_tone1: :rightwards_pushing_hand_light_skin_tone: :rightwards_pushing_hand::skin-tone-1:")
    ("🫸🏼" . ":rightwards_pushing_hand_tone2: :rightwards_pushing_hand_medium_light_skin_tone: :rightwards_pushing_hand::skin-tone-2:")
    ("🫸🏽" . ":rightwards_pushing_hand_tone3: :rightwards_pushing_hand_medium_skin_tone: :rightwards_pushing_hand::skin-tone-3:")
    ("🫸🏾" . ":rightwards_pushing_hand_tone4: :rightwards_pushing_hand_medium_dark_skin_tone: :rightwards_pushing_hand::skin-tone-4:")
    ("🫸🏿" . ":rightwards_pushing_hand_tone5: :rightwards_pushing_hand_dark_skin_tone: :rightwards_pushing_hand::skin-tone-5:")
    ("🤞" . ":fingers_crossed: :hand_with_index_and_middle_finger_crossed:")
    ("🤞🏻" . ":hand_with_index_and_middle_fingers_crossed_tone1: :fingers_crossed::skin-tone-1: :hand_with_index_and_middle_finger_crossed::skin-tone-1:")
    ("🤞🏼" . ":hand_with_index_and_middle_fingers_crossed_tone2: :fingers_crossed::skin-tone-2: :hand_with_index_and_middle_finger_crossed::skin-tone-2:")
    ("🤞🏽" . ":hand_with_index_and_middle_fingers_crossed_tone3: :fingers_crossed::skin-tone-3: :hand_with_index_and_middle_finger_crossed::skin-tone-3:")
    ("🤞🏾" . ":hand_with_index_and_middle_fingers_crossed_tone4: :fingers_crossed::skin-tone-4: :hand_with_index_and_middle_finger_crossed::skin-tone-4:")
    ("🤞🏿" . ":hand_with_index_and_middle_fingers_crossed_tone5: :fingers_crossed::skin-tone-5: :hand_with_index_and_middle_finger_crossed::skin-tone-5:")
    ("\342\234\214\357\270\217" . ":victory_hand:")
    ("\342\234\214🏻" . ":v_tone1: :v::skin-tone-1: :victory_hand::skin-tone-1:")
    ("\342\234\214🏼" . ":v_tone2: :v::skin-tone-2: :victory_hand::skin-tone-2:")
    ("\342\234\214🏽" . ":v_tone3: :v::skin-tone-3: :victory_hand::skin-tone-3:")
    ("\342\234\214🏾" . ":v_tone4: :v::skin-tone-4: :victory_hand::skin-tone-4:")
    ("\342\234\214🏿" . ":v_tone5: :v::skin-tone-5: :victory_hand::skin-tone-5:")
    ("🫰" . ":hand_with_index_finger_and_thumb_crossed:")
    ("🫰🏻" . ":hand_with_index_finger_and_thumb_crossed_tone1: :hand_with_index_finger_and_thumb_crossed_light_skin_tone: :hand_with_index_finger_and_thumb_crossed::skin-tone-1:")
    ("🫰🏼" . ":hand_with_index_finger_and_thumb_crossed_tone2: :hand_with_index_finger_and_thumb_crossed_medium_light_skin_tone: :hand_with_index_finger_and_thumb_crossed::skin-tone-2:")
    ("🫰🏽" . ":hand_with_index_finger_and_thumb_crossed_tone3: :hand_with_index_finger_and_thumb_crossed_medium_skin_tone: :hand_with_index_finger_and_thumb_crossed::skin-tone-3:")
    ("🫰🏾" . ":hand_with_index_finger_and_thumb_crossed_tone4: :hand_with_index_finger_and_thumb_crossed_medium_dark_skin_tone: :hand_with_index_finger_and_thumb_crossed::skin-tone-4:")
    ("🫰🏿" . ":hand_with_index_finger_and_thumb_crossed_tone5: :hand_with_index_finger_and_thumb_crossed_dark_skin_tone: :hand_with_index_finger_and_thumb_crossed::skin-tone-5:")
    ("🤟" . ":love_you_gesture:")
    ("🤟🏻" . ":love_you_gesture_tone1: :love_you_gesture_light_skin_tone: :love_you_gesture::skin-tone-1:")
    ("🤟🏼" . ":love_you_gesture_tone2: :love_you_gesture_medium_light_skin_tone: :love_you_gesture::skin-tone-2:")
    ("🤟🏽" . ":love_you_gesture_tone3: :love_you_gesture_medium_skin_tone: :love_you_gesture::skin-tone-3:")
    ("🤟🏾" . ":love_you_gesture_tone4: :love_you_gesture_medium_dark_skin_tone: :love_you_gesture::skin-tone-4:")
    ("🤟🏿" . ":love_you_gesture_tone5: :love_you_gesture_dark_skin_tone: :love_you_gesture::skin-tone-5:")
    ("🤘" . ":metal: :sign_of_the_horns:")
    ("🤘🏻" . ":metal_tone1: :sign_of_the_horns_tone1: :metal::skin-tone-1: :sign_of_the_horns::skin-tone-1:")
    ("🤘🏼" . ":metal_tone2: :sign_of_the_horns_tone2: :metal::skin-tone-2: :sign_of_the_horns::skin-tone-2:")
    ("🤘🏽" . ":metal_tone3: :sign_of_the_horns_tone3: :metal::skin-tone-3: :sign_of_the_horns::skin-tone-3:")
    ("🤘🏾" . ":metal_tone4: :sign_of_the_horns_tone4: :metal::skin-tone-4: :sign_of_the_horns::skin-tone-4:")
    ("🤘🏿" . ":metal_tone5: :sign_of_the_horns_tone5: :metal::skin-tone-5: :sign_of_the_horns::skin-tone-5:")
    ("👌" . ":ok_hand:")
    ("👌🏻" . ":ok_hand_tone1: :ok_hand::skin-tone-1:")
    ("👌🏼" . ":ok_hand_tone2: :ok_hand::skin-tone-2:")
    ("👌🏽" . ":ok_hand_tone3: :ok_hand::skin-tone-3:")
    ("👌🏾" . ":ok_hand_tone4: :ok_hand::skin-tone-4:")
    ("👌🏿" . ":ok_hand_tone5: :ok_hand::skin-tone-5:")
    ("🤌" . ":pinched_fingers:")
    ("🤌🏼" . ":pinched_fingers_tone2: :pinched_fingers_medium_light_skin_tone: :pinched_fingers::skin-tone-2:")
    ("🤌🏻" . ":pinched_fingers_tone1: :pinched_fingers_light_skin_tone: :pinched_fingers::skin-tone-1:")
    ("🤌🏽" . ":pinched_fingers_tone3: :pinched_fingers_medium_skin_tone: :pinched_fingers::skin-tone-3:")
    ("🤌🏾" . ":pinched_fingers_tone4: :pinched_fingers_medium_dark_skin_tone: :pinched_fingers::skin-tone-4:")
    ("🤌🏿" . ":pinched_fingers_tone5: :pinched_fingers_dark_skin_tone: :pinched_fingers::skin-tone-5:")
    ("🤏" . ":pinching_hand:")
    ("🤏🏻" . ":pinching_hand_tone1: :pinching_hand_light_skin_tone: :pinching_hand::skin-tone-1:")
    ("🤏🏼" . ":pinching_hand_tone2: :pinching_hand_medium_light_skin_tone: :pinching_hand::skin-tone-2:")
    ("🤏🏽" . ":pinching_hand_tone3: :pinching_hand_medium_skin_tone: :pinching_hand::skin-tone-3:")
    ("🤏🏾" . ":pinching_hand_tone4: :pinching_hand_medium_dark_skin_tone: :pinching_hand::skin-tone-4:")
    ("🤏🏿" . ":pinching_hand_tone5: :pinching_hand_dark_skin_tone: :pinching_hand::skin-tone-5:")
    ("🫳" . ":palm_down_hand:")
    ("🫳🏻" . ":palm_down_hand_tone1: :palm_down_hand_light_skin_tone: :palm_down_hand::skin-tone-1:")
    ("🫳🏼" . ":palm_down_hand_tone2: :palm_down_hand_medium_light_skin_tone: :palm_down_hand::skin-tone-2:")
    ("🫳🏽" . ":palm_down_hand_tone3: :palm_down_hand_medium_skin_tone: :palm_down_hand::skin-tone-3:")
    ("🫳🏾" . ":palm_down_hand_tone4: :palm_down_hand_medium_dark_skin_tone: :palm_down_hand::skin-tone-4:")
    ("🫳🏿" . ":palm_down_hand_tone5: :palm_down_hand_dark_skin_tone: :palm_down_hand::skin-tone-5:")
    ("🫴" . ":palm_up_hand:")
    ("🫴🏻" . ":palm_up_hand_tone1: :palm_up_hand_light_skin_tone: :palm_up_hand::skin-tone-1:")
    ("🫴🏼" . ":palm_up_hand_tone2: :palm_up_hand_medium_light_skin_tone: :palm_up_hand::skin-tone-2:")
    ("🫴🏽" . ":palm_up_hand_tone3: :palm_up_hand_medium_skin_tone: :palm_up_hand::skin-tone-3:")
    ("🫴🏾" . ":palm_up_hand_tone4: :palm_up_hand_medium_dark_skin_tone: :palm_up_hand::skin-tone-4:")
    ("🫴🏿" . ":palm_up_hand_tone5: :palm_up_hand_dark_skin_tone: :palm_up_hand::skin-tone-5:")
    ("👈" . ":point_left:")
    ("👈🏻" . ":point_left_tone1: :point_left::skin-tone-1:")
    ("👈🏼" . ":point_left_tone2: :point_left::skin-tone-2:")
    ("👈🏽" . ":point_left_tone3: :point_left::skin-tone-3:")
    ("👈🏾" . ":point_left_tone4: :point_left::skin-tone-4:")
    ("👈🏿" . ":point_left_tone5: :point_left::skin-tone-5:")
    ("👉" . ":point_right:")
    ("👉🏻" . ":point_right_tone1: :point_right::skin-tone-1:")
    ("👉🏼" . ":point_right_tone2: :point_right::skin-tone-2:")
    ("👉🏽" . ":point_right_tone3: :point_right::skin-tone-3:")
    ("👉🏾" . ":point_right_tone4: :point_right::skin-tone-4:")
    ("👉🏿" . ":point_right_tone5: :point_right::skin-tone-5:")
    ("👆" . ":point_up_2:")
    ("👆🏻" . ":point_up_2_tone1: :point_up_2::skin-tone-1:")
    ("👆🏼" . ":point_up_2_tone2: :point_up_2::skin-tone-2:")
    ("👆🏽" . ":point_up_2_tone3: :point_up_2::skin-tone-3:")
    ("👆🏾" . ":point_up_2_tone4: :point_up_2::skin-tone-4:")
    ("👆🏿" . ":point_up_2_tone5: :point_up_2::skin-tone-5:")
    ("👇" . ":point_down:")
    ("👇🏻" . ":point_down_tone1: :point_down::skin-tone-1:")
    ("👇🏼" . ":point_down_tone2: :point_down::skin-tone-2:")
    ("👇🏽" . ":point_down_tone3: :point_down::skin-tone-3:")
    ("👇🏾" . ":point_down_tone4: :point_down::skin-tone-4:")
    ("👇🏿" . ":point_down_tone5: :point_down::skin-tone-5:")
    ("\342\230\235\357\270\217" . ":point_up:")
    ("\342\230\235🏻" . ":point_up_tone1: :point_up::skin-tone-1:")
    ("\342\230\235🏼" . ":point_up_tone2: :point_up::skin-tone-2:")
    ("\342\230\235🏽" . ":point_up_tone3: :point_up::skin-tone-3:")
    ("\342\230\235🏾" . ":point_up_tone4: :point_up::skin-tone-4:")
    ("\342\230\235🏿" . ":point_up_tone5: :point_up::skin-tone-5:")
    ("\342\234\213" . ":raised_hand:")
    ("\342\234\213🏻" . ":raised_hand_tone1: :raised_hand::skin-tone-1:")
    ("\342\234\213🏼" . ":raised_hand_tone2: :raised_hand::skin-tone-2:")
    ("\342\234\213🏽" . ":raised_hand_tone3: :raised_hand::skin-tone-3:")
    ("\342\234\213🏾" . ":raised_hand_tone4: :raised_hand::skin-tone-4:")
    ("\342\234\213🏿" . ":raised_hand_tone5: :raised_hand::skin-tone-5:")
    ("🤚" . ":raised_back_of_hand:")
    ("🤚🏻" . ":raised_back_of_hand_tone1: :raised_back_of_hand::skin-tone-1:")
    ("🤚🏼" . ":raised_back_of_hand_tone2: :raised_back_of_hand::skin-tone-2:")
    ("🤚🏽" . ":raised_back_of_hand_tone3: :raised_back_of_hand::skin-tone-3:")
    ("🤚🏾" . ":raised_back_of_hand_tone4: :raised_back_of_hand::skin-tone-4:")
    ("🤚🏿" . ":raised_back_of_hand_tone5: :raised_back_of_hand::skin-tone-5:")
    ("🖐\357\270\217" . ":hand_splayed: :raised_hand_with_fingers_splayed:")
    ("🖐🏻" . ":hand_splayed_tone1: :raised_hand_with_fingers_splayed_tone1: :hand_splayed::skin-tone-1: :raised_hand_with_fingers_splayed::skin-tone-1:")
    ("🖐🏼" . ":hand_splayed_tone2: :raised_hand_with_fingers_splayed_tone2: :hand_splayed::skin-tone-2: :raised_hand_with_fingers_splayed::skin-tone-2:")
    ("🖐🏽" . ":hand_splayed_tone3: :raised_hand_with_fingers_splayed_tone3: :hand_splayed::skin-tone-3: :raised_hand_with_fingers_splayed::skin-tone-3:")
    ("🖐🏾" . ":hand_splayed_tone4: :raised_hand_with_fingers_splayed_tone4: :hand_splayed::skin-tone-4: :raised_hand_with_fingers_splayed::skin-tone-4:")
    ("🖐🏿" . ":hand_splayed_tone5: :raised_hand_with_fingers_splayed_tone5: :hand_splayed::skin-tone-5: :raised_hand_with_fingers_splayed::skin-tone-5:")
    ("🖖" . ":raised_hand_with_part_between_middle_and_ring_fingers: :vulcan_salute:")
    ("🖖🏻" . ":vulcan_tone1: :raised_hand_with_part_between_middle_and_ring_fingers_tone1: :vulcan::skin-tone-1: :raised_hand_with_part_between_middle_and_ring_fingers::skin-tone-1: :vulcan_salute::skin-tone-1:")
    ("🖖🏼" . ":vulcan_tone2: :raised_hand_with_part_between_middle_and_ring_fingers_tone2: :vulcan::skin-tone-2: :raised_hand_with_part_between_middle_and_ring_fingers::skin-tone-2: :vulcan_salute::skin-tone-2:")
    ("🖖🏽" . ":vulcan_tone3: :raised_hand_with_part_between_middle_and_ring_fingers_tone3: :vulcan::skin-tone-3: :raised_hand_with_part_between_middle_and_ring_fingers::skin-tone-3: :vulcan_salute::skin-tone-3:")
    ("🖖🏾" . ":vulcan_tone4: :raised_hand_with_part_between_middle_and_ring_fingers_tone4: :vulcan::skin-tone-4: :raised_hand_with_part_between_middle_and_ring_fingers::skin-tone-4: :vulcan_salute::skin-tone-4:")
    ("🖖🏿" . ":vulcan_tone5: :raised_hand_with_part_between_middle_and_ring_fingers_tone5: :vulcan::skin-tone-5: :raised_hand_with_part_between_middle_and_ring_fingers::skin-tone-5: :vulcan_salute::skin-tone-5:")
    ("👋" . ":wave: :waving_hand:")
    ("👋🏻" . ":wave_tone1: :wave::skin-tone-1: :waving_hand::skin-tone-1:")
    ("👋🏼" . ":wave_tone2: :wave::skin-tone-2: :waving_hand::skin-tone-2:")
    ("👋🏽" . ":wave_tone3: :wave::skin-tone-3: :waving_hand::skin-tone-3:")
    ("👋🏾" . ":wave_tone4: :wave::skin-tone-4: :waving_hand::skin-tone-4:")
    ("👋🏿" . ":wave_tone5: :wave::skin-tone-5: :waving_hand::skin-tone-5:")
    ("🤙" . ":call_me_hand:")
    ("🤙🏻" . ":call_me_tone1: :call_me_hand_tone1: :call_me::skin-tone-1: :call_me_hand::skin-tone-1:")
    ("🤙🏼" . ":call_me_tone2: :call_me_hand_tone2: :call_me::skin-tone-2: :call_me_hand::skin-tone-2:")
    ("🤙🏽" . ":call_me_tone3: :call_me_hand_tone3: :call_me::skin-tone-3: :call_me_hand::skin-tone-3:")
    ("🤙🏾" . ":call_me_tone4: :call_me_hand_tone4: :call_me::skin-tone-4: :call_me_hand::skin-tone-4:")
    ("🤙🏿" . ":call_me_tone5: :call_me_hand_tone5: :call_me::skin-tone-5: :call_me_hand::skin-tone-5:")
    ("🫲" . ":leftwards_hand:")
    ("🫲🏻" . ":leftwards_hand_tone1: :leftwards_hand_light_skin_tone: :leftwards_hand::skin-tone-1:")
    ("🫲🏼" . ":leftwards_hand_tone2: :leftwards_hand_medium_light_skin_tone: :leftwards_hand::skin-tone-2:")
    ("🫲🏽" . ":leftwards_hand_tone3: :leftwards_hand_medium_skin_tone: :leftwards_hand::skin-tone-3:")
    ("🫲🏾" . ":leftwards_hand_tone4: :leftwards_hand_medium_dark_skin_tone: :leftwards_hand::skin-tone-4:")
    ("🫲🏿" . ":leftwards_hand_tone5: :leftwards_hand_dark_skin_tone: :leftwards_hand::skin-tone-5:")
    ("🫱" . ":rightwards_hand:")
    ("🫱🏻" . ":rightwards_hand_tone1: :rightwards_hand_light_skin_tone: :rightwards_hand::skin-tone-1:")
    ("🫱🏼" . ":rightwards_hand_tone2: :rightwards_hand_medium_light_skin_tone: :rightwards_hand::skin-tone-2:")
    ("🫱🏽" . ":rightwards_hand_tone3: :rightwards_hand_medium_skin_tone: :rightwards_hand::skin-tone-3:")
    ("🫱🏾" . ":rightwards_hand_tone4: :rightwards_hand_medium_dark_skin_tone: :rightwards_hand::skin-tone-4:")
    ("🫱🏿" . ":rightwards_hand_tone5: :rightwards_hand_dark_skin_tone: :rightwards_hand::skin-tone-5:")
    ("💪" . ":muscle: :flexed_biceps:")
    ("💪🏻" . ":muscle_tone1: :muscle::skin-tone-1: :flexed_biceps::skin-tone-1:")
    ("💪🏼" . ":muscle_tone2: :muscle::skin-tone-2: :flexed_biceps::skin-tone-2:")
    ("💪🏽" . ":muscle_tone3: :muscle::skin-tone-3: :flexed_biceps::skin-tone-3:")
    ("💪🏾" . ":muscle_tone4: :muscle::skin-tone-4: :flexed_biceps::skin-tone-4:")
    ("💪🏿" . ":muscle_tone5: :muscle::skin-tone-5: :flexed_biceps::skin-tone-5:")
    ("🦾" . ":mechanical_arm:")
    ("🖕" . ":reversed_hand_with_middle_finger_extended:")
    ("🖕🏻" . ":middle_finger_tone1: :reversed_hand_with_middle_finger_extended_tone1: :middle_finger::skin-tone-1: :reversed_hand_with_middle_finger_extended::skin-tone-1:")
    ("🖕🏼" . ":middle_finger_tone2: :reversed_hand_with_middle_finger_extended_tone2: :middle_finger::skin-tone-2: :reversed_hand_with_middle_finger_extended::skin-tone-2:")
    ("🖕🏽" . ":middle_finger_tone3: :reversed_hand_with_middle_finger_extended_tone3: :middle_finger::skin-tone-3: :reversed_hand_with_middle_finger_extended::skin-tone-3:")
    ("🖕🏾" . ":middle_finger_tone4: :reversed_hand_with_middle_finger_extended_tone4: :middle_finger::skin-tone-4: :reversed_hand_with_middle_finger_extended::skin-tone-4:")
    ("🖕🏿" . ":middle_finger_tone5: :reversed_hand_with_middle_finger_extended_tone5: :middle_finger::skin-tone-5: :reversed_hand_with_middle_finger_extended::skin-tone-5:")
    ("\342\234\215\357\270\217" . ":writing_hand:")
    ("\342\234\215🏻" . ":writing_hand_tone1: :writing_hand::skin-tone-1:")
    ("\342\234\215🏼" . ":writing_hand_tone2: :writing_hand::skin-tone-2:")
    ("\342\234\215🏽" . ":writing_hand_tone3: :writing_hand::skin-tone-3:")
    ("\342\234\215🏾" . ":writing_hand_tone4: :writing_hand::skin-tone-4:")
    ("\342\234\215🏿" . ":writing_hand_tone5: :writing_hand::skin-tone-5:")
    ("🙏" . ":pray: :folded_hands:")
    ("🙏🏻" . ":pray_tone1: :pray::skin-tone-1: :folded_hands::skin-tone-1:")
    ("🙏🏼" . ":pray_tone2: :pray::skin-tone-2: :folded_hands::skin-tone-2:")
    ("🙏🏽" . ":pray_tone3: :pray::skin-tone-3: :folded_hands::skin-tone-3:")
    ("🙏🏾" . ":pray_tone4: :pray::skin-tone-4: :folded_hands::skin-tone-4:")
    ("🙏🏿" . ":pray_tone5: :pray::skin-tone-5: :folded_hands::skin-tone-5:")
    ("🫵" . ":index_pointing_at_the_viewer:")
    ("🫵🏻" . ":index_pointing_at_the_viewer_tone1: :index_pointing_at_the_viewer_light_skin_tone: :index_pointing_at_the_viewer::skin-tone-1:")
    ("🫵🏼" . ":index_pointing_at_the_viewer_tone2: :index_pointing_at_the_viewer_medium_light_skin_tone: :index_pointing_at_the_viewer::skin-tone-2:")
    ("🫵🏽" . ":index_pointing_at_the_viewer_tone3: :index_pointing_at_the_viewer_medium_skin_tone: :index_pointing_at_the_viewer::skin-tone-3:")
    ("🫵🏾" . ":index_pointing_at_the_viewer_tone4: :index_pointing_at_the_viewer_medium_dark_skin_tone: :index_pointing_at_the_viewer::skin-tone-4:")
    ("🫵🏿" . ":index_pointing_at_the_viewer_tone5: :index_pointing_at_the_viewer_dark_skin_tone: :index_pointing_at_the_viewer::skin-tone-5:")
    ("🦶" . ":foot:")
    ("🦶🏻" . ":foot_tone1: :foot_light_skin_tone: :foot::skin-tone-1:")
    ("🦶🏼" . ":foot_tone2: :foot_medium_light_skin_tone: :foot::skin-tone-2:")
    ("🦶🏽" . ":foot_tone3: :foot_medium_skin_tone: :foot::skin-tone-3:")
    ("🦶🏾" . ":foot_tone4: :foot_medium_dark_skin_tone: :foot::skin-tone-4:")
    ("🦶🏿" . ":foot_tone5: :foot_dark_skin_tone: :foot::skin-tone-5:")
    ("🦵" . ":leg:")
    ("🦵🏻" . ":leg_tone1: :leg_light_skin_tone: :leg::skin-tone-1:")
    ("🦵🏼" . ":leg_tone2: :leg_medium_light_skin_tone: :leg::skin-tone-2:")
    ("🦵🏽" . ":leg_tone3: :leg_medium_skin_tone: :leg::skin-tone-3:")
    ("🦵🏾" . ":leg_tone4: :leg_medium_dark_skin_tone: :leg::skin-tone-4:")
    ("🦵🏿" . ":leg_tone5: :leg_dark_skin_tone: :leg::skin-tone-5:")
    ("🦿" . ":mechanical_leg:")
    ("💄" . ":lipstick:")
    ("💋" . ":kiss_mark:")
    ("👄" . ":lips: :mouth:")
    ("🫦" . ":biting_lip:")
    ("🦷" . ":tooth:")
    ("👅" . ":tongue:")
    ("👂" . ":ear:")
    ("👂🏻" . ":ear_tone1: :ear::skin-tone-1:")
    ("👂🏼" . ":ear_tone2: :ear::skin-tone-2:")
    ("👂🏽" . ":ear_tone3: :ear::skin-tone-3:")
    ("👂🏾" . ":ear_tone4: :ear::skin-tone-4:")
    ("👂🏿" . ":ear_tone5: :ear::skin-tone-5:")
    ("🦻" . ":ear_with_hearing_aid:")
    ("🦻🏻" . ":ear_with_hearing_aid_tone1: :ear_with_hearing_aid_light_skin_tone: :ear_with_hearing_aid::skin-tone-1:")
    ("🦻🏼" . ":ear_with_hearing_aid_tone2: :ear_with_hearing_aid_medium_light_skin_tone: :ear_with_hearing_aid::skin-tone-2:")
    ("🦻🏽" . ":ear_with_hearing_aid_tone3: :ear_with_hearing_aid_medium_skin_tone: :ear_with_hearing_aid::skin-tone-3:")
    ("🦻🏾" . ":ear_with_hearing_aid_tone4: :ear_with_hearing_aid_medium_dark_skin_tone: :ear_with_hearing_aid::skin-tone-4:")
    ("🦻🏿" . ":ear_with_hearing_aid_tone5: :ear_with_hearing_aid_dark_skin_tone: :ear_with_hearing_aid::skin-tone-5:")
    ("👃" . ":nose:")
    ("👃🏻" . ":nose_tone1: :nose::skin-tone-1:")
    ("👃🏼" . ":nose_tone2: :nose::skin-tone-2:")
    ("👃🏽" . ":nose_tone3: :nose::skin-tone-3:")
    ("👃🏾" . ":nose_tone4: :nose::skin-tone-4:")
    ("👃🏿" . ":nose_tone5: :nose::skin-tone-5:")
    ("👣" . ":footprints:")
    ("👁\357\270\217" . ":eye:")
    ("👀" . ":eyes:")
    ("🫀" . ":anatomical_heart:")
    ("🫁" . ":lungs:")
    ("🧠" . ":brain:")
    ("🗣\357\270\217" . ":speaking_head_in_silhouette:")
    ("👤" . ":bust_in_silhouette:")
    ("👥" . ":busts_in_silhouette:")
    ("🫂" . ":people_hugging:")
    ("👶" . ":baby:")
    ("👶🏻" . ":baby_tone1: :baby::skin-tone-1:")
    ("👶🏼" . ":baby_tone2: :baby::skin-tone-2:")
    ("👶🏽" . ":baby_tone3: :baby::skin-tone-3:")
    ("👶🏾" . ":baby_tone4: :baby::skin-tone-4:")
    ("👶🏿" . ":baby_tone5: :baby::skin-tone-5:")
    ("🧒" . ":child:")
    ("🧒🏻" . ":child_tone1: :child_light_skin_tone: :child::skin-tone-1:")
    ("🧒🏼" . ":child_tone2: :child_medium_light_skin_tone: :child::skin-tone-2:")
    ("🧒🏽" . ":child_tone3: :child_medium_skin_tone: :child::skin-tone-3:")
    ("🧒🏾" . ":child_tone4: :child_medium_dark_skin_tone: :child::skin-tone-4:")
    ("🧒🏿" . ":child_tone5: :child_dark_skin_tone: :child::skin-tone-5:")
    ("👧" . ":girl:")
    ("👧🏻" . ":girl_tone1: :girl::skin-tone-1:")
    ("👧🏼" . ":girl_tone2: :girl::skin-tone-2:")
    ("👧🏽" . ":girl_tone3: :girl::skin-tone-3:")
    ("👧🏾" . ":girl_tone4: :girl::skin-tone-4:")
    ("👧🏿" . ":girl_tone5: :girl::skin-tone-5:")
    ("👦" . ":boy:")
    ("👦🏻" . ":boy_tone1: :boy::skin-tone-1:")
    ("👦🏼" . ":boy_tone2: :boy::skin-tone-2:")
    ("👦🏽" . ":boy_tone3: :boy::skin-tone-3:")
    ("👦🏾" . ":boy_tone4: :boy::skin-tone-4:")
    ("👦🏿" . ":boy_tone5: :boy::skin-tone-5:")
    ("🧑" . ":adult: :person:")
    ("🧑🏻" . ":adult_tone1: :adult_light_skin_tone: :adult::skin-tone-1: :person::skin-tone-1:")
    ("🧑🏼" . ":adult_tone2: :adult_medium_light_skin_tone: :adult::skin-tone-2: :person::skin-tone-2:")
    ("🧑🏽" . ":adult_tone3: :adult_medium_skin_tone: :adult::skin-tone-3: :person::skin-tone-3:")
    ("🧑🏾" . ":adult_tone4: :adult_medium_dark_skin_tone: :adult::skin-tone-4: :person::skin-tone-4:")
    ("🧑🏿" . ":adult_tone5: :adult_dark_skin_tone: :adult::skin-tone-5: :person::skin-tone-5:")
    ("👩" . ":woman:")
    ("👩🏻" . ":woman_tone1: :woman::skin-tone-1:")
    ("👩🏼" . ":woman_tone2: :woman::skin-tone-2:")
    ("👩🏽" . ":woman_tone3: :woman::skin-tone-3:")
    ("👩🏾" . ":woman_tone4: :woman::skin-tone-4:")
    ("👩🏿" . ":woman_tone5: :woman::skin-tone-5:")
    ("👨" . ":man:")
    ("👨🏻" . ":man_tone1: :man::skin-tone-1:")
    ("👨🏼" . ":man_tone2: :man::skin-tone-2:")
    ("👨🏽" . ":man_tone3: :man::skin-tone-3:")
    ("👨🏾" . ":man_tone4: :man::skin-tone-4:")
    ("👨🏿" . ":man_tone5: :man::skin-tone-5:")
    ("🧑\342\200\215🦱" . ":person_curly_hair:")
    ("🧑🏻\342\200\215🦱" . ":person_tone1_curly_hair: :person_light_skin_tone_curly_hair: :person_curly_hair::skin-tone-1:")
    ("🧑🏼\342\200\215🦱" . ":person_tone2_curly_hair: :person_medium_light_skin_tone_curly_hair: :person_curly_hair::skin-tone-2:")
    ("🧑🏽\342\200\215🦱" . ":person_tone3_curly_hair: :person_medium_skin_tone_curly_hair: :person_curly_hair::skin-tone-3:")
    ("🧑🏾\342\200\215🦱" . ":person_tone4_curly_hair: :person_medium_dark_skin_tone_curly_hair: :person_curly_hair::skin-tone-4:")
    ("🧑🏿\342\200\215🦱" . ":person_tone5_curly_hair: :person_dark_skin_tone_curly_hair: :person_curly_hair::skin-tone-5:")
    ("👩\342\200\215🦱" . ":woman_curly_haired:")
    ("👩🏻\342\200\215🦱" . ":woman_curly_haired_tone1: :woman_curly_haired_light_skin_tone: :woman_curly_haired::skin-tone-1:")
    ("👩🏼\342\200\215🦱" . ":woman_curly_haired_tone2: :woman_curly_haired_medium_light_skin_tone: :woman_curly_haired::skin-tone-2:")
    ("👩🏽\342\200\215🦱" . ":woman_curly_haired_tone3: :woman_curly_haired_medium_skin_tone: :woman_curly_haired::skin-tone-3:")
    ("👩🏾\342\200\215🦱" . ":woman_curly_haired_tone4: :woman_curly_haired_medium_dark_skin_tone: :woman_curly_haired::skin-tone-4:")
    ("👩🏿\342\200\215🦱" . ":woman_curly_haired_tone5: :woman_curly_haired_dark_skin_tone: :woman_curly_haired::skin-tone-5:")
    ("👨\342\200\215🦱" . ":man_curly_haired:")
    ("👨🏻\342\200\215🦱" . ":man_curly_haired_tone1: :man_curly_haired_light_skin_tone: :man_curly_haired::skin-tone-1:")
    ("👨🏼\342\200\215🦱" . ":man_curly_haired_tone2: :man_curly_haired_medium_light_skin_tone: :man_curly_haired::skin-tone-2:")
    ("👨🏽\342\200\215🦱" . ":man_curly_haired_tone3: :man_curly_haired_medium_skin_tone: :man_curly_haired::skin-tone-3:")
    ("👨🏾\342\200\215🦱" . ":man_curly_haired_tone4: :man_curly_haired_medium_dark_skin_tone: :man_curly_haired::skin-tone-4:")
    ("👨🏿\342\200\215🦱" . ":man_curly_haired_tone5: :man_curly_haired_dark_skin_tone: :man_curly_haired::skin-tone-5:")
    ("🧑\342\200\215🦰" . ":person_red_hair:")
    ("🧑🏻\342\200\215🦰" . ":person_tone1_red_hair: :person_light_skin_tone_red_hair: :person_red_hair::skin-tone-1:")
    ("🧑🏼\342\200\215🦰" . ":person_tone2_red_hair: :person_medium_light_skin_tone_red_hair: :person_red_hair::skin-tone-2:")
    ("🧑🏽\342\200\215🦰" . ":person_tone3_red_hair: :person_medium_skin_tone_red_hair: :person_red_hair::skin-tone-3:")
    ("🧑🏾\342\200\215🦰" . ":person_tone4_red_hair: :person_medium_dark_skin_tone_red_hair: :person_red_hair::skin-tone-4:")
    ("🧑🏿\342\200\215🦰" . ":person_tone5_red_hair: :person_dark_skin_tone_red_hair: :person_red_hair::skin-tone-5:")
    ("👩\342\200\215🦰" . ":woman_red_haired:")
    ("👩🏻\342\200\215🦰" . ":woman_red_haired_tone1: :woman_red_haired_light_skin_tone: :woman_red_haired::skin-tone-1:")
    ("👩🏼\342\200\215🦰" . ":woman_red_haired_tone2: :woman_red_haired_medium_light_skin_tone: :woman_red_haired::skin-tone-2:")
    ("👩🏽\342\200\215🦰" . ":woman_red_haired_tone3: :woman_red_haired_medium_skin_tone: :woman_red_haired::skin-tone-3:")
    ("👩🏾\342\200\215🦰" . ":woman_red_haired_tone4: :woman_red_haired_medium_dark_skin_tone: :woman_red_haired::skin-tone-4:")
    ("👩🏿\342\200\215🦰" . ":woman_red_haired_tone5: :woman_red_haired_dark_skin_tone: :woman_red_haired::skin-tone-5:")
    ("👨\342\200\215🦰" . ":man_red_haired:")
    ("👨🏻\342\200\215🦰" . ":man_red_haired_tone1: :man_red_haired_light_skin_tone: :man_red_haired::skin-tone-1: :man_red_hair::skin-tone-1:")
    ("👨🏼\342\200\215🦰" . ":man_red_haired_tone2: :man_red_haired_medium_light_skin_tone: :man_red_haired::skin-tone-2: :man_red_hair::skin-tone-2:")
    ("👨🏽\342\200\215🦰" . ":man_red_haired_tone3: :man_red_haired_medium_skin_tone: :man_red_haired::skin-tone-3: :man_red_hair::skin-tone-3:")
    ("👨🏾\342\200\215🦰" . ":man_red_haired_tone4: :man_red_haired_medium_dark_skin_tone: :man_red_haired::skin-tone-4: :man_red_hair::skin-tone-4:")
    ("👨🏿\342\200\215🦰" . ":man_red_haired_tone5: :man_red_haired_dark_skin_tone: :man_red_haired::skin-tone-5: :man_red_hair::skin-tone-5:")
    ("👱" . ":blond_haired_person: :person_with_blond_hair:")
    ("👱🏻" . ":blond_haired_person_tone1: :person_with_blond_hair_tone1: :blond_haired_person::skin-tone-1: :person_with_blond_hair::skin-tone-1:")
    ("👱🏼" . ":blond_haired_person_tone2: :person_with_blond_hair_tone2: :blond_haired_person::skin-tone-2: :person_with_blond_hair::skin-tone-2:")
    ("👱🏽" . ":blond_haired_person_tone3: :person_with_blond_hair_tone3: :blond_haired_person::skin-tone-3: :person_with_blond_hair::skin-tone-3:")
    ("👱🏾" . ":blond_haired_person_tone4: :person_with_blond_hair_tone4: :blond_haired_person::skin-tone-4: :person_with_blond_hair::skin-tone-4:")
    ("👱🏿" . ":blond_haired_person_tone5: :person_with_blond_hair_tone5: :blond_haired_person::skin-tone-5: :person_with_blond_hair::skin-tone-5:")
    ("👱\342\200\215\342\231\200\357\270\217" . ":blond_haired_woman:")
    ("👱🏻\342\200\215\342\231\200\357\270\217" . ":blond_haired_woman_tone1: :blond_haired_woman_light_skin_tone: :blond_haired_woman::skin-tone-1:")
    ("👱🏼\342\200\215\342\231\200\357\270\217" . ":blond_haired_woman_tone2: :blond_haired_woman_medium_light_skin_tone: :blond_haired_woman::skin-tone-2:")
    ("👱🏽\342\200\215\342\231\200\357\270\217" . ":blond_haired_woman_tone3: :blond_haired_woman_medium_skin_tone: :blond_haired_woman::skin-tone-3:")
    ("👱🏾\342\200\215\342\231\200\357\270\217" . ":blond_haired_woman_tone4: :blond_haired_woman_medium_dark_skin_tone: :blond_haired_woman::skin-tone-4:")
    ("👱🏿\342\200\215\342\231\200\357\270\217" . ":blond_haired_woman_tone5: :blond_haired_woman_dark_skin_tone: :blond_haired_woman::skin-tone-5:")
    ("👱\342\200\215\342\231\202\357\270\217" . ":blond_haired_man:")
    ("👱🏻\342\200\215\342\231\202\357\270\217" . ":blond_haired_man_tone1: :blond_haired_man_light_skin_tone: :blond_haired_man::skin-tone-1:")
    ("👱🏼\342\200\215\342\231\202\357\270\217" . ":blond_haired_man_tone2: :blond_haired_man_medium_light_skin_tone: :blond_haired_man::skin-tone-2:")
    ("👱🏽\342\200\215\342\231\202\357\270\217" . ":blond_haired_man_tone3: :blond_haired_man_medium_skin_tone: :blond_haired_man::skin-tone-3:")
    ("👱🏾\342\200\215\342\231\202\357\270\217" . ":blond_haired_man_tone4: :blond_haired_man_medium_dark_skin_tone: :blond_haired_man::skin-tone-4:")
    ("👱🏿\342\200\215\342\231\202\357\270\217" . ":blond_haired_man_tone5: :blond_haired_man_dark_skin_tone: :blond_haired_man::skin-tone-5:")
    ("🧑\342\200\215🦳" . ":person_white_hair:")
    ("🧑🏻\342\200\215🦳" . ":person_tone1_white_hair: :person_light_skin_tone_white_hair: :person_white_hair::skin-tone-1:")
    ("🧑🏼\342\200\215🦳" . ":person_tone2_white_hair: :person_medium_light_skin_tone_white_hair: :person_white_hair::skin-tone-2:")
    ("🧑🏽\342\200\215🦳" . ":person_tone3_white_hair: :person_medium_skin_tone_white_hair: :person_white_hair::skin-tone-3:")
    ("🧑🏾\342\200\215🦳" . ":person_tone4_white_hair: :person_medium_dark_skin_tone_white_hair: :person_white_hair::skin-tone-4:")
    ("🧑🏿\342\200\215🦳" . ":person_tone5_white_hair: :person_dark_skin_tone_white_hair: :person_white_hair::skin-tone-5:")
    ("👩\342\200\215🦳" . ":woman_white_haired:")
    ("👩🏻\342\200\215🦳" . ":woman_white_haired_tone1: :woman_white_haired_light_skin_tone: :woman_white_haired::skin-tone-1:")
    ("👩🏼\342\200\215🦳" . ":woman_white_haired_tone2: :woman_white_haired_medium_light_skin_tone: :woman_white_haired::skin-tone-2:")
    ("👩🏽\342\200\215🦳" . ":woman_white_haired_tone3: :woman_white_haired_medium_skin_tone: :woman_white_haired::skin-tone-3:")
    ("👩🏾\342\200\215🦳" . ":woman_white_haired_tone4: :woman_white_haired_medium_dark_skin_tone: :woman_white_haired::skin-tone-4:")
    ("👩🏿\342\200\215🦳" . ":woman_white_haired_tone5: :woman_white_haired_dark_skin_tone: :woman_white_haired::skin-tone-5:")
    ("👨\342\200\215🦳" . ":man_white_haired:")
    ("👨🏻\342\200\215🦳" . ":man_white_haired_tone1: :man_white_haired_light_skin_tone: :man_white_haired::skin-tone-1:")
    ("👨🏼\342\200\215🦳" . ":man_white_haired_tone2: :man_white_haired_medium_light_skin_tone: :man_white_haired::skin-tone-2:")
    ("👨🏽\342\200\215🦳" . ":man_white_haired_tone3: :man_white_haired_medium_skin_tone: :man_white_haired::skin-tone-3:")
    ("👨🏾\342\200\215🦳" . ":man_white_haired_tone4: :man_white_haired_medium_dark_skin_tone: :man_white_haired::skin-tone-4:")
    ("👨🏿\342\200\215🦳" . ":man_white_haired_tone5: :man_white_haired_dark_skin_tone: :man_white_haired::skin-tone-5:")
    ("🧑\342\200\215🦲" . ":person_bald:")
    ("🧑🏻\342\200\215🦲" . ":person_tone1_bald: :person_light_skin_tone_bald: :person_bald::skin-tone-1:")
    ("🧑🏼\342\200\215🦲" . ":person_tone2_bald: :person_medium_light_skin_tone_bald: :person_bald::skin-tone-2:")
    ("🧑🏽\342\200\215🦲" . ":person_tone3_bald: :person_medium_skin_tone_bald: :person_bald::skin-tone-3:")
    ("🧑🏾\342\200\215🦲" . ":person_tone4_bald: :person_medium_dark_skin_tone_bald: :person_bald::skin-tone-4:")
    ("🧑🏿\342\200\215🦲" . ":person_tone5_bald: :person_dark_skin_tone_bald: :person_bald::skin-tone-5:")
    ("👩\342\200\215🦲" . ":woman_bald:")
    ("👩🏻\342\200\215🦲" . ":woman_bald_tone1: :woman_bald_light_skin_tone: :woman_bald::skin-tone-1:")
    ("👩🏼\342\200\215🦲" . ":woman_bald_tone2: :woman_bald_medium_light_skin_tone: :woman_bald::skin-tone-2:")
    ("👩🏽\342\200\215🦲" . ":woman_bald_tone3: :woman_bald_medium_skin_tone: :woman_bald::skin-tone-3:")
    ("👩🏾\342\200\215🦲" . ":woman_bald_tone4: :woman_bald_medium_dark_skin_tone: :woman_bald::skin-tone-4:")
    ("👩🏿\342\200\215🦲" . ":woman_bald_tone5: :woman_bald_dark_skin_tone: :woman_bald::skin-tone-5:")
    ("👨\342\200\215🦲" . ":man_bald:")
    ("👨🏻\342\200\215🦲" . ":man_bald_tone1: :man_bald_light_skin_tone: :man_bald::skin-tone-1:")
    ("👨🏼\342\200\215🦲" . ":man_bald_tone2: :man_bald_medium_light_skin_tone: :man_bald::skin-tone-2:")
    ("👨🏽\342\200\215🦲" . ":man_bald_tone3: :man_bald_medium_skin_tone: :man_bald::skin-tone-3:")
    ("👨🏾\342\200\215🦲" . ":man_bald_tone4: :man_bald_medium_dark_skin_tone: :man_bald::skin-tone-4:")
    ("👨🏿\342\200\215🦲" . ":man_bald_tone5: :man_bald_dark_skin_tone: :man_bald::skin-tone-5:")
    ("🧔" . ":bearded_person: :person_beard:")
    ("🧔🏻" . ":bearded_person_tone1: :bearded_person_light_skin_tone: :bearded_person::skin-tone-1: :person_beard::skin-tone-1:")
    ("🧔🏼" . ":bearded_person_tone2: :bearded_person_medium_light_skin_tone: :bearded_person::skin-tone-2: :person_beard::skin-tone-2:")
    ("🧔🏽" . ":bearded_person_tone3: :bearded_person_medium_skin_tone: :bearded_person::skin-tone-3: :person_beard::skin-tone-3:")
    ("🧔🏾" . ":bearded_person_tone4: :bearded_person_medium_dark_skin_tone: :bearded_person::skin-tone-4: :person_beard::skin-tone-4:")
    ("🧔🏿" . ":bearded_person_tone5: :bearded_person_dark_skin_tone: :bearded_person::skin-tone-5: :person_beard::skin-tone-5:")
    ("🧔\342\200\215\342\231\200\357\270\217" . ":woman_beard:")
    ("🧔🏻\342\200\215\342\231\200\357\270\217" . ":woman_tone1_beard: :woman_light_skin_tone_beard: :woman_beard::skin-tone-1:")
    ("🧔🏼\342\200\215\342\231\200\357\270\217" . ":woman_tone2_beard: :woman_medium_light_skin_tone_beard: :woman_beard::skin-tone-2:")
    ("🧔🏽\342\200\215\342\231\200\357\270\217" . ":woman_tone3_beard: :woman_medium_skin_tone_beard: :woman_beard::skin-tone-3:")
    ("🧔🏾\342\200\215\342\231\200\357\270\217" . ":woman_tone4_beard: :woman_medium_dark_skin_tone_beard: :woman_beard::skin-tone-4:")
    ("🧔🏿\342\200\215\342\231\200\357\270\217" . ":woman_tone5_beard: :woman_dark_skin_tone_beard: :woman_beard::skin-tone-5:")
    ("🧔\342\200\215\342\231\202\357\270\217" . ":man_beard:")
    ("🧔🏻\342\200\215\342\231\202\357\270\217" . ":man_tone1_beard: :man_light_skin_tone_beard: :man_beard::skin-tone-1:")
    ("🧔🏼\342\200\215\342\231\202\357\270\217" . ":man_tone2_beard: :man_medium_light_skin_tone_beard: :man_beard::skin-tone-2:")
    ("🧔🏽\342\200\215\342\231\202\357\270\217" . ":man_tone3_beard: :man_medium_skin_tone_beard: :man_beard::skin-tone-3:")
    ("🧔🏾\342\200\215\342\231\202\357\270\217" . ":man_tone4_beard: :man_medium_dark_skin_tone_beard: :man_beard::skin-tone-4:")
    ("🧔🏿\342\200\215\342\231\202\357\270\217" . ":man_tone5_beard: :man_dark_skin_tone_beard: :man_beard::skin-tone-5:")
    ("🧓" . ":older_adult: :older_person:")
    ("🧓🏻" . ":older_adult_tone1: :older_adult_light_skin_tone: :older_adult::skin-tone-1: :older_person::skin-tone-1:")
    ("🧓🏼" . ":older_adult_tone2: :older_adult_medium_light_skin_tone: :older_adult::skin-tone-2: :older_person::skin-tone-2:")
    ("🧓🏽" . ":older_adult_tone3: :older_adult_medium_skin_tone: :older_adult::skin-tone-3: :older_person::skin-tone-3:")
    ("🧓🏾" . ":older_adult_tone4: :older_adult_medium_dark_skin_tone: :older_adult::skin-tone-4: :older_person::skin-tone-4:")
    ("🧓🏿" . ":older_adult_tone5: :older_adult_dark_skin_tone: :older_adult::skin-tone-5: :older_person::skin-tone-5:")
    ("👵" . ":older_woman: :grandma: :old_woman:")
    ("👵🏻" . ":older_woman_tone1: :grandma_tone1: :older_woman::skin-tone-1: :grandma::skin-tone-1: :old_woman::skin-tone-1:")
    ("👵🏼" . ":older_woman_tone2: :grandma_tone2: :older_woman::skin-tone-2: :grandma::skin-tone-2: :old_woman::skin-tone-2:")
    ("👵🏽" . ":older_woman_tone3: :grandma_tone3: :older_woman::skin-tone-3: :grandma::skin-tone-3: :old_woman::skin-tone-3:")
    ("👵🏾" . ":older_woman_tone4: :grandma_tone4: :older_woman::skin-tone-4: :grandma::skin-tone-4: :old_woman::skin-tone-4:")
    ("👵🏿" . ":older_woman_tone5: :grandma_tone5: :older_woman::skin-tone-5: :grandma::skin-tone-5: :old_woman::skin-tone-5:")
    ("👴" . ":older_man: :old_man:")
    ("👴🏻" . ":older_man_tone1: :older_man::skin-tone-1: :old_man::skin-tone-1:")
    ("👴🏼" . ":older_man_tone2: :older_man::skin-tone-2: :old_man::skin-tone-2:")
    ("👴🏽" . ":older_man_tone3: :older_man::skin-tone-3: :old_man::skin-tone-3:")
    ("👴🏾" . ":older_man_tone4: :older_man::skin-tone-4: :old_man::skin-tone-4:")
    ("👴🏿" . ":older_man_tone5: :older_man::skin-tone-5: :old_man::skin-tone-5:")
    ("👲" . ":man_with_chinese_cap: :man_with_gua_pi_mao:")
    ("👲🏻" . ":man_with_chinese_cap_tone1: :man_with_gua_pi_mao_tone1: :man_with_chinese_cap::skin-tone-1: :man_with_gua_pi_mao::skin-tone-1:")
    ("👲🏼" . ":man_with_chinese_cap_tone2: :man_with_gua_pi_mao_tone2: :man_with_chinese_cap::skin-tone-2: :man_with_gua_pi_mao::skin-tone-2:")
    ("👲🏽" . ":man_with_chinese_cap_tone3: :man_with_gua_pi_mao_tone3: :man_with_chinese_cap::skin-tone-3: :man_with_gua_pi_mao::skin-tone-3:")
    ("👲🏾" . ":man_with_chinese_cap_tone4: :man_with_gua_pi_mao_tone4: :man_with_chinese_cap::skin-tone-4: :man_with_gua_pi_mao::skin-tone-4:")
    ("👲🏿" . ":man_with_chinese_cap_tone5: :man_with_gua_pi_mao_tone5: :man_with_chinese_cap::skin-tone-5: :man_with_gua_pi_mao::skin-tone-5:")
    ("👳" . ":person_wearing_turban: :man_with_turban:")
    ("👳🏻" . ":person_wearing_turban_tone1: :man_with_turban_tone1: :person_wearing_turban::skin-tone-1: :man_with_turban::skin-tone-1:")
    ("👳🏼" . ":person_wearing_turban_tone2: :man_with_turban_tone2: :person_wearing_turban::skin-tone-2: :man_with_turban::skin-tone-2:")
    ("👳🏽" . ":person_wearing_turban_tone3: :man_with_turban_tone3: :person_wearing_turban::skin-tone-3: :man_with_turban::skin-tone-3:")
    ("👳🏾" . ":person_wearing_turban_tone4: :man_with_turban_tone4: :person_wearing_turban::skin-tone-4: :man_with_turban::skin-tone-4:")
    ("👳🏿" . ":person_wearing_turban_tone5: :man_with_turban_tone5: :person_wearing_turban::skin-tone-5: :man_with_turban::skin-tone-5:")
    ("👳\342\200\215\342\231\200\357\270\217" . ":woman_wearing_turban:")
    ("👳🏻\342\200\215\342\231\200\357\270\217" . ":woman_wearing_turban_tone1: :woman_wearing_turban_light_skin_tone: :woman_wearing_turban::skin-tone-1:")
    ("👳🏼\342\200\215\342\231\200\357\270\217" . ":woman_wearing_turban_tone2: :woman_wearing_turban_medium_light_skin_tone: :woman_wearing_turban::skin-tone-2:")
    ("👳🏽\342\200\215\342\231\200\357\270\217" . ":woman_wearing_turban_tone3: :woman_wearing_turban_medium_skin_tone: :woman_wearing_turban::skin-tone-3:")
    ("👳🏾\342\200\215\342\231\200\357\270\217" . ":woman_wearing_turban_tone4: :woman_wearing_turban_medium_dark_skin_tone: :woman_wearing_turban::skin-tone-4:")
    ("👳🏿\342\200\215\342\231\200\357\270\217" . ":woman_wearing_turban_tone5: :woman_wearing_turban_dark_skin_tone: :woman_wearing_turban::skin-tone-5:")
    ("👳\342\200\215\342\231\202\357\270\217" . ":man_wearing_turban:")
    ("👳🏻\342\200\215\342\231\202\357\270\217" . ":man_wearing_turban_tone1: :man_wearing_turban_light_skin_tone: :man_wearing_turban::skin-tone-1:")
    ("👳🏼\342\200\215\342\231\202\357\270\217" . ":man_wearing_turban_tone2: :man_wearing_turban_medium_light_skin_tone: :man_wearing_turban::skin-tone-2:")
    ("👳🏽\342\200\215\342\231\202\357\270\217" . ":man_wearing_turban_tone3: :man_wearing_turban_medium_skin_tone: :man_wearing_turban::skin-tone-3:")
    ("👳🏾\342\200\215\342\231\202\357\270\217" . ":man_wearing_turban_tone4: :man_wearing_turban_medium_dark_skin_tone: :man_wearing_turban::skin-tone-4:")
    ("👳🏿\342\200\215\342\231\202\357\270\217" . ":man_wearing_turban_tone5: :man_wearing_turban_dark_skin_tone: :man_wearing_turban::skin-tone-5:")
    ("🧕" . ":woman_with_headscarf:")
    ("🧕🏻" . ":woman_with_headscarf_tone1: :woman_with_headscarf_light_skin_tone: :woman_with_headscarf::skin-tone-1:")
    ("🧕🏼" . ":woman_with_headscarf_tone2: :woman_with_headscarf_medium_light_skin_tone: :woman_with_headscarf::skin-tone-2:")
    ("🧕🏽" . ":woman_with_headscarf_tone3: :woman_with_headscarf_medium_skin_tone: :woman_with_headscarf::skin-tone-3:")
    ("🧕🏾" . ":woman_with_headscarf_tone4: :woman_with_headscarf_medium_dark_skin_tone: :woman_with_headscarf::skin-tone-4:")
    ("🧕🏿" . ":woman_with_headscarf_tone5: :woman_with_headscarf_dark_skin_tone: :woman_with_headscarf::skin-tone-5:")
    ("👮" . ":police_officer: :cop:")
    ("👮🏻" . ":police_officer_tone1: :cop_tone1: :police_officer::skin-tone-1: :cop::skin-tone-1:")
    ("👮🏼" . ":police_officer_tone2: :cop_tone2: :police_officer::skin-tone-2: :cop::skin-tone-2:")
    ("👮🏽" . ":police_officer_tone3: :cop_tone3: :police_officer::skin-tone-3: :cop::skin-tone-3:")
    ("👮🏾" . ":police_officer_tone4: :cop_tone4: :police_officer::skin-tone-4: :cop::skin-tone-4:")
    ("👮🏿" . ":police_officer_tone5: :cop_tone5: :police_officer::skin-tone-5: :cop::skin-tone-5:")
    ("👮\342\200\215\342\231\200\357\270\217" . ":woman_police_officer:")
    ("👮🏻\342\200\215\342\231\200\357\270\217" . ":woman_police_officer_tone1: :woman_police_officer_light_skin_tone: :woman_police_officer::skin-tone-1:")
    ("👮🏼\342\200\215\342\231\200\357\270\217" . ":woman_police_officer_tone2: :woman_police_officer_medium_light_skin_tone: :woman_police_officer::skin-tone-2:")
    ("👮🏽\342\200\215\342\231\200\357\270\217" . ":woman_police_officer_tone3: :woman_police_officer_medium_skin_tone: :woman_police_officer::skin-tone-3:")
    ("👮🏾\342\200\215\342\231\200\357\270\217" . ":woman_police_officer_tone4: :woman_police_officer_medium_dark_skin_tone: :woman_police_officer::skin-tone-4:")
    ("👮🏿\342\200\215\342\231\200\357\270\217" . ":woman_police_officer_tone5: :woman_police_officer_dark_skin_tone: :woman_police_officer::skin-tone-5:")
    ("👮\342\200\215\342\231\202\357\270\217" . ":man_police_officer:")
    ("👮🏻\342\200\215\342\231\202\357\270\217" . ":man_police_officer_tone1: :man_police_officer_light_skin_tone: :man_police_officer::skin-tone-1:")
    ("👮🏼\342\200\215\342\231\202\357\270\217" . ":man_police_officer_tone2: :man_police_officer_medium_light_skin_tone: :man_police_officer::skin-tone-2:")
    ("👮🏽\342\200\215\342\231\202\357\270\217" . ":man_police_officer_tone3: :man_police_officer_medium_skin_tone: :man_police_officer::skin-tone-3:")
    ("👮🏾\342\200\215\342\231\202\357\270\217" . ":man_police_officer_tone4: :man_police_officer_medium_dark_skin_tone: :man_police_officer::skin-tone-4:")
    ("👮🏿\342\200\215\342\231\202\357\270\217" . ":man_police_officer_tone5: :man_police_officer_dark_skin_tone: :man_police_officer::skin-tone-5:")
    ("👷" . ":construction_worker:")
    ("👷🏻" . ":construction_worker_tone1: :construction_worker::skin-tone-1:")
    ("👷🏼" . ":construction_worker_tone2: :construction_worker::skin-tone-2:")
    ("👷🏽" . ":construction_worker_tone3: :construction_worker::skin-tone-3:")
    ("👷🏾" . ":construction_worker_tone4: :construction_worker::skin-tone-4:")
    ("👷🏿" . ":construction_worker_tone5: :construction_worker::skin-tone-5:")
    ("👷\342\200\215\342\231\200\357\270\217" . ":woman_construction_worker:")
    ("👷🏻\342\200\215\342\231\200\357\270\217" . ":woman_construction_worker_tone1: :woman_construction_worker_light_skin_tone: :woman_construction_worker::skin-tone-1:")
    ("👷🏼\342\200\215\342\231\200\357\270\217" . ":woman_construction_worker_tone2: :woman_construction_worker_medium_light_skin_tone: :woman_construction_worker::skin-tone-2:")
    ("👷🏽\342\200\215\342\231\200\357\270\217" . ":woman_construction_worker_tone3: :woman_construction_worker_medium_skin_tone: :woman_construction_worker::skin-tone-3:")
    ("👷🏾\342\200\215\342\231\200\357\270\217" . ":woman_construction_worker_tone4: :woman_construction_worker_medium_dark_skin_tone: :woman_construction_worker::skin-tone-4:")
    ("👷🏿\342\200\215\342\231\200\357\270\217" . ":woman_construction_worker_tone5: :woman_construction_worker_dark_skin_tone: :woman_construction_worker::skin-tone-5:")
    ("👷\342\200\215\342\231\202\357\270\217" . ":man_construction_worker:")
    ("👷🏻\342\200\215\342\231\202\357\270\217" . ":man_construction_worker_tone1: :man_construction_worker_light_skin_tone: :man_construction_worker::skin-tone-1:")
    ("👷🏼\342\200\215\342\231\202\357\270\217" . ":man_construction_worker_tone2: :man_construction_worker_medium_light_skin_tone: :man_construction_worker::skin-tone-2:")
    ("👷🏽\342\200\215\342\231\202\357\270\217" . ":man_construction_worker_tone3: :man_construction_worker_medium_skin_tone: :man_construction_worker::skin-tone-3:")
    ("👷🏾\342\200\215\342\231\202\357\270\217" . ":man_construction_worker_tone4: :man_construction_worker_medium_dark_skin_tone: :man_construction_worker::skin-tone-4:")
    ("👷🏿\342\200\215\342\231\202\357\270\217" . ":man_construction_worker_tone5: :man_construction_worker_dark_skin_tone: :man_construction_worker::skin-tone-5:")
    ("💂" . ":guardsman:")
    ("💂🏻" . ":guard_tone1: :guardsman_tone1: :guard::skin-tone-1: :guardsman::skin-tone-1:")
    ("💂🏼" . ":guard_tone2: :guardsman_tone2: :guard::skin-tone-2: :guardsman::skin-tone-2:")
    ("💂🏽" . ":guard_tone3: :guardsman_tone3: :guard::skin-tone-3: :guardsman::skin-tone-3:")
    ("💂🏾" . ":guard_tone4: :guardsman_tone4: :guard::skin-tone-4: :guardsman::skin-tone-4:")
    ("💂🏿" . ":guard_tone5: :guardsman_tone5: :guard::skin-tone-5: :guardsman::skin-tone-5:")
    ("💂\342\200\215\342\231\200\357\270\217" . ":woman_guard:")
    ("💂🏻\342\200\215\342\231\200\357\270\217" . ":woman_guard_tone1: :woman_guard_light_skin_tone: :woman_guard::skin-tone-1:")
    ("💂🏼\342\200\215\342\231\200\357\270\217" . ":woman_guard_tone2: :woman_guard_medium_light_skin_tone: :woman_guard::skin-tone-2:")
    ("💂🏽\342\200\215\342\231\200\357\270\217" . ":woman_guard_tone3: :woman_guard_medium_skin_tone: :woman_guard::skin-tone-3:")
    ("💂🏾\342\200\215\342\231\200\357\270\217" . ":woman_guard_tone4: :woman_guard_medium_dark_skin_tone: :woman_guard::skin-tone-4:")
    ("💂🏿\342\200\215\342\231\200\357\270\217" . ":woman_guard_tone5: :woman_guard_dark_skin_tone: :woman_guard::skin-tone-5:")
    ("💂\342\200\215\342\231\202\357\270\217" . ":man_guard:")
    ("💂🏻\342\200\215\342\231\202\357\270\217" . ":man_guard_tone1: :man_guard_light_skin_tone: :man_guard::skin-tone-1:")
    ("💂🏼\342\200\215\342\231\202\357\270\217" . ":man_guard_tone2: :man_guard_medium_light_skin_tone: :man_guard::skin-tone-2:")
    ("💂🏽\342\200\215\342\231\202\357\270\217" . ":man_guard_tone3: :man_guard_medium_skin_tone: :man_guard::skin-tone-3:")
    ("💂🏾\342\200\215\342\231\202\357\270\217" . ":man_guard_tone4: :man_guard_medium_dark_skin_tone: :man_guard::skin-tone-4:")
    ("💂🏿\342\200\215\342\231\202\357\270\217" . ":man_guard_tone5: :man_guard_dark_skin_tone: :man_guard::skin-tone-5:")
    ("🕵\357\270\217" . ":detective: :sleuth_or_spy:")
    ("🕵🏻" . ":detective_tone1: :sleuth_or_spy_tone1: :detective::skin-tone-1: :sleuth_or_spy::skin-tone-1:")
    ("🕵🏼" . ":detective_tone2: :sleuth_or_spy_tone2: :detective::skin-tone-2: :sleuth_or_spy::skin-tone-2:")
    ("🕵🏽" . ":detective_tone3: :sleuth_or_spy_tone3: :detective::skin-tone-3: :sleuth_or_spy::skin-tone-3:")
    ("🕵🏾" . ":detective_tone4: :sleuth_or_spy_tone4: :detective::skin-tone-4: :sleuth_or_spy::skin-tone-4:")
    ("🕵🏿" . ":detective_tone5: :sleuth_or_spy_tone5: :detective::skin-tone-5: :sleuth_or_spy::skin-tone-5:")
    ("🕵\357\270\217\342\200\215\342\231\200\357\270\217" . ":woman_detective:")
    ("🕵🏻\342\200\215\342\231\200\357\270\217" . ":woman_detective_tone1: :woman_detective_light_skin_tone: :woman_detective::skin-tone-1:")
    ("🕵🏼\342\200\215\342\231\200\357\270\217" . ":woman_detective_tone2: :woman_detective_medium_light_skin_tone: :woman_detective::skin-tone-2:")
    ("🕵🏽\342\200\215\342\231\200\357\270\217" . ":woman_detective_tone3: :woman_detective_medium_skin_tone: :woman_detective::skin-tone-3:")
    ("🕵🏾\342\200\215\342\231\200\357\270\217" . ":woman_detective_tone4: :woman_detective_medium_dark_skin_tone: :woman_detective::skin-tone-4:")
    ("🕵🏿\342\200\215\342\231\200\357\270\217" . ":woman_detective_tone5: :woman_detective_dark_skin_tone: :woman_detective::skin-tone-5:")
    ("🕵\357\270\217\342\200\215\342\231\202\357\270\217" . ":man_detective:")
    ("🕵🏻\342\200\215\342\231\202\357\270\217" . ":man_detective_tone1: :man_detective_light_skin_tone: :man_detective::skin-tone-1:")
    ("🕵🏼\342\200\215\342\231\202\357\270\217" . ":man_detective_tone2: :man_detective_medium_light_skin_tone: :man_detective::skin-tone-2:")
    ("🕵🏽\342\200\215\342\231\202\357\270\217" . ":man_detective_tone3: :man_detective_medium_skin_tone: :man_detective::skin-tone-3:")
    ("🕵🏾\342\200\215\342\231\202\357\270\217" . ":man_detective_tone4: :man_detective_medium_dark_skin_tone: :man_detective::skin-tone-4:")
    ("🕵🏿\342\200\215\342\231\202\357\270\217" . ":man_detective_tone5: :man_detective_dark_skin_tone: :man_detective::skin-tone-5:")
    ("🧑\342\200\215\342\232\225\357\270\217" . ":health_worker:")
    ("🧑🏻\342\200\215\342\232\225\357\270\217" . ":health_worker_tone1: :health_worker_light_skin_tone: :health_worker::skin-tone-1:")
    ("🧑🏼\342\200\215\342\232\225\357\270\217" . ":health_worker_tone2: :health_worker_medium_light_skin_tone: :health_worker::skin-tone-2:")
    ("🧑🏽\342\200\215\342\232\225\357\270\217" . ":health_worker_tone3: :health_worker_medium_skin_tone: :health_worker::skin-tone-3:")
    ("🧑🏾\342\200\215\342\232\225\357\270\217" . ":health_worker_tone4: :health_worker_medium_dark_skin_tone: :health_worker::skin-tone-4:")
    ("🧑🏿\342\200\215\342\232\225\357\270\217" . ":health_worker_tone5: :health_worker_dark_skin_tone: :health_worker::skin-tone-5:")
    ("👩\342\200\215\342\232\225\357\270\217" . ":woman_health_worker:")
    ("👩🏻\342\200\215\342\232\225\357\270\217" . ":woman_health_worker_tone1: :woman_health_worker_light_skin_tone: :woman_health_worker::skin-tone-1:")
    ("👩🏼\342\200\215\342\232\225\357\270\217" . ":woman_health_worker_tone2: :woman_health_worker_medium_light_skin_tone: :woman_health_worker::skin-tone-2:")
    ("👩🏽\342\200\215\342\232\225\357\270\217" . ":woman_health_worker_tone3: :woman_health_worker_medium_skin_tone: :woman_health_worker::skin-tone-3:")
    ("👩🏾\342\200\215\342\232\225\357\270\217" . ":woman_health_worker_tone4: :woman_health_worker_medium_dark_skin_tone: :woman_health_worker::skin-tone-4:")
    ("👩🏿\342\200\215\342\232\225\357\270\217" . ":woman_health_worker_tone5: :woman_health_worker_dark_skin_tone: :woman_health_worker::skin-tone-5:")
    ("👨\342\200\215\342\232\225\357\270\217" . ":man_health_worker:")
    ("👨🏻\342\200\215\342\232\225\357\270\217" . ":man_health_worker_tone1: :man_health_worker_light_skin_tone: :man_health_worker::skin-tone-1:")
    ("👨🏼\342\200\215\342\232\225\357\270\217" . ":man_health_worker_tone2: :man_health_worker_medium_light_skin_tone: :man_health_worker::skin-tone-2:")
    ("👨🏽\342\200\215\342\232\225\357\270\217" . ":man_health_worker_tone3: :man_health_worker_medium_skin_tone: :man_health_worker::skin-tone-3:")
    ("👨🏾\342\200\215\342\232\225\357\270\217" . ":man_health_worker_tone4: :man_health_worker_medium_dark_skin_tone: :man_health_worker::skin-tone-4:")
    ("👨🏿\342\200\215\342\232\225\357\270\217" . ":man_health_worker_tone5: :man_health_worker_dark_skin_tone: :man_health_worker::skin-tone-5:")
    ("🧑\342\200\215🌾" . ":farmer:")
    ("🧑🏻\342\200\215🌾" . ":farmer_tone1: :farmer_light_skin_tone: :farmer::skin-tone-1:")
    ("🧑🏼\342\200\215🌾" . ":farmer_tone2: :farmer_medium_light_skin_tone: :farmer::skin-tone-2:")
    ("🧑🏽\342\200\215🌾" . ":farmer_tone3: :farmer_medium_skin_tone: :farmer::skin-tone-3:")
    ("🧑🏾\342\200\215🌾" . ":farmer_tone4: :farmer_medium_dark_skin_tone: :farmer::skin-tone-4:")
    ("🧑🏿\342\200\215🌾" . ":farmer_tone5: :farmer_dark_skin_tone: :farmer::skin-tone-5:")
    ("👩\342\200\215🌾" . ":woman_farmer:")
    ("👩🏻\342\200\215🌾" . ":woman_farmer_tone1: :woman_farmer_light_skin_tone: :woman_farmer::skin-tone-1:")
    ("👩🏼\342\200\215🌾" . ":woman_farmer_tone2: :woman_farmer_medium_light_skin_tone: :woman_farmer::skin-tone-2:")
    ("👩🏽\342\200\215🌾" . ":woman_farmer_tone3: :woman_farmer_medium_skin_tone: :woman_farmer::skin-tone-3:")
    ("👩🏾\342\200\215🌾" . ":woman_farmer_tone4: :woman_farmer_medium_dark_skin_tone: :woman_farmer::skin-tone-4:")
    ("👩🏿\342\200\215🌾" . ":woman_farmer_tone5: :woman_farmer_dark_skin_tone: :woman_farmer::skin-tone-5:")
    ("👨\342\200\215🌾" . ":man_farmer:")
    ("👨🏻\342\200\215🌾" . ":man_farmer_tone1: :man_farmer_light_skin_tone: :man_farmer::skin-tone-1:")
    ("👨🏼\342\200\215🌾" . ":man_farmer_tone2: :man_farmer_medium_light_skin_tone: :man_farmer::skin-tone-2:")
    ("👨🏽\342\200\215🌾" . ":man_farmer_tone3: :man_farmer_medium_skin_tone: :man_farmer::skin-tone-3:")
    ("👨🏾\342\200\215🌾" . ":man_farmer_tone4: :man_farmer_medium_dark_skin_tone: :man_farmer::skin-tone-4:")
    ("👨🏿\342\200\215🌾" . ":man_farmer_tone5: :man_farmer_dark_skin_tone: :man_farmer::skin-tone-5:")
    ("🧑\342\200\215🍳" . ":cook:")
    ("🧑🏻\342\200\215🍳" . ":cook_tone1: :cook_light_skin_tone: :cook::skin-tone-1:")
    ("🧑🏼\342\200\215🍳" . ":cook_tone2: :cook_medium_light_skin_tone: :cook::skin-tone-2:")
    ("🧑🏽\342\200\215🍳" . ":cook_tone3: :cook_medium_skin_tone: :cook::skin-tone-3:")
    ("🧑🏾\342\200\215🍳" . ":cook_tone4: :cook_medium_dark_skin_tone: :cook::skin-tone-4:")
    ("🧑🏿\342\200\215🍳" . ":cook_tone5: :cook_dark_skin_tone: :cook::skin-tone-5:")
    ("👩\342\200\215🍳" . ":woman_cook:")
    ("👩🏻\342\200\215🍳" . ":woman_cook_tone1: :woman_cook_light_skin_tone: :woman_cook::skin-tone-1:")
    ("👩🏼\342\200\215🍳" . ":woman_cook_tone2: :woman_cook_medium_light_skin_tone: :woman_cook::skin-tone-2:")
    ("👩🏽\342\200\215🍳" . ":woman_cook_tone3: :woman_cook_medium_skin_tone: :woman_cook::skin-tone-3:")
    ("👩🏾\342\200\215🍳" . ":woman_cook_tone4: :woman_cook_medium_dark_skin_tone: :woman_cook::skin-tone-4:")
    ("👩🏿\342\200\215🍳" . ":woman_cook_tone5: :woman_cook_dark_skin_tone: :woman_cook::skin-tone-5:")
    ("👨\342\200\215🍳" . ":man_cook:")
    ("👨🏻\342\200\215🍳" . ":man_cook_tone1: :man_cook_light_skin_tone: :man_cook::skin-tone-1:")
    ("👨🏼\342\200\215🍳" . ":man_cook_tone2: :man_cook_medium_light_skin_tone: :man_cook::skin-tone-2:")
    ("👨🏽\342\200\215🍳" . ":man_cook_tone3: :man_cook_medium_skin_tone: :man_cook::skin-tone-3:")
    ("👨🏾\342\200\215🍳" . ":man_cook_tone4: :man_cook_medium_dark_skin_tone: :man_cook::skin-tone-4:")
    ("👨🏿\342\200\215🍳" . ":man_cook_tone5: :man_cook_dark_skin_tone: :man_cook::skin-tone-5:")
    ("🧑\342\200\215🎓" . ":student:")
    ("🧑🏻\342\200\215🎓" . ":student_tone1: :student_light_skin_tone: :student::skin-tone-1:")
    ("🧑🏼\342\200\215🎓" . ":student_tone2: :student_medium_light_skin_tone: :student::skin-tone-2:")
    ("🧑🏽\342\200\215🎓" . ":student_tone3: :student_medium_skin_tone: :student::skin-tone-3:")
    ("🧑🏾\342\200\215🎓" . ":student_tone4: :student_medium_dark_skin_tone: :student::skin-tone-4:")
    ("🧑🏿\342\200\215🎓" . ":student_tone5: :student_dark_skin_tone: :student::skin-tone-5:")
    ("👩\342\200\215🎓" . ":woman_student:")
    ("👩🏻\342\200\215🎓" . ":woman_student_tone1: :woman_student_light_skin_tone: :woman_student::skin-tone-1:")
    ("👩🏼\342\200\215🎓" . ":woman_student_tone2: :woman_student_medium_light_skin_tone: :woman_student::skin-tone-2:")
    ("👩🏽\342\200\215🎓" . ":woman_student_tone3: :woman_student_medium_skin_tone: :woman_student::skin-tone-3:")
    ("👩🏾\342\200\215🎓" . ":woman_student_tone4: :woman_student_medium_dark_skin_tone: :woman_student::skin-tone-4:")
    ("👩🏿\342\200\215🎓" . ":woman_student_tone5: :woman_student_dark_skin_tone: :woman_student::skin-tone-5:")
    ("👨\342\200\215🎓" . ":man_student:")
    ("👨🏻\342\200\215🎓" . ":man_student_tone1: :man_student_light_skin_tone: :man_student::skin-tone-1:")
    ("👨🏼\342\200\215🎓" . ":man_student_tone2: :man_student_medium_light_skin_tone: :man_student::skin-tone-2:")
    ("👨🏽\342\200\215🎓" . ":man_student_tone3: :man_student_medium_skin_tone: :man_student::skin-tone-3:")
    ("👨🏾\342\200\215🎓" . ":man_student_tone4: :man_student_medium_dark_skin_tone: :man_student::skin-tone-4:")
    ("👨🏿\342\200\215🎓" . ":man_student_tone5: :man_student_dark_skin_tone: :man_student::skin-tone-5:")
    ("🧑\342\200\215🎤" . ":singer:")
    ("🧑🏻\342\200\215🎤" . ":singer_tone1: :singer_light_skin_tone: :singer::skin-tone-1:")
    ("🧑🏼\342\200\215🎤" . ":singer_tone2: :singer_medium_light_skin_tone: :singer::skin-tone-2:")
    ("🧑🏽\342\200\215🎤" . ":singer_tone3: :singer_medium_skin_tone: :singer::skin-tone-3:")
    ("🧑🏾\342\200\215🎤" . ":singer_tone4: :singer_medium_dark_skin_tone: :singer::skin-tone-4:")
    ("🧑🏿\342\200\215🎤" . ":singer_tone5: :singer_dark_skin_tone: :singer::skin-tone-5:")
    ("👩\342\200\215🎤" . ":woman_singer:")
    ("👩🏻\342\200\215🎤" . ":woman_singer_tone1: :woman_singer_light_skin_tone: :woman_singer::skin-tone-1:")
    ("👩🏼\342\200\215🎤" . ":woman_singer_tone2: :woman_singer_medium_light_skin_tone: :woman_singer::skin-tone-2:")
    ("👩🏽\342\200\215🎤" . ":woman_singer_tone3: :woman_singer_medium_skin_tone: :woman_singer::skin-tone-3:")
    ("👩🏾\342\200\215🎤" . ":woman_singer_tone4: :woman_singer_medium_dark_skin_tone: :woman_singer::skin-tone-4:")
    ("👩🏿\342\200\215🎤" . ":woman_singer_tone5: :woman_singer_dark_skin_tone: :woman_singer::skin-tone-5:")
    ("👨\342\200\215🎤" . ":man_singer:")
    ("👨🏻\342\200\215🎤" . ":man_singer_tone1: :man_singer_light_skin_tone: :man_singer::skin-tone-1:")
    ("👨🏼\342\200\215🎤" . ":man_singer_tone2: :man_singer_medium_light_skin_tone: :man_singer::skin-tone-2:")
    ("👨🏽\342\200\215🎤" . ":man_singer_tone3: :man_singer_medium_skin_tone: :man_singer::skin-tone-3:")
    ("👨🏾\342\200\215🎤" . ":man_singer_tone4: :man_singer_medium_dark_skin_tone: :man_singer::skin-tone-4:")
    ("👨🏿\342\200\215🎤" . ":man_singer_tone5: :man_singer_dark_skin_tone: :man_singer::skin-tone-5:")
    ("🧑\342\200\215🏫" . ":teacher:")
    ("🧑🏻\342\200\215🏫" . ":teacher_tone1: :teacher_light_skin_tone: :teacher::skin-tone-1:")
    ("🧑🏼\342\200\215🏫" . ":teacher_tone2: :teacher_medium_light_skin_tone: :teacher::skin-tone-2:")
    ("🧑🏽\342\200\215🏫" . ":teacher_tone3: :teacher_medium_skin_tone: :teacher::skin-tone-3:")
    ("🧑🏾\342\200\215🏫" . ":teacher_tone4: :teacher_medium_dark_skin_tone: :teacher::skin-tone-4:")
    ("🧑🏿\342\200\215🏫" . ":teacher_tone5: :teacher_dark_skin_tone: :teacher::skin-tone-5:")
    ("👩\342\200\215🏫" . ":woman_teacher:")
    ("👩🏻\342\200\215🏫" . ":woman_teacher_tone1: :woman_teacher_light_skin_tone: :woman_teacher::skin-tone-1:")
    ("👩🏼\342\200\215🏫" . ":woman_teacher_tone2: :woman_teacher_medium_light_skin_tone: :woman_teacher::skin-tone-2:")
    ("👩🏽\342\200\215🏫" . ":woman_teacher_tone3: :woman_teacher_medium_skin_tone: :woman_teacher::skin-tone-3:")
    ("👩🏾\342\200\215🏫" . ":woman_teacher_tone4: :woman_teacher_medium_dark_skin_tone: :woman_teacher::skin-tone-4:")
    ("👩🏿\342\200\215🏫" . ":woman_teacher_tone5: :woman_teacher_dark_skin_tone: :woman_teacher::skin-tone-5:")
    ("👨\342\200\215🏫" . ":man_teacher:")
    ("👨🏻\342\200\215🏫" . ":man_teacher_tone1: :man_teacher_light_skin_tone: :man_teacher::skin-tone-1:")
    ("👨🏼\342\200\215🏫" . ":man_teacher_tone2: :man_teacher_medium_light_skin_tone: :man_teacher::skin-tone-2:")
    ("👨🏽\342\200\215🏫" . ":man_teacher_tone3: :man_teacher_medium_skin_tone: :man_teacher::skin-tone-3:")
    ("👨🏾\342\200\215🏫" . ":man_teacher_tone4: :man_teacher_medium_dark_skin_tone: :man_teacher::skin-tone-4:")
    ("👨🏿\342\200\215🏫" . ":man_teacher_tone5: :man_teacher_dark_skin_tone: :man_teacher::skin-tone-5:")
    ("🧑\342\200\215🏭" . ":factory_worker:")
    ("🧑🏻\342\200\215🏭" . ":factory_worker_tone1: :factory_worker_light_skin_tone: :factory_worker::skin-tone-1:")
    ("🧑🏼\342\200\215🏭" . ":factory_worker_tone2: :factory_worker_medium_light_skin_tone: :factory_worker::skin-tone-2:")
    ("🧑🏽\342\200\215🏭" . ":factory_worker_tone3: :factory_worker_medium_skin_tone: :factory_worker::skin-tone-3:")
    ("🧑🏾\342\200\215🏭" . ":factory_worker_tone4: :factory_worker_medium_dark_skin_tone: :factory_worker::skin-tone-4:")
    ("🧑🏿\342\200\215🏭" . ":factory_worker_tone5: :factory_worker_dark_skin_tone: :factory_worker::skin-tone-5:")
    ("👩\342\200\215🏭" . ":woman_factory_worker:")
    ("👩🏻\342\200\215🏭" . ":woman_factory_worker_tone1: :woman_factory_worker_light_skin_tone: :woman_factory_worker::skin-tone-1:")
    ("👩🏼\342\200\215🏭" . ":woman_factory_worker_tone2: :woman_factory_worker_medium_light_skin_tone: :woman_factory_worker::skin-tone-2:")
    ("👩🏽\342\200\215🏭" . ":woman_factory_worker_tone3: :woman_factory_worker_medium_skin_tone: :woman_factory_worker::skin-tone-3:")
    ("👩🏾\342\200\215🏭" . ":woman_factory_worker_tone4: :woman_factory_worker_medium_dark_skin_tone: :woman_factory_worker::skin-tone-4:")
    ("👩🏿\342\200\215🏭" . ":woman_factory_worker_tone5: :woman_factory_worker_dark_skin_tone: :woman_factory_worker::skin-tone-5:")
    ("👨\342\200\215🏭" . ":man_factory_worker:")
    ("👨🏻\342\200\215🏭" . ":man_factory_worker_tone1: :man_factory_worker_light_skin_tone: :man_factory_worker::skin-tone-1:")
    ("👨🏼\342\200\215🏭" . ":man_factory_worker_tone2: :man_factory_worker_medium_light_skin_tone: :man_factory_worker::skin-tone-2:")
    ("👨🏽\342\200\215🏭" . ":man_factory_worker_tone3: :man_factory_worker_medium_skin_tone: :man_factory_worker::skin-tone-3:")
    ("👨🏾\342\200\215🏭" . ":man_factory_worker_tone4: :man_factory_worker_medium_dark_skin_tone: :man_factory_worker::skin-tone-4:")
    ("👨🏿\342\200\215🏭" . ":man_factory_worker_tone5: :man_factory_worker_dark_skin_tone: :man_factory_worker::skin-tone-5:")
    ("🧑\342\200\215💻" . ":technologist:")
    ("🧑🏻\342\200\215💻" . ":technologist_tone1: :technologist_light_skin_tone: :technologist::skin-tone-1:")
    ("🧑🏼\342\200\215💻" . ":technologist_tone2: :technologist_medium_light_skin_tone: :technologist::skin-tone-2:")
    ("🧑🏽\342\200\215💻" . ":technologist_tone3: :technologist_medium_skin_tone: :technologist::skin-tone-3:")
    ("🧑🏾\342\200\215💻" . ":technologist_tone4: :technologist_medium_dark_skin_tone: :technologist::skin-tone-4:")
    ("🧑🏿\342\200\215💻" . ":technologist_tone5: :technologist_dark_skin_tone: :technologist::skin-tone-5:")
    ("👩\342\200\215💻" . ":woman_technologist:")
    ("👩🏻\342\200\215💻" . ":woman_technologist_tone1: :woman_technologist_light_skin_tone: :woman_technologist::skin-tone-1:")
    ("👩🏼\342\200\215💻" . ":woman_technologist_tone2: :woman_technologist_medium_light_skin_tone: :woman_technologist::skin-tone-2:")
    ("👩🏽\342\200\215💻" . ":woman_technologist_tone3: :woman_technologist_medium_skin_tone: :woman_technologist::skin-tone-3:")
    ("👩🏾\342\200\215💻" . ":woman_technologist_tone4: :woman_technologist_medium_dark_skin_tone: :woman_technologist::skin-tone-4:")
    ("👩🏿\342\200\215💻" . ":woman_technologist_tone5: :woman_technologist_dark_skin_tone: :woman_technologist::skin-tone-5:")
    ("👨\342\200\215💻" . ":man_technologist:")
    ("👨🏻\342\200\215💻" . ":man_technologist_tone1: :man_technologist_light_skin_tone: :man_technologist::skin-tone-1:")
    ("👨🏼\342\200\215💻" . ":man_technologist_tone2: :man_technologist_medium_light_skin_tone: :man_technologist::skin-tone-2:")
    ("👨🏽\342\200\215💻" . ":man_technologist_tone3: :man_technologist_medium_skin_tone: :man_technologist::skin-tone-3:")
    ("👨🏾\342\200\215💻" . ":man_technologist_tone4: :man_technologist_medium_dark_skin_tone: :man_technologist::skin-tone-4:")
    ("👨🏿\342\200\215💻" . ":man_technologist_tone5: :man_technologist_dark_skin_tone: :man_technologist::skin-tone-5:")
    ("🧑\342\200\215💼" . ":office_worker:")
    ("🧑🏻\342\200\215💼" . ":office_worker_tone1: :office_worker_light_skin_tone: :office_worker::skin-tone-1:")
    ("🧑🏼\342\200\215💼" . ":office_worker_tone2: :office_worker_medium_light_skin_tone: :office_worker::skin-tone-2:")
    ("🧑🏽\342\200\215💼" . ":office_worker_tone3: :office_worker_medium_skin_tone: :office_worker::skin-tone-3:")
    ("🧑🏾\342\200\215💼" . ":office_worker_tone4: :office_worker_medium_dark_skin_tone: :office_worker::skin-tone-4:")
    ("🧑🏿\342\200\215💼" . ":office_worker_tone5: :office_worker_dark_skin_tone: :office_worker::skin-tone-5:")
    ("👩\342\200\215💼" . ":woman_office_worker:")
    ("👩🏻\342\200\215💼" . ":woman_office_worker_tone1: :woman_office_worker_light_skin_tone: :woman_office_worker::skin-tone-1:")
    ("👩🏼\342\200\215💼" . ":woman_office_worker_tone2: :woman_office_worker_medium_light_skin_tone: :woman_office_worker::skin-tone-2:")
    ("👩🏽\342\200\215💼" . ":woman_office_worker_tone3: :woman_office_worker_medium_skin_tone: :woman_office_worker::skin-tone-3:")
    ("👩🏾\342\200\215💼" . ":woman_office_worker_tone4: :woman_office_worker_medium_dark_skin_tone: :woman_office_worker::skin-tone-4:")
    ("👩🏿\342\200\215💼" . ":woman_office_worker_tone5: :woman_office_worker_dark_skin_tone: :woman_office_worker::skin-tone-5:")
    ("👨\342\200\215💼" . ":man_office_worker:")
    ("👨🏻\342\200\215💼" . ":man_office_worker_tone1: :man_office_worker_light_skin_tone: :man_office_worker::skin-tone-1:")
    ("👨🏼\342\200\215💼" . ":man_office_worker_tone2: :man_office_worker_medium_light_skin_tone: :man_office_worker::skin-tone-2:")
    ("👨🏽\342\200\215💼" . ":man_office_worker_tone3: :man_office_worker_medium_skin_tone: :man_office_worker::skin-tone-3:")
    ("👨🏾\342\200\215💼" . ":man_office_worker_tone4: :man_office_worker_medium_dark_skin_tone: :man_office_worker::skin-tone-4:")
    ("👨🏿\342\200\215💼" . ":man_office_worker_tone5: :man_office_worker_dark_skin_tone: :man_office_worker::skin-tone-5:")
    ("🧑\342\200\215🔧" . ":mechanic:")
    ("🧑🏻\342\200\215🔧" . ":mechanic_tone1: :mechanic_light_skin_tone: :mechanic::skin-tone-1:")
    ("🧑🏼\342\200\215🔧" . ":mechanic_tone2: :mechanic_medium_light_skin_tone: :mechanic::skin-tone-2:")
    ("🧑🏽\342\200\215🔧" . ":mechanic_tone3: :mechanic_medium_skin_tone: :mechanic::skin-tone-3:")
    ("🧑🏾\342\200\215🔧" . ":mechanic_tone4: :mechanic_medium_dark_skin_tone: :mechanic::skin-tone-4:")
    ("🧑🏿\342\200\215🔧" . ":mechanic_tone5: :mechanic_dark_skin_tone: :mechanic::skin-tone-5:")
    ("👩\342\200\215🔧" . ":woman_mechanic:")
    ("👩🏻\342\200\215🔧" . ":woman_mechanic_tone1: :woman_mechanic_light_skin_tone: :woman_mechanic::skin-tone-1:")
    ("👩🏼\342\200\215🔧" . ":woman_mechanic_tone2: :woman_mechanic_medium_light_skin_tone: :woman_mechanic::skin-tone-2:")
    ("👩🏽\342\200\215🔧" . ":woman_mechanic_tone3: :woman_mechanic_medium_skin_tone: :woman_mechanic::skin-tone-3:")
    ("👩🏾\342\200\215🔧" . ":woman_mechanic_tone4: :woman_mechanic_medium_dark_skin_tone: :woman_mechanic::skin-tone-4:")
    ("👩🏿\342\200\215🔧" . ":woman_mechanic_tone5: :woman_mechanic_dark_skin_tone: :woman_mechanic::skin-tone-5:")
    ("👨\342\200\215🔧" . ":man_mechanic:")
    ("👨🏻\342\200\215🔧" . ":man_mechanic_tone1: :man_mechanic_light_skin_tone: :man_mechanic::skin-tone-1:")
    ("👨🏼\342\200\215🔧" . ":man_mechanic_tone2: :man_mechanic_medium_light_skin_tone: :man_mechanic::skin-tone-2:")
    ("👨🏽\342\200\215🔧" . ":man_mechanic_tone3: :man_mechanic_medium_skin_tone: :man_mechanic::skin-tone-3:")
    ("👨🏾\342\200\215🔧" . ":man_mechanic_tone4: :man_mechanic_medium_dark_skin_tone: :man_mechanic::skin-tone-4:")
    ("👨🏿\342\200\215🔧" . ":man_mechanic_tone5: :man_mechanic_dark_skin_tone: :man_mechanic::skin-tone-5:")
    ("🧑\342\200\215🔬" . ":scientist:")
    ("🧑🏻\342\200\215🔬" . ":scientist_tone1: :scientist_light_skin_tone: :scientist::skin-tone-1:")
    ("🧑🏼\342\200\215🔬" . ":scientist_tone2: :scientist_medium_light_skin_tone: :scientist::skin-tone-2:")
    ("🧑🏽\342\200\215🔬" . ":scientist_tone3: :scientist_medium_skin_tone: :scientist::skin-tone-3:")
    ("🧑🏾\342\200\215🔬" . ":scientist_tone4: :scientist_medium_dark_skin_tone: :scientist::skin-tone-4:")
    ("🧑🏿\342\200\215🔬" . ":scientist_tone5: :scientist_dark_skin_tone: :scientist::skin-tone-5:")
    ("👩\342\200\215🔬" . ":woman_scientist:")
    ("👩🏻\342\200\215🔬" . ":woman_scientist_tone1: :woman_scientist_light_skin_tone: :woman_scientist::skin-tone-1:")
    ("👩🏼\342\200\215🔬" . ":woman_scientist_tone2: :woman_scientist_medium_light_skin_tone: :woman_scientist::skin-tone-2:")
    ("👩🏽\342\200\215🔬" . ":woman_scientist_tone3: :woman_scientist_medium_skin_tone: :woman_scientist::skin-tone-3:")
    ("👩🏾\342\200\215🔬" . ":woman_scientist_tone4: :woman_scientist_medium_dark_skin_tone: :woman_scientist::skin-tone-4:")
    ("👩🏿\342\200\215🔬" . ":woman_scientist_tone5: :woman_scientist_dark_skin_tone: :woman_scientist::skin-tone-5:")
    ("👨\342\200\215🔬" . ":man_scientist:")
    ("👨🏻\342\200\215🔬" . ":man_scientist_tone1: :man_scientist_light_skin_tone: :man_scientist::skin-tone-1:")
    ("👨🏼\342\200\215🔬" . ":man_scientist_tone2: :man_scientist_medium_light_skin_tone: :man_scientist::skin-tone-2:")
    ("👨🏽\342\200\215🔬" . ":man_scientist_tone3: :man_scientist_medium_skin_tone: :man_scientist::skin-tone-3:")
    ("👨🏾\342\200\215🔬" . ":man_scientist_tone4: :man_scientist_medium_dark_skin_tone: :man_scientist::skin-tone-4:")
    ("👨🏿\342\200\215🔬" . ":man_scientist_tone5: :man_scientist_dark_skin_tone: :man_scientist::skin-tone-5:")
    ("🧑\342\200\215🎨" . ":artist:")
    ("🧑🏻\342\200\215🎨" . ":artist_tone1: :artist_light_skin_tone: :artist::skin-tone-1:")
    ("🧑🏼\342\200\215🎨" . ":artist_tone2: :artist_medium_light_skin_tone: :artist::skin-tone-2:")
    ("🧑🏽\342\200\215🎨" . ":artist_tone3: :artist_medium_skin_tone: :artist::skin-tone-3:")
    ("🧑🏾\342\200\215🎨" . ":artist_tone4: :artist_medium_dark_skin_tone: :artist::skin-tone-4:")
    ("🧑🏿\342\200\215🎨" . ":artist_tone5: :artist_dark_skin_tone: :artist::skin-tone-5:")
    ("👩\342\200\215🎨" . ":woman_artist:")
    ("👩🏻\342\200\215🎨" . ":woman_artist_tone1: :woman_artist_light_skin_tone: :woman_artist::skin-tone-1:")
    ("👩🏼\342\200\215🎨" . ":woman_artist_tone2: :woman_artist_medium_light_skin_tone: :woman_artist::skin-tone-2:")
    ("👩🏽\342\200\215🎨" . ":woman_artist_tone3: :woman_artist_medium_skin_tone: :woman_artist::skin-tone-3:")
    ("👩🏾\342\200\215🎨" . ":woman_artist_tone4: :woman_artist_medium_dark_skin_tone: :woman_artist::skin-tone-4:")
    ("👩🏿\342\200\215🎨" . ":woman_artist_tone5: :woman_artist_dark_skin_tone: :woman_artist::skin-tone-5:")
    ("👨\342\200\215🎨" . ":man_artist:")
    ("👨🏻\342\200\215🎨" . ":man_artist_tone1: :man_artist_light_skin_tone: :man_artist::skin-tone-1:")
    ("👨🏼\342\200\215🎨" . ":man_artist_tone2: :man_artist_medium_light_skin_tone: :man_artist::skin-tone-2:")
    ("👨🏽\342\200\215🎨" . ":man_artist_tone3: :man_artist_medium_skin_tone: :man_artist::skin-tone-3:")
    ("👨🏾\342\200\215🎨" . ":man_artist_tone4: :man_artist_medium_dark_skin_tone: :man_artist::skin-tone-4:")
    ("👨🏿\342\200\215🎨" . ":man_artist_tone5: :man_artist_dark_skin_tone: :man_artist::skin-tone-5:")
    ("🧑\342\200\215🚒" . ":firefighter:")
    ("🧑🏻\342\200\215🚒" . ":firefighter_tone1: :firefighter_light_skin_tone: :firefighter::skin-tone-1:")
    ("🧑🏼\342\200\215🚒" . ":firefighter_tone2: :firefighter_medium_light_skin_tone: :firefighter::skin-tone-2:")
    ("🧑🏽\342\200\215🚒" . ":firefighter_tone3: :firefighter_medium_skin_tone: :firefighter::skin-tone-3:")
    ("🧑🏾\342\200\215🚒" . ":firefighter_tone4: :firefighter_medium_dark_skin_tone: :firefighter::skin-tone-4:")
    ("🧑🏿\342\200\215🚒" . ":firefighter_tone5: :firefighter_dark_skin_tone: :firefighter::skin-tone-5:")
    ("👩\342\200\215🚒" . ":woman_firefighter:")
    ("👩🏻\342\200\215🚒" . ":woman_firefighter_tone1: :woman_firefighter_light_skin_tone: :woman_firefighter::skin-tone-1:")
    ("👩🏼\342\200\215🚒" . ":woman_firefighter_tone2: :woman_firefighter_medium_light_skin_tone: :woman_firefighter::skin-tone-2:")
    ("👩🏽\342\200\215🚒" . ":woman_firefighter_tone3: :woman_firefighter_medium_skin_tone: :woman_firefighter::skin-tone-3:")
    ("👩🏾\342\200\215🚒" . ":woman_firefighter_tone4: :woman_firefighter_medium_dark_skin_tone: :woman_firefighter::skin-tone-4:")
    ("👩🏿\342\200\215🚒" . ":woman_firefighter_tone5: :woman_firefighter_dark_skin_tone: :woman_firefighter::skin-tone-5:")
    ("👨\342\200\215🚒" . ":man_firefighter:")
    ("👨🏻\342\200\215🚒" . ":man_firefighter_tone1: :man_firefighter_light_skin_tone: :man_firefighter::skin-tone-1:")
    ("👨🏼\342\200\215🚒" . ":man_firefighter_tone2: :man_firefighter_medium_light_skin_tone: :man_firefighter::skin-tone-2:")
    ("👨🏽\342\200\215🚒" . ":man_firefighter_tone3: :man_firefighter_medium_skin_tone: :man_firefighter::skin-tone-3:")
    ("👨🏾\342\200\215🚒" . ":man_firefighter_tone4: :man_firefighter_medium_dark_skin_tone: :man_firefighter::skin-tone-4:")
    ("👨🏿\342\200\215🚒" . ":man_firefighter_tone5: :man_firefighter_dark_skin_tone: :man_firefighter::skin-tone-5:")
    ("🧑\342\200\215\342\234\210\357\270\217" . ":pilot:")
    ("🧑🏻\342\200\215\342\234\210\357\270\217" . ":pilot_tone1: :pilot_light_skin_tone: :pilot::skin-tone-1:")
    ("🧑🏼\342\200\215\342\234\210\357\270\217" . ":pilot_tone2: :pilot_medium_light_skin_tone: :pilot::skin-tone-2:")
    ("🧑🏽\342\200\215\342\234\210\357\270\217" . ":pilot_tone3: :pilot_medium_skin_tone: :pilot::skin-tone-3:")
    ("🧑🏾\342\200\215\342\234\210\357\270\217" . ":pilot_tone4: :pilot_medium_dark_skin_tone: :pilot::skin-tone-4:")
    ("🧑🏿\342\200\215\342\234\210\357\270\217" . ":pilot_tone5: :pilot_dark_skin_tone: :pilot::skin-tone-5:")
    ("👩\342\200\215\342\234\210\357\270\217" . ":woman_pilot:")
    ("👩🏻\342\200\215\342\234\210\357\270\217" . ":woman_pilot_tone1: :woman_pilot_light_skin_tone: :woman_pilot::skin-tone-1:")
    ("👩🏼\342\200\215\342\234\210\357\270\217" . ":woman_pilot_tone2: :woman_pilot_medium_light_skin_tone: :woman_pilot::skin-tone-2:")
    ("👩🏽\342\200\215\342\234\210\357\270\217" . ":woman_pilot_tone3: :woman_pilot_medium_skin_tone: :woman_pilot::skin-tone-3:")
    ("👩🏾\342\200\215\342\234\210\357\270\217" . ":woman_pilot_tone4: :woman_pilot_medium_dark_skin_tone: :woman_pilot::skin-tone-4:")
    ("👩🏿\342\200\215\342\234\210\357\270\217" . ":woman_pilot_tone5: :woman_pilot_dark_skin_tone: :woman_pilot::skin-tone-5:")
    ("👨\342\200\215\342\234\210\357\270\217" . ":man_pilot:")
    ("👨🏻\342\200\215\342\234\210\357\270\217" . ":man_pilot_tone1: :man_pilot_light_skin_tone: :man_pilot::skin-tone-1:")
    ("👨🏼\342\200\215\342\234\210\357\270\217" . ":man_pilot_tone2: :man_pilot_medium_light_skin_tone: :man_pilot::skin-tone-2:")
    ("👨🏽\342\200\215\342\234\210\357\270\217" . ":man_pilot_tone3: :man_pilot_medium_skin_tone: :man_pilot::skin-tone-3:")
    ("👨🏾\342\200\215\342\234\210\357\270\217" . ":man_pilot_tone4: :man_pilot_medium_dark_skin_tone: :man_pilot::skin-tone-4:")
    ("👨🏿\342\200\215\342\234\210\357\270\217" . ":man_pilot_tone5: :man_pilot_dark_skin_tone: :man_pilot::skin-tone-5:")
    ("🧑\342\200\215🚀" . ":astronaut:")
    ("🧑🏻\342\200\215🚀" . ":astronaut_tone1: :astronaut_light_skin_tone: :astronaut::skin-tone-1:")
    ("🧑🏼\342\200\215🚀" . ":astronaut_tone2: :astronaut_medium_light_skin_tone: :astronaut::skin-tone-2:")
    ("🧑🏽\342\200\215🚀" . ":astronaut_tone3: :astronaut_medium_skin_tone: :astronaut::skin-tone-3:")
    ("🧑🏾\342\200\215🚀" . ":astronaut_tone4: :astronaut_medium_dark_skin_tone: :astronaut::skin-tone-4:")
    ("🧑🏿\342\200\215🚀" . ":astronaut_tone5: :astronaut_dark_skin_tone: :astronaut::skin-tone-5:")
    ("👩\342\200\215🚀" . ":woman_astronaut:")
    ("👩🏻\342\200\215🚀" . ":woman_astronaut_tone1: :woman_astronaut_light_skin_tone: :woman_astronaut::skin-tone-1:")
    ("👩🏼\342\200\215🚀" . ":woman_astronaut_tone2: :woman_astronaut_medium_light_skin_tone: :woman_astronaut::skin-tone-2:")
    ("👩🏽\342\200\215🚀" . ":woman_astronaut_tone3: :woman_astronaut_medium_skin_tone: :woman_astronaut::skin-tone-3:")
    ("👩🏾\342\200\215🚀" . ":woman_astronaut_tone4: :woman_astronaut_medium_dark_skin_tone: :woman_astronaut::skin-tone-4:")
    ("👩🏿\342\200\215🚀" . ":woman_astronaut_tone5: :woman_astronaut_dark_skin_tone: :woman_astronaut::skin-tone-5:")
    ("👨\342\200\215🚀" . ":man_astronaut:")
    ("👨🏻\342\200\215🚀" . ":man_astronaut_tone1: :man_astronaut_light_skin_tone: :man_astronaut::skin-tone-1:")
    ("👨🏼\342\200\215🚀" . ":man_astronaut_tone2: :man_astronaut_medium_light_skin_tone: :man_astronaut::skin-tone-2:")
    ("👨🏽\342\200\215🚀" . ":man_astronaut_tone3: :man_astronaut_medium_skin_tone: :man_astronaut::skin-tone-3:")
    ("👨🏾\342\200\215🚀" . ":man_astronaut_tone4: :man_astronaut_medium_dark_skin_tone: :man_astronaut::skin-tone-4:")
    ("👨🏿\342\200\215🚀" . ":man_astronaut_tone5: :man_astronaut_dark_skin_tone: :man_astronaut::skin-tone-5:")
    ("🧑\342\200\215\342\232\226\357\270\217" . ":judge:")
    ("🧑🏻\342\200\215\342\232\226\357\270\217" . ":judge_tone1: :judge_light_skin_tone: :judge::skin-tone-1:")
    ("🧑🏼\342\200\215\342\232\226\357\270\217" . ":judge_tone2: :judge_medium_light_skin_tone: :judge::skin-tone-2:")
    ("🧑🏽\342\200\215\342\232\226\357\270\217" . ":judge_tone3: :judge_medium_skin_tone: :judge::skin-tone-3:")
    ("🧑🏾\342\200\215\342\232\226\357\270\217" . ":judge_tone4: :judge_medium_dark_skin_tone: :judge::skin-tone-4:")
    ("🧑🏿\342\200\215\342\232\226\357\270\217" . ":judge_tone5: :judge_dark_skin_tone: :judge::skin-tone-5:")
    ("👩\342\200\215\342\232\226\357\270\217" . ":woman_judge:")
    ("👩🏻\342\200\215\342\232\226\357\270\217" . ":woman_judge_tone1: :woman_judge_light_skin_tone: :woman_judge::skin-tone-1:")
    ("👩🏼\342\200\215\342\232\226\357\270\217" . ":woman_judge_tone2: :woman_judge_medium_light_skin_tone: :woman_judge::skin-tone-2:")
    ("👩🏽\342\200\215\342\232\226\357\270\217" . ":woman_judge_tone3: :woman_judge_medium_skin_tone: :woman_judge::skin-tone-3:")
    ("👩🏾\342\200\215\342\232\226\357\270\217" . ":woman_judge_tone4: :woman_judge_medium_dark_skin_tone: :woman_judge::skin-tone-4:")
    ("👩🏿\342\200\215\342\232\226\357\270\217" . ":woman_judge_tone5: :woman_judge_dark_skin_tone: :woman_judge::skin-tone-5:")
    ("👨\342\200\215\342\232\226\357\270\217" . ":man_judge:")
    ("👨🏻\342\200\215\342\232\226\357\270\217" . ":man_judge_tone1: :man_judge_light_skin_tone: :man_judge::skin-tone-1:")
    ("👨🏼\342\200\215\342\232\226\357\270\217" . ":man_judge_tone2: :man_judge_medium_light_skin_tone: :man_judge::skin-tone-2:")
    ("👨🏽\342\200\215\342\232\226\357\270\217" . ":man_judge_tone3: :man_judge_medium_skin_tone: :man_judge::skin-tone-3:")
    ("👨🏾\342\200\215\342\232\226\357\270\217" . ":man_judge_tone4: :man_judge_medium_dark_skin_tone: :man_judge::skin-tone-4:")
    ("👨🏿\342\200\215\342\232\226\357\270\217" . ":man_judge_tone5: :man_judge_dark_skin_tone: :man_judge::skin-tone-5:")
    ("👰" . ":person_with_veil:")
    ("👰🏻" . ":person_with_veil_tone1: :person_with_veil::skin-tone-1:")
    ("👰🏼" . ":person_with_veil_tone2: :person_with_veil::skin-tone-2:")
    ("👰🏽" . ":person_with_veil_tone3: :person_with_veil::skin-tone-3:")
    ("👰🏾" . ":person_with_veil_tone4: :person_with_veil::skin-tone-4:")
    ("👰🏿" . ":person_with_veil_tone5: :person_with_veil::skin-tone-5:")
    ("👰\342\200\215\342\231\200\357\270\217" . ":woman_with_veil: :bride_with_veil:")
    ("👰🏻\342\200\215\342\231\200\357\270\217" . ":woman_with_veil_tone1: :woman_with_veil_light_skin_tone: :woman_with_veil::skin-tone-1: :bride_with_veil::skin-tone-1:")
    ("👰🏼\342\200\215\342\231\200\357\270\217" . ":woman_with_veil_tone2: :woman_with_veil_medium_light_skin_tone: :woman_with_veil::skin-tone-2: :bride_with_veil::skin-tone-2:")
    ("👰🏽\342\200\215\342\231\200\357\270\217" . ":woman_with_veil_tone3: :woman_with_veil_medium_skin_tone: :woman_with_veil::skin-tone-3: :bride_with_veil::skin-tone-3:")
    ("👰🏾\342\200\215\342\231\200\357\270\217" . ":woman_with_veil_tone4: :woman_with_veil_medium_dark_skin_tone: :woman_with_veil::skin-tone-4: :bride_with_veil::skin-tone-4:")
    ("👰🏿\342\200\215\342\231\200\357\270\217" . ":woman_with_veil_tone5: :woman_with_veil_dark_skin_tone: :woman_with_veil::skin-tone-5: :bride_with_veil::skin-tone-5:")
    ("👰\342\200\215\342\231\202\357\270\217" . ":man_with_veil:")
    ("👰🏻\342\200\215\342\231\202\357\270\217" . ":man_with_veil_tone1: :man_with_veil_light_skin_tone: :man_with_veil::skin-tone-1:")
    ("👰🏼\342\200\215\342\231\202\357\270\217" . ":man_with_veil_tone2: :man_with_veil_medium_light_skin_tone: :man_with_veil::skin-tone-2:")
    ("👰🏽\342\200\215\342\231\202\357\270\217" . ":man_with_veil_tone3: :man_with_veil_medium_skin_tone: :man_with_veil::skin-tone-3:")
    ("👰🏾\342\200\215\342\231\202\357\270\217" . ":man_with_veil_tone4: :man_with_veil_medium_dark_skin_tone: :man_with_veil::skin-tone-4:")
    ("👰🏿\342\200\215\342\231\202\357\270\217" . ":man_with_veil_tone5: :man_with_veil_dark_skin_tone: :man_with_veil::skin-tone-5:")
    ("🤵" . ":person_in_tuxedo:")
    ("🤵🏻" . ":person_in_tuxedo_tone1: :person_in_tuxedo::skin-tone-1:")
    ("🤵🏼" . ":person_in_tuxedo_tone2: :person_in_tuxedo::skin-tone-2:")
    ("🤵🏽" . ":person_in_tuxedo_tone3: :person_in_tuxedo::skin-tone-3:")
    ("🤵🏾" . ":person_in_tuxedo_tone4: :person_in_tuxedo::skin-tone-4:")
    ("🤵🏿" . ":person_in_tuxedo_tone5: :person_in_tuxedo::skin-tone-5:")
    ("🤵\342\200\215\342\231\200\357\270\217" . ":woman_in_tuxedo:")
    ("🤵🏻\342\200\215\342\231\200\357\270\217" . ":woman_in_tuxedo_tone1: :woman_in_tuxedo_light_skin_tone: :woman_in_tuxedo::skin-tone-1:")
    ("🤵🏼\342\200\215\342\231\200\357\270\217" . ":woman_in_tuxedo_tone2: :woman_in_tuxedo_medium_light_skin_tone: :woman_in_tuxedo::skin-tone-2:")
    ("🤵🏽\342\200\215\342\231\200\357\270\217" . ":woman_in_tuxedo_tone3: :woman_in_tuxedo_medium_skin_tone: :woman_in_tuxedo::skin-tone-3:")
    ("🤵🏾\342\200\215\342\231\200\357\270\217" . ":woman_in_tuxedo_tone4: :woman_in_tuxedo_medium_dark_skin_tone: :woman_in_tuxedo::skin-tone-4:")
    ("🤵🏿\342\200\215\342\231\200\357\270\217" . ":woman_in_tuxedo_tone5: :woman_in_tuxedo_dark_skin_tone: :woman_in_tuxedo::skin-tone-5:")
    ("🤵\342\200\215\342\231\202\357\270\217" . ":man_in_tuxedo:")
    ("🤵🏻\342\200\215\342\231\202\357\270\217" . ":man_in_tuxedo_tone1: :man_in_tuxedo_light_skin_tone: :man_in_tuxedo::skin-tone-1:")
    ("🤵🏼\342\200\215\342\231\202\357\270\217" . ":man_in_tuxedo_tone2: :man_in_tuxedo_medium_light_skin_tone: :man_in_tuxedo::skin-tone-2:")
    ("🤵🏽\342\200\215\342\231\202\357\270\217" . ":man_in_tuxedo_tone3: :man_in_tuxedo_medium_skin_tone: :man_in_tuxedo::skin-tone-3:")
    ("🤵🏾\342\200\215\342\231\202\357\270\217" . ":man_in_tuxedo_tone4: :man_in_tuxedo_medium_dark_skin_tone: :man_in_tuxedo::skin-tone-4:")
    ("🤵🏿\342\200\215\342\231\202\357\270\217" . ":man_in_tuxedo_tone5: :man_in_tuxedo_dark_skin_tone: :man_in_tuxedo::skin-tone-5:")
    ("🫅" . ":person_with_crown:")
    ("🫅🏻" . ":person_with_crown_tone1: :person_with_crown_light_skin_tone: :person_with_crown::skin-tone-1:")
    ("🫅🏼" . ":person_with_crown_tone2: :person_with_crown_medium_light_skin_tone: :person_with_crown::skin-tone-2:")
    ("🫅🏽" . ":person_with_crown_tone3: :person_with_crown_medium_skin_tone: :person_with_crown::skin-tone-3:")
    ("🫅🏾" . ":person_with_crown_tone4: :person_with_crown_medium_dark_skin_tone: :person_with_crown::skin-tone-4:")
    ("🫅🏿" . ":person_with_crown_tone5: :person_with_crown_dark_skin_tone: :person_with_crown::skin-tone-5:")
    ("👸" . ":princess:")
    ("👸🏻" . ":princess_tone1: :princess::skin-tone-1:")
    ("👸🏼" . ":princess_tone2: :princess::skin-tone-2:")
    ("👸🏽" . ":princess_tone3: :princess::skin-tone-3:")
    ("👸🏾" . ":princess_tone4: :princess::skin-tone-4:")
    ("👸🏿" . ":princess_tone5: :princess::skin-tone-5:")
    ("🤴" . ":prince:")
    ("🤴🏻" . ":prince_tone1: :prince::skin-tone-1:")
    ("🤴🏼" . ":prince_tone2: :prince::skin-tone-2:")
    ("🤴🏽" . ":prince_tone3: :prince::skin-tone-3:")
    ("🤴🏾" . ":prince_tone4: :prince::skin-tone-4:")
    ("🤴🏿" . ":prince_tone5: :prince::skin-tone-5:")
    ("🦸" . ":superhero:")
    ("🦸🏻" . ":superhero_tone1: :superhero_light_skin_tone: :superhero::skin-tone-1:")
    ("🦸🏼" . ":superhero_tone2: :superhero_medium_light_skin_tone: :superhero::skin-tone-2:")
    ("🦸🏽" . ":superhero_tone3: :superhero_medium_skin_tone: :superhero::skin-tone-3:")
    ("🦸🏾" . ":superhero_tone4: :superhero_medium_dark_skin_tone: :superhero::skin-tone-4:")
    ("🦸🏿" . ":superhero_tone5: :superhero_dark_skin_tone: :superhero::skin-tone-5:")
    ("🦸\342\200\215\342\231\200\357\270\217" . ":woman_superhero:")
    ("🦸🏻\342\200\215\342\231\200\357\270\217" . ":woman_superhero_tone1: :woman_superhero_light_skin_tone: :woman_superhero::skin-tone-1:")
    ("🦸🏼\342\200\215\342\231\200\357\270\217" . ":woman_superhero_tone2: :woman_superhero_medium_light_skin_tone: :woman_superhero::skin-tone-2:")
    ("🦸🏽\342\200\215\342\231\200\357\270\217" . ":woman_superhero_tone3: :woman_superhero_medium_skin_tone: :woman_superhero::skin-tone-3:")
    ("🦸🏾\342\200\215\342\231\200\357\270\217" . ":woman_superhero_tone4: :woman_superhero_medium_dark_skin_tone: :woman_superhero::skin-tone-4:")
    ("🦸🏿\342\200\215\342\231\200\357\270\217" . ":woman_superhero_tone5: :woman_superhero_dark_skin_tone: :woman_superhero::skin-tone-5:")
    ("🦸\342\200\215\342\231\202\357\270\217" . ":man_superhero:")
    ("🦸🏻\342\200\215\342\231\202\357\270\217" . ":man_superhero_tone1: :man_superhero_light_skin_tone: :man_superhero::skin-tone-1:")
    ("🦸🏼\342\200\215\342\231\202\357\270\217" . ":man_superhero_tone2: :man_superhero_medium_light_skin_tone: :man_superhero::skin-tone-2:")
    ("🦸🏽\342\200\215\342\231\202\357\270\217" . ":man_superhero_tone3: :man_superhero_medium_skin_tone: :man_superhero::skin-tone-3:")
    ("🦸🏾\342\200\215\342\231\202\357\270\217" . ":man_superhero_tone4: :man_superhero_medium_dark_skin_tone: :man_superhero::skin-tone-4:")
    ("🦸🏿\342\200\215\342\231\202\357\270\217" . ":man_superhero_tone5: :man_superhero_dark_skin_tone: :man_superhero::skin-tone-5:")
    ("🦹" . ":supervillain:")
    ("🦹🏻" . ":supervillain_tone1: :supervillain_light_skin_tone: :supervillain::skin-tone-1:")
    ("🦹🏼" . ":supervillain_tone2: :supervillain_medium_light_skin_tone: :supervillain::skin-tone-2:")
    ("🦹🏽" . ":supervillain_tone3: :supervillain_medium_skin_tone: :supervillain::skin-tone-3:")
    ("🦹🏾" . ":supervillain_tone4: :supervillain_medium_dark_skin_tone: :supervillain::skin-tone-4:")
    ("🦹🏿" . ":supervillain_tone5: :supervillain_dark_skin_tone: :supervillain::skin-tone-5:")
    ("🦹\342\200\215\342\231\200\357\270\217" . ":woman_supervillain:")
    ("🦹🏻\342\200\215\342\231\200\357\270\217" . ":woman_supervillain_tone1: :woman_supervillain_light_skin_tone: :woman_supervillain::skin-tone-1:")
    ("🦹🏼\342\200\215\342\231\200\357\270\217" . ":woman_supervillain_tone2: :woman_supervillain_medium_light_skin_tone: :woman_supervillain::skin-tone-2:")
    ("🦹🏽\342\200\215\342\231\200\357\270\217" . ":woman_supervillain_tone3: :woman_supervillain_medium_skin_tone: :woman_supervillain::skin-tone-3:")
    ("🦹🏾\342\200\215\342\231\200\357\270\217" . ":woman_supervillain_tone4: :woman_supervillain_medium_dark_skin_tone: :woman_supervillain::skin-tone-4:")
    ("🦹🏿\342\200\215\342\231\200\357\270\217" . ":woman_supervillain_tone5: :woman_supervillain_dark_skin_tone: :woman_supervillain::skin-tone-5:")
    ("🦹\342\200\215\342\231\202\357\270\217" . ":man_supervillain:")
    ("🦹🏻\342\200\215\342\231\202\357\270\217" . ":man_supervillain_tone1: :man_supervillain_light_skin_tone: :man_supervillain::skin-tone-1:")
    ("🦹🏼\342\200\215\342\231\202\357\270\217" . ":man_supervillain_tone2: :man_supervillain_medium_light_skin_tone: :man_supervillain::skin-tone-2:")
    ("🦹🏽\342\200\215\342\231\202\357\270\217" . ":man_supervillain_tone3: :man_supervillain_medium_skin_tone: :man_supervillain::skin-tone-3:")
    ("🦹🏾\342\200\215\342\231\202\357\270\217" . ":man_supervillain_tone4: :man_supervillain_medium_dark_skin_tone: :man_supervillain::skin-tone-4:")
    ("🦹🏿\342\200\215\342\231\202\357\270\217" . ":man_supervillain_tone5: :man_supervillain_dark_skin_tone: :man_supervillain::skin-tone-5:")
    ("🥷" . ":ninja:")
    ("🥷🏻" . ":ninja_tone1: :ninja_light_skin_tone: :ninja::skin-tone-1:")
    ("🥷🏼" . ":ninja_tone2: :ninja_medium_light_skin_tone: :ninja::skin-tone-2:")
    ("🥷🏽" . ":ninja_tone3: :ninja_medium_skin_tone: :ninja::skin-tone-3:")
    ("🥷🏾" . ":ninja_tone4: :ninja_medium_dark_skin_tone: :ninja::skin-tone-4:")
    ("🥷🏿" . ":ninja_tone5: :ninja_dark_skin_tone: :ninja::skin-tone-5:")
    ("🧑\342\200\215🎄" . ":mx_claus:")
    ("🧑🏻\342\200\215🎄" . ":mx_claus_tone1: :mx_claus_light_skin_tone: :mx_claus::skin-tone-1:")
    ("🧑🏼\342\200\215🎄" . ":mx_claus_tone2: :mx_claus_medium_light_skin_tone: :mx_claus::skin-tone-2:")
    ("🧑🏽\342\200\215🎄" . ":mx_claus_tone3: :mx_claus_medium_skin_tone: :mx_claus::skin-tone-3:")
    ("🧑🏾\342\200\215🎄" . ":mx_claus_tone4: :mx_claus_medium_dark_skin_tone: :mx_claus::skin-tone-4:")
    ("🧑🏿\342\200\215🎄" . ":mx_claus_tone5: :mx_claus_dark_skin_tone: :mx_claus::skin-tone-5:")
    ("🤶" . ":mrs_claus: :mother_christmas:")
    ("🤶🏻" . ":mrs_claus_tone1: :mother_christmas_tone1: :mrs_claus::skin-tone-1: :mother_christmas::skin-tone-1:")
    ("🤶🏼" . ":mrs_claus_tone2: :mother_christmas_tone2: :mrs_claus::skin-tone-2: :mother_christmas::skin-tone-2:")
    ("🤶🏽" . ":mrs_claus_tone3: :mother_christmas_tone3: :mrs_claus::skin-tone-3: :mother_christmas::skin-tone-3:")
    ("🤶🏾" . ":mrs_claus_tone4: :mother_christmas_tone4: :mrs_claus::skin-tone-4: :mother_christmas::skin-tone-4:")
    ("🤶🏿" . ":mrs_claus_tone5: :mother_christmas_tone5: :mrs_claus::skin-tone-5: :mother_christmas::skin-tone-5:")
    ("🎅" . ":santa_claus:")
    ("🎅🏻" . ":santa_tone1: :santa::skin-tone-1: :santa_claus::skin-tone-1:")
    ("🎅🏼" . ":santa_tone2: :santa::skin-tone-2: :santa_claus::skin-tone-2:")
    ("🎅🏽" . ":santa_tone3: :santa::skin-tone-3: :santa_claus::skin-tone-3:")
    ("🎅🏾" . ":santa_tone4: :santa::skin-tone-4: :santa_claus::skin-tone-4:")
    ("🎅🏿" . ":santa_tone5: :santa::skin-tone-5: :santa_claus::skin-tone-5:")
    ("🧙" . ":mage:")
    ("🧙🏻" . ":mage_tone1: :mage_light_skin_tone: :mage::skin-tone-1:")
    ("🧙🏼" . ":mage_tone2: :mage_medium_light_skin_tone: :mage::skin-tone-2:")
    ("🧙🏽" . ":mage_tone3: :mage_medium_skin_tone: :mage::skin-tone-3:")
    ("🧙🏾" . ":mage_tone4: :mage_medium_dark_skin_tone: :mage::skin-tone-4:")
    ("🧙🏿" . ":mage_tone5: :mage_dark_skin_tone: :mage::skin-tone-5:")
    ("🧙\342\200\215\342\231\200\357\270\217" . ":woman_mage:")
    ("🧙🏻\342\200\215\342\231\200\357\270\217" . ":woman_mage_tone1: :woman_mage_light_skin_tone: :woman_mage::skin-tone-1:")
    ("🧙🏼\342\200\215\342\231\200\357\270\217" . ":woman_mage_tone2: :woman_mage_medium_light_skin_tone: :woman_mage::skin-tone-2:")
    ("🧙🏽\342\200\215\342\231\200\357\270\217" . ":woman_mage_tone3: :woman_mage_medium_skin_tone: :woman_mage::skin-tone-3:")
    ("🧙🏾\342\200\215\342\231\200\357\270\217" . ":woman_mage_tone4: :woman_mage_medium_dark_skin_tone: :woman_mage::skin-tone-4:")
    ("🧙🏿\342\200\215\342\231\200\357\270\217" . ":woman_mage_tone5: :woman_mage_dark_skin_tone: :woman_mage::skin-tone-5:")
    ("🧙\342\200\215\342\231\202\357\270\217" . ":man_mage:")
    ("🧙🏻\342\200\215\342\231\202\357\270\217" . ":man_mage_tone1: :man_mage_light_skin_tone: :man_mage::skin-tone-1:")
    ("🧙🏼\342\200\215\342\231\202\357\270\217" . ":man_mage_tone2: :man_mage_medium_light_skin_tone: :man_mage::skin-tone-2:")
    ("🧙🏽\342\200\215\342\231\202\357\270\217" . ":man_mage_tone3: :man_mage_medium_skin_tone: :man_mage::skin-tone-3:")
    ("🧙🏾\342\200\215\342\231\202\357\270\217" . ":man_mage_tone4: :man_mage_medium_dark_skin_tone: :man_mage::skin-tone-4:")
    ("🧙🏿\342\200\215\342\231\202\357\270\217" . ":man_mage_tone5: :man_mage_dark_skin_tone: :man_mage::skin-tone-5:")
    ("🧝" . ":elf:")
    ("🧝🏻" . ":elf_tone1: :elf_light_skin_tone: :elf::skin-tone-1:")
    ("🧝🏼" . ":elf_tone2: :elf_medium_light_skin_tone: :elf::skin-tone-2:")
    ("🧝🏽" . ":elf_tone3: :elf_medium_skin_tone: :elf::skin-tone-3:")
    ("🧝🏾" . ":elf_tone4: :elf_medium_dark_skin_tone: :elf::skin-tone-4:")
    ("🧝🏿" . ":elf_tone5: :elf_dark_skin_tone: :elf::skin-tone-5:")
    ("🧝\342\200\215\342\231\200\357\270\217" . ":woman_elf:")
    ("🧝🏻\342\200\215\342\231\200\357\270\217" . ":woman_elf_tone1: :woman_elf_light_skin_tone: :woman_elf::skin-tone-1:")
    ("🧝🏼\342\200\215\342\231\200\357\270\217" . ":woman_elf_tone2: :woman_elf_medium_light_skin_tone: :woman_elf::skin-tone-2:")
    ("🧝🏽\342\200\215\342\231\200\357\270\217" . ":woman_elf_tone3: :woman_elf_medium_skin_tone: :woman_elf::skin-tone-3:")
    ("🧝🏾\342\200\215\342\231\200\357\270\217" . ":woman_elf_tone4: :woman_elf_medium_dark_skin_tone: :woman_elf::skin-tone-4:")
    ("🧝🏿\342\200\215\342\231\200\357\270\217" . ":woman_elf_tone5: :woman_elf_dark_skin_tone: :woman_elf::skin-tone-5:")
    ("🧝\342\200\215\342\231\202\357\270\217" . ":man_elf:")
    ("🧝🏻\342\200\215\342\231\202\357\270\217" . ":man_elf_tone1: :man_elf_light_skin_tone: :man_elf::skin-tone-1:")
    ("🧝🏼\342\200\215\342\231\202\357\270\217" . ":man_elf_tone2: :man_elf_medium_light_skin_tone: :man_elf::skin-tone-2:")
    ("🧝🏽\342\200\215\342\231\202\357\270\217" . ":man_elf_tone3: :man_elf_medium_skin_tone: :man_elf::skin-tone-3:")
    ("🧝🏾\342\200\215\342\231\202\357\270\217" . ":man_elf_tone4: :man_elf_medium_dark_skin_tone: :man_elf::skin-tone-4:")
    ("🧝🏿\342\200\215\342\231\202\357\270\217" . ":man_elf_tone5: :man_elf_dark_skin_tone: :man_elf::skin-tone-5:")
    ("🧌" . ":troll:")
    ("🧛" . ":vampire:")
    ("🧛🏻" . ":vampire_tone1: :vampire_light_skin_tone: :vampire::skin-tone-1:")
    ("🧛🏼" . ":vampire_tone2: :vampire_medium_light_skin_tone: :vampire::skin-tone-2:")
    ("🧛🏽" . ":vampire_tone3: :vampire_medium_skin_tone: :vampire::skin-tone-3:")
    ("🧛🏾" . ":vampire_tone4: :vampire_medium_dark_skin_tone: :vampire::skin-tone-4:")
    ("🧛🏿" . ":vampire_tone5: :vampire_dark_skin_tone: :vampire::skin-tone-5:")
    ("🧛\342\200\215\342\231\200\357\270\217" . ":woman_vampire:")
    ("🧛🏻\342\200\215\342\231\200\357\270\217" . ":woman_vampire_tone1: :woman_vampire_light_skin_tone: :woman_vampire::skin-tone-1:")
    ("🧛🏼\342\200\215\342\231\200\357\270\217" . ":woman_vampire_tone2: :woman_vampire_medium_light_skin_tone: :woman_vampire::skin-tone-2:")
    ("🧛🏽\342\200\215\342\231\200\357\270\217" . ":woman_vampire_tone3: :woman_vampire_medium_skin_tone: :woman_vampire::skin-tone-3:")
    ("🧛🏾\342\200\215\342\231\200\357\270\217" . ":woman_vampire_tone4: :woman_vampire_medium_dark_skin_tone: :woman_vampire::skin-tone-4:")
    ("🧛🏿\342\200\215\342\231\200\357\270\217" . ":woman_vampire_tone5: :woman_vampire_dark_skin_tone: :woman_vampire::skin-tone-5:")
    ("🧛\342\200\215\342\231\202\357\270\217" . ":man_vampire:")
    ("🧛🏻\342\200\215\342\231\202\357\270\217" . ":man_vampire_tone1: :man_vampire_light_skin_tone: :man_vampire::skin-tone-1:")
    ("🧛🏼\342\200\215\342\231\202\357\270\217" . ":man_vampire_tone2: :man_vampire_medium_light_skin_tone: :man_vampire::skin-tone-2:")
    ("🧛🏽\342\200\215\342\231\202\357\270\217" . ":man_vampire_tone3: :man_vampire_medium_skin_tone: :man_vampire::skin-tone-3:")
    ("🧛🏾\342\200\215\342\231\202\357\270\217" . ":man_vampire_tone4: :man_vampire_medium_dark_skin_tone: :man_vampire::skin-tone-4:")
    ("🧛🏿\342\200\215\342\231\202\357\270\217" . ":man_vampire_tone5: :man_vampire_dark_skin_tone: :man_vampire::skin-tone-5:")
    ("🧟" . ":zombie:")
    ("🧟\342\200\215\342\231\200\357\270\217" . ":woman_zombie:")
    ("🧟\342\200\215\342\231\202\357\270\217" . ":man_zombie:")
    ("🧞" . ":genie:")
    ("🧞\342\200\215\342\231\200\357\270\217" . ":woman_genie:")
    ("🧞\342\200\215\342\231\202\357\270\217" . ":man_genie:")
    ("🧜" . ":merperson:")
    ("🧜🏻" . ":merperson_tone1: :merperson_light_skin_tone: :merperson::skin-tone-1:")
    ("🧜🏼" . ":merperson_tone2: :merperson_medium_light_skin_tone: :merperson::skin-tone-2:")
    ("🧜🏽" . ":merperson_tone3: :merperson_medium_skin_tone: :merperson::skin-tone-3:")
    ("🧜🏾" . ":merperson_tone4: :merperson_medium_dark_skin_tone: :merperson::skin-tone-4:")
    ("🧜🏿" . ":merperson_tone5: :merperson_dark_skin_tone: :merperson::skin-tone-5:")
    ("🧜\342\200\215\342\231\200\357\270\217" . ":mermaid:")
    ("🧜🏻\342\200\215\342\231\200\357\270\217" . ":mermaid_tone1: :mermaid_light_skin_tone: :mermaid::skin-tone-1:")
    ("🧜🏼\342\200\215\342\231\200\357\270\217" . ":mermaid_tone2: :mermaid_medium_light_skin_tone: :mermaid::skin-tone-2:")
    ("🧜🏽\342\200\215\342\231\200\357\270\217" . ":mermaid_tone3: :mermaid_medium_skin_tone: :mermaid::skin-tone-3:")
    ("🧜🏾\342\200\215\342\231\200\357\270\217" . ":mermaid_tone4: :mermaid_medium_dark_skin_tone: :mermaid::skin-tone-4:")
    ("🧜🏿\342\200\215\342\231\200\357\270\217" . ":mermaid_tone5: :mermaid_dark_skin_tone: :mermaid::skin-tone-5:")
    ("🧜\342\200\215\342\231\202\357\270\217" . ":merman:")
    ("🧜🏻\342\200\215\342\231\202\357\270\217" . ":merman_tone1: :merman_light_skin_tone: :merman::skin-tone-1:")
    ("🧜🏼\342\200\215\342\231\202\357\270\217" . ":merman_tone2: :merman_medium_light_skin_tone: :merman::skin-tone-2:")
    ("🧜🏽\342\200\215\342\231\202\357\270\217" . ":merman_tone3: :merman_medium_skin_tone: :merman::skin-tone-3:")
    ("🧜🏾\342\200\215\342\231\202\357\270\217" . ":merman_tone4: :merman_medium_dark_skin_tone: :merman::skin-tone-4:")
    ("🧜🏿\342\200\215\342\231\202\357\270\217" . ":merman_tone5: :merman_dark_skin_tone: :merman::skin-tone-5:")
    ("🧚" . ":fairy:")
    ("🧚🏻" . ":fairy_tone1: :fairy_light_skin_tone: :fairy::skin-tone-1:")
    ("🧚🏼" . ":fairy_tone2: :fairy_medium_light_skin_tone: :fairy::skin-tone-2:")
    ("🧚🏽" . ":fairy_tone3: :fairy_medium_skin_tone: :fairy::skin-tone-3:")
    ("🧚🏾" . ":fairy_tone4: :fairy_medium_dark_skin_tone: :fairy::skin-tone-4:")
    ("🧚🏿" . ":fairy_tone5: :fairy_dark_skin_tone: :fairy::skin-tone-5:")
    ("🧚\342\200\215\342\231\200\357\270\217" . ":woman_fairy:")
    ("🧚🏻\342\200\215\342\231\200\357\270\217" . ":woman_fairy_tone1: :woman_fairy_light_skin_tone: :woman_fairy::skin-tone-1:")
    ("🧚🏼\342\200\215\342\231\200\357\270\217" . ":woman_fairy_tone2: :woman_fairy_medium_light_skin_tone: :woman_fairy::skin-tone-2:")
    ("🧚🏽\342\200\215\342\231\200\357\270\217" . ":woman_fairy_tone3: :woman_fairy_medium_skin_tone: :woman_fairy::skin-tone-3:")
    ("🧚🏾\342\200\215\342\231\200\357\270\217" . ":woman_fairy_tone4: :woman_fairy_medium_dark_skin_tone: :woman_fairy::skin-tone-4:")
    ("🧚🏿\342\200\215\342\231\200\357\270\217" . ":woman_fairy_tone5: :woman_fairy_dark_skin_tone: :woman_fairy::skin-tone-5:")
    ("🧚\342\200\215\342\231\202\357\270\217" . ":man_fairy:")
    ("🧚🏻\342\200\215\342\231\202\357\270\217" . ":man_fairy_tone1: :man_fairy_light_skin_tone: :man_fairy::skin-tone-1:")
    ("🧚🏼\342\200\215\342\231\202\357\270\217" . ":man_fairy_tone2: :man_fairy_medium_light_skin_tone: :man_fairy::skin-tone-2:")
    ("🧚🏽\342\200\215\342\231\202\357\270\217" . ":man_fairy_tone3: :man_fairy_medium_skin_tone: :man_fairy::skin-tone-3:")
    ("🧚🏾\342\200\215\342\231\202\357\270\217" . ":man_fairy_tone4: :man_fairy_medium_dark_skin_tone: :man_fairy::skin-tone-4:")
    ("🧚🏿\342\200\215\342\231\202\357\270\217" . ":man_fairy_tone5: :man_fairy_dark_skin_tone: :man_fairy::skin-tone-5:")
    ("👼" . ":baby_angel:")
    ("👼🏻" . ":angel_tone1: :baby_angel::skin-tone-1:")
    ("👼🏼" . ":angel_tone2: :baby_angel::skin-tone-2:")
    ("👼🏽" . ":angel_tone3: :baby_angel::skin-tone-3:")
    ("👼🏾" . ":angel_tone4: :baby_angel::skin-tone-4:")
    ("👼🏿" . ":angel_tone5: :baby_angel::skin-tone-5:")
    ("🫄" . ":pregnant_person:")
    ("🫄🏻" . ":pregnant_person_tone1: :pregnant_person_light_skin_tone: :pregnant_person::skin-tone-1:")
    ("🫄🏼" . ":pregnant_person_tone2: :pregnant_person_medium_light_skin_tone: :pregnant_person::skin-tone-2:")
    ("🫄🏽" . ":pregnant_person_tone3: :pregnant_person_medium_skin_tone: :pregnant_person::skin-tone-3:")
    ("🫄🏾" . ":pregnant_person_tone4: :pregnant_person_medium_dark_skin_tone: :pregnant_person::skin-tone-4:")
    ("🫄🏿" . ":pregnant_person_tone5: :pregnant_person_dark_skin_tone: :pregnant_person::skin-tone-5:")
    ("🤰" . ":pregnant_woman: :expecting_woman:")
    ("🤰🏻" . ":pregnant_woman_tone1: :expecting_woman_tone1: :pregnant_woman::skin-tone-1: :expecting_woman::skin-tone-1:")
    ("🤰🏼" . ":pregnant_woman_tone2: :expecting_woman_tone2: :pregnant_woman::skin-tone-2: :expecting_woman::skin-tone-2:")
    ("🤰🏽" . ":pregnant_woman_tone3: :expecting_woman_tone3: :pregnant_woman::skin-tone-3: :expecting_woman::skin-tone-3:")
    ("🤰🏾" . ":pregnant_woman_tone4: :expecting_woman_tone4: :pregnant_woman::skin-tone-4: :expecting_woman::skin-tone-4:")
    ("🤰🏿" . ":pregnant_woman_tone5: :expecting_woman_tone5: :pregnant_woman::skin-tone-5: :expecting_woman::skin-tone-5:")
    ("🫃" . ":pregnant_man:")
    ("🫃🏻" . ":pregnant_man_tone1: :pregnant_man_light_skin_tone: :pregnant_man::skin-tone-1:")
    ("🫃🏼" . ":pregnant_man_tone2: :pregnant_man_medium_light_skin_tone: :pregnant_man::skin-tone-2:")
    ("🫃🏽" . ":pregnant_man_tone3: :pregnant_man_medium_skin_tone: :pregnant_man::skin-tone-3:")
    ("🫃🏾" . ":pregnant_man_tone4: :pregnant_man_medium_dark_skin_tone: :pregnant_man::skin-tone-4:")
    ("🫃🏿" . ":pregnant_man_tone5: :pregnant_man_dark_skin_tone: :pregnant_man::skin-tone-5:")
    ("🤱" . ":breast_feeding:")
    ("🤱🏻" . ":breast_feeding_tone1: :breast_feeding_light_skin_tone: :breast_feeding::skin-tone-1:")
    ("🤱🏼" . ":breast_feeding_tone2: :breast_feeding_medium_light_skin_tone: :breast_feeding::skin-tone-2:")
    ("🤱🏽" . ":breast_feeding_tone3: :breast_feeding_medium_skin_tone: :breast_feeding::skin-tone-3:")
    ("🤱🏾" . ":breast_feeding_tone4: :breast_feeding_medium_dark_skin_tone: :breast_feeding::skin-tone-4:")
    ("🤱🏿" . ":breast_feeding_tone5: :breast_feeding_dark_skin_tone: :breast_feeding::skin-tone-5:")
    ("🧑\342\200\215🍼" . ":person_feeding_baby:")
    ("🧑🏻\342\200\215🍼" . ":person_feeding_baby_tone1: :person_feeding_baby_light_skin_tone: :person_feeding_baby::skin-tone-1:")
    ("🧑🏼\342\200\215🍼" . ":person_feeding_baby_tone2: :person_feeding_baby_medium_light_skin_tone: :person_feeding_baby::skin-tone-2:")
    ("🧑🏽\342\200\215🍼" . ":person_feeding_baby_tone3: :person_feeding_baby_medium_skin_tone: :person_feeding_baby::skin-tone-3:")
    ("🧑🏾\342\200\215🍼" . ":person_feeding_baby_tone4: :person_feeding_baby_medium_dark_skin_tone: :person_feeding_baby::skin-tone-4:")
    ("🧑🏿\342\200\215🍼" . ":person_feeding_baby_tone5: :person_feeding_baby_dark_skin_tone: :person_feeding_baby::skin-tone-5:")
    ("👩\342\200\215🍼" . ":woman_feeding_baby:")
    ("👩🏻\342\200\215🍼" . ":woman_feeding_baby_tone1: :woman_feeding_baby_light_skin_tone: :woman_feeding_baby::skin-tone-1:")
    ("👩🏼\342\200\215🍼" . ":woman_feeding_baby_tone2: :woman_feeding_baby_medium_light_skin_tone: :woman_feeding_baby::skin-tone-2:")
    ("👩🏽\342\200\215🍼" . ":woman_feeding_baby_tone3: :woman_feeding_baby_medium_skin_tone: :woman_feeding_baby::skin-tone-3:")
    ("👩🏾\342\200\215🍼" . ":woman_feeding_baby_tone4: :woman_feeding_baby_medium_dark_skin_tone: :woman_feeding_baby::skin-tone-4:")
    ("👩🏿\342\200\215🍼" . ":woman_feeding_baby_tone5: :woman_feeding_baby_dark_skin_tone: :woman_feeding_baby::skin-tone-5:")
    ("👨\342\200\215🍼" . ":man_feeding_baby:")
    ("👨🏻\342\200\215🍼" . ":man_feeding_baby_tone1: :man_feeding_baby_light_skin_tone: :man_feeding_baby::skin-tone-1:")
    ("👨🏼\342\200\215🍼" . ":man_feeding_baby_tone2: :man_feeding_baby_medium_light_skin_tone: :man_feeding_baby::skin-tone-2:")
    ("👨🏽\342\200\215🍼" . ":man_feeding_baby_tone3: :man_feeding_baby_medium_skin_tone: :man_feeding_baby::skin-tone-3:")
    ("👨🏾\342\200\215🍼" . ":man_feeding_baby_tone4: :man_feeding_baby_medium_dark_skin_tone: :man_feeding_baby::skin-tone-4:")
    ("👨🏿\342\200\215🍼" . ":man_feeding_baby_tone5: :man_feeding_baby_dark_skin_tone: :man_feeding_baby::skin-tone-5:")
    ("🙇" . ":person_bowing:")
    ("🙇🏻" . ":person_bowing_tone1: :bow_tone1: :person_bowing::skin-tone-1: :bow::skin-tone-1:")
    ("🙇🏼" . ":person_bowing_tone2: :bow_tone2: :person_bowing::skin-tone-2: :bow::skin-tone-2:")
    ("🙇🏽" . ":person_bowing_tone3: :bow_tone3: :person_bowing::skin-tone-3: :bow::skin-tone-3:")
    ("🙇🏾" . ":person_bowing_tone4: :bow_tone4: :person_bowing::skin-tone-4: :bow::skin-tone-4:")
    ("🙇🏿" . ":person_bowing_tone5: :bow_tone5: :person_bowing::skin-tone-5: :bow::skin-tone-5:")
    ("🙇\342\200\215\342\231\200\357\270\217" . ":woman_bowing:")
    ("🙇🏻\342\200\215\342\231\200\357\270\217" . ":woman_bowing_tone1: :woman_bowing_light_skin_tone: :woman_bowing::skin-tone-1:")
    ("🙇🏼\342\200\215\342\231\200\357\270\217" . ":woman_bowing_tone2: :woman_bowing_medium_light_skin_tone: :woman_bowing::skin-tone-2:")
    ("🙇🏽\342\200\215\342\231\200\357\270\217" . ":woman_bowing_tone3: :woman_bowing_medium_skin_tone: :woman_bowing::skin-tone-3:")
    ("🙇🏾\342\200\215\342\231\200\357\270\217" . ":woman_bowing_tone4: :woman_bowing_medium_dark_skin_tone: :woman_bowing::skin-tone-4:")
    ("🙇🏿\342\200\215\342\231\200\357\270\217" . ":woman_bowing_tone5: :woman_bowing_dark_skin_tone: :woman_bowing::skin-tone-5:")
    ("🙇\342\200\215\342\231\202\357\270\217" . ":man_bowing:")
    ("🙇🏻\342\200\215\342\231\202\357\270\217" . ":man_bowing_tone1: :man_bowing_light_skin_tone: :man_bowing::skin-tone-1:")
    ("🙇🏼\342\200\215\342\231\202\357\270\217" . ":man_bowing_tone2: :man_bowing_medium_light_skin_tone: :man_bowing::skin-tone-2:")
    ("🙇🏽\342\200\215\342\231\202\357\270\217" . ":man_bowing_tone3: :man_bowing_medium_skin_tone: :man_bowing::skin-tone-3:")
    ("🙇🏾\342\200\215\342\231\202\357\270\217" . ":man_bowing_tone4: :man_bowing_medium_dark_skin_tone: :man_bowing::skin-tone-4:")
    ("🙇🏿\342\200\215\342\231\202\357\270\217" . ":man_bowing_tone5: :man_bowing_dark_skin_tone: :man_bowing::skin-tone-5:")
    ("💁" . ":person_tipping_hand: :information_desk_person:")
    ("💁🏻" . ":person_tipping_hand_tone1: :information_desk_person_tone1: :person_tipping_hand::skin-tone-1: :information_desk_person::skin-tone-1:")
    ("💁🏼" . ":person_tipping_hand_tone2: :information_desk_person_tone2: :person_tipping_hand::skin-tone-2: :information_desk_person::skin-tone-2:")
    ("💁🏽" . ":person_tipping_hand_tone3: :information_desk_person_tone3: :person_tipping_hand::skin-tone-3: :information_desk_person::skin-tone-3:")
    ("💁🏾" . ":person_tipping_hand_tone4: :information_desk_person_tone4: :person_tipping_hand::skin-tone-4: :information_desk_person::skin-tone-4:")
    ("💁🏿" . ":person_tipping_hand_tone5: :information_desk_person_tone5: :person_tipping_hand::skin-tone-5: :information_desk_person::skin-tone-5:")
    ("💁\342\200\215\342\231\200\357\270\217" . ":woman_tipping_hand:")
    ("💁🏻\342\200\215\342\231\200\357\270\217" . ":woman_tipping_hand_tone1: :woman_tipping_hand_light_skin_tone: :woman_tipping_hand::skin-tone-1:")
    ("💁🏼\342\200\215\342\231\200\357\270\217" . ":woman_tipping_hand_tone2: :woman_tipping_hand_medium_light_skin_tone: :woman_tipping_hand::skin-tone-2:")
    ("💁🏽\342\200\215\342\231\200\357\270\217" . ":woman_tipping_hand_tone3: :woman_tipping_hand_medium_skin_tone: :woman_tipping_hand::skin-tone-3:")
    ("💁🏾\342\200\215\342\231\200\357\270\217" . ":woman_tipping_hand_tone4: :woman_tipping_hand_medium_dark_skin_tone: :woman_tipping_hand::skin-tone-4:")
    ("💁🏿\342\200\215\342\231\200\357\270\217" . ":woman_tipping_hand_tone5: :woman_tipping_hand_dark_skin_tone: :woman_tipping_hand::skin-tone-5:")
    ("💁\342\200\215\342\231\202\357\270\217" . ":man_tipping_hand:")
    ("💁🏻\342\200\215\342\231\202\357\270\217" . ":man_tipping_hand_tone1: :man_tipping_hand_light_skin_tone: :man_tipping_hand::skin-tone-1:")
    ("💁🏼\342\200\215\342\231\202\357\270\217" . ":man_tipping_hand_tone2: :man_tipping_hand_medium_light_skin_tone: :man_tipping_hand::skin-tone-2:")
    ("💁🏽\342\200\215\342\231\202\357\270\217" . ":man_tipping_hand_tone3: :man_tipping_hand_medium_skin_tone: :man_tipping_hand::skin-tone-3:")
    ("💁🏾\342\200\215\342\231\202\357\270\217" . ":man_tipping_hand_tone4: :man_tipping_hand_medium_dark_skin_tone: :man_tipping_hand::skin-tone-4:")
    ("💁🏿\342\200\215\342\231\202\357\270\217" . ":man_tipping_hand_tone5: :man_tipping_hand_dark_skin_tone: :man_tipping_hand::skin-tone-5:")
    ("🙅" . ":person_gesturing_no: :no_good:")
    ("🙅🏻" . ":person_gesturing_no_tone1: :no_good_tone1: :person_gesturing_no::skin-tone-1: :no_good::skin-tone-1:")
    ("🙅🏼" . ":person_gesturing_no_tone2: :no_good_tone2: :person_gesturing_no::skin-tone-2: :no_good::skin-tone-2:")
    ("🙅🏽" . ":person_gesturing_no_tone3: :no_good_tone3: :person_gesturing_no::skin-tone-3: :no_good::skin-tone-3:")
    ("🙅🏾" . ":person_gesturing_no_tone4: :no_good_tone4: :person_gesturing_no::skin-tone-4: :no_good::skin-tone-4:")
    ("🙅🏿" . ":person_gesturing_no_tone5: :no_good_tone5: :person_gesturing_no::skin-tone-5: :no_good::skin-tone-5:")
    ("🙅\342\200\215\342\231\200\357\270\217" . ":woman_gesturing_no:")
    ("🙅🏻\342\200\215\342\231\200\357\270\217" . ":woman_gesturing_no_tone1: :woman_gesturing_no_light_skin_tone: :woman_gesturing_no::skin-tone-1:")
    ("🙅🏼\342\200\215\342\231\200\357\270\217" . ":woman_gesturing_no_tone2: :woman_gesturing_no_medium_light_skin_tone: :woman_gesturing_no::skin-tone-2:")
    ("🙅🏽\342\200\215\342\231\200\357\270\217" . ":woman_gesturing_no_tone3: :woman_gesturing_no_medium_skin_tone: :woman_gesturing_no::skin-tone-3:")
    ("🙅🏾\342\200\215\342\231\200\357\270\217" . ":woman_gesturing_no_tone4: :woman_gesturing_no_medium_dark_skin_tone: :woman_gesturing_no::skin-tone-4:")
    ("🙅🏿\342\200\215\342\231\200\357\270\217" . ":woman_gesturing_no_tone5: :woman_gesturing_no_dark_skin_tone: :woman_gesturing_no::skin-tone-5:")
    ("🙅\342\200\215\342\231\202\357\270\217" . ":man_gesturing_no:")
    ("🙅🏻\342\200\215\342\231\202\357\270\217" . ":man_gesturing_no_tone1: :man_gesturing_no_light_skin_tone: :man_gesturing_no::skin-tone-1:")
    ("🙅🏼\342\200\215\342\231\202\357\270\217" . ":man_gesturing_no_tone2: :man_gesturing_no_medium_light_skin_tone: :man_gesturing_no::skin-tone-2:")
    ("🙅🏽\342\200\215\342\231\202\357\270\217" . ":man_gesturing_no_tone3: :man_gesturing_no_medium_skin_tone: :man_gesturing_no::skin-tone-3:")
    ("🙅🏾\342\200\215\342\231\202\357\270\217" . ":man_gesturing_no_tone4: :man_gesturing_no_medium_dark_skin_tone: :man_gesturing_no::skin-tone-4:")
    ("🙅🏿\342\200\215\342\231\202\357\270\217" . ":man_gesturing_no_tone5: :man_gesturing_no_dark_skin_tone: :man_gesturing_no::skin-tone-5:")
    ("🙆" . ":person_gesturing_ok:")
    ("🙆🏻" . ":person_gesturing_ok_tone1: :person_gesturing_ok::skin-tone-1:")
    ("🙆🏼" . ":person_gesturing_ok_tone2: :person_gesturing_ok::skin-tone-2:")
    ("🙆🏽" . ":person_gesturing_ok_tone3: :person_gesturing_ok::skin-tone-3:")
    ("🙆🏾" . ":person_gesturing_ok_tone4: :person_gesturing_ok::skin-tone-4:")
    ("🙆🏿" . ":person_gesturing_ok_tone5: :person_gesturing_ok::skin-tone-5:")
    ("🙆\342\200\215\342\231\200\357\270\217" . ":woman_gesturing_ok:")
    ("🙆🏻\342\200\215\342\231\200\357\270\217" . ":woman_gesturing_ok_tone1: :woman_gesturing_ok_light_skin_tone: :woman_gesturing_ok::skin-tone-1:")
    ("🙆🏼\342\200\215\342\231\200\357\270\217" . ":woman_gesturing_ok_tone2: :woman_gesturing_ok_medium_light_skin_tone: :woman_gesturing_ok::skin-tone-2:")
    ("🙆🏽\342\200\215\342\231\200\357\270\217" . ":woman_gesturing_ok_tone3: :woman_gesturing_ok_medium_skin_tone: :woman_gesturing_ok::skin-tone-3:")
    ("🙆🏾\342\200\215\342\231\200\357\270\217" . ":woman_gesturing_ok_tone4: :woman_gesturing_ok_medium_dark_skin_tone: :woman_gesturing_ok::skin-tone-4:")
    ("🙆🏿\342\200\215\342\231\200\357\270\217" . ":woman_gesturing_ok_tone5: :woman_gesturing_ok_dark_skin_tone: :woman_gesturing_ok::skin-tone-5:")
    ("🙆\342\200\215\342\231\202\357\270\217" . ":man_gesturing_ok:")
    ("🙆🏻\342\200\215\342\231\202\357\270\217" . ":man_gesturing_ok_tone1: :man_gesturing_ok_light_skin_tone: :man_gesturing_ok::skin-tone-1:")
    ("🙆🏼\342\200\215\342\231\202\357\270\217" . ":man_gesturing_ok_tone2: :man_gesturing_ok_medium_light_skin_tone: :man_gesturing_ok::skin-tone-2:")
    ("🙆🏽\342\200\215\342\231\202\357\270\217" . ":man_gesturing_ok_tone3: :man_gesturing_ok_medium_skin_tone: :man_gesturing_ok::skin-tone-3:")
    ("🙆🏾\342\200\215\342\231\202\357\270\217" . ":man_gesturing_ok_tone4: :man_gesturing_ok_medium_dark_skin_tone: :man_gesturing_ok::skin-tone-4:")
    ("🙆🏿\342\200\215\342\231\202\357\270\217" . ":man_gesturing_ok_tone5: :man_gesturing_ok_dark_skin_tone: :man_gesturing_ok::skin-tone-5:")
    ("🙋" . ":person_raising_hand:")
    ("🙋🏻" . ":person_raising_hand_tone1: :person_raising_hand::skin-tone-1:")
    ("🙋🏼" . ":person_raising_hand_tone2: :person_raising_hand::skin-tone-2:")
    ("🙋🏽" . ":person_raising_hand_tone3: :person_raising_hand::skin-tone-3:")
    ("🙋🏾" . ":person_raising_hand_tone4: :person_raising_hand::skin-tone-4:")
    ("🙋🏿" . ":person_raising_hand_tone5: :person_raising_hand::skin-tone-5:")
    ("🙋\342\200\215\342\231\200\357\270\217" . ":woman_raising_hand:")
    ("🙋🏻\342\200\215\342\231\200\357\270\217" . ":woman_raising_hand_tone1: :woman_raising_hand_light_skin_tone: :woman_raising_hand::skin-tone-1:")
    ("🙋🏼\342\200\215\342\231\200\357\270\217" . ":woman_raising_hand_tone2: :woman_raising_hand_medium_light_skin_tone: :woman_raising_hand::skin-tone-2:")
    ("🙋🏽\342\200\215\342\231\200\357\270\217" . ":woman_raising_hand_tone3: :woman_raising_hand_medium_skin_tone: :woman_raising_hand::skin-tone-3:")
    ("🙋🏾\342\200\215\342\231\200\357\270\217" . ":woman_raising_hand_tone4: :woman_raising_hand_medium_dark_skin_tone: :woman_raising_hand::skin-tone-4:")
    ("🙋🏿\342\200\215\342\231\200\357\270\217" . ":woman_raising_hand_tone5: :woman_raising_hand_dark_skin_tone: :woman_raising_hand::skin-tone-5:")
    ("🙋\342\200\215\342\231\202\357\270\217" . ":man_raising_hand:")
    ("🙋🏻\342\200\215\342\231\202\357\270\217" . ":man_raising_hand_tone1: :man_raising_hand_light_skin_tone: :man_raising_hand::skin-tone-1:")
    ("🙋🏼\342\200\215\342\231\202\357\270\217" . ":man_raising_hand_tone2: :man_raising_hand_medium_light_skin_tone: :man_raising_hand::skin-tone-2:")
    ("🙋🏽\342\200\215\342\231\202\357\270\217" . ":man_raising_hand_tone3: :man_raising_hand_medium_skin_tone: :man_raising_hand::skin-tone-3:")
    ("🙋🏾\342\200\215\342\231\202\357\270\217" . ":man_raising_hand_tone4: :man_raising_hand_medium_dark_skin_tone: :man_raising_hand::skin-tone-4:")
    ("🙋🏿\342\200\215\342\231\202\357\270\217" . ":man_raising_hand_tone5: :man_raising_hand_dark_skin_tone: :man_raising_hand::skin-tone-5:")
    ("🧏" . ":deaf_person:")
    ("🧏🏻" . ":deaf_person_tone1: :deaf_person_light_skin_tone: :deaf_person::skin-tone-1:")
    ("🧏🏼" . ":deaf_person_tone2: :deaf_person_medium_light_skin_tone: :deaf_person::skin-tone-2:")
    ("🧏🏽" . ":deaf_person_tone3: :deaf_person_medium_skin_tone: :deaf_person::skin-tone-3:")
    ("🧏🏾" . ":deaf_person_tone4: :deaf_person_medium_dark_skin_tone: :deaf_person::skin-tone-4:")
    ("🧏🏿" . ":deaf_person_tone5: :deaf_person_dark_skin_tone: :deaf_person::skin-tone-5:")
    ("🧏\342\200\215\342\231\200\357\270\217" . ":deaf_woman:")
    ("🧏🏻\342\200\215\342\231\200\357\270\217" . ":deaf_woman_tone1: :deaf_woman_light_skin_tone: :deaf_woman::skin-tone-1:")
    ("🧏🏼\342\200\215\342\231\200\357\270\217" . ":deaf_woman_tone2: :deaf_woman_medium_light_skin_tone: :deaf_woman::skin-tone-2:")
    ("🧏🏽\342\200\215\342\231\200\357\270\217" . ":deaf_woman_tone3: :deaf_woman_medium_skin_tone: :deaf_woman::skin-tone-3:")
    ("🧏🏾\342\200\215\342\231\200\357\270\217" . ":deaf_woman_tone4: :deaf_woman_medium_dark_skin_tone: :deaf_woman::skin-tone-4:")
    ("🧏🏿\342\200\215\342\231\200\357\270\217" . ":deaf_woman_tone5: :deaf_woman_dark_skin_tone: :deaf_woman::skin-tone-5:")
    ("🧏\342\200\215\342\231\202\357\270\217" . ":deaf_man:")
    ("🧏🏻\342\200\215\342\231\202\357\270\217" . ":deaf_man_tone1: :deaf_man_light_skin_tone: :deaf_man::skin-tone-1:")
    ("🧏🏼\342\200\215\342\231\202\357\270\217" . ":deaf_man_tone2: :deaf_man_medium_light_skin_tone: :deaf_man::skin-tone-2:")
    ("🧏🏽\342\200\215\342\231\202\357\270\217" . ":deaf_man_tone3: :deaf_man_medium_skin_tone: :deaf_man::skin-tone-3:")
    ("🧏🏾\342\200\215\342\231\202\357\270\217" . ":deaf_man_tone4: :deaf_man_medium_dark_skin_tone: :deaf_man::skin-tone-4:")
    ("🧏🏿\342\200\215\342\231\202\357\270\217" . ":deaf_man_tone5: :deaf_man_dark_skin_tone: :deaf_man::skin-tone-5:")
    ("🤦" . ":person_facepalming: :face_palm:")
    ("🤦🏻" . ":person_facepalming_tone1: :face_palm_tone1: :facepalm_tone1: :person_facepalming::skin-tone-1: :face_palm::skin-tone-1: :facepalm::skin-tone-1:")
    ("🤦🏼" . ":person_facepalming_tone2: :face_palm_tone2: :facepalm_tone2: :person_facepalming::skin-tone-2: :face_palm::skin-tone-2: :facepalm::skin-tone-2:")
    ("🤦🏽" . ":person_facepalming_tone3: :face_palm_tone3: :facepalm_tone3: :person_facepalming::skin-tone-3: :face_palm::skin-tone-3: :facepalm::skin-tone-3:")
    ("🤦🏾" . ":person_facepalming_tone4: :face_palm_tone4: :facepalm_tone4: :person_facepalming::skin-tone-4: :face_palm::skin-tone-4: :facepalm::skin-tone-4:")
    ("🤦🏿" . ":person_facepalming_tone5: :face_palm_tone5: :facepalm_tone5: :person_facepalming::skin-tone-5: :face_palm::skin-tone-5: :facepalm::skin-tone-5:")
    ("🤦\342\200\215\342\231\200\357\270\217" . ":woman_facepalming:")
    ("🤦🏻\342\200\215\342\231\200\357\270\217" . ":woman_facepalming_tone1: :woman_facepalming_light_skin_tone: :woman_facepalming::skin-tone-1:")
    ("🤦🏼\342\200\215\342\231\200\357\270\217" . ":woman_facepalming_tone2: :woman_facepalming_medium_light_skin_tone: :woman_facepalming::skin-tone-2:")
    ("🤦🏽\342\200\215\342\231\200\357\270\217" . ":woman_facepalming_tone3: :woman_facepalming_medium_skin_tone: :woman_facepalming::skin-tone-3:")
    ("🤦🏾\342\200\215\342\231\200\357\270\217" . ":woman_facepalming_tone4: :woman_facepalming_medium_dark_skin_tone: :woman_facepalming::skin-tone-4:")
    ("🤦🏿\342\200\215\342\231\200\357\270\217" . ":woman_facepalming_tone5: :woman_facepalming_dark_skin_tone: :woman_facepalming::skin-tone-5:")
    ("🤦\342\200\215\342\231\202\357\270\217" . ":man_facepalming:")
    ("🤦🏻\342\200\215\342\231\202\357\270\217" . ":man_facepalming_tone1: :man_facepalming_light_skin_tone: :man_facepalming::skin-tone-1:")
    ("🤦🏼\342\200\215\342\231\202\357\270\217" . ":man_facepalming_tone2: :man_facepalming_medium_light_skin_tone: :man_facepalming::skin-tone-2:")
    ("🤦🏽\342\200\215\342\231\202\357\270\217" . ":man_facepalming_tone3: :man_facepalming_medium_skin_tone: :man_facepalming::skin-tone-3:")
    ("🤦🏾\342\200\215\342\231\202\357\270\217" . ":man_facepalming_tone4: :man_facepalming_medium_dark_skin_tone: :man_facepalming::skin-tone-4:")
    ("🤦🏿\342\200\215\342\231\202\357\270\217" . ":man_facepalming_tone5: :man_facepalming_dark_skin_tone: :man_facepalming::skin-tone-5:")
    ("🤷" . ":person_shrugging:")
    ("🤷🏻" . ":person_shrugging_tone1: :shrug_tone1: :person_shrugging::skin-tone-1: :shrug::skin-tone-1:")
    ("🤷🏼" . ":person_shrugging_tone2: :shrug_tone2: :person_shrugging::skin-tone-2: :shrug::skin-tone-2:")
    ("🤷🏽" . ":person_shrugging_tone3: :shrug_tone3: :person_shrugging::skin-tone-3: :shrug::skin-tone-3:")
    ("🤷🏾" . ":person_shrugging_tone4: :shrug_tone4: :person_shrugging::skin-tone-4: :shrug::skin-tone-4:")
    ("🤷🏿" . ":person_shrugging_tone5: :shrug_tone5: :person_shrugging::skin-tone-5: :shrug::skin-tone-5:")
    ("🤷\342\200\215\342\231\200\357\270\217" . ":woman_shrugging:")
    ("🤷🏻\342\200\215\342\231\200\357\270\217" . ":woman_shrugging_tone1: :woman_shrugging_light_skin_tone: :woman_shrugging::skin-tone-1:")
    ("🤷🏼\342\200\215\342\231\200\357\270\217" . ":woman_shrugging_tone2: :woman_shrugging_medium_light_skin_tone: :woman_shrugging::skin-tone-2:")
    ("🤷🏽\342\200\215\342\231\200\357\270\217" . ":woman_shrugging_tone3: :woman_shrugging_medium_skin_tone: :woman_shrugging::skin-tone-3:")
    ("🤷🏾\342\200\215\342\231\200\357\270\217" . ":woman_shrugging_tone4: :woman_shrugging_medium_dark_skin_tone: :woman_shrugging::skin-tone-4:")
    ("🤷🏿\342\200\215\342\231\200\357\270\217" . ":woman_shrugging_tone5: :woman_shrugging_dark_skin_tone: :woman_shrugging::skin-tone-5:")
    ("🤷\342\200\215\342\231\202\357\270\217" . ":man_shrugging:")
    ("🤷🏻\342\200\215\342\231\202\357\270\217" . ":man_shrugging_tone1: :man_shrugging_light_skin_tone: :man_shrugging::skin-tone-1:")
    ("🤷🏼\342\200\215\342\231\202\357\270\217" . ":man_shrugging_tone2: :man_shrugging_medium_light_skin_tone: :man_shrugging::skin-tone-2:")
    ("🤷🏽\342\200\215\342\231\202\357\270\217" . ":man_shrugging_tone3: :man_shrugging_medium_skin_tone: :man_shrugging::skin-tone-3:")
    ("🤷🏾\342\200\215\342\231\202\357\270\217" . ":man_shrugging_tone4: :man_shrugging_medium_dark_skin_tone: :man_shrugging::skin-tone-4:")
    ("🤷🏿\342\200\215\342\231\202\357\270\217" . ":man_shrugging_tone5: :man_shrugging_dark_skin_tone: :man_shrugging::skin-tone-5:")
    ("🙎" . ":person_pouting: :person_with_pouting_face:")
    ("🙎🏻" . ":person_pouting_tone1: :person_with_pouting_face_tone1: :person_pouting::skin-tone-1: :person_with_pouting_face::skin-tone-1:")
    ("🙎🏼" . ":person_pouting_tone2: :person_with_pouting_face_tone2: :person_pouting::skin-tone-2: :person_with_pouting_face::skin-tone-2:")
    ("🙎🏽" . ":person_pouting_tone3: :person_with_pouting_face_tone3: :person_pouting::skin-tone-3: :person_with_pouting_face::skin-tone-3:")
    ("🙎🏾" . ":person_pouting_tone4: :person_with_pouting_face_tone4: :person_pouting::skin-tone-4: :person_with_pouting_face::skin-tone-4:")
    ("🙎🏿" . ":person_pouting_tone5: :person_with_pouting_face_tone5: :person_pouting::skin-tone-5: :person_with_pouting_face::skin-tone-5:")
    ("🙎\342\200\215\342\231\200\357\270\217" . ":woman_pouting:")
    ("🙎🏻\342\200\215\342\231\200\357\270\217" . ":woman_pouting_tone1: :woman_pouting_light_skin_tone: :woman_pouting::skin-tone-1:")
    ("🙎🏼\342\200\215\342\231\200\357\270\217" . ":woman_pouting_tone2: :woman_pouting_medium_light_skin_tone: :woman_pouting::skin-tone-2:")
    ("🙎🏽\342\200\215\342\231\200\357\270\217" . ":woman_pouting_tone3: :woman_pouting_medium_skin_tone: :woman_pouting::skin-tone-3:")
    ("🙎🏾\342\200\215\342\231\200\357\270\217" . ":woman_pouting_tone4: :woman_pouting_medium_dark_skin_tone: :woman_pouting::skin-tone-4:")
    ("🙎🏿\342\200\215\342\231\200\357\270\217" . ":woman_pouting_tone5: :woman_pouting_dark_skin_tone: :woman_pouting::skin-tone-5:")
    ("🙎\342\200\215\342\231\202\357\270\217" . ":man_pouting:")
    ("🙎🏻\342\200\215\342\231\202\357\270\217" . ":man_pouting_tone1: :man_pouting_light_skin_tone: :man_pouting::skin-tone-1:")
    ("🙎🏼\342\200\215\342\231\202\357\270\217" . ":man_pouting_tone2: :man_pouting_medium_light_skin_tone: :man_pouting::skin-tone-2:")
    ("🙎🏽\342\200\215\342\231\202\357\270\217" . ":man_pouting_tone3: :man_pouting_medium_skin_tone: :man_pouting::skin-tone-3:")
    ("🙎🏾\342\200\215\342\231\202\357\270\217" . ":man_pouting_tone4: :man_pouting_medium_dark_skin_tone: :man_pouting::skin-tone-4:")
    ("🙎🏿\342\200\215\342\231\202\357\270\217" . ":man_pouting_tone5: :man_pouting_dark_skin_tone: :man_pouting::skin-tone-5:")
    ("🙍" . ":person_frowning:")
    ("🙍🏻" . ":person_frowning_tone1: :person_frowning::skin-tone-1:")
    ("🙍🏼" . ":person_frowning_tone2: :person_frowning::skin-tone-2:")
    ("🙍🏽" . ":person_frowning_tone3: :person_frowning::skin-tone-3:")
    ("🙍🏾" . ":person_frowning_tone4: :person_frowning::skin-tone-4:")
    ("🙍🏿" . ":person_frowning_tone5: :person_frowning::skin-tone-5:")
    ("🙍\342\200\215\342\231\200\357\270\217" . ":woman_frowning:")
    ("🙍🏻\342\200\215\342\231\200\357\270\217" . ":woman_frowning_tone1: :woman_frowning_light_skin_tone: :woman_frowning::skin-tone-1:")
    ("🙍🏼\342\200\215\342\231\200\357\270\217" . ":woman_frowning_tone2: :woman_frowning_medium_light_skin_tone: :woman_frowning::skin-tone-2:")
    ("🙍🏽\342\200\215\342\231\200\357\270\217" . ":woman_frowning_tone3: :woman_frowning_medium_skin_tone: :woman_frowning::skin-tone-3:")
    ("🙍🏾\342\200\215\342\231\200\357\270\217" . ":woman_frowning_tone4: :woman_frowning_medium_dark_skin_tone: :woman_frowning::skin-tone-4:")
    ("🙍🏿\342\200\215\342\231\200\357\270\217" . ":woman_frowning_tone5: :woman_frowning_dark_skin_tone: :woman_frowning::skin-tone-5:")
    ("🙍\342\200\215\342\231\202\357\270\217" . ":man_frowning:")
    ("🙍🏻\342\200\215\342\231\202\357\270\217" . ":man_frowning_tone1: :man_frowning_light_skin_tone: :man_frowning::skin-tone-1:")
    ("🙍🏼\342\200\215\342\231\202\357\270\217" . ":man_frowning_tone2: :man_frowning_medium_light_skin_tone: :man_frowning::skin-tone-2:")
    ("🙍🏽\342\200\215\342\231\202\357\270\217" . ":man_frowning_tone3: :man_frowning_medium_skin_tone: :man_frowning::skin-tone-3:")
    ("🙍🏾\342\200\215\342\231\202\357\270\217" . ":man_frowning_tone4: :man_frowning_medium_dark_skin_tone: :man_frowning::skin-tone-4:")
    ("🙍🏿\342\200\215\342\231\202\357\270\217" . ":man_frowning_tone5: :man_frowning_dark_skin_tone: :man_frowning::skin-tone-5:")
    ("💇" . ":person_getting_haircut:")
    ("💇🏻" . ":person_getting_haircut_tone1: :person_getting_haircut::skin-tone-1:")
    ("💇🏼" . ":person_getting_haircut_tone2: :person_getting_haircut::skin-tone-2:")
    ("💇🏽" . ":person_getting_haircut_tone3: :person_getting_haircut::skin-tone-3:")
    ("💇🏾" . ":person_getting_haircut_tone4: :person_getting_haircut::skin-tone-4:")
    ("💇🏿" . ":person_getting_haircut_tone5: :person_getting_haircut::skin-tone-5:")
    ("💇\342\200\215\342\231\200\357\270\217" . ":woman_getting_haircut:")
    ("💇🏻\342\200\215\342\231\200\357\270\217" . ":woman_getting_haircut_tone1: :woman_getting_haircut_light_skin_tone: :woman_getting_haircut::skin-tone-1:")
    ("💇🏼\342\200\215\342\231\200\357\270\217" . ":woman_getting_haircut_tone2: :woman_getting_haircut_medium_light_skin_tone: :woman_getting_haircut::skin-tone-2:")
    ("💇🏽\342\200\215\342\231\200\357\270\217" . ":woman_getting_haircut_tone3: :woman_getting_haircut_medium_skin_tone: :woman_getting_haircut::skin-tone-3:")
    ("💇🏾\342\200\215\342\231\200\357\270\217" . ":woman_getting_haircut_tone4: :woman_getting_haircut_medium_dark_skin_tone: :woman_getting_haircut::skin-tone-4:")
    ("💇🏿\342\200\215\342\231\200\357\270\217" . ":woman_getting_haircut_tone5: :woman_getting_haircut_dark_skin_tone: :woman_getting_haircut::skin-tone-5:")
    ("💇\342\200\215\342\231\202\357\270\217" . ":man_getting_haircut:")
    ("💇🏻\342\200\215\342\231\202\357\270\217" . ":man_getting_haircut_tone1: :man_getting_haircut_light_skin_tone: :man_getting_haircut::skin-tone-1:")
    ("💇🏼\342\200\215\342\231\202\357\270\217" . ":man_getting_haircut_tone2: :man_getting_haircut_medium_light_skin_tone: :man_getting_haircut::skin-tone-2:")
    ("💇🏽\342\200\215\342\231\202\357\270\217" . ":man_getting_haircut_tone3: :man_getting_haircut_medium_skin_tone: :man_getting_haircut::skin-tone-3:")
    ("💇🏾\342\200\215\342\231\202\357\270\217" . ":man_getting_haircut_tone4: :man_getting_haircut_medium_dark_skin_tone: :man_getting_haircut::skin-tone-4:")
    ("💇🏿\342\200\215\342\231\202\357\270\217" . ":man_getting_haircut_tone5: :man_getting_haircut_dark_skin_tone: :man_getting_haircut::skin-tone-5:")
    ("💆" . ":person_getting_massage:")
    ("💆🏻" . ":person_getting_massage_tone1: :person_getting_massage::skin-tone-1:")
    ("💆🏼" . ":person_getting_massage_tone2: :person_getting_massage::skin-tone-2:")
    ("💆🏽" . ":person_getting_massage_tone3: :person_getting_massage::skin-tone-3:")
    ("💆🏾" . ":person_getting_massage_tone4: :person_getting_massage::skin-tone-4:")
    ("💆🏿" . ":person_getting_massage_tone5: :person_getting_massage::skin-tone-5:")
    ("💆\342\200\215\342\231\200\357\270\217" . ":woman_getting_face_massage:")
    ("💆🏻\342\200\215\342\231\200\357\270\217" . ":woman_getting_face_massage_tone1: :woman_getting_face_massage_light_skin_tone: :woman_getting_face_massage::skin-tone-1:")
    ("💆🏼\342\200\215\342\231\200\357\270\217" . ":woman_getting_face_massage_tone2: :woman_getting_face_massage_medium_light_skin_tone: :woman_getting_face_massage::skin-tone-2:")
    ("💆🏽\342\200\215\342\231\200\357\270\217" . ":woman_getting_face_massage_tone3: :woman_getting_face_massage_medium_skin_tone: :woman_getting_face_massage::skin-tone-3:")
    ("💆🏾\342\200\215\342\231\200\357\270\217" . ":woman_getting_face_massage_tone4: :woman_getting_face_massage_medium_dark_skin_tone: :woman_getting_face_massage::skin-tone-4:")
    ("💆🏿\342\200\215\342\231\200\357\270\217" . ":woman_getting_face_massage_tone5: :woman_getting_face_massage_dark_skin_tone: :woman_getting_face_massage::skin-tone-5:")
    ("💆\342\200\215\342\231\202\357\270\217" . ":man_getting_face_massage:")
    ("💆🏻\342\200\215\342\231\202\357\270\217" . ":man_getting_face_massage_tone1: :man_getting_face_massage_light_skin_tone: :man_getting_face_massage::skin-tone-1:")
    ("💆🏼\342\200\215\342\231\202\357\270\217" . ":man_getting_face_massage_tone2: :man_getting_face_massage_medium_light_skin_tone: :man_getting_face_massage::skin-tone-2:")
    ("💆🏽\342\200\215\342\231\202\357\270\217" . ":man_getting_face_massage_tone3: :man_getting_face_massage_medium_skin_tone: :man_getting_face_massage::skin-tone-3:")
    ("💆🏾\342\200\215\342\231\202\357\270\217" . ":man_getting_face_massage_tone4: :man_getting_face_massage_medium_dark_skin_tone: :man_getting_face_massage::skin-tone-4:")
    ("💆🏿\342\200\215\342\231\202\357\270\217" . ":man_getting_face_massage_tone5: :man_getting_face_massage_dark_skin_tone: :man_getting_face_massage::skin-tone-5:")
    ("🧖" . ":person_in_steamy_room:")
    ("🧖🏻" . ":person_in_steamy_room_tone1: :person_in_steamy_room_light_skin_tone: :person_in_steamy_room::skin-tone-1:")
    ("🧖🏼" . ":person_in_steamy_room_tone2: :person_in_steamy_room_medium_light_skin_tone: :person_in_steamy_room::skin-tone-2:")
    ("🧖🏽" . ":person_in_steamy_room_tone3: :person_in_steamy_room_medium_skin_tone: :person_in_steamy_room::skin-tone-3:")
    ("🧖🏾" . ":person_in_steamy_room_tone4: :person_in_steamy_room_medium_dark_skin_tone: :person_in_steamy_room::skin-tone-4:")
    ("🧖🏿" . ":person_in_steamy_room_tone5: :person_in_steamy_room_dark_skin_tone: :person_in_steamy_room::skin-tone-5:")
    ("🧖\342\200\215\342\231\200\357\270\217" . ":woman_in_steamy_room:")
    ("🧖🏻\342\200\215\342\231\200\357\270\217" . ":woman_in_steamy_room_tone1: :woman_in_steamy_room_light_skin_tone: :woman_in_steamy_room::skin-tone-1:")
    ("🧖🏼\342\200\215\342\231\200\357\270\217" . ":woman_in_steamy_room_tone2: :woman_in_steamy_room_medium_light_skin_tone: :woman_in_steamy_room::skin-tone-2:")
    ("🧖🏽\342\200\215\342\231\200\357\270\217" . ":woman_in_steamy_room_tone3: :woman_in_steamy_room_medium_skin_tone: :woman_in_steamy_room::skin-tone-3:")
    ("🧖🏾\342\200\215\342\231\200\357\270\217" . ":woman_in_steamy_room_tone4: :woman_in_steamy_room_medium_dark_skin_tone: :woman_in_steamy_room::skin-tone-4:")
    ("🧖🏿\342\200\215\342\231\200\357\270\217" . ":woman_in_steamy_room_tone5: :woman_in_steamy_room_dark_skin_tone: :woman_in_steamy_room::skin-tone-5:")
    ("🧖\342\200\215\342\231\202\357\270\217" . ":man_in_steamy_room:")
    ("🧖🏻\342\200\215\342\231\202\357\270\217" . ":man_in_steamy_room_tone1: :man_in_steamy_room_light_skin_tone: :man_in_steamy_room::skin-tone-1:")
    ("🧖🏼\342\200\215\342\231\202\357\270\217" . ":man_in_steamy_room_tone2: :man_in_steamy_room_medium_light_skin_tone: :man_in_steamy_room::skin-tone-2:")
    ("🧖🏽\342\200\215\342\231\202\357\270\217" . ":man_in_steamy_room_tone3: :man_in_steamy_room_medium_skin_tone: :man_in_steamy_room::skin-tone-3:")
    ("🧖🏾\342\200\215\342\231\202\357\270\217" . ":man_in_steamy_room_tone4: :man_in_steamy_room_medium_dark_skin_tone: :man_in_steamy_room::skin-tone-4:")
    ("🧖🏿\342\200\215\342\231\202\357\270\217" . ":man_in_steamy_room_tone5: :man_in_steamy_room_dark_skin_tone: :man_in_steamy_room::skin-tone-5:")
    ("💅" . ":nail_care: :nail_polish:")
    ("💅🏻" . ":nail_care_tone1: :nail_care::skin-tone-1: :nail_polish::skin-tone-1:")
    ("💅🏼" . ":nail_care_tone2: :nail_care::skin-tone-2: :nail_polish::skin-tone-2:")
    ("💅🏽" . ":nail_care_tone3: :nail_care::skin-tone-3: :nail_polish::skin-tone-3:")
    ("💅🏾" . ":nail_care_tone4: :nail_care::skin-tone-4: :nail_polish::skin-tone-4:")
    ("💅🏿" . ":nail_care_tone5: :nail_care::skin-tone-5: :nail_polish::skin-tone-5:")
    ("🤳" . ":selfie:")
    ("🤳🏻" . ":selfie_tone1: :selfie::skin-tone-1:")
    ("🤳🏼" . ":selfie_tone2: :selfie::skin-tone-2:")
    ("🤳🏽" . ":selfie_tone3: :selfie::skin-tone-3:")
    ("🤳🏾" . ":selfie_tone4: :selfie::skin-tone-4:")
    ("🤳🏿" . ":selfie_tone5: :selfie::skin-tone-5:")
    ("💃" . ":dancer: :woman_dancing:")
    ("💃🏻" . ":dancer_tone1: :dancer::skin-tone-1: :woman_dancing::skin-tone-1:")
    ("💃🏼" . ":dancer_tone2: :dancer::skin-tone-2: :woman_dancing::skin-tone-2:")
    ("💃🏽" . ":dancer_tone3: :dancer::skin-tone-3: :woman_dancing::skin-tone-3:")
    ("💃🏾" . ":dancer_tone4: :dancer::skin-tone-4: :woman_dancing::skin-tone-4:")
    ("💃🏿" . ":dancer_tone5: :dancer::skin-tone-5: :woman_dancing::skin-tone-5:")
    ("🕺" . ":man_dancing: :male_dancer:")
    ("🕺🏻" . ":man_dancing_tone1: :male_dancer_tone1: :man_dancing::skin-tone-1: :male_dancer::skin-tone-1:")
    ("🕺🏼" . ":man_dancing_tone2: :male_dancer_tone2: :man_dancing::skin-tone-2: :male_dancer::skin-tone-2:")
    ("🕺🏽" . ":man_dancing_tone3: :male_dancer_tone3: :man_dancing::skin-tone-3: :male_dancer::skin-tone-3:")
    ("🕺🏿" . ":man_dancing_tone5: :male_dancer_tone5: :man_dancing::skin-tone-5: :male_dancer::skin-tone-5:")
    ("🕺🏾" . ":man_dancing_tone4: :male_dancer_tone4: :man_dancing::skin-tone-4: :male_dancer::skin-tone-4:")
    ("👯" . ":people_with_bunny_ears_partying: :dancers:")
    ("👯\342\200\215\342\231\200\357\270\217" . ":women_with_bunny_ears_partying:")
    ("👯\342\200\215\342\231\202\357\270\217" . ":men_with_bunny_ears_partying:")
    ("🕴\357\270\217" . ":levitate: :man_in_business_suit_levitating:")
    ("🕴🏻" . ":levitate_tone1: :man_in_business_suit_levitating_tone1: :man_in_business_suit_levitating_light_skin_tone: :levitate::skin-tone-1: :man_in_business_suit_levitating::skin-tone-1:")
    ("🕴🏼" . ":levitate_tone2: :man_in_business_suit_levitating_tone2: :man_in_business_suit_levitating_medium_light_skin_tone: :levitate::skin-tone-2: :man_in_business_suit_levitating::skin-tone-2:")
    ("🕴🏽" . ":levitate_tone3: :man_in_business_suit_levitating_tone3: :man_in_business_suit_levitating_medium_skin_tone: :levitate::skin-tone-3: :man_in_business_suit_levitating::skin-tone-3:")
    ("🕴🏾" . ":levitate_tone4: :man_in_business_suit_levitating_tone4: :man_in_business_suit_levitating_medium_dark_skin_tone: :levitate::skin-tone-4: :man_in_business_suit_levitating::skin-tone-4:")
    ("🕴🏿" . ":levitate_tone5: :man_in_business_suit_levitating_tone5: :man_in_business_suit_levitating_dark_skin_tone: :levitate::skin-tone-5: :man_in_business_suit_levitating::skin-tone-5:")
    ("🧑\342\200\215🦽" . ":person_in_manual_wheelchair:")
    ("🧑🏻\342\200\215🦽" . ":person_in_manual_wheelchair_tone1: :person_in_manual_wheelchair_light_skin_tone: :person_in_manual_wheelchair::skin-tone-1:")
    ("🧑🏼\342\200\215🦽" . ":person_in_manual_wheelchair_tone2: :person_in_manual_wheelchair_medium_light_skin_tone: :person_in_manual_wheelchair::skin-tone-2:")
    ("🧑🏽\342\200\215🦽" . ":person_in_manual_wheelchair_tone3: :person_in_manual_wheelchair_medium_skin_tone: :person_in_manual_wheelchair::skin-tone-3:")
    ("🧑🏾\342\200\215🦽" . ":person_in_manual_wheelchair_tone4: :person_in_manual_wheelchair_medium_dark_skin_tone: :person_in_manual_wheelchair::skin-tone-4:")
    ("🧑🏿\342\200\215🦽" . ":person_in_manual_wheelchair_tone5: :person_in_manual_wheelchair_dark_skin_tone: :person_in_manual_wheelchair::skin-tone-5:")
    ("👩\342\200\215🦽" . ":woman_in_manual_wheelchair:")
    ("👩🏻\342\200\215🦽" . ":woman_in_manual_wheelchair_tone1: :woman_in_manual_wheelchair_light_skin_tone: :woman_in_manual_wheelchair::skin-tone-1:")
    ("👩🏼\342\200\215🦽" . ":woman_in_manual_wheelchair_tone2: :woman_in_manual_wheelchair_medium_light_skin_tone: :woman_in_manual_wheelchair::skin-tone-2:")
    ("👩🏽\342\200\215🦽" . ":woman_in_manual_wheelchair_tone3: :woman_in_manual_wheelchair_medium_skin_tone: :woman_in_manual_wheelchair::skin-tone-3:")
    ("👩🏾\342\200\215🦽" . ":woman_in_manual_wheelchair_tone4: :woman_in_manual_wheelchair_medium_dark_skin_tone: :woman_in_manual_wheelchair::skin-tone-4:")
    ("👩🏿\342\200\215🦽" . ":woman_in_manual_wheelchair_tone5: :woman_in_manual_wheelchair_dark_skin_tone: :woman_in_manual_wheelchair::skin-tone-5:")
    ("👨\342\200\215🦽" . ":man_in_manual_wheelchair:")
    ("👨🏻\342\200\215🦽" . ":man_in_manual_wheelchair_tone1: :man_in_manual_wheelchair_light_skin_tone: :man_in_manual_wheelchair::skin-tone-1:")
    ("👨🏼\342\200\215🦽" . ":man_in_manual_wheelchair_tone2: :man_in_manual_wheelchair_medium_light_skin_tone: :man_in_manual_wheelchair::skin-tone-2:")
    ("👨🏽\342\200\215🦽" . ":man_in_manual_wheelchair_tone3: :man_in_manual_wheelchair_medium_skin_tone: :man_in_manual_wheelchair::skin-tone-3:")
    ("👨🏾\342\200\215🦽" . ":man_in_manual_wheelchair_tone4: :man_in_manual_wheelchair_medium_dark_skin_tone: :man_in_manual_wheelchair::skin-tone-4:")
    ("👨🏿\342\200\215🦽" . ":man_in_manual_wheelchair_tone5: :man_in_manual_wheelchair_dark_skin_tone: :man_in_manual_wheelchair::skin-tone-5:")
    ("🧑\342\200\215🦽\342\200\215\342\236\241\357\270\217" . ":person_in_manual_wheelchair_facing_right:")
    ("🧑🏻\342\200\215🦽\342\200\215\342\236\241\357\270\217" . ":person_in_manual_wheelchair_facing_right_tone1: :person_in_manual_wheelchair_facing_right_light_skin_tone: :person_in_manual_wheelchair_facing_right::skin-tone-1:")
    ("🧑🏼\342\200\215🦽\342\200\215\342\236\241\357\270\217" . ":person_in_manual_wheelchair_facing_right_tone2: :person_in_manual_wheelchair_facing_right_medium_light_skin_tone: :person_in_manual_wheelchair_facing_right::skin-tone-2:")
    ("🧑🏽\342\200\215🦽\342\200\215\342\236\241\357\270\217" . ":person_in_manual_wheelchair_facing_right_tone3: :person_in_manual_wheelchair_facing_right_medium_skin_tone: :person_in_manual_wheelchair_facing_right::skin-tone-3:")
    ("🧑🏾\342\200\215🦽\342\200\215\342\236\241\357\270\217" . ":person_in_manual_wheelchair_facing_right_tone4: :person_in_manual_wheelchair_facing_right_medium_dark_skin_tone: :person_in_manual_wheelchair_facing_right::skin-tone-4:")
    ("🧑🏿\342\200\215🦽\342\200\215\342\236\241\357\270\217" . ":person_in_manual_wheelchair_facing_right_tone5: :person_in_manual_wheelchair_facing_right_dark_skin_tone: :person_in_manual_wheelchair_facing_right::skin-tone-5:")
    ("👨\342\200\215🦽\342\200\215\342\236\241\357\270\217" . ":man_in_manual_wheelchair_facing_right:")
    ("👨🏼\342\200\215🦽\342\200\215\342\236\241\357\270\217" . ":man_in_manual_wheelchair_facing_right_tone2: :man_in_manual_wheelchair_facing_right_medium_light_skin_tone: :man_in_manual_wheelchair_facing_right::skin-tone-2:")
    ("👨🏻\342\200\215🦽\342\200\215\342\236\241\357\270\217" . ":man_in_manual_wheelchair_facing_right_tone1: :man_in_manual_wheelchair_facing_right_light_skin_tone: :man_in_manual_wheelchair_facing_right::skin-tone-1:")
    ("👨🏽\342\200\215🦽\342\200\215\342\236\241\357\270\217" . ":man_in_manual_wheelchair_facing_right_tone3: :man_in_manual_wheelchair_facing_right_medium_skin_tone: :man_in_manual_wheelchair_facing_right::skin-tone-3:")
    ("👨🏾\342\200\215🦽\342\200\215\342\236\241\357\270\217" . ":man_in_manual_wheelchair_facing_right_tone4: :man_in_manual_wheelchair_facing_right_medium_dark_skin_tone: :man_in_manual_wheelchair_facing_right::skin-tone-4:")
    ("👨🏿\342\200\215🦽\342\200\215\342\236\241\357\270\217" . ":man_in_manual_wheelchair_facing_right_tone5: :man_in_manual_wheelchair_facing_right_dark_skin_tone: :man_in_manual_wheelchair_facing_right::skin-tone-5:")
    ("👩\342\200\215🦽\342\200\215\342\236\241\357\270\217" . ":woman_in_manual_wheelchair_facing_right:")
    ("👩🏻\342\200\215🦽\342\200\215\342\236\241\357\270\217" . ":woman_in_manual_wheelchair_facing_right_tone1: :woman_in_manual_wheelchair_facing_right_light_skin_tone: :woman_in_manual_wheelchair_facing_right::skin-tone-1:")
    ("👩🏼\342\200\215🦽\342\200\215\342\236\241\357\270\217" . ":woman_in_manual_wheelchair_facing_right_tone2: :woman_in_manual_wheelchair_facing_right_medium_light_skin_tone: :woman_in_manual_wheelchair_facing_right::skin-tone-2:")
    ("👩🏽\342\200\215🦽\342\200\215\342\236\241\357\270\217" . ":woman_in_manual_wheelchair_facing_right_tone3: :woman_in_manual_wheelchair_facing_right_medium_skin_tone: :woman_in_manual_wheelchair_facing_right::skin-tone-3:")
    ("👩🏾\342\200\215🦽\342\200\215\342\236\241\357\270\217" . ":woman_in_manual_wheelchair_facing_right_tone4: :woman_in_manual_wheelchair_facing_right_medium_dark_skin_tone: :woman_in_manual_wheelchair_facing_right::skin-tone-4:")
    ("👩🏿\342\200\215🦽\342\200\215\342\236\241\357\270\217" . ":woman_in_manual_wheelchair_facing_right_tone5: :woman_in_manual_wheelchair_facing_right_dark_skin_tone: :woman_in_manual_wheelchair_facing_right::skin-tone-5:")
    ("🧑\342\200\215🦼" . ":person_in_motorized_wheelchair:")
    ("🧑🏻\342\200\215🦼" . ":person_in_motorized_wheelchair_tone1: :person_in_motorized_wheelchair_light_skin_tone: :person_in_motorized_wheelchair::skin-tone-1:")
    ("🧑🏼\342\200\215🦼" . ":person_in_motorized_wheelchair_tone2: :person_in_motorized_wheelchair_medium_light_skin_tone: :person_in_motorized_wheelchair::skin-tone-2:")
    ("🧑🏽\342\200\215🦼" . ":person_in_motorized_wheelchair_tone3: :person_in_motorized_wheelchair_medium_skin_tone: :person_in_motorized_wheelchair::skin-tone-3:")
    ("🧑🏾\342\200\215🦼" . ":person_in_motorized_wheelchair_tone4: :person_in_motorized_wheelchair_medium_dark_skin_tone: :person_in_motorized_wheelchair::skin-tone-4:")
    ("🧑🏿\342\200\215🦼" . ":person_in_motorized_wheelchair_tone5: :person_in_motorized_wheelchair_dark_skin_tone: :person_in_motorized_wheelchair::skin-tone-5:")
    ("👩\342\200\215🦼" . ":woman_in_motorized_wheelchair:")
    ("👩🏻\342\200\215🦼" . ":woman_in_motorized_wheelchair_tone1: :woman_in_motorized_wheelchair_light_skin_tone: :woman_in_motorized_wheelchair::skin-tone-1:")
    ("👩🏼\342\200\215🦼" . ":woman_in_motorized_wheelchair_tone2: :woman_in_motorized_wheelchair_medium_light_skin_tone: :woman_in_motorized_wheelchair::skin-tone-2:")
    ("👩🏽\342\200\215🦼" . ":woman_in_motorized_wheelchair_tone3: :woman_in_motorized_wheelchair_medium_skin_tone: :woman_in_motorized_wheelchair::skin-tone-3:")
    ("👩🏾\342\200\215🦼" . ":woman_in_motorized_wheelchair_tone4: :woman_in_motorized_wheelchair_medium_dark_skin_tone: :woman_in_motorized_wheelchair::skin-tone-4:")
    ("👩🏿\342\200\215🦼" . ":woman_in_motorized_wheelchair_tone5: :woman_in_motorized_wheelchair_dark_skin_tone: :woman_in_motorized_wheelchair::skin-tone-5:")
    ("👨\342\200\215🦼" . ":man_in_motorized_wheelchair:")
    ("👨🏻\342\200\215🦼" . ":man_in_motorized_wheelchair_tone1: :man_in_motorized_wheelchair_light_skin_tone: :man_in_motorized_wheelchair::skin-tone-1:")
    ("👨🏼\342\200\215🦼" . ":man_in_motorized_wheelchair_tone2: :man_in_motorized_wheelchair_medium_light_skin_tone: :man_in_motorized_wheelchair::skin-tone-2:")
    ("👨🏽\342\200\215🦼" . ":man_in_motorized_wheelchair_tone3: :man_in_motorized_wheelchair_medium_skin_tone: :man_in_motorized_wheelchair::skin-tone-3:")
    ("👨🏾\342\200\215🦼" . ":man_in_motorized_wheelchair_tone4: :man_in_motorized_wheelchair_medium_dark_skin_tone: :man_in_motorized_wheelchair::skin-tone-4:")
    ("👨🏿\342\200\215🦼" . ":man_in_motorized_wheelchair_tone5: :man_in_motorized_wheelchair_dark_skin_tone: :man_in_motorized_wheelchair::skin-tone-5:")
    ("🧑\342\200\215🦼\342\200\215\342\236\241\357\270\217" . ":person_in_motorized_wheelchair_facing_right:")
    ("🧑🏻\342\200\215🦼\342\200\215\342\236\241\357\270\217" . ":person_in_motorized_wheelchair_facing_right_tone1: :person_in_motorized_wheelchair_facing_right_light_skin_tone: :person_in_motorized_wheelchair_facing_right::skin-tone-1:")
    ("🧑🏼\342\200\215🦼\342\200\215\342\236\241\357\270\217" . ":person_in_motorized_wheelchair_facing_right_tone2: :person_in_motorized_wheelchair_facing_right_medium_light_skin_tone: :person_in_motorized_wheelchair_facing_right::skin-tone-2:")
    ("🧑🏽\342\200\215🦼\342\200\215\342\236\241\357\270\217" . ":person_in_motorized_wheelchair_facing_right_tone3: :person_in_motorized_wheelchair_facing_right_medium_skin_tone: :person_in_motorized_wheelchair_facing_right::skin-tone-3:")
    ("🧑🏾\342\200\215🦼\342\200\215\342\236\241\357\270\217" . ":person_in_motorized_wheelchair_facing_right_tone4: :person_in_motorized_wheelchair_facing_right_medium_dark_skin_tone: :person_in_motorized_wheelchair_facing_right::skin-tone-4:")
    ("🧑🏿\342\200\215🦼\342\200\215\342\236\241\357\270\217" . ":person_in_motorized_wheelchair_facing_right_tone5: :person_in_motorized_wheelchair_facing_right_dark_skin_tone: :person_in_motorized_wheelchair_facing_right::skin-tone-5:")
    ("👨\342\200\215🦼\342\200\215\342\236\241\357\270\217" . ":man_in_motorized_wheelchair_facing_right:")
    ("👨🏻\342\200\215🦼\342\200\215\342\236\241\357\270\217" . ":man_in_motorized_wheelchair_facing_right_tone1: :man_in_motorized_wheelchair_facing_right_light_skin_tone: :man_in_motorized_wheelchair_facing_right::skin-tone-1:")
    ("👨🏼\342\200\215🦼\342\200\215\342\236\241\357\270\217" . ":man_in_motorized_wheelchair_facing_right_tone2: :man_in_motorized_wheelchair_facing_right_medium_light_skin_tone: :man_in_motorized_wheelchair_facing_right::skin-tone-2:")
    ("👨🏽\342\200\215🦼\342\200\215\342\236\241\357\270\217" . ":man_in_motorized_wheelchair_facing_right_tone3: :man_in_motorized_wheelchair_facing_right_medium_skin_tone: :man_in_motorized_wheelchair_facing_right::skin-tone-3:")
    ("👨🏾\342\200\215🦼\342\200\215\342\236\241\357\270\217" . ":man_in_motorized_wheelchair_facing_right_tone4: :man_in_motorized_wheelchair_facing_right_medium_dark_skin_tone: :man_in_motorized_wheelchair_facing_right::skin-tone-4:")
    ("👨🏿\342\200\215🦼\342\200\215\342\236\241\357\270\217" . ":man_in_motorized_wheelchair_facing_right_tone5: :man_in_motorized_wheelchair_facing_right_dark_skin_tone: :man_in_motorized_wheelchair_facing_right::skin-tone-5:")
    ("👩\342\200\215🦼\342\200\215\342\236\241\357\270\217" . ":woman_in_motorized_wheelchair_facing_right:")
    ("👩🏻\342\200\215🦼\342\200\215\342\236\241\357\270\217" . ":woman_in_motorized_wheelchair_facing_right_tone1: :woman_in_motorized_wheelchair_facing_right_light_skin_tone: :woman_in_motorized_wheelchair_facing_right::skin-tone-1:")
    ("👩🏼\342\200\215🦼\342\200\215\342\236\241\357\270\217" . ":woman_in_motorized_wheelchair_facing_right_tone2: :woman_in_motorized_wheelchair_facing_right_medium_light_skin_tone: :woman_in_motorized_wheelchair_facing_right::skin-tone-2:")
    ("👩🏽\342\200\215🦼\342\200\215\342\236\241\357\270\217" . ":woman_in_motorized_wheelchair_facing_right_tone3: :woman_in_motorized_wheelchair_facing_right_medium_skin_tone: :woman_in_motorized_wheelchair_facing_right::skin-tone-3:")
    ("👩🏾\342\200\215🦼\342\200\215\342\236\241\357\270\217" . ":woman_in_motorized_wheelchair_facing_right_tone4: :woman_in_motorized_wheelchair_facing_right_medium_dark_skin_tone: :woman_in_motorized_wheelchair_facing_right::skin-tone-4:")
    ("👩🏿\342\200\215🦼\342\200\215\342\236\241\357\270\217" . ":woman_in_motorized_wheelchair_facing_right_tone5: :woman_in_motorized_wheelchair_facing_right_dark_skin_tone: :woman_in_motorized_wheelchair_facing_right::skin-tone-5:")
    ("🚶" . ":person_walking:")
    ("🚶🏻" . ":person_walking_tone1: :person_walking::skin-tone-1:")
    ("🚶🏼" . ":person_walking_tone2: :person_walking::skin-tone-2:")
    ("🚶🏽" . ":person_walking_tone3: :person_walking::skin-tone-3:")
    ("🚶🏾" . ":person_walking_tone4: :person_walking::skin-tone-4:")
    ("🚶🏿" . ":person_walking_tone5: :person_walking::skin-tone-5:")
    ("🚶\342\200\215\342\231\200\357\270\217" . ":woman_walking:")
    ("🚶🏻\342\200\215\342\231\200\357\270\217" . ":woman_walking_tone1: :woman_walking_light_skin_tone: :woman_walking::skin-tone-1:")
    ("🚶🏼\342\200\215\342\231\200\357\270\217" . ":woman_walking_tone2: :woman_walking_medium_light_skin_tone: :woman_walking::skin-tone-2:")
    ("🚶🏽\342\200\215\342\231\200\357\270\217" . ":woman_walking_tone3: :woman_walking_medium_skin_tone: :woman_walking::skin-tone-3:")
    ("🚶🏾\342\200\215\342\231\200\357\270\217" . ":woman_walking_tone4: :woman_walking_medium_dark_skin_tone: :woman_walking::skin-tone-4:")
    ("🚶🏿\342\200\215\342\231\200\357\270\217" . ":woman_walking_tone5: :woman_walking_dark_skin_tone: :woman_walking::skin-tone-5:")
    ("🚶\342\200\215\342\231\202\357\270\217" . ":man_walking:")
    ("🚶🏻\342\200\215\342\231\202\357\270\217" . ":man_walking_tone1: :man_walking_light_skin_tone: :man_walking::skin-tone-1:")
    ("🚶🏼\342\200\215\342\231\202\357\270\217" . ":man_walking_tone2: :man_walking_medium_light_skin_tone: :man_walking::skin-tone-2:")
    ("🚶🏽\342\200\215\342\231\202\357\270\217" . ":man_walking_tone3: :man_walking_medium_skin_tone: :man_walking::skin-tone-3:")
    ("🚶🏾\342\200\215\342\231\202\357\270\217" . ":man_walking_tone4: :man_walking_medium_dark_skin_tone: :man_walking::skin-tone-4:")
    ("🚶🏿\342\200\215\342\231\202\357\270\217" . ":man_walking_tone5: :man_walking_dark_skin_tone: :man_walking::skin-tone-5:")
    ("🚶\342\200\215\342\236\241\357\270\217" . ":person_walking_facing_right:")
    ("🚶🏻\342\200\215\342\236\241\357\270\217" . ":person_walking_facing_right_tone1: :person_walking_facing_right_light_skin_tone: :person_walking_facing_right::skin-tone-1:")
    ("🚶🏼\342\200\215\342\236\241\357\270\217" . ":person_walking_facing_right_tone2: :person_walking_facing_right_medium_light_skin_tone: :person_walking_facing_right::skin-tone-2:")
    ("🚶🏽\342\200\215\342\236\241\357\270\217" . ":person_walking_facing_right_tone3: :person_walking_facing_right_medium_skin_tone: :person_walking_facing_right::skin-tone-3:")
    ("🚶🏾\342\200\215\342\236\241\357\270\217" . ":person_walking_facing_right_tone4: :person_walking_facing_right_medium_dark_skin_tone: :person_walking_facing_right::skin-tone-4:")
    ("🚶🏿\342\200\215\342\236\241\357\270\217" . ":person_walking_facing_right_tone5: :person_walking_facing_right_dark_skin_tone: :person_walking_facing_right::skin-tone-5:")
    ("🚶\342\200\215\342\231\200\357\270\217\342\200\215\342\236\241\357\270\217" . ":woman_walking_facing_right:")
    ("🚶🏻\342\200\215\342\231\200\357\270\217\342\200\215\342\236\241\357\270\217" . ":woman_walking_facing_right_tone1: :woman_walking_facing_right_light_skin_tone: :woman_walking_facing_right::skin-tone-1:")
    ("🚶🏼\342\200\215\342\231\200\357\270\217\342\200\215\342\236\241\357\270\217" . ":woman_walking_facing_right_tone2: :woman_walking_facing_right_medium_light_skin_tone: :woman_walking_facing_right::skin-tone-2:")
    ("🚶🏽\342\200\215\342\231\200\357\270\217\342\200\215\342\236\241\357\270\217" . ":woman_walking_facing_right_tone3: :woman_walking_facing_right_medium_skin_tone: :woman_walking_facing_right::skin-tone-3:")
    ("🚶🏾\342\200\215\342\231\200\357\270\217\342\200\215\342\236\241\357\270\217" . ":woman_walking_facing_right_tone4: :woman_walking_facing_right_medium_dark_skin_tone: :woman_walking_facing_right::skin-tone-4:")
    ("🚶🏿\342\200\215\342\231\200\357\270\217\342\200\215\342\236\241\357\270\217" . ":woman_walking_facing_right_tone5: :woman_walking_facing_right_dark_skin_tone: :woman_walking_facing_right::skin-tone-5:")
    ("🚶\342\200\215\342\231\202\357\270\217\342\200\215\342\236\241\357\270\217" . ":man_walking_facing_right:")
    ("🚶🏻\342\200\215\342\231\202\357\270\217\342\200\215\342\236\241\357\270\217" . ":man_walking_facing_right_tone1: :man_walking_facing_right_light_skin_tone: :man_walking_facing_right::skin-tone-1:")
    ("🚶🏼\342\200\215\342\231\202\357\270\217\342\200\215\342\236\241\357\270\217" . ":man_walking_facing_right_tone2: :man_walking_facing_right_medium_light_skin_tone: :man_walking_facing_right::skin-tone-2:")
    ("🚶🏽\342\200\215\342\231\202\357\270\217\342\200\215\342\236\241\357\270\217" . ":man_walking_facing_right_tone3: :man_walking_facing_right_medium_skin_tone: :man_walking_facing_right::skin-tone-3:")
    ("🚶🏾\342\200\215\342\231\202\357\270\217\342\200\215\342\236\241\357\270\217" . ":man_walking_facing_right_tone4: :man_walking_facing_right_medium_dark_skin_tone: :man_walking_facing_right::skin-tone-4:")
    ("🚶🏿\342\200\215\342\231\202\357\270\217\342\200\215\342\236\241\357\270\217" . ":man_walking_facing_right_tone5: :man_walking_facing_right_dark_skin_tone: :man_walking_facing_right::skin-tone-5:")
    ("🧑\342\200\215🦯" . ":person_with_probing_cane:")
    ("🧑🏻\342\200\215🦯" . ":person_with_probing_cane_tone1: :person_with_probing_cane_light_skin_tone: :person_with_probing_cane::skin-tone-1:")
    ("🧑🏼\342\200\215🦯" . ":person_with_probing_cane_tone2: :person_with_probing_cane_medium_light_skin_tone: :person_with_probing_cane::skin-tone-2:")
    ("🧑🏽\342\200\215🦯" . ":person_with_probing_cane_tone3: :person_with_probing_cane_medium_skin_tone: :person_with_probing_cane::skin-tone-3:")
    ("🧑🏾\342\200\215🦯" . ":person_with_probing_cane_tone4: :person_with_probing_cane_medium_dark_skin_tone: :person_with_probing_cane::skin-tone-4:")
    ("🧑🏿\342\200\215🦯" . ":person_with_probing_cane_tone5: :person_with_probing_cane_dark_skin_tone: :person_with_probing_cane::skin-tone-5:")
    ("👩\342\200\215🦯" . ":woman_with_probing_cane:")
    ("👩🏻\342\200\215🦯" . ":woman_with_probing_cane_tone1: :woman_with_probing_cane_light_skin_tone: :woman_with_probing_cane::skin-tone-1:")
    ("👩🏼\342\200\215🦯" . ":woman_with_probing_cane_tone2: :woman_with_probing_cane_medium_light_skin_tone: :woman_with_probing_cane::skin-tone-2:")
    ("👩🏽\342\200\215🦯" . ":woman_with_probing_cane_tone3: :woman_with_probing_cane_medium_skin_tone: :woman_with_probing_cane::skin-tone-3:")
    ("👩🏾\342\200\215🦯" . ":woman_with_probing_cane_tone4: :woman_with_probing_cane_medium_dark_skin_tone: :woman_with_probing_cane::skin-tone-4:")
    ("👩🏿\342\200\215🦯" . ":woman_with_probing_cane_tone5: :woman_with_probing_cane_dark_skin_tone: :woman_with_probing_cane::skin-tone-5:")
    ("👨\342\200\215🦯" . ":man_with_probing_cane:")
    ("👨🏻\342\200\215🦯" . ":man_with_probing_cane_tone1: :man_with_probing_cane_light_skin_tone: :man_with_probing_cane::skin-tone-1:")
    ("👨🏼\342\200\215🦯" . ":man_with_probing_cane_tone2: :man_with_probing_cane_medium_light_skin_tone: :man_with_probing_cane::skin-tone-2:")
    ("👨🏽\342\200\215🦯" . ":man_with_probing_cane_tone3: :man_with_probing_cane_medium_skin_tone: :man_with_probing_cane::skin-tone-3:")
    ("👨🏾\342\200\215🦯" . ":man_with_probing_cane_tone4: :man_with_probing_cane_medium_dark_skin_tone: :man_with_probing_cane::skin-tone-4:")
    ("👨🏿\342\200\215🦯" . ":man_with_probing_cane_tone5: :man_with_probing_cane_dark_skin_tone: :man_with_probing_cane::skin-tone-5:")
    ("🧑\342\200\215🦯\342\200\215\342\236\241\357\270\217" . ":person_with_white_cane_facing_right:")
    ("🧑🏻\342\200\215🦯\342\200\215\342\236\241\357\270\217" . ":person_with_white_cane_facing_right_tone1: :person_with_white_cane_facing_right_light_skin_tone: :person_with_white_cane_facing_right::skin-tone-1:")
    ("🧑🏼\342\200\215🦯\342\200\215\342\236\241\357\270\217" . ":person_with_white_cane_facing_right_tone2: :person_with_white_cane_facing_right_medium_light_skin_tone: :person_with_white_cane_facing_right::skin-tone-2:")
    ("🧑🏽\342\200\215🦯\342\200\215\342\236\241\357\270\217" . ":person_with_white_cane_facing_right_tone3: :person_with_white_cane_facing_right_medium_skin_tone: :person_with_white_cane_facing_right::skin-tone-3:")
    ("🧑🏾\342\200\215🦯\342\200\215\342\236\241\357\270\217" . ":person_with_white_cane_facing_right_tone4: :person_with_white_cane_facing_right_medium_dark_skin_tone: :person_with_white_cane_facing_right::skin-tone-4:")
    ("🧑🏿\342\200\215🦯\342\200\215\342\236\241\357\270\217" . ":person_with_white_cane_facing_right_tone5: :person_with_white_cane_facing_right_dark_skin_tone: :person_with_white_cane_facing_right::skin-tone-5:")
    ("👨\342\200\215🦯\342\200\215\342\236\241\357\270\217" . ":man_with_white_cane_facing_right:")
    ("👨🏻\342\200\215🦯\342\200\215\342\236\241\357\270\217" . ":man_with_white_cane_facing_right_tone1: :man_with_white_cane_facing_right_light_skin_tone: :man_with_white_cane_facing_right::skin-tone-1:")
    ("👨🏼\342\200\215🦯\342\200\215\342\236\241\357\270\217" . ":man_with_white_cane_facing_right_tone2: :man_with_white_cane_facing_right_medium_light_skin_tone: :man_with_white_cane_facing_right::skin-tone-2:")
    ("👨🏽\342\200\215🦯\342\200\215\342\236\241\357\270\217" . ":man_with_white_cane_facing_right_tone3: :man_with_white_cane_facing_right_medium_skin_tone: :man_with_white_cane_facing_right::skin-tone-3:")
    ("👨🏾\342\200\215🦯\342\200\215\342\236\241\357\270\217" . ":man_with_white_cane_facing_right_tone4: :man_with_white_cane_facing_right_medium_dark_skin_tone: :man_with_white_cane_facing_right::skin-tone-4:")
    ("👨🏿\342\200\215🦯\342\200\215\342\236\241\357\270\217" . ":man_with_white_cane_facing_right_tone5: :man_with_white_cane_facing_right_dark_skin_tone: :man_with_white_cane_facing_right::skin-tone-5:")
    ("👩\342\200\215🦯\342\200\215\342\236\241\357\270\217" . ":woman_with_white_cane_facing_right:")
    ("👩🏻\342\200\215🦯\342\200\215\342\236\241\357\270\217" . ":woman_with_white_cane_facing_right_tone1: :woman_with_white_cane_facing_right_light_skin_tone: :woman_with_white_cane_facing_right::skin-tone-1:")
    ("👩🏼\342\200\215🦯\342\200\215\342\236\241\357\270\217" . ":woman_with_white_cane_facing_right_tone2: :woman_with_white_cane_facing_right_medium_light_skin_tone: :woman_with_white_cane_facing_right::skin-tone-2:")
    ("👩🏽\342\200\215🦯\342\200\215\342\236\241\357\270\217" . ":woman_with_white_cane_facing_right_tone3: :woman_with_white_cane_facing_right_medium_skin_tone: :woman_with_white_cane_facing_right::skin-tone-3:")
    ("👩🏾\342\200\215🦯\342\200\215\342\236\241\357\270\217" . ":woman_with_white_cane_facing_right_tone4: :woman_with_white_cane_facing_right_medium_dark_skin_tone: :woman_with_white_cane_facing_right::skin-tone-4:")
    ("👩🏿\342\200\215🦯\342\200\215\342\236\241\357\270\217" . ":woman_with_white_cane_facing_right_tone5: :woman_with_white_cane_facing_right_dark_skin_tone: :woman_with_white_cane_facing_right::skin-tone-5:")
    ("🧎" . ":person_kneeling:")
    ("🧎🏻" . ":person_kneeling_tone1: :person_kneeling_light_skin_tone: :person_kneeling::skin-tone-1:")
    ("🧎🏼" . ":person_kneeling_tone2: :person_kneeling_medium_light_skin_tone: :person_kneeling::skin-tone-2:")
    ("🧎🏽" . ":person_kneeling_tone3: :person_kneeling_medium_skin_tone: :person_kneeling::skin-tone-3:")
    ("🧎🏾" . ":person_kneeling_tone4: :person_kneeling_medium_dark_skin_tone: :person_kneeling::skin-tone-4:")
    ("🧎🏿" . ":person_kneeling_tone5: :person_kneeling_dark_skin_tone: :person_kneeling::skin-tone-5:")
    ("🧎\342\200\215\342\231\200\357\270\217" . ":woman_kneeling:")
    ("🧎🏻\342\200\215\342\231\200\357\270\217" . ":woman_kneeling_tone1: :woman_kneeling_light_skin_tone: :woman_kneeling::skin-tone-1:")
    ("🧎🏼\342\200\215\342\231\200\357\270\217" . ":woman_kneeling_tone2: :woman_kneeling_medium_light_skin_tone: :woman_kneeling::skin-tone-2:")
    ("🧎🏽\342\200\215\342\231\200\357\270\217" . ":woman_kneeling_tone3: :woman_kneeling_medium_skin_tone: :woman_kneeling::skin-tone-3:")
    ("🧎🏾\342\200\215\342\231\200\357\270\217" . ":woman_kneeling_tone4: :woman_kneeling_medium_dark_skin_tone: :woman_kneeling::skin-tone-4:")
    ("🧎🏿\342\200\215\342\231\200\357\270\217" . ":woman_kneeling_tone5: :woman_kneeling_dark_skin_tone: :woman_kneeling::skin-tone-5:")
    ("🧎\342\200\215\342\231\202\357\270\217" . ":man_kneeling:")
    ("🧎🏻\342\200\215\342\231\202\357\270\217" . ":man_kneeling_tone1: :man_kneeling_light_skin_tone: :man_kneeling::skin-tone-1:")
    ("🧎🏼\342\200\215\342\231\202\357\270\217" . ":man_kneeling_tone2: :man_kneeling_medium_light_skin_tone: :man_kneeling::skin-tone-2:")
    ("🧎🏽\342\200\215\342\231\202\357\270\217" . ":man_kneeling_tone3: :man_kneeling_medium_skin_tone: :man_kneeling::skin-tone-3:")
    ("🧎🏾\342\200\215\342\231\202\357\270\217" . ":man_kneeling_tone4: :man_kneeling_medium_dark_skin_tone: :man_kneeling::skin-tone-4:")
    ("🧎🏿\342\200\215\342\231\202\357\270\217" . ":man_kneeling_tone5: :man_kneeling_dark_skin_tone: :man_kneeling::skin-tone-5:")
    ("🧎\342\200\215\342\236\241\357\270\217" . ":person_kneeling_facing_right:")
    ("🧎🏻\342\200\215\342\236\241\357\270\217" . ":person_kneeling_facing_right_tone1: :person_kneeling_facing_right_light_skin_tone: :person_kneeling_facing_right::skin-tone-1:")
    ("🧎🏼\342\200\215\342\236\241\357\270\217" . ":person_kneeling_facing_right_tone2: :person_kneeling_facing_right_medium_light_skin_tone: :person_kneeling_facing_right::skin-tone-2:")
    ("🧎🏽\342\200\215\342\236\241\357\270\217" . ":person_kneeling_facing_right_tone3: :person_kneeling_facing_right_medium_skin_tone: :person_kneeling_facing_right::skin-tone-3:")
    ("🧎🏾\342\200\215\342\236\241\357\270\217" . ":person_kneeling_facing_right_tone4: :person_kneeling_facing_right_medium_dark_skin_tone: :person_kneeling_facing_right::skin-tone-4:")
    ("🧎🏿\342\200\215\342\236\241\357\270\217" . ":person_kneeling_facing_right_tone5: :person_kneeling_facing_right_dark_skin_tone: :person_kneeling_facing_right::skin-tone-5:")
    ("🧎\342\200\215\342\231\200\357\270\217\342\200\215\342\236\241\357\270\217" . ":woman_kneeling_facing_right:")
    ("🧎🏻\342\200\215\342\231\200\357\270\217\342\200\215\342\236\241\357\270\217" . ":woman_kneeling_facing_right_tone1: :woman_kneeling_facing_right_light_skin_tone: :woman_kneeling_facing_right::skin-tone-1:")
    ("🧎🏼\342\200\215\342\231\200\357\270\217\342\200\215\342\236\241\357\270\217" . ":woman_kneeling_facing_right_tone2: :woman_kneeling_facing_right_medium_light_skin_tone: :woman_kneeling_facing_right::skin-tone-2:")
    ("🧎🏽\342\200\215\342\231\200\357\270\217\342\200\215\342\236\241\357\270\217" . ":woman_kneeling_facing_right_tone3: :woman_kneeling_facing_right_medium_skin_tone: :woman_kneeling_facing_right::skin-tone-3:")
    ("🧎🏾\342\200\215\342\231\200\357\270\217\342\200\215\342\236\241\357\270\217" . ":woman_kneeling_facing_right_tone4: :woman_kneeling_facing_right_medium_dark_skin_tone: :woman_kneeling_facing_right::skin-tone-4:")
    ("🧎🏿\342\200\215\342\231\200\357\270\217\342\200\215\342\236\241\357\270\217" . ":woman_kneeling_facing_right_tone5: :woman_kneeling_facing_right_dark_skin_tone: :woman_kneeling_facing_right::skin-tone-5:")
    ("🧎\342\200\215\342\231\202\357\270\217\342\200\215\342\236\241\357\270\217" . ":man_kneeling_facing_right:")
    ("🧎🏻\342\200\215\342\231\202\357\270\217\342\200\215\342\236\241\357\270\217" . ":man_kneeling_facing_right_tone1: :man_kneeling_facing_right_light_skin_tone: :man_kneeling_facing_right::skin-tone-1:")
    ("🧎🏼\342\200\215\342\231\202\357\270\217\342\200\215\342\236\241\357\270\217" . ":man_kneeling_facing_right_tone2: :man_kneeling_facing_right_medium_light_skin_tone: :man_kneeling_facing_right::skin-tone-2:")
    ("🧎🏽\342\200\215\342\231\202\357\270\217\342\200\215\342\236\241\357\270\217" . ":man_kneeling_facing_right_tone3: :man_kneeling_facing_right_medium_skin_tone: :man_kneeling_facing_right::skin-tone-3:")
    ("🧎🏾\342\200\215\342\231\202\357\270\217\342\200\215\342\236\241\357\270\217" . ":man_kneeling_facing_right_tone4: :man_kneeling_facing_right_medium_dark_skin_tone: :man_kneeling_facing_right::skin-tone-4:")
    ("🧎🏿\342\200\215\342\231\202\357\270\217\342\200\215\342\236\241\357\270\217" . ":man_kneeling_facing_right_tone5: :man_kneeling_facing_right_dark_skin_tone: :man_kneeling_facing_right::skin-tone-5:")
    ("🏃" . ":person_running: :runner:")
    ("🏃🏻" . ":person_running_tone1: :runner_tone1: :person_running::skin-tone-1: :runner::skin-tone-1:")
    ("🏃🏼" . ":person_running_tone2: :runner_tone2: :person_running::skin-tone-2: :runner::skin-tone-2:")
    ("🏃🏽" . ":person_running_tone3: :runner_tone3: :person_running::skin-tone-3: :runner::skin-tone-3:")
    ("🏃🏾" . ":person_running_tone4: :runner_tone4: :person_running::skin-tone-4: :runner::skin-tone-4:")
    ("🏃🏿" . ":person_running_tone5: :runner_tone5: :person_running::skin-tone-5: :runner::skin-tone-5:")
    ("🏃\342\200\215\342\231\200\357\270\217" . ":woman_running:")
    ("🏃🏻\342\200\215\342\231\200\357\270\217" . ":woman_running_tone1: :woman_running_light_skin_tone: :woman_running::skin-tone-1:")
    ("🏃🏼\342\200\215\342\231\200\357\270\217" . ":woman_running_tone2: :woman_running_medium_light_skin_tone: :woman_running::skin-tone-2:")
    ("🏃🏽\342\200\215\342\231\200\357\270\217" . ":woman_running_tone3: :woman_running_medium_skin_tone: :woman_running::skin-tone-3:")
    ("🏃🏾\342\200\215\342\231\200\357\270\217" . ":woman_running_tone4: :woman_running_medium_dark_skin_tone: :woman_running::skin-tone-4:")
    ("🏃🏿\342\200\215\342\231\200\357\270\217" . ":woman_running_tone5: :woman_running_dark_skin_tone: :woman_running::skin-tone-5:")
    ("🏃\342\200\215\342\231\202\357\270\217" . ":man_running:")
    ("🏃🏻\342\200\215\342\231\202\357\270\217" . ":man_running_tone1: :man_running_light_skin_tone: :man_running::skin-tone-1:")
    ("🏃🏼\342\200\215\342\231\202\357\270\217" . ":man_running_tone2: :man_running_medium_light_skin_tone: :man_running::skin-tone-2:")
    ("🏃🏽\342\200\215\342\231\202\357\270\217" . ":man_running_tone3: :man_running_medium_skin_tone: :man_running::skin-tone-3:")
    ("🏃🏾\342\200\215\342\231\202\357\270\217" . ":man_running_tone4: :man_running_medium_dark_skin_tone: :man_running::skin-tone-4:")
    ("🏃🏿\342\200\215\342\231\202\357\270\217" . ":man_running_tone5: :man_running_dark_skin_tone: :man_running::skin-tone-5:")
    ("🏃\342\200\215\342\236\241\357\270\217" . ":person_running_facing_right:")
    ("🏃🏻\342\200\215\342\236\241\357\270\217" . ":person_running_facing_right_tone1: :person_running_facing_right_light_skin_tone: :person_running_facing_right::skin-tone-1:")
    ("🏃🏼\342\200\215\342\236\241\357\270\217" . ":person_running_facing_right_tone2: :person_running_facing_right_medium_light_skin_tone: :person_running_facing_right::skin-tone-2:")
    ("🏃🏽\342\200\215\342\236\241\357\270\217" . ":person_running_facing_right_tone3: :person_running_facing_right_medium_skin_tone: :person_running_facing_right::skin-tone-3:")
    ("🏃🏾\342\200\215\342\236\241\357\270\217" . ":person_running_facing_right_tone4: :person_running_facing_right_medium_dark_skin_tone: :person_running_facing_right::skin-tone-4:")
    ("🏃🏿\342\200\215\342\236\241\357\270\217" . ":person_running_facing_right_tone5: :person_running_facing_right_dark_skin_tone: :person_running_facing_right::skin-tone-5:")
    ("🏃\342\200\215\342\231\200\357\270\217\342\200\215\342\236\241\357\270\217" . ":woman_running_facing_right:")
    ("🏃🏻\342\200\215\342\231\200\357\270\217\342\200\215\342\236\241\357\270\217" . ":woman_running_facing_right_tone1: :woman_running_facing_right_light_skin_tone: :woman_running_facing_right::skin-tone-1:")
    ("🏃🏼\342\200\215\342\231\200\357\270\217\342\200\215\342\236\241\357\270\217" . ":woman_running_facing_right_tone2: :woman_running_facing_right_medium_light_skin_tone: :woman_running_facing_right::skin-tone-2:")
    ("🏃🏽\342\200\215\342\231\200\357\270\217\342\200\215\342\236\241\357\270\217" . ":woman_running_facing_right_tone3: :woman_running_facing_right_medium_skin_tone: :woman_running_facing_right::skin-tone-3:")
    ("🏃🏾\342\200\215\342\231\200\357\270\217\342\200\215\342\236\241\357\270\217" . ":woman_running_facing_right_tone4: :woman_running_facing_right_medium_dark_skin_tone: :woman_running_facing_right::skin-tone-4:")
    ("🏃🏿\342\200\215\342\231\200\357\270\217\342\200\215\342\236\241\357\270\217" . ":woman_running_facing_right_tone5: :woman_running_facing_right_dark_skin_tone: :woman_running_facing_right::skin-tone-5:")
    ("🏃\342\200\215\342\231\202\357\270\217\342\200\215\342\236\241\357\270\217" . ":man_running_facing_right:")
    ("🏃🏻\342\200\215\342\231\202\357\270\217\342\200\215\342\236\241\357\270\217" . ":man_running_facing_right_tone1: :man_running_facing_right_light_skin_tone: :man_running_facing_right::skin-tone-1:")
    ("🏃🏼\342\200\215\342\231\202\357\270\217\342\200\215\342\236\241\357\270\217" . ":man_running_facing_right_tone2: :man_running_facing_right_medium_light_skin_tone: :man_running_facing_right::skin-tone-2:")
    ("🏃🏽\342\200\215\342\231\202\357\270\217\342\200\215\342\236\241\357\270\217" . ":man_running_facing_right_tone3: :man_running_facing_right_medium_skin_tone: :man_running_facing_right::skin-tone-3:")
    ("🏃🏾\342\200\215\342\231\202\357\270\217\342\200\215\342\236\241\357\270\217" . ":man_running_facing_right_tone4: :man_running_facing_right_medium_dark_skin_tone: :man_running_facing_right::skin-tone-4:")
    ("🏃🏿\342\200\215\342\231\202\357\270\217\342\200\215\342\236\241\357\270\217" . ":man_running_facing_right_tone5: :man_running_facing_right_dark_skin_tone: :man_running_facing_right::skin-tone-5:")
    ("🧍" . ":person_standing:")
    ("🧍🏻" . ":person_standing_tone1: :person_standing_light_skin_tone: :person_standing::skin-tone-1:")
    ("🧍🏼" . ":person_standing_tone2: :person_standing_medium_light_skin_tone: :person_standing::skin-tone-2:")
    ("🧍🏽" . ":person_standing_tone3: :person_standing_medium_skin_tone: :person_standing::skin-tone-3:")
    ("🧍🏾" . ":person_standing_tone4: :person_standing_medium_dark_skin_tone: :person_standing::skin-tone-4:")
    ("🧍🏿" . ":person_standing_tone5: :person_standing_dark_skin_tone: :person_standing::skin-tone-5:")
    ("🧍\342\200\215\342\231\200\357\270\217" . ":woman_standing:")
    ("🧍🏻\342\200\215\342\231\200\357\270\217" . ":woman_standing_tone1: :woman_standing_light_skin_tone: :woman_standing::skin-tone-1:")
    ("🧍🏼\342\200\215\342\231\200\357\270\217" . ":woman_standing_tone2: :woman_standing_medium_light_skin_tone: :woman_standing::skin-tone-2:")
    ("🧍🏽\342\200\215\342\231\200\357\270\217" . ":woman_standing_tone3: :woman_standing_medium_skin_tone: :woman_standing::skin-tone-3:")
    ("🧍🏾\342\200\215\342\231\200\357\270\217" . ":woman_standing_tone4: :woman_standing_medium_dark_skin_tone: :woman_standing::skin-tone-4:")
    ("🧍🏿\342\200\215\342\231\200\357\270\217" . ":woman_standing_tone5: :woman_standing_dark_skin_tone: :woman_standing::skin-tone-5:")
    ("🧍\342\200\215\342\231\202\357\270\217" . ":man_standing:")
    ("🧍🏻\342\200\215\342\231\202\357\270\217" . ":man_standing_tone1: :man_standing_light_skin_tone: :man_standing::skin-tone-1:")
    ("🧍🏼\342\200\215\342\231\202\357\270\217" . ":man_standing_tone2: :man_standing_medium_light_skin_tone: :man_standing::skin-tone-2:")
    ("🧍🏽\342\200\215\342\231\202\357\270\217" . ":man_standing_tone3: :man_standing_medium_skin_tone: :man_standing::skin-tone-3:")
    ("🧍🏾\342\200\215\342\231\202\357\270\217" . ":man_standing_tone4: :man_standing_medium_dark_skin_tone: :man_standing::skin-tone-4:")
    ("🧍🏿\342\200\215\342\231\202\357\270\217" . ":man_standing_tone5: :man_standing_dark_skin_tone: :man_standing::skin-tone-5:")
    ("🧑\342\200\215🤝\342\200\215🧑" . ":people_holding_hands:")
    ("🧑🏻\342\200\215🤝\342\200\215🧑🏻" . ":people_holding_hands_tone1: :people_holding_hands_light_skin_tone:")
    ("🧑🏻\342\200\215🤝\342\200\215🧑🏼" . ":people_holding_hands_tone1_tone2: :people_holding_hands_light_skin_tone_medium_light_skin_tone:")
    ("🧑🏻\342\200\215🤝\342\200\215🧑🏽" . ":people_holding_hands_tone1_tone3: :people_holding_hands_light_skin_tone_medium_skin_tone:")
    ("🧑🏻\342\200\215🤝\342\200\215🧑🏾" . ":people_holding_hands_tone1_tone4: :people_holding_hands_light_skin_tone_medium_dark_skin_tone:")
    ("🧑🏻\342\200\215🤝\342\200\215🧑🏿" . ":people_holding_hands_tone1_tone5: :people_holding_hands_light_skin_tone_dark_skin_tone:")
    ("🧑🏼\342\200\215🤝\342\200\215🧑🏻" . ":people_holding_hands_tone2_tone1: :people_holding_hands_medium_light_skin_tone_light_skin_tone:")
    ("🧑🏼\342\200\215🤝\342\200\215🧑🏼" . ":people_holding_hands_tone2: :people_holding_hands_medium_light_skin_tone:")
    ("🧑🏼\342\200\215🤝\342\200\215🧑🏽" . ":people_holding_hands_tone2_tone3: :people_holding_hands_medium_light_skin_tone_medium_skin_tone:")
    ("🧑🏼\342\200\215🤝\342\200\215🧑🏾" . ":people_holding_hands_tone2_tone4: :people_holding_hands_medium_light_skin_tone_medium_dark_skin_tone:")
    ("🧑🏼\342\200\215🤝\342\200\215🧑🏿" . ":people_holding_hands_tone2_tone5: :people_holding_hands_medium_light_skin_tone_dark_skin_tone:")
    ("🧑🏽\342\200\215🤝\342\200\215🧑🏻" . ":people_holding_hands_tone3_tone1: :people_holding_hands_medium_skin_tone_light_skin_tone:")
    ("🧑🏽\342\200\215🤝\342\200\215🧑🏼" . ":people_holding_hands_tone3_tone2: :people_holding_hands_medium_skin_tone_medium_light_skin_tone:")
    ("🧑🏽\342\200\215🤝\342\200\215🧑🏽" . ":people_holding_hands_tone3: :people_holding_hands_medium_skin_tone:")
    ("🧑🏽\342\200\215🤝\342\200\215🧑🏾" . ":people_holding_hands_tone3_tone4: :people_holding_hands_medium_skin_tone_medium_dark_skin_tone:")
    ("🧑🏽\342\200\215🤝\342\200\215🧑🏿" . ":people_holding_hands_tone3_tone5: :people_holding_hands_medium_skin_tone_dark_skin_tone:")
    ("🧑🏾\342\200\215🤝\342\200\215🧑🏻" . ":people_holding_hands_tone4_tone1: :people_holding_hands_medium_dark_skin_tone_light_skin_tone:")
    ("🧑🏾\342\200\215🤝\342\200\215🧑🏼" . ":people_holding_hands_tone4_tone2: :people_holding_hands_medium_dark_skin_tone_medium_light_skin_tone:")
    ("🧑🏾\342\200\215🤝\342\200\215🧑🏽" . ":people_holding_hands_tone4_tone3: :people_holding_hands_medium_dark_skin_tone_medium_skin_tone:")
    ("🧑🏾\342\200\215🤝\342\200\215🧑🏾" . ":people_holding_hands_tone4: :people_holding_hands_medium_dark_skin_tone:")
    ("🧑🏾\342\200\215🤝\342\200\215🧑🏿" . ":people_holding_hands_tone4_tone5: :people_holding_hands_medium_dark_skin_tone_dark_skin_tone:")
    ("🧑🏿\342\200\215🤝\342\200\215🧑🏻" . ":people_holding_hands_tone5_tone1: :people_holding_hands_dark_skin_tone_light_skin_tone:")
    ("🧑🏿\342\200\215🤝\342\200\215🧑🏼" . ":people_holding_hands_tone5_tone2: :people_holding_hands_dark_skin_tone_medium_light_skin_tone:")
    ("🧑🏿\342\200\215🤝\342\200\215🧑🏽" . ":people_holding_hands_tone5_tone3: :people_holding_hands_dark_skin_tone_medium_skin_tone:")
    ("🧑🏿\342\200\215🤝\342\200\215🧑🏾" . ":people_holding_hands_tone5_tone4: :people_holding_hands_dark_skin_tone_medium_dark_skin_tone:")
    ("🧑🏿\342\200\215🤝\342\200\215🧑🏿" . ":people_holding_hands_tone5: :people_holding_hands_dark_skin_tone:")
    ("👫" . ":couple:")
    ("👫🏻" . ":woman_and_man_holding_hands_tone1: :woman_and_man_holding_hands_light_skin_tone:")
    ("👩🏻\342\200\215🤝\342\200\215👨🏼" . ":woman_and_man_holding_hands_tone1_tone2: :woman_and_man_holding_hands_light_skin_tone_medium_light_skin_tone:")
    ("👩🏻\342\200\215🤝\342\200\215👨🏽" . ":woman_and_man_holding_hands_tone1_tone3: :woman_and_man_holding_hands_light_skin_tone_medium_skin_tone:")
    ("👩🏻\342\200\215🤝\342\200\215👨🏾" . ":woman_and_man_holding_hands_tone1_tone4: :woman_and_man_holding_hands_light_skin_tone_medium_dark_skin_tone:")
    ("👩🏻\342\200\215🤝\342\200\215👨🏿" . ":woman_and_man_holding_hands_tone1_tone5: :woman_and_man_holding_hands_light_skin_tone_dark_skin_tone:")
    ("👩🏼\342\200\215🤝\342\200\215👨🏻" . ":woman_and_man_holding_hands_tone2_tone1: :woman_and_man_holding_hands_medium_light_skin_tone_light_skin_tone:")
    ("👫🏼" . ":woman_and_man_holding_hands_tone2: :woman_and_man_holding_hands_medium_light_skin_tone:")
    ("👩🏼\342\200\215🤝\342\200\215👨🏽" . ":woman_and_man_holding_hands_tone2_tone3: :woman_and_man_holding_hands_medium_light_skin_tone_medium_skin_tone:")
    ("👩🏼\342\200\215🤝\342\200\215👨🏾" . ":woman_and_man_holding_hands_tone2_tone4: :woman_and_man_holding_hands_medium_light_skin_tone_medium_dark_skin_tone:")
    ("👩🏼\342\200\215🤝\342\200\215👨🏿" . ":woman_and_man_holding_hands_tone2_tone5: :woman_and_man_holding_hands_medium_light_skin_tone_dark_skin_tone:")
    ("👩🏽\342\200\215🤝\342\200\215👨🏻" . ":woman_and_man_holding_hands_tone3_tone1: :woman_and_man_holding_hands_medium_skin_tone_light_skin_tone:")
    ("👩🏽\342\200\215🤝\342\200\215👨🏼" . ":woman_and_man_holding_hands_tone3_tone2: :woman_and_man_holding_hands_medium_skin_tone_medium_light_skin_tone:")
    ("👫🏽" . ":woman_and_man_holding_hands_tone3: :woman_and_man_holding_hands_medium_skin_tone:")
    ("👩🏽\342\200\215🤝\342\200\215👨🏾" . ":woman_and_man_holding_hands_tone3_tone4: :woman_and_man_holding_hands_medium_skin_tone_medium_dark_skin_tone:")
    ("👩🏽\342\200\215🤝\342\200\215👨🏿" . ":woman_and_man_holding_hands_tone3_tone5: :woman_and_man_holding_hands_medium_skin_tone_dark_skin_tone:")
    ("👩🏾\342\200\215🤝\342\200\215👨🏻" . ":woman_and_man_holding_hands_tone4_tone1: :woman_and_man_holding_hands_medium_dark_skin_tone_light_skin_tone:")
    ("👩🏾\342\200\215🤝\342\200\215👨🏼" . ":woman_and_man_holding_hands_tone4_tone2: :woman_and_man_holding_hands_medium_dark_skin_tone_medium_light_skin_tone:")
    ("👩🏾\342\200\215🤝\342\200\215👨🏽" . ":woman_and_man_holding_hands_tone4_tone3: :woman_and_man_holding_hands_medium_dark_skin_tone_medium_skin_tone:")
    ("👫🏾" . ":woman_and_man_holding_hands_tone4: :woman_and_man_holding_hands_medium_dark_skin_tone:")
    ("👩🏾\342\200\215🤝\342\200\215👨🏿" . ":woman_and_man_holding_hands_tone4_tone5: :woman_and_man_holding_hands_medium_dark_skin_tone_dark_skin_tone:")
    ("👩🏿\342\200\215🤝\342\200\215👨🏻" . ":woman_and_man_holding_hands_tone5_tone1: :woman_and_man_holding_hands_dark_skin_tone_light_skin_tone:")
    ("👩🏿\342\200\215🤝\342\200\215👨🏼" . ":woman_and_man_holding_hands_tone5_tone2: :woman_and_man_holding_hands_dark_skin_tone_medium_light_skin_tone:")
    ("👩🏿\342\200\215🤝\342\200\215👨🏽" . ":woman_and_man_holding_hands_tone5_tone3: :woman_and_man_holding_hands_dark_skin_tone_medium_skin_tone:")
    ("👩🏿\342\200\215🤝\342\200\215👨🏾" . ":woman_and_man_holding_hands_tone5_tone4: :woman_and_man_holding_hands_dark_skin_tone_medium_dark_skin_tone:")
    ("👫🏿" . ":woman_and_man_holding_hands_tone5: :woman_and_man_holding_hands_dark_skin_tone:")
    ("👭" . ":two_women_holding_hands:")
    ("👭🏻" . ":women_holding_hands_tone1: :women_holding_hands_light_skin_tone:")
    ("👩🏻\342\200\215🤝\342\200\215👩🏼" . ":women_holding_hands_tone1_tone2: :women_holding_hands_light_skin_tone_medium_light_skin_tone:")
    ("👩🏻\342\200\215🤝\342\200\215👩🏽" . ":women_holding_hands_tone1_tone3: :women_holding_hands_light_skin_tone_medium_skin_tone:")
    ("👩🏻\342\200\215🤝\342\200\215👩🏾" . ":women_holding_hands_tone1_tone4: :women_holding_hands_light_skin_tone_medium_dark_skin_tone:")
    ("👩🏻\342\200\215🤝\342\200\215👩🏿" . ":women_holding_hands_tone1_tone5: :women_holding_hands_light_skin_tone_dark_skin_tone:")
    ("👩🏼\342\200\215🤝\342\200\215👩🏻" . ":women_holding_hands_tone2_tone1: :women_holding_hands_medium_light_skin_tone_light_skin_tone:")
    ("👭🏼" . ":women_holding_hands_tone2: :women_holding_hands_medium_light_skin_tone:")
    ("👩🏼\342\200\215🤝\342\200\215👩🏽" . ":women_holding_hands_tone2_tone3: :women_holding_hands_medium_light_skin_tone_medium_skin_tone:")
    ("👩🏼\342\200\215🤝\342\200\215👩🏾" . ":women_holding_hands_tone2_tone4: :women_holding_hands_medium_light_skin_tone_medium_dark_skin_tone:")
    ("👩🏼\342\200\215🤝\342\200\215👩🏿" . ":women_holding_hands_tone2_tone5: :women_holding_hands_medium_light_skin_tone_dark_skin_tone:")
    ("👩🏽\342\200\215🤝\342\200\215👩🏻" . ":women_holding_hands_tone3_tone1: :women_holding_hands_medium_skin_tone_light_skin_tone:")
    ("👩🏽\342\200\215🤝\342\200\215👩🏼" . ":women_holding_hands_tone3_tone2: :women_holding_hands_medium_skin_tone_medium_light_skin_tone:")
    ("👭🏽" . ":women_holding_hands_tone3: :women_holding_hands_medium_skin_tone:")
    ("👩🏽\342\200\215🤝\342\200\215👩🏾" . ":women_holding_hands_tone3_tone4: :women_holding_hands_medium_skin_tone_medium_dark_skin_tone:")
    ("👩🏽\342\200\215🤝\342\200\215👩🏿" . ":women_holding_hands_tone3_tone5: :women_holding_hands_medium_skin_tone_dark_skin_tone:")
    ("👩🏾\342\200\215🤝\342\200\215👩🏻" . ":women_holding_hands_tone4_tone1: :women_holding_hands_medium_dark_skin_tone_light_skin_tone:")
    ("👩🏾\342\200\215🤝\342\200\215👩🏼" . ":women_holding_hands_tone4_tone2: :women_holding_hands_medium_dark_skin_tone_medium_light_skin_tone:")
    ("👩🏾\342\200\215🤝\342\200\215👩🏽" . ":women_holding_hands_tone4_tone3: :women_holding_hands_medium_dark_skin_tone_medium_skin_tone:")
    ("👭🏾" . ":women_holding_hands_tone4: :women_holding_hands_medium_dark_skin_tone:")
    ("👩🏾\342\200\215🤝\342\200\215👩🏿" . ":women_holding_hands_tone4_tone5: :women_holding_hands_medium_dark_skin_tone_dark_skin_tone:")
    ("👩🏿\342\200\215🤝\342\200\215👩🏻" . ":women_holding_hands_tone5_tone1: :women_holding_hands_dark_skin_tone_light_skin_tone:")
    ("👩🏿\342\200\215🤝\342\200\215👩🏼" . ":women_holding_hands_tone5_tone2: :women_holding_hands_dark_skin_tone_medium_light_skin_tone:")
    ("👩🏿\342\200\215🤝\342\200\215👩🏽" . ":women_holding_hands_tone5_tone3: :women_holding_hands_dark_skin_tone_medium_skin_tone:")
    ("👩🏿\342\200\215🤝\342\200\215👩🏾" . ":women_holding_hands_tone5_tone4: :women_holding_hands_dark_skin_tone_medium_dark_skin_tone:")
    ("👭🏿" . ":women_holding_hands_tone5: :women_holding_hands_dark_skin_tone:")
    ("👬" . ":two_men_holding_hands:")
    ("👬🏻" . ":men_holding_hands_tone1: :men_holding_hands_light_skin_tone:")
    ("👨🏻\342\200\215🤝\342\200\215👨🏼" . ":men_holding_hands_tone1_tone2: :men_holding_hands_light_skin_tone_medium_light_skin_tone:")
    ("👨🏻\342\200\215🤝\342\200\215👨🏽" . ":men_holding_hands_tone1_tone3: :men_holding_hands_light_skin_tone_medium_skin_tone:")
    ("👨🏻\342\200\215🤝\342\200\215👨🏾" . ":men_holding_hands_tone1_tone4: :men_holding_hands_light_skin_tone_medium_dark_skin_tone:")
    ("👨🏻\342\200\215🤝\342\200\215👨🏿" . ":men_holding_hands_tone1_tone5: :men_holding_hands_light_skin_tone_dark_skin_tone:")
    ("👨🏼\342\200\215🤝\342\200\215👨🏻" . ":men_holding_hands_tone2_tone1: :men_holding_hands_medium_light_skin_tone_light_skin_tone:")
    ("👬🏼" . ":men_holding_hands_tone2: :men_holding_hands_medium_light_skin_tone:")
    ("👨🏼\342\200\215🤝\342\200\215👨🏽" . ":men_holding_hands_tone2_tone3: :men_holding_hands_medium_light_skin_tone_medium_skin_tone:")
    ("👨🏼\342\200\215🤝\342\200\215👨🏾" . ":men_holding_hands_tone2_tone4: :men_holding_hands_medium_light_skin_tone_medium_dark_skin_tone:")
    ("👨🏼\342\200\215🤝\342\200\215👨🏿" . ":men_holding_hands_tone2_tone5: :men_holding_hands_medium_light_skin_tone_dark_skin_tone:")
    ("👨🏽\342\200\215🤝\342\200\215👨🏻" . ":men_holding_hands_tone3_tone1: :men_holding_hands_medium_skin_tone_light_skin_tone:")
    ("👨🏽\342\200\215🤝\342\200\215👨🏼" . ":men_holding_hands_tone3_tone2: :men_holding_hands_medium_skin_tone_medium_light_skin_tone:")
    ("👬🏽" . ":men_holding_hands_tone3: :men_holding_hands_medium_skin_tone:")
    ("👨🏽\342\200\215🤝\342\200\215👨🏾" . ":men_holding_hands_tone3_tone4: :men_holding_hands_medium_skin_tone_medium_dark_skin_tone:")
    ("👨🏽\342\200\215🤝\342\200\215👨🏿" . ":men_holding_hands_tone3_tone5: :men_holding_hands_medium_skin_tone_dark_skin_tone:")
    ("👨🏾\342\200\215🤝\342\200\215👨🏻" . ":men_holding_hands_tone4_tone1: :men_holding_hands_medium_dark_skin_tone_light_skin_tone:")
    ("👨🏾\342\200\215🤝\342\200\215👨🏼" . ":men_holding_hands_tone4_tone2: :men_holding_hands_medium_dark_skin_tone_medium_light_skin_tone:")
    ("👨🏾\342\200\215🤝\342\200\215👨🏽" . ":men_holding_hands_tone4_tone3: :men_holding_hands_medium_dark_skin_tone_medium_skin_tone:")
    ("👬🏾" . ":men_holding_hands_tone4: :men_holding_hands_medium_dark_skin_tone:")
    ("👨🏾\342\200\215🤝\342\200\215👨🏿" . ":men_holding_hands_tone4_tone5: :men_holding_hands_medium_dark_skin_tone_dark_skin_tone:")
    ("👨🏿\342\200\215🤝\342\200\215👨🏻" . ":men_holding_hands_tone5_tone1: :men_holding_hands_dark_skin_tone_light_skin_tone:")
    ("👨🏿\342\200\215🤝\342\200\215👨🏼" . ":men_holding_hands_tone5_tone2: :men_holding_hands_dark_skin_tone_medium_light_skin_tone:")
    ("👨🏿\342\200\215🤝\342\200\215👨🏽" . ":men_holding_hands_tone5_tone3: :men_holding_hands_dark_skin_tone_medium_skin_tone:")
    ("👨🏿\342\200\215🤝\342\200\215👨🏾" . ":men_holding_hands_tone5_tone4: :men_holding_hands_dark_skin_tone_medium_dark_skin_tone:")
    ("👬🏿" . ":men_holding_hands_tone5: :men_holding_hands_dark_skin_tone:")
    ("💑" . ":couple_with_heart:")
    ("💑🏻" . ":couple_with_heart_tone1: :couple_with_heart_light_skin_tone:")
    ("🧑🏻\342\200\215\342\235\244\357\270\217\342\200\215🧑🏼" . ":couple_with_heart_person_person_tone1_tone2: :couple_with_heart_person_person_light_skin_tone_medium_light_skin_tone:")
    ("🧑🏻\342\200\215\342\235\244\357\270\217\342\200\215🧑🏽" . ":couple_with_heart_person_person_tone1_tone3: :couple_with_heart_person_person_light_skin_tone_medium_skin_tone:")
    ("🧑🏻\342\200\215\342\235\244\357\270\217\342\200\215🧑🏾" . ":couple_with_heart_person_person_tone1_tone4: :couple_with_heart_person_person_light_skin_tone_medium_dark_skin_tone:")
    ("🧑🏻\342\200\215\342\235\244\357\270\217\342\200\215🧑🏿" . ":couple_with_heart_person_person_tone1_tone5: :couple_with_heart_person_person_light_skin_tone_dark_skin_tone:")
    ("🧑🏼\342\200\215\342\235\244\357\270\217\342\200\215🧑🏻" . ":couple_with_heart_person_person_tone2_tone1: :couple_with_heart_person_person_medium_light_skin_tone_light_skin_tone:")
    ("💑🏼" . ":couple_with_heart_tone2: :couple_with_heart_medium_light_skin_tone:")
    ("🧑🏼\342\200\215\342\235\244\357\270\217\342\200\215🧑🏽" . ":couple_with_heart_person_person_tone2_tone3: :couple_with_heart_person_person_medium_light_skin_tone_medium_skin_tone:")
    ("🧑🏼\342\200\215\342\235\244\357\270\217\342\200\215🧑🏾" . ":couple_with_heart_person_person_tone2_tone4: :couple_with_heart_person_person_medium_light_skin_tone_medium_dark_skin_tone:")
    ("🧑🏼\342\200\215\342\235\244\357\270\217\342\200\215🧑🏿" . ":couple_with_heart_person_person_tone2_tone5: :couple_with_heart_person_person_medium_light_skin_tone_dark_skin_tone:")
    ("🧑🏽\342\200\215\342\235\244\357\270\217\342\200\215🧑🏻" . ":couple_with_heart_person_person_tone3_tone1: :couple_with_heart_person_person_medium_skin_tone_light_skin_tone:")
    ("🧑🏽\342\200\215\342\235\244\357\270\217\342\200\215🧑🏼" . ":couple_with_heart_person_person_tone3_tone2: :couple_with_heart_person_person_medium_skin_tone_medium_light_skin_tone:")
    ("💑🏽" . ":couple_with_heart_tone3: :couple_with_heart_medium_skin_tone:")
    ("🧑🏽\342\200\215\342\235\244\357\270\217\342\200\215🧑🏾" . ":couple_with_heart_person_person_tone3_tone4: :couple_with_heart_person_person_medium_skin_tone_medium_dark_skin_tone:")
    ("🧑🏽\342\200\215\342\235\244\357\270\217\342\200\215🧑🏿" . ":couple_with_heart_person_person_tone3_tone5: :couple_with_heart_person_person_medium_skin_tone_dark_skin_tone:")
    ("🧑🏾\342\200\215\342\235\244\357\270\217\342\200\215🧑🏻" . ":couple_with_heart_person_person_tone4_tone1: :couple_with_heart_person_person_medium_dark_skin_tone_light_skin_tone:")
    ("🧑🏾\342\200\215\342\235\244\357\270\217\342\200\215🧑🏼" . ":couple_with_heart_person_person_tone4_tone2: :couple_with_heart_person_person_medium_dark_skin_tone_medium_light_skin_tone:")
    ("🧑🏾\342\200\215\342\235\244\357\270\217\342\200\215🧑🏽" . ":couple_with_heart_person_person_tone4_tone3: :couple_with_heart_person_person_medium_dark_skin_tone_medium_skin_tone:")
    ("💑🏾" . ":couple_with_heart_tone4: :couple_with_heart_medium_dark_skin_tone:")
    ("🧑🏾\342\200\215\342\235\244\357\270\217\342\200\215🧑🏿" . ":couple_with_heart_person_person_tone4_tone5: :couple_with_heart_person_person_medium_dark_skin_tone_dark_skin_tone:")
    ("🧑🏿\342\200\215\342\235\244\357\270\217\342\200\215🧑🏻" . ":couple_with_heart_person_person_tone5_tone1: :couple_with_heart_person_person_dark_skin_tone_light_skin_tone:")
    ("🧑🏿\342\200\215\342\235\244\357\270\217\342\200\215🧑🏼" . ":couple_with_heart_person_person_tone5_tone2: :couple_with_heart_person_person_dark_skin_tone_medium_light_skin_tone:")
    ("🧑🏿\342\200\215\342\235\244\357\270\217\342\200\215🧑🏽" . ":couple_with_heart_person_person_tone5_tone3: :couple_with_heart_person_person_dark_skin_tone_medium_skin_tone:")
    ("🧑🏿\342\200\215\342\235\244\357\270\217\342\200\215🧑🏾" . ":couple_with_heart_person_person_tone5_tone4: :couple_with_heart_person_person_dark_skin_tone_medium_dark_skin_tone:")
    ("💑🏿" . ":couple_with_heart_tone5: :couple_with_heart_dark_skin_tone:")
    ("👩\342\200\215\342\235\244\357\270\217\342\200\215👨" . ":couple_with_heart_woman_man:")
    ("👩🏻\342\200\215\342\235\244\357\270\217\342\200\215👨🏻" . ":couple_with_heart_woman_man_tone1: :couple_with_heart_woman_man_light_skin_tone:")
    ("👩🏻\342\200\215\342\235\244\357\270\217\342\200\215👨🏼" . ":couple_with_heart_woman_man_tone1_tone2: :couple_with_heart_woman_man_light_skin_tone_medium_light_skin_tone:")
    ("👩🏻\342\200\215\342\235\244\357\270\217\342\200\215👨🏽" . ":couple_with_heart_woman_man_tone1_tone3: :couple_with_heart_woman_man_light_skin_tone_medium_skin_tone:")
    ("👩🏻\342\200\215\342\235\244\357\270\217\342\200\215👨🏾" . ":couple_with_heart_woman_man_tone1_tone4: :couple_with_heart_woman_man_light_skin_tone_medium_dark_skin_tone:")
    ("👩🏻\342\200\215\342\235\244\357\270\217\342\200\215👨🏿" . ":couple_with_heart_woman_man_tone1_tone5: :couple_with_heart_woman_man_light_skin_tone_dark_skin_tone:")
    ("👩🏼\342\200\215\342\235\244\357\270\217\342\200\215👨🏻" . ":couple_with_heart_woman_man_tone2_tone1: :couple_with_heart_woman_man_medium_light_skin_tone_light_skin_tone:")
    ("👩🏼\342\200\215\342\235\244\357\270\217\342\200\215👨🏼" . ":couple_with_heart_woman_man_tone2: :couple_with_heart_woman_man_medium_light_skin_tone:")
    ("👩🏼\342\200\215\342\235\244\357\270\217\342\200\215👨🏽" . ":couple_with_heart_woman_man_tone2_tone3: :couple_with_heart_woman_man_medium_light_skin_tone_medium_skin_tone:")
    ("👩🏼\342\200\215\342\235\244\357\270\217\342\200\215👨🏾" . ":couple_with_heart_woman_man_tone2_tone4: :couple_with_heart_woman_man_medium_light_skin_tone_medium_dark_skin_tone:")
    ("👩🏼\342\200\215\342\235\244\357\270\217\342\200\215👨🏿" . ":couple_with_heart_woman_man_tone2_tone5: :couple_with_heart_woman_man_medium_light_skin_tone_dark_skin_tone:")
    ("👩🏽\342\200\215\342\235\244\357\270\217\342\200\215👨🏻" . ":couple_with_heart_woman_man_tone3_tone1: :couple_with_heart_woman_man_medium_skin_tone_light_skin_tone:")
    ("👩🏽\342\200\215\342\235\244\357\270\217\342\200\215👨🏼" . ":couple_with_heart_woman_man_tone3_tone2: :couple_with_heart_woman_man_medium_skin_tone_medium_light_skin_tone:")
    ("👩🏽\342\200\215\342\235\244\357\270\217\342\200\215👨🏽" . ":couple_with_heart_woman_man_tone3: :couple_with_heart_woman_man_medium_skin_tone:")
    ("👩🏽\342\200\215\342\235\244\357\270\217\342\200\215👨🏾" . ":couple_with_heart_woman_man_tone3_tone4: :couple_with_heart_woman_man_medium_skin_tone_medium_dark_skin_tone:")
    ("👩🏽\342\200\215\342\235\244\357\270\217\342\200\215👨🏿" . ":couple_with_heart_woman_man_tone3_tone5: :couple_with_heart_woman_man_medium_skin_tone_dark_skin_tone:")
    ("👩🏾\342\200\215\342\235\244\357\270\217\342\200\215👨🏻" . ":couple_with_heart_woman_man_tone4_tone1: :couple_with_heart_woman_man_medium_dark_skin_tone_light_skin_tone:")
    ("👩🏾\342\200\215\342\235\244\357\270\217\342\200\215👨🏼" . ":couple_with_heart_woman_man_tone4_tone2: :couple_with_heart_woman_man_medium_dark_skin_tone_medium_light_skin_tone:")
    ("👩🏾\342\200\215\342\235\244\357\270\217\342\200\215👨🏽" . ":couple_with_heart_woman_man_tone4_tone3: :couple_with_heart_woman_man_medium_dark_skin_tone_medium_skin_tone:")
    ("👩🏾\342\200\215\342\235\244\357\270\217\342\200\215👨🏾" . ":couple_with_heart_woman_man_tone4: :couple_with_heart_woman_man_medium_dark_skin_tone:")
    ("👩🏾\342\200\215\342\235\244\357\270\217\342\200\215👨🏿" . ":couple_with_heart_woman_man_tone4_tone5: :couple_with_heart_woman_man_medium_dark_skin_tone_dark_skin_tone:")
    ("👩🏿\342\200\215\342\235\244\357\270\217\342\200\215👨🏻" . ":couple_with_heart_woman_man_tone5_tone1: :couple_with_heart_woman_man_dark_skin_tone_light_skin_tone:")
    ("👩🏿\342\200\215\342\235\244\357\270\217\342\200\215👨🏼" . ":couple_with_heart_woman_man_tone5_tone2: :couple_with_heart_woman_man_dark_skin_tone_medium_light_skin_tone:")
    ("👩🏿\342\200\215\342\235\244\357\270\217\342\200\215👨🏽" . ":couple_with_heart_woman_man_tone5_tone3: :couple_with_heart_woman_man_dark_skin_tone_medium_skin_tone:")
    ("👩🏿\342\200\215\342\235\244\357\270\217\342\200\215👨🏾" . ":couple_with_heart_woman_man_tone5_tone4: :couple_with_heart_woman_man_dark_skin_tone_medium_dark_skin_tone:")
    ("👩🏿\342\200\215\342\235\244\357\270\217\342\200\215👨🏿" . ":couple_with_heart_woman_man_tone5: :couple_with_heart_woman_man_dark_skin_tone:")
    ("👩\342\200\215\342\235\244\357\270\217\342\200\215👩" . ":couple_ww: :couple_with_heart_ww:")
    ("👩🏻\342\200\215\342\235\244\357\270\217\342\200\215👩🏻" . ":couple_with_heart_woman_woman_tone1: :couple_with_heart_woman_woman_light_skin_tone:")
    ("👩🏻\342\200\215\342\235\244\357\270\217\342\200\215👩🏼" . ":couple_with_heart_woman_woman_tone1_tone2: :couple_with_heart_woman_woman_light_skin_tone_medium_light_skin_tone:")
    ("👩🏻\342\200\215\342\235\244\357\270\217\342\200\215👩🏽" . ":couple_with_heart_woman_woman_tone1_tone3: :couple_with_heart_woman_woman_light_skin_tone_medium_skin_tone:")
    ("👩🏻\342\200\215\342\235\244\357\270\217\342\200\215👩🏾" . ":couple_with_heart_woman_woman_tone1_tone4: :couple_with_heart_woman_woman_light_skin_tone_medium_dark_skin_tone:")
    ("👩🏻\342\200\215\342\235\244\357\270\217\342\200\215👩🏿" . ":couple_with_heart_woman_woman_tone1_tone5: :couple_with_heart_woman_woman_light_skin_tone_dark_skin_tone:")
    ("👩🏼\342\200\215\342\235\244\357\270\217\342\200\215👩🏻" . ":couple_with_heart_woman_woman_tone2_tone1: :couple_with_heart_woman_woman_medium_light_skin_tone_light_skin_tone:")
    ("👩🏼\342\200\215\342\235\244\357\270\217\342\200\215👩🏼" . ":couple_with_heart_woman_woman_tone2: :couple_with_heart_woman_woman_medium_light_skin_tone:")
    ("👩🏼\342\200\215\342\235\244\357\270\217\342\200\215👩🏽" . ":couple_with_heart_woman_woman_tone2_tone3: :couple_with_heart_woman_woman_medium_light_skin_tone_medium_skin_tone:")
    ("👩🏼\342\200\215\342\235\244\357\270\217\342\200\215👩🏾" . ":couple_with_heart_woman_woman_tone2_tone4: :couple_with_heart_woman_woman_medium_light_skin_tone_medium_dark_skin_tone:")
    ("👩🏼\342\200\215\342\235\244\357\270\217\342\200\215👩🏿" . ":couple_with_heart_woman_woman_tone2_tone5: :couple_with_heart_woman_woman_medium_light_skin_tone_dark_skin_tone:")
    ("👩🏽\342\200\215\342\235\244\357\270\217\342\200\215👩🏻" . ":couple_with_heart_woman_woman_tone3_tone1: :couple_with_heart_woman_woman_medium_skin_tone_light_skin_tone:")
    ("👩🏽\342\200\215\342\235\244\357\270\217\342\200\215👩🏼" . ":couple_with_heart_woman_woman_tone3_tone2: :couple_with_heart_woman_woman_medium_skin_tone_medium_light_skin_tone:")
    ("👩🏽\342\200\215\342\235\244\357\270\217\342\200\215👩🏽" . ":couple_with_heart_woman_woman_tone3: :couple_with_heart_woman_woman_medium_skin_tone:")
    ("👩🏽\342\200\215\342\235\244\357\270\217\342\200\215👩🏾" . ":couple_with_heart_woman_woman_tone3_tone4: :couple_with_heart_woman_woman_medium_skin_tone_medium_dark_skin_tone:")
    ("👩🏽\342\200\215\342\235\244\357\270\217\342\200\215👩🏿" . ":couple_with_heart_woman_woman_tone3_tone5: :couple_with_heart_woman_woman_medium_skin_tone_dark_skin_tone:")
    ("👩🏾\342\200\215\342\235\244\357\270\217\342\200\215👩🏻" . ":couple_with_heart_woman_woman_tone4_tone1: :couple_with_heart_woman_woman_medium_dark_skin_tone_light_skin_tone:")
    ("👩🏾\342\200\215\342\235\244\357\270\217\342\200\215👩🏼" . ":couple_with_heart_woman_woman_tone4_tone2: :couple_with_heart_woman_woman_medium_dark_skin_tone_medium_light_skin_tone:")
    ("👩🏾\342\200\215\342\235\244\357\270\217\342\200\215👩🏽" . ":couple_with_heart_woman_woman_tone4_tone3: :couple_with_heart_woman_woman_medium_dark_skin_tone_medium_skin_tone:")
    ("👩🏾\342\200\215\342\235\244\357\270\217\342\200\215👩🏾" . ":couple_with_heart_woman_woman_tone4: :couple_with_heart_woman_woman_medium_dark_skin_tone:")
    ("👩🏾\342\200\215\342\235\244\357\270\217\342\200\215👩🏿" . ":couple_with_heart_woman_woman_tone4_tone5: :couple_with_heart_woman_woman_medium_dark_skin_tone_dark_skin_tone:")
    ("👩🏿\342\200\215\342\235\244\357\270\217\342\200\215👩🏻" . ":couple_with_heart_woman_woman_tone5_tone1: :couple_with_heart_woman_woman_dark_skin_tone_light_skin_tone:")
    ("👩🏿\342\200\215\342\235\244\357\270\217\342\200\215👩🏼" . ":couple_with_heart_woman_woman_tone5_tone2: :couple_with_heart_woman_woman_dark_skin_tone_medium_light_skin_tone:")
    ("👩🏿\342\200\215\342\235\244\357\270\217\342\200\215👩🏽" . ":couple_with_heart_woman_woman_tone5_tone3: :couple_with_heart_woman_woman_dark_skin_tone_medium_skin_tone:")
    ("👩🏿\342\200\215\342\235\244\357\270\217\342\200\215👩🏾" . ":couple_with_heart_woman_woman_tone5_tone4: :couple_with_heart_woman_woman_dark_skin_tone_medium_dark_skin_tone:")
    ("👩🏿\342\200\215\342\235\244\357\270\217\342\200\215👩🏿" . ":couple_with_heart_woman_woman_tone5: :couple_with_heart_woman_woman_dark_skin_tone:")
    ("👨\342\200\215\342\235\244\357\270\217\342\200\215👨" . ":couple_mm: :couple_with_heart_mm:")
    ("👨🏻\342\200\215\342\235\244\357\270\217\342\200\215👨🏻" . ":couple_with_heart_man_man_tone1: :couple_with_heart_man_man_light_skin_tone:")
    ("👨🏻\342\200\215\342\235\244\357\270\217\342\200\215👨🏼" . ":couple_with_heart_man_man_tone1_tone2: :couple_with_heart_man_man_light_skin_tone_medium_light_skin_tone:")
    ("👨🏻\342\200\215\342\235\244\357\270\217\342\200\215👨🏽" . ":couple_with_heart_man_man_tone1_tone3: :couple_with_heart_man_man_light_skin_tone_medium_skin_tone:")
    ("👨🏻\342\200\215\342\235\244\357\270\217\342\200\215👨🏾" . ":couple_with_heart_man_man_tone1_tone4: :couple_with_heart_man_man_light_skin_tone_medium_dark_skin_tone:")
    ("👨🏻\342\200\215\342\235\244\357\270\217\342\200\215👨🏿" . ":couple_with_heart_man_man_tone1_tone5: :couple_with_heart_man_man_light_skin_tone_dark_skin_tone:")
    ("👨🏼\342\200\215\342\235\244\357\270\217\342\200\215👨🏻" . ":couple_with_heart_man_man_tone2_tone1: :couple_with_heart_man_man_medium_light_skin_tone_light_skin_tone:")
    ("👨🏼\342\200\215\342\235\244\357\270\217\342\200\215👨🏼" . ":couple_with_heart_man_man_tone2: :couple_with_heart_man_man_medium_light_skin_tone:")
    ("👨🏼\342\200\215\342\235\244\357\270\217\342\200\215👨🏽" . ":couple_with_heart_man_man_tone2_tone3: :couple_with_heart_man_man_medium_light_skin_tone_medium_skin_tone:")
    ("👨🏼\342\200\215\342\235\244\357\270\217\342\200\215👨🏾" . ":couple_with_heart_man_man_tone2_tone4: :couple_with_heart_man_man_medium_light_skin_tone_medium_dark_skin_tone:")
    ("👨🏼\342\200\215\342\235\244\357\270\217\342\200\215👨🏿" . ":couple_with_heart_man_man_tone2_tone5: :couple_with_heart_man_man_medium_light_skin_tone_dark_skin_tone:")
    ("👨🏽\342\200\215\342\235\244\357\270\217\342\200\215👨🏻" . ":couple_with_heart_man_man_tone3_tone1: :couple_with_heart_man_man_medium_skin_tone_light_skin_tone:")
    ("👨🏽\342\200\215\342\235\244\357\270\217\342\200\215👨🏼" . ":couple_with_heart_man_man_tone3_tone2: :couple_with_heart_man_man_medium_skin_tone_medium_light_skin_tone:")
    ("👨🏽\342\200\215\342\235\244\357\270\217\342\200\215👨🏽" . ":couple_with_heart_man_man_tone3: :couple_with_heart_man_man_medium_skin_tone:")
    ("👨🏽\342\200\215\342\235\244\357\270\217\342\200\215👨🏾" . ":couple_with_heart_man_man_tone3_tone4: :couple_with_heart_man_man_medium_skin_tone_medium_dark_skin_tone:")
    ("👨🏽\342\200\215\342\235\244\357\270\217\342\200\215👨🏿" . ":couple_with_heart_man_man_tone3_tone5: :couple_with_heart_man_man_medium_skin_tone_dark_skin_tone:")
    ("👨🏾\342\200\215\342\235\244\357\270\217\342\200\215👨🏻" . ":couple_with_heart_man_man_tone4_tone1: :couple_with_heart_man_man_medium_dark_skin_tone_light_skin_tone:")
    ("👨🏾\342\200\215\342\235\244\357\270\217\342\200\215👨🏼" . ":couple_with_heart_man_man_tone4_tone2: :couple_with_heart_man_man_medium_dark_skin_tone_medium_light_skin_tone:")
    ("👨🏾\342\200\215\342\235\244\357\270\217\342\200\215👨🏽" . ":couple_with_heart_man_man_tone4_tone3: :couple_with_heart_man_man_medium_dark_skin_tone_medium_skin_tone:")
    ("👨🏾\342\200\215\342\235\244\357\270\217\342\200\215👨🏾" . ":couple_with_heart_man_man_tone4: :couple_with_heart_man_man_medium_dark_skin_tone:")
    ("👨🏾\342\200\215\342\235\244\357\270\217\342\200\215👨🏿" . ":couple_with_heart_man_man_tone4_tone5: :couple_with_heart_man_man_medium_dark_skin_tone_dark_skin_tone:")
    ("👨🏿\342\200\215\342\235\244\357\270\217\342\200\215👨🏻" . ":couple_with_heart_man_man_tone5_tone1: :couple_with_heart_man_man_dark_skin_tone_light_skin_tone:")
    ("👨🏿\342\200\215\342\235\244\357\270\217\342\200\215👨🏼" . ":couple_with_heart_man_man_tone5_tone2: :couple_with_heart_man_man_dark_skin_tone_medium_light_skin_tone:")
    ("👨🏿\342\200\215\342\235\244\357\270\217\342\200\215👨🏽" . ":couple_with_heart_man_man_tone5_tone3: :couple_with_heart_man_man_dark_skin_tone_medium_skin_tone:")
    ("👨🏿\342\200\215\342\235\244\357\270\217\342\200\215👨🏾" . ":couple_with_heart_man_man_tone5_tone4: :couple_with_heart_man_man_dark_skin_tone_medium_dark_skin_tone:")
    ("👨🏿\342\200\215\342\235\244\357\270\217\342\200\215👨🏿" . ":couple_with_heart_man_man_tone5: :couple_with_heart_man_man_dark_skin_tone:")
    ("💏" . ":couplekiss:")
    ("💏🏻" . ":kiss_tone1: :kiss_light_skin_tone:")
    ("🧑🏻\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215🧑🏼" . ":kiss_person_person_tone1_tone2: :kiss_person_person_light_skin_tone_medium_light_skin_tone:")
    ("🧑🏻\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215🧑🏽" . ":kiss_person_person_tone1_tone3: :kiss_person_person_light_skin_tone_medium_skin_tone:")
    ("🧑🏻\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215🧑🏾" . ":kiss_person_person_tone1_tone4: :kiss_person_person_light_skin_tone_medium_dark_skin_tone:")
    ("🧑🏻\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215🧑🏿" . ":kiss_person_person_tone1_tone5: :kiss_person_person_light_skin_tone_dark_skin_tone:")
    ("🧑🏼\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215🧑🏻" . ":kiss_person_person_tone2_tone1: :kiss_person_person_medium_light_skin_tone_light_skin_tone:")
    ("💏🏼" . ":kiss_tone2: :kiss_medium_light_skin_tone:")
    ("🧑🏼\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215🧑🏽" . ":kiss_person_person_tone2_tone3: :kiss_person_person_medium_light_skin_tone_medium_skin_tone:")
    ("🧑🏼\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215🧑🏾" . ":kiss_person_person_tone2_tone4: :kiss_person_person_medium_light_skin_tone_medium_dark_skin_tone:")
    ("🧑🏼\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215🧑🏿" . ":kiss_person_person_tone2_tone5: :kiss_person_person_medium_light_skin_tone_dark_skin_tone:")
    ("🧑🏽\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215🧑🏻" . ":kiss_person_person_tone3_tone1: :kiss_person_person_medium_skin_tone_light_skin_tone:")
    ("🧑🏽\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215🧑🏼" . ":kiss_person_person_tone3_tone2: :kiss_person_person_medium_skin_tone_medium_light_skin_tone:")
    ("💏🏽" . ":kiss_tone3: :kiss_medium_skin_tone:")
    ("🧑🏽\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215🧑🏾" . ":kiss_person_person_tone3_tone4: :kiss_person_person_medium_skin_tone_medium_dark_skin_tone:")
    ("🧑🏽\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215🧑🏿" . ":kiss_person_person_tone3_tone5: :kiss_person_person_medium_skin_tone_dark_skin_tone:")
    ("🧑🏾\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215🧑🏻" . ":kiss_person_person_tone4_tone1: :kiss_person_person_medium_dark_skin_tone_light_skin_tone:")
    ("🧑🏾\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215🧑🏼" . ":kiss_person_person_tone4_tone2: :kiss_person_person_medium_dark_skin_tone_medium_light_skin_tone:")
    ("🧑🏾\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215🧑🏽" . ":kiss_person_person_tone4_tone3: :kiss_person_person_medium_dark_skin_tone_medium_skin_tone:")
    ("💏🏾" . ":kiss_tone4: :kiss_medium_dark_skin_tone:")
    ("🧑🏾\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215🧑🏿" . ":kiss_person_person_tone4_tone5: :kiss_person_person_medium_dark_skin_tone_dark_skin_tone:")
    ("🧑🏿\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215🧑🏻" . ":kiss_person_person_tone5_tone1: :kiss_person_person_dark_skin_tone_light_skin_tone:")
    ("🧑🏿\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215🧑🏼" . ":kiss_person_person_tone5_tone2: :kiss_person_person_dark_skin_tone_medium_light_skin_tone:")
    ("🧑🏿\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215🧑🏽" . ":kiss_person_person_tone5_tone3: :kiss_person_person_dark_skin_tone_medium_skin_tone:")
    ("🧑🏿\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215🧑🏾" . ":kiss_person_person_tone5_tone4: :kiss_person_person_dark_skin_tone_medium_dark_skin_tone:")
    ("💏🏿" . ":kiss_tone5: :kiss_dark_skin_tone:")
    ("👩\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215👨" . ":kiss_woman_man:")
    ("👩🏻\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215👨🏻" . ":kiss_woman_man_tone1: :kiss_woman_man_light_skin_tone:")
    ("👩🏻\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215👨🏼" . ":kiss_woman_man_tone1_tone2: :kiss_woman_man_light_skin_tone_medium_light_skin_tone:")
    ("👩🏻\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215👨🏽" . ":kiss_woman_man_tone1_tone3: :kiss_woman_man_light_skin_tone_medium_skin_tone:")
    ("👩🏻\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215👨🏾" . ":kiss_woman_man_tone1_tone4: :kiss_woman_man_light_skin_tone_medium_dark_skin_tone:")
    ("👩🏻\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215👨🏿" . ":kiss_woman_man_tone1_tone5: :kiss_woman_man_light_skin_tone_dark_skin_tone:")
    ("👩🏼\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215👨🏻" . ":kiss_woman_man_tone2_tone1: :kiss_woman_man_medium_light_skin_tone_light_skin_tone:")
    ("👩🏼\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215👨🏼" . ":kiss_woman_man_tone2: :kiss_woman_man_medium_light_skin_tone:")
    ("👩🏼\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215👨🏽" . ":kiss_woman_man_tone2_tone3: :kiss_woman_man_medium_light_skin_tone_medium_skin_tone:")
    ("👩🏼\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215👨🏾" . ":kiss_woman_man_tone2_tone4: :kiss_woman_man_medium_light_skin_tone_medium_dark_skin_tone:")
    ("👩🏼\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215👨🏿" . ":kiss_woman_man_tone2_tone5: :kiss_woman_man_medium_light_skin_tone_dark_skin_tone:")
    ("👩🏽\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215👨🏻" . ":kiss_woman_man_tone3_tone1: :kiss_woman_man_medium_skin_tone_light_skin_tone:")
    ("👩🏽\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215👨🏼" . ":kiss_woman_man_tone3_tone2: :kiss_woman_man_medium_skin_tone_medium_light_skin_tone:")
    ("👩🏽\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215👨🏽" . ":kiss_woman_man_tone3: :kiss_woman_man_medium_skin_tone:")
    ("👩🏽\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215👨🏾" . ":kiss_woman_man_tone3_tone4: :kiss_woman_man_medium_skin_tone_medium_dark_skin_tone:")
    ("👩🏽\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215👨🏿" . ":kiss_woman_man_tone3_tone5: :kiss_woman_man_medium_skin_tone_dark_skin_tone:")
    ("👩🏾\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215👨🏻" . ":kiss_woman_man_tone4_tone1: :kiss_woman_man_medium_dark_skin_tone_light_skin_tone:")
    ("👩🏾\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215👨🏼" . ":kiss_woman_man_tone4_tone2: :kiss_woman_man_medium_dark_skin_tone_medium_light_skin_tone:")
    ("👩🏾\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215👨🏽" . ":kiss_woman_man_tone4_tone3: :kiss_woman_man_medium_dark_skin_tone_medium_skin_tone:")
    ("👩🏾\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215👨🏾" . ":kiss_woman_man_tone4: :kiss_woman_man_medium_dark_skin_tone:")
    ("👩🏾\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215👨🏿" . ":kiss_woman_man_tone4_tone5: :kiss_woman_man_medium_dark_skin_tone_dark_skin_tone:")
    ("👩🏿\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215👨🏻" . ":kiss_woman_man_tone5_tone1: :kiss_woman_man_dark_skin_tone_light_skin_tone:")
    ("👩🏿\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215👨🏼" . ":kiss_woman_man_tone5_tone2: :kiss_woman_man_dark_skin_tone_medium_light_skin_tone:")
    ("👩🏿\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215👨🏽" . ":kiss_woman_man_tone5_tone3: :kiss_woman_man_dark_skin_tone_medium_skin_tone:")
    ("👩🏿\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215👨🏾" . ":kiss_woman_man_tone5_tone4: :kiss_woman_man_dark_skin_tone_medium_dark_skin_tone:")
    ("👩🏿\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215👨🏿" . ":kiss_woman_man_tone5: :kiss_woman_man_dark_skin_tone:")
    ("👩\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215👩" . ":couplekiss_ww:")
    ("👩🏻\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215👩🏻" . ":kiss_woman_woman_tone1: :kiss_woman_woman_light_skin_tone:")
    ("👩🏻\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215👩🏼" . ":kiss_woman_woman_tone1_tone2: :kiss_woman_woman_light_skin_tone_medium_light_skin_tone:")
    ("👩🏻\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215👩🏽" . ":kiss_woman_woman_tone1_tone3: :kiss_woman_woman_light_skin_tone_medium_skin_tone:")
    ("👩🏻\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215👩🏾" . ":kiss_woman_woman_tone1_tone4: :kiss_woman_woman_light_skin_tone_medium_dark_skin_tone:")
    ("👩🏻\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215👩🏿" . ":kiss_woman_woman_tone1_tone5: :kiss_woman_woman_light_skin_tone_dark_skin_tone:")
    ("👩🏼\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215👩🏻" . ":kiss_woman_woman_tone2_tone1: :kiss_woman_woman_medium_light_skin_tone_light_skin_tone:")
    ("👩🏼\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215👩🏼" . ":kiss_woman_woman_tone2: :kiss_woman_woman_medium_light_skin_tone:")
    ("👩🏼\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215👩🏽" . ":kiss_woman_woman_tone2_tone3: :kiss_woman_woman_medium_light_skin_tone_medium_skin_tone:")
    ("👩🏼\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215👩🏾" . ":kiss_woman_woman_tone2_tone4: :kiss_woman_woman_medium_light_skin_tone_medium_dark_skin_tone:")
    ("👩🏼\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215👩🏿" . ":kiss_woman_woman_tone2_tone5: :kiss_woman_woman_medium_light_skin_tone_dark_skin_tone:")
    ("👩🏽\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215👩🏻" . ":kiss_woman_woman_tone3_tone1: :kiss_woman_woman_medium_skin_tone_light_skin_tone:")
    ("👩🏽\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215👩🏼" . ":kiss_woman_woman_tone3_tone2: :kiss_woman_woman_medium_skin_tone_medium_light_skin_tone:")
    ("👩🏽\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215👩🏽" . ":kiss_woman_woman_tone3: :kiss_woman_woman_medium_skin_tone:")
    ("👩🏽\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215👩🏾" . ":kiss_woman_woman_tone3_tone4: :kiss_woman_woman_medium_skin_tone_medium_dark_skin_tone:")
    ("👩🏽\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215👩🏿" . ":kiss_woman_woman_tone3_tone5: :kiss_woman_woman_medium_skin_tone_dark_skin_tone:")
    ("👩🏾\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215👩🏻" . ":kiss_woman_woman_tone4_tone1: :kiss_woman_woman_medium_dark_skin_tone_light_skin_tone:")
    ("👩🏾\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215👩🏼" . ":kiss_woman_woman_tone4_tone2: :kiss_woman_woman_medium_dark_skin_tone_medium_light_skin_tone:")
    ("👩🏾\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215👩🏽" . ":kiss_woman_woman_tone4_tone3: :kiss_woman_woman_medium_dark_skin_tone_medium_skin_tone:")
    ("👩🏾\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215👩🏾" . ":kiss_woman_woman_tone4: :kiss_woman_woman_medium_dark_skin_tone:")
    ("👩🏾\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215👩🏿" . ":kiss_woman_woman_tone4_tone5: :kiss_woman_woman_medium_dark_skin_tone_dark_skin_tone:")
    ("👩🏿\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215👩🏻" . ":kiss_woman_woman_tone5_tone1: :kiss_woman_woman_dark_skin_tone_light_skin_tone:")
    ("👩🏿\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215👩🏼" . ":kiss_woman_woman_tone5_tone2: :kiss_woman_woman_dark_skin_tone_medium_light_skin_tone:")
    ("👩🏿\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215👩🏽" . ":kiss_woman_woman_tone5_tone3: :kiss_woman_woman_dark_skin_tone_medium_skin_tone:")
    ("👩🏿\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215👩🏾" . ":kiss_woman_woman_tone5_tone4: :kiss_woman_woman_dark_skin_tone_medium_dark_skin_tone:")
    ("👩🏿\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215👩🏿" . ":kiss_woman_woman_tone5: :kiss_woman_woman_dark_skin_tone:")
    ("👨\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215👨" . ":couplekiss_mm: :kiss_man_man:")
    ("👨🏻\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215👨🏻" . ":kiss_man_man_tone1: :kiss_man_man_light_skin_tone:")
    ("👨🏻\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215👨🏼" . ":kiss_man_man_tone1_tone2: :kiss_man_man_light_skin_tone_medium_light_skin_tone:")
    ("👨🏻\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215👨🏽" . ":kiss_man_man_tone1_tone3: :kiss_man_man_light_skin_tone_medium_skin_tone:")
    ("👨🏻\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215👨🏾" . ":kiss_man_man_tone1_tone4: :kiss_man_man_light_skin_tone_medium_dark_skin_tone:")
    ("👨🏻\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215👨🏿" . ":kiss_man_man_tone1_tone5: :kiss_man_man_light_skin_tone_dark_skin_tone:")
    ("👨🏼\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215👨🏻" . ":kiss_man_man_tone2_tone1: :kiss_man_man_medium_light_skin_tone_light_skin_tone:")
    ("👨🏼\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215👨🏼" . ":kiss_man_man_tone2: :kiss_man_man_medium_light_skin_tone:")
    ("👨🏼\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215👨🏽" . ":kiss_man_man_tone2_tone3: :kiss_man_man_medium_light_skin_tone_medium_skin_tone:")
    ("👨🏼\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215👨🏾" . ":kiss_man_man_tone2_tone4: :kiss_man_man_medium_light_skin_tone_medium_dark_skin_tone:")
    ("👨🏼\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215👨🏿" . ":kiss_man_man_tone2_tone5: :kiss_man_man_medium_light_skin_tone_dark_skin_tone:")
    ("👨🏽\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215👨🏻" . ":kiss_man_man_tone3_tone1: :kiss_man_man_medium_skin_tone_light_skin_tone:")
    ("👨🏽\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215👨🏼" . ":kiss_man_man_tone3_tone2: :kiss_man_man_medium_skin_tone_medium_light_skin_tone:")
    ("👨🏽\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215👨🏽" . ":kiss_man_man_tone3: :kiss_man_man_medium_skin_tone:")
    ("👨🏽\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215👨🏾" . ":kiss_man_man_tone3_tone4: :kiss_man_man_medium_skin_tone_medium_dark_skin_tone:")
    ("👨🏽\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215👨🏿" . ":kiss_man_man_tone3_tone5: :kiss_man_man_medium_skin_tone_dark_skin_tone:")
    ("👨🏾\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215👨🏻" . ":kiss_man_man_tone4_tone1: :kiss_man_man_medium_dark_skin_tone_light_skin_tone:")
    ("👨🏾\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215👨🏼" . ":kiss_man_man_tone4_tone2: :kiss_man_man_medium_dark_skin_tone_medium_light_skin_tone:")
    ("👨🏾\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215👨🏽" . ":kiss_man_man_tone4_tone3: :kiss_man_man_medium_dark_skin_tone_medium_skin_tone:")
    ("👨🏾\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215👨🏾" . ":kiss_man_man_tone4: :kiss_man_man_medium_dark_skin_tone:")
    ("👨🏾\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215👨🏿" . ":kiss_man_man_tone4_tone5: :kiss_man_man_medium_dark_skin_tone_dark_skin_tone:")
    ("👨🏿\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215👨🏻" . ":kiss_man_man_tone5_tone1: :kiss_man_man_dark_skin_tone_light_skin_tone:")
    ("👨🏿\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215👨🏼" . ":kiss_man_man_tone5_tone2: :kiss_man_man_dark_skin_tone_medium_light_skin_tone:")
    ("👨🏿\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215👨🏽" . ":kiss_man_man_tone5_tone3: :kiss_man_man_dark_skin_tone_medium_skin_tone:")
    ("👨🏿\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215👨🏾" . ":kiss_man_man_tone5_tone4: :kiss_man_man_dark_skin_tone_medium_dark_skin_tone:")
    ("👨🏿\342\200\215\342\235\244\357\270\217\342\200\215💋\342\200\215👨🏿" . ":kiss_man_man_tone5: :kiss_man_man_dark_skin_tone:")
    ("🧑\342\200\215🧑\342\200\215🧒\342\200\215🧒" . ":family_adult_adult_child_child:")
    ("🧑\342\200\215🧑\342\200\215🧒" . ":family_adult_adult_child:")
    ("🧑\342\200\215🧒\342\200\215🧒" . ":family_adult_child_child:")
    ("🧑\342\200\215🧒" . ":family_adult_child:")
    ("👪" . ":family:")
    ("👨\342\200\215👩\342\200\215👦" . ":family_man_woman_boy:")
    ("👨\342\200\215👩\342\200\215👧" . ":family_mwg:")
    ("👨\342\200\215👩\342\200\215👧\342\200\215👦" . ":family_mwgb:")
    ("👨\342\200\215👩\342\200\215👦\342\200\215👦" . ":family_mwbb:")
    ("👨\342\200\215👩\342\200\215👧\342\200\215👧" . ":family_mwgg:")
    ("👩\342\200\215👩\342\200\215👦" . ":family_wwb:")
    ("👩\342\200\215👩\342\200\215👧" . ":family_wwg:")
    ("👩\342\200\215👩\342\200\215👧\342\200\215👦" . ":family_wwgb:")
    ("👩\342\200\215👩\342\200\215👦\342\200\215👦" . ":family_wwbb:")
    ("👩\342\200\215👩\342\200\215👧\342\200\215👧" . ":family_wwgg:")
    ("👨\342\200\215👨\342\200\215👦" . ":family_mmb:")
    ("👨\342\200\215👨\342\200\215👧" . ":family_mmg:")
    ("👨\342\200\215👨\342\200\215👧\342\200\215👦" . ":family_mmgb:")
    ("👨\342\200\215👨\342\200\215👦\342\200\215👦" . ":family_mmbb:")
    ("👨\342\200\215👨\342\200\215👧\342\200\215👧" . ":family_mmgg:")
    ("👩\342\200\215👦" . ":family_woman_boy:")
    ("👩\342\200\215👧" . ":family_woman_girl:")
    ("👩\342\200\215👧\342\200\215👦" . ":family_woman_girl_boy:")
    ("👩\342\200\215👦\342\200\215👦" . ":family_woman_boy_boy:")
    ("👩\342\200\215👧\342\200\215👧" . ":family_woman_girl_girl:")
    ("👨\342\200\215👦" . ":family_man_boy:")
    ("👨\342\200\215👧" . ":family_man_girl:")
    ("👨\342\200\215👧\342\200\215👦" . ":family_man_girl_boy:")
    ("👨\342\200\215👦\342\200\215👦" . ":family_man_boy_boy:")
    ("👨\342\200\215👧\342\200\215👧" . ":family_man_girl_girl:")
    ("🪢" . ":knot:")
    ("🧶" . ":yarn:")
    ("🧵" . ":thread:")
    ("🪡" . ":sewing_needle:")
    ("🧥" . ":coat:")
    ("🥼" . ":lab_coat:")
    ("🦺" . ":safety_vest:")
    ("👚" . ":womans_clothes:")
    ("👕" . ":t_shirt:")
    ("👖" . ":jeans:")
    ("🩲" . ":briefs:")
    ("🩳" . ":shorts:")
    ("👔" . ":necktie:")
    ("👗" . ":dress:")
    ("👙" . ":bikini:")
    ("🩱" . ":one_piece_swimsuit:")
    ("👘" . ":kimono:")
    ("🥻" . ":sari:")
    ("🩴" . ":thong_sandal:")
    ("🥿" . ":womans_flat_shoe:")
    ("👠" . ":high_heel:")
    ("👡" . ":womans_sandal:")
    ("👢" . ":womans_boot:")
    ("👞" . ":mans_shoe:")
    ("👟" . ":athletic_shoe: :running_shoe:")
    ("🥾" . ":hiking_boot:")
    ("🧦" . ":socks:")
    ("🧤" . ":gloves:")
    ("🧣" . ":scarf:")
    ("🎩" . ":tophat: :top_hat:")
    ("🧢" . ":billed_cap:")
    ("👒" . ":womans_hat:")
    ("🎓" . ":mortar_board:")
    ("\342\233\221\357\270\217" . ":helmet_with_cross: :helmet_with_white_cross:")
    ("🪖" . ":military_helmet:")
    ("👑" . ":crown:")
    ("💍" . ":ring:")
    ("👝" . ":pouch: :clutch_bag:")
    ("👛" . ":purse:")
    ("👜" . ":handbag:")
    ("💼" . ":briefcase:")
    ("🎒" . ":school_satchel: :backpack:")
    ("🧳" . ":luggage:")
    ("👓" . ":eyeglasses:")
    ("🕶\357\270\217" . ":dark_sunglasses:")
    ("🥽" . ":goggles:")
    ("🌂" . ":closed_umbrella:")
    ("🐶" . ":dog_face:")
    ("🐱" . ":cat_face:")
    ("🐭" . ":mouse_face:")
    ("🐹" . ":hamster:")
    ("🐰" . ":rabbit_face:")
    ("🦊" . ":fox_face:")
    ("🐻" . ":bear:")
    ("🐼" . ":panda_face:")
    ("🐻\342\200\215\342\235\204\357\270\217" . ":polar_bear:")
    ("🐨" . ":koala:")
    ("🐯" . ":tiger_face:")
    ("🦁" . ":lion_face:")
    ("🐮" . ":cow_face:")
    ("🐷" . ":pig_face:")
    ("🐽" . ":pig_nose:")
    ("🐸" . ":frog:")
    ("🐵" . ":monkey_face:")
    ("🙈" . ":see_no_evil:")
    ("🙉" . ":hear_no_evil:")
    ("🙊" . ":speak_no_evil:")
    ("🐒" . ":monkey:")
    ("🐔" . ":chicken:")
    ("🐧" . ":penguin:")
    ("🐦" . ":bird:")
    ("🐤" . ":baby_chick:")
    ("🐣" . ":hatching_chick:")
    ("🐥" . ":hatched_chick:")
    ("🪿" . ":goose:")
    ("🦆" . ":duck:")
    ("🐦\342\200\215\342\254\233" . ":black_bird:")
    ("🦅" . ":eagle:")
    ("🦉" . ":owl:")
    ("🦇" . ":bat:")
    ("🐺" . ":wolf:")
    ("🐗" . ":boar:")
    ("🐴" . ":horse_face:")
    ("🦄" . ":unicorn_face:")
    ("🫎" . ":moose:")
    ("🐝" . ":honeybee:")
    ("🪱" . ":worm:")
    ("🐛" . ":bug:")
    ("🦋" . ":butterfly:")
    ("🐌" . ":snail:")
    ("🐞" . ":lady_beetle:")
    ("🐜" . ":ant:")
    ("🪰" . ":fly:")
    ("🪲" . ":beetle:")
    ("🪳" . ":cockroach:")
    ("🦟" . ":mosquito:")
    ("🦗" . ":cricket:")
    ("🕷\357\270\217" . ":spider:")
    ("🕸\357\270\217" . ":spider_web:")
    ("🦂" . ":scorpion:")
    ("🐢" . ":turtle:")
    ("🐍" . ":snake:")
    ("🦎" . ":lizard:")
    ("🦖" . ":t_rex:")
    ("🦕" . ":sauropod:")
    ("🐙" . ":octopus:")
    ("🦑" . ":squid:")
    ("🪼" . ":jellyfish:")
    ("🦐" . ":shrimp:")
    ("🦞" . ":lobster:")
    ("🦀" . ":crab:")
    ("🐡" . ":blowfish:")
    ("🐠" . ":tropical_fish:")
    ("🐟" . ":fish:")
    ("🐬" . ":dolphin:")
    ("🐳" . ":whale:")
    ("🐋" . ":whale2:")
    ("🦈" . ":shark:")
    ("🦭" . ":seal:")
    ("🐊" . ":crocodile:")
    ("🐅" . ":tiger2:")
    ("🐆" . ":leopard:")
    ("🦓" . ":zebra:")
    ("🦍" . ":gorilla:")
    ("🦧" . ":orangutan:")
    ("🦣" . ":mammoth:")
    ("🐘" . ":elephant:")
    ("🦛" . ":hippopotamus:")
    ("🦏" . ":rhinoceros:")
    ("🐪" . ":dromedary_camel:")
    ("🐫" . ":camel:")
    ("🦒" . ":giraffe:")
    ("🦘" . ":kangaroo:")
    ("🦬" . ":bison:")
    ("🐃" . ":water_buffalo:")
    ("🐂" . ":ox:")
    ("🐄" . ":cow2:")
    ("🫏" . ":donkey:")
    ("🐎" . ":racehorse:")
    ("🐖" . ":pig2:")
    ("🐏" . ":ram:")
    ("🐑" . ":sheep: :ewe:")
    ("🦙" . ":llama:")
    ("🐐" . ":goat:")
    ("🦌" . ":deer:")
    ("🐕" . ":dog2:")
    ("🐩" . ":poodle:")
    ("🦮" . ":guide_dog:")
    ("🐕\342\200\215🦺" . ":service_dog:")
    ("🐈" . ":cat2:")
    ("🐈\342\200\215\342\254\233" . ":black_cat:")
    ("🪶" . ":feather:")
    ("🪽" . ":wing:")
    ("🐓" . ":rooster:")
    ("🦃" . ":turkey:")
    ("🦤" . ":dodo:")
    ("🦚" . ":peacock:")
    ("🦜" . ":parrot:")
    ("🦢" . ":swan:")
    ("🦩" . ":flamingo:")
    ("🕊\357\270\217" . ":dove_of_peace:")
    ("🐇" . ":rabbit2:")
    ("🦝" . ":raccoon:")
    ("🦨" . ":skunk:")
    ("🦡" . ":badger:")
    ("🦫" . ":beaver:")
    ("🦦" . ":otter:")
    ("🦥" . ":sloth:")
    ("🐁" . ":mouse2:")
    ("🐀" . ":rat:")
    ("🐿\357\270\217" . ":chipmunk:")
    ("🦔" . ":hedgehog:")
    ("🐾" . ":feet: :paw_prints:")
    ("🐉" . ":dragon:")
    ("🐲" . ":dragon_face:")
    ("🐦\342\200\215🔥" . ":phoenix:")
    ("🌵" . ":cactus:")
    ("🎄" . ":christmas_tree:")
    ("🌲" . ":evergreen_tree:")
    ("🌳" . ":deciduous_tree:")
    ("🌴" . ":palm_tree:")
    ("🪵" . ":wood:")
    ("🌱" . ":seedling:")
    ("🌿" . ":herb:")
    ("\342\230\230\357\270\217" . ":shamrock:")
    ("🍀" . ":four_leaf_clover:")
    ("🎍" . ":bamboo:")
    ("🪴" . ":potted_plant:")
    ("🎋" . ":tanabata_tree:")
    ("🍃" . ":leaves:")
    ("🍂" . ":fallen_leaf:")
    ("🍁" . ":maple_leaf:")
    ("🪺" . ":nest_with_eggs:")
    ("🪹" . ":empty_nest:")
    ("🍄" . ":mushroom:")
    ("🍄\342\200\215🟫" . ":brown_mushroom:")
    ("🐚" . ":spiral_shell:")
    ("🪸" . ":coral:")
    ("🪨" . ":rock:")
    ("🌾" . ":ear_of_rice: :sheaf_of_rice:")
    ("💐" . ":bouquet:")
    ("🌷" . ":tulip:")
    ("🌹" . ":rose:")
    ("🥀" . ":wilted_rose: :wilted_flower:")
    ("🪻" . ":hyacinth:")
    ("🪷" . ":lotus:")
    ("🌺" . ":hibiscus:")
    ("🌸" . ":cherry_blossom:")
    ("🌼" . ":blossom:")
    ("🌻" . ":sunflower:")
    ("🌞" . ":sun_with_face:")
    ("🌝" . ":full_moon_with_face:")
    ("🌛" . ":first_quarter_moon_with_face:")
    ("🌜" . ":last_quarter_moon_with_face:")
    ("🌚" . ":new_moon_with_face: :new_moon_face:")
    ("🌕" . ":full_moon:")
    ("🌖" . ":waning_gibbous_moon:")
    ("🌗" . ":last_quarter_moon:")
    ("🌘" . ":waning_crescent_moon:")
    ("🌑" . ":new_moon:")
    ("🌒" . ":waxing_crescent_moon:")
    ("🌓" . ":first_quarter_moon:")
    ("🌔" . ":waxing_gibbous_moon:")
    ("🌙" . ":crescent_moon:")
    ("🌎" . ":earth_americas:")
    ("🌍" . ":earth_africa:")
    ("🌏" . ":earth_asia:")
    ("🪐" . ":ringed_planet:")
    ("💫" . ":dizzy:")
    ("\342\255\220" . ":star:")
    ("🌟" . ":star2: :glowing_star:")
    ("\342\234\250" . ":sparkles:")
    ("\342\232\241" . ":zap: :high_voltage:")
    ("\342\230\204\357\270\217" . ":comet:")
    ("💥" . ":boom: :collision:")
    ("🔥" . ":fire: :flame:")
    ("🌪\357\270\217" . ":cloud_tornado: :cloud_with_tornado:")
    ("🌈" . ":rainbow:")
    ("\342\230\200\357\270\217" . ":sunny:")
    ("🌤\357\270\217" . ":white_sun_small_cloud: :white_sun_with_small_cloud:")
    ("\342\233\205" . ":partly_sunny:")
    ("🌥\357\270\217" . ":white_sun_cloud: :white_sun_behind_cloud:")
    ("\342\230\201\357\270\217" . ":cloud:")
    ("🌦\357\270\217" . ":white_sun_rain_cloud: :white_sun_behind_cloud_with_rain:")
    ("🌧\357\270\217" . ":cloud_rain: :cloud_with_rain:")
    ("\342\233\210\357\270\217" . ":thunder_cloud_rain: :thunder_cloud_and_rain:")
    ("🌩\357\270\217" . ":cloud_lightning: :cloud_with_lightning:")
    ("🌨\357\270\217" . ":cloud_snow: :cloud_with_snow:")
    ("\342\235\204\357\270\217" . ":snowflake:")
    ("\342\230\203\357\270\217" . ":snowman2:")
    ("\342\233\204" . ":snowman:")
    ("🌬\357\270\217" . ":wind_blowing_face: :wind_face:")
    ("💨" . ":dashing_away:")
    ("💧" . ":droplet:")
    ("💦" . ":sweat_drops:")
    ("🫧" . ":bubbles:")
    ("\342\230\224" . ":umbrella:")
    ("\342\230\202\357\270\217" . ":umbrella2:")
    ("🌊" . ":ocean: :water_wave:")
    ("🌫\357\270\217" . ":fog:")
    ("🍏" . ":green_apple:")
    ("🍎" . ":red_apple:")
    ("🍐" . ":pear:")
    ("🍊" . ":tangerine:")
    ("🍋" . ":lemon:")
    ("🍋\342\200\215🟩" . ":lime:")
    ("🍌" . ":banana:")
    ("🍉" . ":watermelon:")
    ("🍇" . ":grapes:")
    ("🍓" . ":strawberry:")
    ("🫐" . ":blueberries:")
    ("🍈" . ":melon:")
    ("🍒" . ":cherries:")
    ("🍑" . ":peach:")
    ("🥭" . ":mango:")
    ("🍍" . ":pineapple:")
    ("🥥" . ":coconut:")
    ("🥝" . ":kiwifruit: :kiwi_fruit:")
    ("🍅" . ":tomato:")
    ("🍆" . ":eggplant:")
    ("🥑" . ":avocado:")
    ("🫛" . ":pea_pod:")
    ("🥦" . ":broccoli:")
    ("🥬" . ":leafy_green:")
    ("🥒" . ":cucumber:")
    ("🌶\357\270\217" . ":hot_pepper:")
    ("🫑" . ":bell_pepper:")
    ("🌽" . ":ear_of_corn:")
    ("🥕" . ":carrot:")
    ("🫒" . ":olive:")
    ("🧄" . ":garlic:")
    ("🧅" . ":onion:")
    ("🥔" . ":potato:")
    ("🍠" . ":sweet_potato:")
    ("🫚" . ":ginger_root:")
    ("🥐" . ":croissant:")
    ("🥯" . ":bagel:")
    ("🍞" . ":bread:")
    ("🥖" . ":french_bread: :baguette_bread:")
    ("🥨" . ":pretzel:")
    ("🧀" . ":cheese_wedge:")
    ("🥚" . ":egg:")
    ("🍳" . ":cooking:")
    ("🧈" . ":butter:")
    ("🥞" . ":pancakes:")
    ("🧇" . ":waffle:")
    ("🥓" . ":bacon:")
    ("🥩" . ":cut_of_meat:")
    ("🍗" . ":poultry_leg:")
    ("🍖" . ":meat_on_bone:")
    ("🦴" . ":bone:")
    ("🌭" . ":hotdog: :hot_dog:")
    ("🍔" . ":hamburger:")
    ("🍟" . ":french_fries:")
    ("🍕" . ":pizza:")
    ("🫓" . ":flatbread:")
    ("🥪" . ":sandwich:")
    ("🥙" . ":stuffed_flatbread: :stuffed_pita:")
    ("🧆" . ":falafel:")
    ("🌮" . ":taco:")
    ("🌯" . ":burrito:")
    ("🫔" . ":tamale:")
    ("🥗" . ":green_salad:")
    ("🥘" . ":shallow_pan_of_food: :paella:")
    ("🫕" . ":fondue:")
    ("🥫" . ":canned_food:")
    ("🫙" . ":jar:")
    ("🍝" . ":spaghetti:")
    ("🍜" . ":ramen: :steaming_bowl:")
    ("🍲" . ":stew: :pot_of_food:")
    ("🍛" . ":curry_rice:")
    ("🍣" . ":sushi:")
    ("🍱" . ":bento_box:")
    ("🥟" . ":dumpling:")
    ("🦪" . ":oyster:")
    ("🍤" . ":fried_shrimp:")
    ("🍙" . ":rice_ball:")
    ("🍚" . ":cooked_rice:")
    ("🍘" . ":rice_cracker:")
    ("🍥" . ":fish_cake:")
    ("🥠" . ":fortune_cookie:")
    ("🥮" . ":moon_cake:")
    ("🍢" . ":oden:")
    ("🍡" . ":dango:")
    ("🍧" . ":shaved_ice:")
    ("🍨" . ":ice_cream:")
    ("🍦" . ":icecream:")
    ("🥧" . ":pie:")
    ("🧁" . ":cupcake:")
    ("🍰" . ":shortcake:")
    ("🎂" . ":birthday_cake:")
    ("🍮" . ":custard: :pudding: :flan:")
    ("🍭" . ":lollipop:")
    ("🍬" . ":candy:")
    ("🍫" . ":chocolate_bar:")
    ("🍿" . ":popcorn:")
    ("🍩" . ":doughnut:")
    ("🍪" . ":cookie:")
    ("🌰" . ":chestnut:")
    ("🥜" . ":peanuts: :shelled_peanut:")
    ("🫘" . ":beans:")
    ("🍯" . ":honey_pot:")
    ("🥛" . ":glass_of_milk:")
    ("🫗" . ":pouring_liquid:")
    ("🍼" . ":baby_bottle:")
    ("🫖" . ":teapot:")
    ("\342\230\225" . ":coffee: :hot_beverage:")
    ("🍵" . ":tea:")
    ("🧉" . ":mate:")
    ("🧃" . ":beverage_box:")
    ("🥤" . ":cup_with_straw:")
    ("🧋" . ":bubble_tea:")
    ("🍶" . ":sake:")
    ("🍺" . ":beer_mug:")
    ("🍻" . ":beers:")
    ("🥂" . ":champagne_glass: :clinking_glass:")
    ("🍷" . ":wine_glass:")
    ("🥃" . ":tumbler_glass: :whisky:")
    ("🍸" . ":cocktail:")
    ("🍹" . ":tropical_drink:")
    ("🍾" . ":champagne: :bottle_with_popping_cork:")
    ("🧊" . ":ice_cube:")
    ("🥄" . ":spoon:")
    ("🍴" . ":fork_and_knife:")
    ("🍽\357\270\217" . ":fork_knife_plate: :fork_and_knife_with_plate:")
    ("🥣" . ":bowl_with_spoon:")
    ("🥡" . ":takeout_box:")
    ("🥢" . ":chopsticks:")
    ("🧂" . ":salt:")
    ("\342\232\275" . ":soccer_ball: :actual_football:")
    ("🏀" . ":basketball:")
    ("🏈" . ":football:")
    ("\342\232\276" . ":baseball:")
    ("🥎" . ":softball:")
    ("🎾" . ":tennis:")
    ("🏐" . ":volleyball:")
    ("🏉" . ":rugby_football:")
    ("🥏" . ":flying_disc:")
    ("🎱" . ":8ball:")
    ("🪀" . ":yo_yo:")
    ("🏓" . ":ping_pong: :table_tennis:")
    ("🏸" . ":badminton:")
    ("🏒" . ":ice_hockey:")
    ("🏑" . ":field_hockey:")
    ("🥍" . ":lacrosse:")
    ("🏏" . ":cricket_game: :cricket_bat_ball:")
    ("🪃" . ":boomerang:")
    ("🥅" . ":goal_net:")
    ("\342\233\263" . ":golf: :flag_in_hole:")
    ("🪁" . ":kite:")
    ("🛝" . ":playground_slide:")
    ("🏹" . ":bow_and_arrow: :archery:")
    ("🎣" . ":fishing_pole_and_fish:")
    ("🤿" . ":diving_mask:")
    ("🥊" . ":boxing_gloves:")
    ("🥋" . ":martial_arts_uniform: :karate_uniform:")
    ("🎽" . ":running_shirt_with_sash:")
    ("🛹" . ":skateboard:")
    ("🛼" . ":roller_skate:")
    ("🛷" . ":sled:")
    ("\342\233\270\357\270\217" . ":ice_skate:")
    ("🥌" . ":curling_stone:")
    ("🎿" . ":skis:")
    ("\342\233\267\357\270\217" . ":skier:")
    ("🏂" . ":snowboarder:")
    ("🏂🏻" . ":snowboarder_tone1: :snowboarder_light_skin_tone: :snowboarder::skin-tone-1:")
    ("🏂🏼" . ":snowboarder_tone2: :snowboarder_medium_light_skin_tone: :snowboarder::skin-tone-2:")
    ("🏂🏽" . ":snowboarder_tone3: :snowboarder_medium_skin_tone: :snowboarder::skin-tone-3:")
    ("🏂🏾" . ":snowboarder_tone4: :snowboarder_medium_dark_skin_tone: :snowboarder::skin-tone-4:")
    ("🏂🏿" . ":snowboarder_tone5: :snowboarder_dark_skin_tone: :snowboarder::skin-tone-5:")
    ("🪂" . ":parachute:")
    ("🏋\357\270\217" . ":person_lifting_weights: :weight_lifter:")
    ("🏋🏻" . ":person_lifting_weights_tone1: :weight_lifter_tone1: :person_lifting_weights::skin-tone-1: :weight_lifter::skin-tone-1:")
    ("🏋🏼" . ":person_lifting_weights_tone2: :weight_lifter_tone2: :person_lifting_weights::skin-tone-2: :weight_lifter::skin-tone-2:")
    ("🏋🏽" . ":person_lifting_weights_tone3: :weight_lifter_tone3: :person_lifting_weights::skin-tone-3: :weight_lifter::skin-tone-3:")
    ("🏋🏾" . ":person_lifting_weights_tone4: :weight_lifter_tone4: :person_lifting_weights::skin-tone-4: :weight_lifter::skin-tone-4:")
    ("🏋🏿" . ":person_lifting_weights_tone5: :weight_lifter_tone5: :person_lifting_weights::skin-tone-5: :weight_lifter::skin-tone-5:")
    ("🏋\357\270\217\342\200\215\342\231\200\357\270\217" . ":woman_lifting_weights:")
    ("🏋🏻\342\200\215\342\231\200\357\270\217" . ":woman_lifting_weights_tone1: :woman_lifting_weights_light_skin_tone: :woman_lifting_weights::skin-tone-1:")
    ("🏋🏼\342\200\215\342\231\200\357\270\217" . ":woman_lifting_weights_tone2: :woman_lifting_weights_medium_light_skin_tone: :woman_lifting_weights::skin-tone-2:")
    ("🏋🏽\342\200\215\342\231\200\357\270\217" . ":woman_lifting_weights_tone3: :woman_lifting_weights_medium_skin_tone: :woman_lifting_weights::skin-tone-3:")
    ("🏋🏾\342\200\215\342\231\200\357\270\217" . ":woman_lifting_weights_tone4: :woman_lifting_weights_medium_dark_skin_tone: :woman_lifting_weights::skin-tone-4:")
    ("🏋🏿\342\200\215\342\231\200\357\270\217" . ":woman_lifting_weights_tone5: :woman_lifting_weights_dark_skin_tone: :woman_lifting_weights::skin-tone-5:")
    ("🏋\357\270\217\342\200\215\342\231\202\357\270\217" . ":man_lifting_weights:")
    ("🏋🏻\342\200\215\342\231\202\357\270\217" . ":man_lifting_weights_tone1: :man_lifting_weights_light_skin_tone: :man_lifting_weights::skin-tone-1:")
    ("🏋🏼\342\200\215\342\231\202\357\270\217" . ":man_lifting_weights_tone2: :man_lifting_weights_medium_light_skin_tone: :man_lifting_weights::skin-tone-2:")
    ("🏋🏽\342\200\215\342\231\202\357\270\217" . ":man_lifting_weights_tone3: :man_lifting_weights_medium_skin_tone: :man_lifting_weights::skin-tone-3:")
    ("🏋🏾\342\200\215\342\231\202\357\270\217" . ":man_lifting_weights_tone4: :man_lifting_weights_medium_dark_skin_tone: :man_lifting_weights::skin-tone-4:")
    ("🏋🏿\342\200\215\342\231\202\357\270\217" . ":man_lifting_weights_tone5: :man_lifting_weights_dark_skin_tone: :man_lifting_weights::skin-tone-5:")
    ("🤼" . ":people_wrestling: :wrestlers:")
    ("🤼\342\200\215\342\231\200\357\270\217" . ":women_wrestling:")
    ("🤼\342\200\215\342\231\202\357\270\217" . ":men_wrestling:")
    ("🤸" . ":person_doing_cartwheel:")
    ("🤸🏻" . ":person_doing_cartwheel_tone1: :person_doing_cartwheel::skin-tone-1:")
    ("🤸🏼" . ":person_doing_cartwheel_tone2: :person_doing_cartwheel::skin-tone-2:")
    ("🤸🏽" . ":person_doing_cartwheel_tone3: :person_doing_cartwheel::skin-tone-3:")
    ("🤸🏾" . ":person_doing_cartwheel_tone4: :person_doing_cartwheel::skin-tone-4:")
    ("🤸🏿" . ":person_doing_cartwheel_tone5: :person_doing_cartwheel::skin-tone-5:")
    ("🤸\342\200\215\342\231\200\357\270\217" . ":woman_cartwheeling:")
    ("🤸🏻\342\200\215\342\231\200\357\270\217" . ":woman_cartwheeling_tone1: :woman_cartwheeling_light_skin_tone: :woman_cartwheeling::skin-tone-1:")
    ("🤸🏼\342\200\215\342\231\200\357\270\217" . ":woman_cartwheeling_tone2: :woman_cartwheeling_medium_light_skin_tone: :woman_cartwheeling::skin-tone-2:")
    ("🤸🏽\342\200\215\342\231\200\357\270\217" . ":woman_cartwheeling_tone3: :woman_cartwheeling_medium_skin_tone: :woman_cartwheeling::skin-tone-3:")
    ("🤸🏾\342\200\215\342\231\200\357\270\217" . ":woman_cartwheeling_tone4: :woman_cartwheeling_medium_dark_skin_tone: :woman_cartwheeling::skin-tone-4:")
    ("🤸🏿\342\200\215\342\231\200\357\270\217" . ":woman_cartwheeling_tone5: :woman_cartwheeling_dark_skin_tone: :woman_cartwheeling::skin-tone-5:")
    ("🤸\342\200\215\342\231\202\357\270\217" . ":man_cartwheeling:")
    ("🤸🏻\342\200\215\342\231\202\357\270\217" . ":man_cartwheeling_tone1: :man_cartwheeling_light_skin_tone: :man_cartwheeling::skin-tone-1:")
    ("🤸🏼\342\200\215\342\231\202\357\270\217" . ":man_cartwheeling_tone2: :man_cartwheeling_medium_light_skin_tone: :man_cartwheeling::skin-tone-2:")
    ("🤸🏽\342\200\215\342\231\202\357\270\217" . ":man_cartwheeling_tone3: :man_cartwheeling_medium_skin_tone: :man_cartwheeling::skin-tone-3:")
    ("🤸🏾\342\200\215\342\231\202\357\270\217" . ":man_cartwheeling_tone4: :man_cartwheeling_medium_dark_skin_tone: :man_cartwheeling::skin-tone-4:")
    ("🤸🏿\342\200\215\342\231\202\357\270\217" . ":man_cartwheeling_tone5: :man_cartwheeling_dark_skin_tone: :man_cartwheeling::skin-tone-5:")
    ("\342\233\271\357\270\217" . ":person_bouncing_ball: :basketball_player: :person_with_ball:")
    ("\342\233\271🏻" . ":person_bouncing_ball_tone1: :basketball_player_tone1: :person_with_ball_tone1: :person_bouncing_ball::skin-tone-1: :basketball_player::skin-tone-1: :person_with_ball::skin-tone-1:")
    ("\342\233\271🏼" . ":person_bouncing_ball_tone2: :basketball_player_tone2: :person_with_ball_tone2: :person_bouncing_ball::skin-tone-2: :basketball_player::skin-tone-2: :person_with_ball::skin-tone-2:")
    ("\342\233\271🏽" . ":person_bouncing_ball_tone3: :basketball_player_tone3: :person_with_ball_tone3: :person_bouncing_ball::skin-tone-3: :basketball_player::skin-tone-3: :person_with_ball::skin-tone-3:")
    ("\342\233\271🏾" . ":person_bouncing_ball_tone4: :basketball_player_tone4: :person_with_ball_tone4: :person_bouncing_ball::skin-tone-4: :basketball_player::skin-tone-4: :person_with_ball::skin-tone-4:")
    ("\342\233\271🏿" . ":person_bouncing_ball_tone5: :basketball_player_tone5: :person_with_ball_tone5: :person_bouncing_ball::skin-tone-5: :basketball_player::skin-tone-5: :person_with_ball::skin-tone-5:")
    ("\342\233\271\357\270\217\342\200\215\342\231\200\357\270\217" . ":woman_bouncing_ball:")
    ("\342\233\271🏻\342\200\215\342\231\200\357\270\217" . ":woman_bouncing_ball_tone1: :woman_bouncing_ball_light_skin_tone: :woman_bouncing_ball::skin-tone-1:")
    ("\342\233\271🏼\342\200\215\342\231\200\357\270\217" . ":woman_bouncing_ball_tone2: :woman_bouncing_ball_medium_light_skin_tone: :woman_bouncing_ball::skin-tone-2:")
    ("\342\233\271🏽\342\200\215\342\231\200\357\270\217" . ":woman_bouncing_ball_tone3: :woman_bouncing_ball_medium_skin_tone: :woman_bouncing_ball::skin-tone-3:")
    ("\342\233\271🏾\342\200\215\342\231\200\357\270\217" . ":woman_bouncing_ball_tone4: :woman_bouncing_ball_medium_dark_skin_tone: :woman_bouncing_ball::skin-tone-4:")
    ("\342\233\271🏿\342\200\215\342\231\200\357\270\217" . ":woman_bouncing_ball_tone5: :woman_bouncing_ball_dark_skin_tone: :woman_bouncing_ball::skin-tone-5:")
    ("\342\233\271\357\270\217\342\200\215\342\231\202\357\270\217" . ":man_bouncing_ball:")
    ("\342\233\271🏻\342\200\215\342\231\202\357\270\217" . ":man_bouncing_ball_tone1: :man_bouncing_ball_light_skin_tone: :man_bouncing_ball::skin-tone-1:")
    ("\342\233\271🏼\342\200\215\342\231\202\357\270\217" . ":man_bouncing_ball_tone2: :man_bouncing_ball_medium_light_skin_tone: :man_bouncing_ball::skin-tone-2:")
    ("\342\233\271🏽\342\200\215\342\231\202\357\270\217" . ":man_bouncing_ball_tone3: :man_bouncing_ball_medium_skin_tone: :man_bouncing_ball::skin-tone-3:")
    ("\342\233\271🏾\342\200\215\342\231\202\357\270\217" . ":man_bouncing_ball_tone4: :man_bouncing_ball_medium_dark_skin_tone: :man_bouncing_ball::skin-tone-4:")
    ("\342\233\271🏿\342\200\215\342\231\202\357\270\217" . ":man_bouncing_ball_tone5: :man_bouncing_ball_dark_skin_tone: :man_bouncing_ball::skin-tone-5:")
    ("🤺" . ":person_fencing: :fencer:")
    ("🤾" . ":person_playing_handball:")
    ("🤾🏻" . ":person_playing_handball_tone1: :person_playing_handball::skin-tone-1:")
    ("🤾🏼" . ":person_playing_handball_tone2: :person_playing_handball::skin-tone-2:")
    ("🤾🏽" . ":person_playing_handball_tone3: :person_playing_handball::skin-tone-3:")
    ("🤾🏾" . ":person_playing_handball_tone4: :person_playing_handball::skin-tone-4:")
    ("🤾🏿" . ":person_playing_handball_tone5: :person_playing_handball::skin-tone-5:")
    ("🤾\342\200\215\342\231\200\357\270\217" . ":woman_playing_handball:")
    ("🤾🏻\342\200\215\342\231\200\357\270\217" . ":woman_playing_handball_tone1: :woman_playing_handball_light_skin_tone: :woman_playing_handball::skin-tone-1:")
    ("🤾🏼\342\200\215\342\231\200\357\270\217" . ":woman_playing_handball_tone2: :woman_playing_handball_medium_light_skin_tone: :woman_playing_handball::skin-tone-2:")
    ("🤾🏽\342\200\215\342\231\200\357\270\217" . ":woman_playing_handball_tone3: :woman_playing_handball_medium_skin_tone: :woman_playing_handball::skin-tone-3:")
    ("🤾🏾\342\200\215\342\231\200\357\270\217" . ":woman_playing_handball_tone4: :woman_playing_handball_medium_dark_skin_tone: :woman_playing_handball::skin-tone-4:")
    ("🤾🏿\342\200\215\342\231\200\357\270\217" . ":woman_playing_handball_tone5: :woman_playing_handball_dark_skin_tone: :woman_playing_handball::skin-tone-5:")
    ("🤾\342\200\215\342\231\202\357\270\217" . ":man_playing_handball:")
    ("🤾🏻\342\200\215\342\231\202\357\270\217" . ":man_playing_handball_tone1: :man_playing_handball_light_skin_tone: :man_playing_handball::skin-tone-1:")
    ("🤾🏼\342\200\215\342\231\202\357\270\217" . ":man_playing_handball_tone2: :man_playing_handball_medium_light_skin_tone: :man_playing_handball::skin-tone-2:")
    ("🤾🏽\342\200\215\342\231\202\357\270\217" . ":man_playing_handball_tone3: :man_playing_handball_medium_skin_tone: :man_playing_handball::skin-tone-3:")
    ("🤾🏾\342\200\215\342\231\202\357\270\217" . ":man_playing_handball_tone4: :man_playing_handball_medium_dark_skin_tone: :man_playing_handball::skin-tone-4:")
    ("🤾🏿\342\200\215\342\231\202\357\270\217" . ":man_playing_handball_tone5: :man_playing_handball_dark_skin_tone: :man_playing_handball::skin-tone-5:")
    ("🏌\357\270\217" . ":person_golfing: :golfer:")
    ("🏌🏻" . ":person_golfing_tone1: :person_golfing_light_skin_tone: :person_golfing::skin-tone-1: :golfer::skin-tone-1:")
    ("🏌🏼" . ":person_golfing_tone2: :person_golfing_medium_light_skin_tone: :person_golfing::skin-tone-2: :golfer::skin-tone-2:")
    ("🏌🏽" . ":person_golfing_tone3: :person_golfing_medium_skin_tone: :person_golfing::skin-tone-3: :golfer::skin-tone-3:")
    ("🏌🏾" . ":person_golfing_tone4: :person_golfing_medium_dark_skin_tone: :person_golfing::skin-tone-4: :golfer::skin-tone-4:")
    ("🏌🏿" . ":person_golfing_tone5: :person_golfing_dark_skin_tone: :person_golfing::skin-tone-5: :golfer::skin-tone-5:")
    ("🏌\357\270\217\342\200\215\342\231\200\357\270\217" . ":woman_golfing:")
    ("🏌🏻\342\200\215\342\231\200\357\270\217" . ":woman_golfing_tone1: :woman_golfing_light_skin_tone: :woman_golfing::skin-tone-1:")
    ("🏌🏼\342\200\215\342\231\200\357\270\217" . ":woman_golfing_tone2: :woman_golfing_medium_light_skin_tone: :woman_golfing::skin-tone-2:")
    ("🏌🏽\342\200\215\342\231\200\357\270\217" . ":woman_golfing_tone3: :woman_golfing_medium_skin_tone: :woman_golfing::skin-tone-3:")
    ("🏌🏾\342\200\215\342\231\200\357\270\217" . ":woman_golfing_tone4: :woman_golfing_medium_dark_skin_tone: :woman_golfing::skin-tone-4:")
    ("🏌🏿\342\200\215\342\231\200\357\270\217" . ":woman_golfing_tone5: :woman_golfing_dark_skin_tone: :woman_golfing::skin-tone-5:")
    ("🏌\357\270\217\342\200\215\342\231\202\357\270\217" . ":man_golfing:")
    ("🏌🏻\342\200\215\342\231\202\357\270\217" . ":man_golfing_tone1: :man_golfing_light_skin_tone: :man_golfing::skin-tone-1:")
    ("🏌🏼\342\200\215\342\231\202\357\270\217" . ":man_golfing_tone2: :man_golfing_medium_light_skin_tone: :man_golfing::skin-tone-2:")
    ("🏌🏽\342\200\215\342\231\202\357\270\217" . ":man_golfing_tone3: :man_golfing_medium_skin_tone: :man_golfing::skin-tone-3:")
    ("🏌🏾\342\200\215\342\231\202\357\270\217" . ":man_golfing_tone4: :man_golfing_medium_dark_skin_tone: :man_golfing::skin-tone-4:")
    ("🏌🏿\342\200\215\342\231\202\357\270\217" . ":man_golfing_tone5: :man_golfing_dark_skin_tone: :man_golfing::skin-tone-5:")
    ("🏇" . ":horse_racing:")
    ("🏇🏻" . ":horse_racing_tone1: :horse_racing::skin-tone-1:")
    ("🏇🏼" . ":horse_racing_tone2: :horse_racing::skin-tone-2:")
    ("🏇🏽" . ":horse_racing_tone3: :horse_racing::skin-tone-3:")
    ("🏇🏾" . ":horse_racing_tone4: :horse_racing::skin-tone-4:")
    ("🏇🏿" . ":horse_racing_tone5: :horse_racing::skin-tone-5:")
    ("🧘" . ":person_in_lotus_position:")
    ("🧘🏻" . ":person_in_lotus_position_tone1: :person_in_lotus_position_light_skin_tone: :person_in_lotus_position::skin-tone-1:")
    ("🧘🏼" . ":person_in_lotus_position_tone2: :person_in_lotus_position_medium_light_skin_tone: :person_in_lotus_position::skin-tone-2:")
    ("🧘🏽" . ":person_in_lotus_position_tone3: :person_in_lotus_position_medium_skin_tone: :person_in_lotus_position::skin-tone-3:")
    ("🧘🏾" . ":person_in_lotus_position_tone4: :person_in_lotus_position_medium_dark_skin_tone: :person_in_lotus_position::skin-tone-4:")
    ("🧘🏿" . ":person_in_lotus_position_tone5: :person_in_lotus_position_dark_skin_tone: :person_in_lotus_position::skin-tone-5:")
    ("🧘\342\200\215\342\231\200\357\270\217" . ":woman_in_lotus_position:")
    ("🧘🏻\342\200\215\342\231\200\357\270\217" . ":woman_in_lotus_position_tone1: :woman_in_lotus_position_light_skin_tone: :woman_in_lotus_position::skin-tone-1:")
    ("🧘🏼\342\200\215\342\231\200\357\270\217" . ":woman_in_lotus_position_tone2: :woman_in_lotus_position_medium_light_skin_tone: :woman_in_lotus_position::skin-tone-2:")
    ("🧘🏽\342\200\215\342\231\200\357\270\217" . ":woman_in_lotus_position_tone3: :woman_in_lotus_position_medium_skin_tone: :woman_in_lotus_position::skin-tone-3:")
    ("🧘🏾\342\200\215\342\231\200\357\270\217" . ":woman_in_lotus_position_tone4: :woman_in_lotus_position_medium_dark_skin_tone: :woman_in_lotus_position::skin-tone-4:")
    ("🧘🏿\342\200\215\342\231\200\357\270\217" . ":woman_in_lotus_position_tone5: :woman_in_lotus_position_dark_skin_tone: :woman_in_lotus_position::skin-tone-5:")
    ("🧘\342\200\215\342\231\202\357\270\217" . ":man_in_lotus_position:")
    ("🧘🏻\342\200\215\342\231\202\357\270\217" . ":man_in_lotus_position_tone1: :man_in_lotus_position_light_skin_tone: :man_in_lotus_position::skin-tone-1:")
    ("🧘🏼\342\200\215\342\231\202\357\270\217" . ":man_in_lotus_position_tone2: :man_in_lotus_position_medium_light_skin_tone: :man_in_lotus_position::skin-tone-2:")
    ("🧘🏽\342\200\215\342\231\202\357\270\217" . ":man_in_lotus_position_tone3: :man_in_lotus_position_medium_skin_tone: :man_in_lotus_position::skin-tone-3:")
    ("🧘🏾\342\200\215\342\231\202\357\270\217" . ":man_in_lotus_position_tone4: :man_in_lotus_position_medium_dark_skin_tone: :man_in_lotus_position::skin-tone-4:")
    ("🧘🏿\342\200\215\342\231\202\357\270\217" . ":man_in_lotus_position_tone5: :man_in_lotus_position_dark_skin_tone: :man_in_lotus_position::skin-tone-5:")
    ("🏄" . ":person_surfing: :surfer:")
    ("🏄🏻" . ":person_surfing_tone1: :surfer_tone1: :person_surfing::skin-tone-1: :surfer::skin-tone-1:")
    ("🏄🏼" . ":person_surfing_tone2: :surfer_tone2: :person_surfing::skin-tone-2: :surfer::skin-tone-2:")
    ("🏄🏽" . ":person_surfing_tone3: :surfer_tone3: :person_surfing::skin-tone-3: :surfer::skin-tone-3:")
    ("🏄🏾" . ":person_surfing_tone4: :surfer_tone4: :person_surfing::skin-tone-4: :surfer::skin-tone-4:")
    ("🏄🏿" . ":person_surfing_tone5: :surfer_tone5: :person_surfing::skin-tone-5: :surfer::skin-tone-5:")
    ("🏄\342\200\215\342\231\200\357\270\217" . ":woman_surfing:")
    ("🏄🏻\342\200\215\342\231\200\357\270\217" . ":woman_surfing_tone1: :woman_surfing_light_skin_tone: :woman_surfing::skin-tone-1:")
    ("🏄🏼\342\200\215\342\231\200\357\270\217" . ":woman_surfing_tone2: :woman_surfing_medium_light_skin_tone: :woman_surfing::skin-tone-2:")
    ("🏄🏽\342\200\215\342\231\200\357\270\217" . ":woman_surfing_tone3: :woman_surfing_medium_skin_tone: :woman_surfing::skin-tone-3:")
    ("🏄🏾\342\200\215\342\231\200\357\270\217" . ":woman_surfing_tone4: :woman_surfing_medium_dark_skin_tone: :woman_surfing::skin-tone-4:")
    ("🏄🏿\342\200\215\342\231\200\357\270\217" . ":woman_surfing_tone5: :woman_surfing_dark_skin_tone: :woman_surfing::skin-tone-5:")
    ("🏄\342\200\215\342\231\202\357\270\217" . ":man_surfing:")
    ("🏄🏻\342\200\215\342\231\202\357\270\217" . ":man_surfing_tone1: :man_surfing_light_skin_tone: :man_surfing::skin-tone-1:")
    ("🏄🏼\342\200\215\342\231\202\357\270\217" . ":man_surfing_tone2: :man_surfing_medium_light_skin_tone: :man_surfing::skin-tone-2:")
    ("🏄🏽\342\200\215\342\231\202\357\270\217" . ":man_surfing_tone3: :man_surfing_medium_skin_tone: :man_surfing::skin-tone-3:")
    ("🏄🏾\342\200\215\342\231\202\357\270\217" . ":man_surfing_tone4: :man_surfing_medium_dark_skin_tone: :man_surfing::skin-tone-4:")
    ("🏄🏿\342\200\215\342\231\202\357\270\217" . ":man_surfing_tone5: :man_surfing_dark_skin_tone: :man_surfing::skin-tone-5:")
    ("🏊" . ":person_swimming: :swimmer:")
    ("🏊🏻" . ":person_swimming_tone1: :swimmer_tone1: :person_swimming::skin-tone-1: :swimmer::skin-tone-1:")
    ("🏊🏼" . ":person_swimming_tone2: :swimmer_tone2: :person_swimming::skin-tone-2: :swimmer::skin-tone-2:")
    ("🏊🏽" . ":person_swimming_tone3: :swimmer_tone3: :person_swimming::skin-tone-3: :swimmer::skin-tone-3:")
    ("🏊🏾" . ":person_swimming_tone4: :swimmer_tone4: :person_swimming::skin-tone-4: :swimmer::skin-tone-4:")
    ("🏊🏿" . ":person_swimming_tone5: :swimmer_tone5: :person_swimming::skin-tone-5: :swimmer::skin-tone-5:")
    ("🏊\342\200\215\342\231\200\357\270\217" . ":woman_swimming:")
    ("🏊🏻\342\200\215\342\231\200\357\270\217" . ":woman_swimming_tone1: :woman_swimming_light_skin_tone: :woman_swimming::skin-tone-1:")
    ("🏊🏼\342\200\215\342\231\200\357\270\217" . ":woman_swimming_tone2: :woman_swimming_medium_light_skin_tone: :woman_swimming::skin-tone-2:")
    ("🏊🏽\342\200\215\342\231\200\357\270\217" . ":woman_swimming_tone3: :woman_swimming_medium_skin_tone: :woman_swimming::skin-tone-3:")
    ("🏊🏾\342\200\215\342\231\200\357\270\217" . ":woman_swimming_tone4: :woman_swimming_medium_dark_skin_tone: :woman_swimming::skin-tone-4:")
    ("🏊🏿\342\200\215\342\231\200\357\270\217" . ":woman_swimming_tone5: :woman_swimming_dark_skin_tone: :woman_swimming::skin-tone-5:")
    ("🏊\342\200\215\342\231\202\357\270\217" . ":man_swimming:")
    ("🏊🏻\342\200\215\342\231\202\357\270\217" . ":man_swimming_tone1: :man_swimming_light_skin_tone: :man_swimming::skin-tone-1:")
    ("🏊🏼\342\200\215\342\231\202\357\270\217" . ":man_swimming_tone2: :man_swimming_medium_light_skin_tone: :man_swimming::skin-tone-2:")
    ("🏊🏽\342\200\215\342\231\202\357\270\217" . ":man_swimming_tone3: :man_swimming_medium_skin_tone: :man_swimming::skin-tone-3:")
    ("🏊🏾\342\200\215\342\231\202\357\270\217" . ":man_swimming_tone4: :man_swimming_medium_dark_skin_tone: :man_swimming::skin-tone-4:")
    ("🏊🏿\342\200\215\342\231\202\357\270\217" . ":man_swimming_tone5: :man_swimming_dark_skin_tone: :man_swimming::skin-tone-5:")
    ("🤽" . ":person_playing_water_polo:")
    ("🤽🏻" . ":person_playing_water_polo_tone1: :person_playing_water_polo::skin-tone-1:")
    ("🤽🏼" . ":person_playing_water_polo_tone2: :person_playing_water_polo::skin-tone-2:")
    ("🤽🏽" . ":person_playing_water_polo_tone3: :person_playing_water_polo::skin-tone-3:")
    ("🤽🏾" . ":person_playing_water_polo_tone4: :person_playing_water_polo::skin-tone-4:")
    ("🤽🏿" . ":person_playing_water_polo_tone5: :person_playing_water_polo::skin-tone-5:")
    ("🤽\342\200\215\342\231\200\357\270\217" . ":woman_playing_water_polo:")
    ("🤽🏻\342\200\215\342\231\200\357\270\217" . ":woman_playing_water_polo_tone1: :woman_playing_water_polo_light_skin_tone: :woman_playing_water_polo::skin-tone-1:")
    ("🤽🏼\342\200\215\342\231\200\357\270\217" . ":woman_playing_water_polo_tone2: :woman_playing_water_polo_medium_light_skin_tone: :woman_playing_water_polo::skin-tone-2:")
    ("🤽🏽\342\200\215\342\231\200\357\270\217" . ":woman_playing_water_polo_tone3: :woman_playing_water_polo_medium_skin_tone: :woman_playing_water_polo::skin-tone-3:")
    ("🤽🏾\342\200\215\342\231\200\357\270\217" . ":woman_playing_water_polo_tone4: :woman_playing_water_polo_medium_dark_skin_tone: :woman_playing_water_polo::skin-tone-4:")
    ("🤽🏿\342\200\215\342\231\200\357\270\217" . ":woman_playing_water_polo_tone5: :woman_playing_water_polo_dark_skin_tone: :woman_playing_water_polo::skin-tone-5:")
    ("🤽\342\200\215\342\231\202\357\270\217" . ":man_playing_water_polo:")
    ("🤽🏻\342\200\215\342\231\202\357\270\217" . ":man_playing_water_polo_tone1: :man_playing_water_polo_light_skin_tone: :man_playing_water_polo::skin-tone-1:")
    ("🤽🏼\342\200\215\342\231\202\357\270\217" . ":man_playing_water_polo_tone2: :man_playing_water_polo_medium_light_skin_tone: :man_playing_water_polo::skin-tone-2:")
    ("🤽🏽\342\200\215\342\231\202\357\270\217" . ":man_playing_water_polo_tone3: :man_playing_water_polo_medium_skin_tone: :man_playing_water_polo::skin-tone-3:")
    ("🤽🏾\342\200\215\342\231\202\357\270\217" . ":man_playing_water_polo_tone4: :man_playing_water_polo_medium_dark_skin_tone: :man_playing_water_polo::skin-tone-4:")
    ("🤽🏿\342\200\215\342\231\202\357\270\217" . ":man_playing_water_polo_tone5: :man_playing_water_polo_dark_skin_tone: :man_playing_water_polo::skin-tone-5:")
    ("🚣" . ":person_rowing_boat: :rowboat:")
    ("🚣🏻" . ":person_rowing_boat_tone1: :rowboat_tone1: :person_rowing_boat::skin-tone-1: :rowboat::skin-tone-1:")
    ("🚣🏼" . ":person_rowing_boat_tone2: :rowboat_tone2: :person_rowing_boat::skin-tone-2: :rowboat::skin-tone-2:")
    ("🚣🏽" . ":person_rowing_boat_tone3: :rowboat_tone3: :person_rowing_boat::skin-tone-3: :rowboat::skin-tone-3:")
    ("🚣🏾" . ":person_rowing_boat_tone4: :rowboat_tone4: :person_rowing_boat::skin-tone-4: :rowboat::skin-tone-4:")
    ("🚣🏿" . ":person_rowing_boat_tone5: :rowboat_tone5: :person_rowing_boat::skin-tone-5: :rowboat::skin-tone-5:")
    ("🚣\342\200\215\342\231\200\357\270\217" . ":woman_rowing_boat:")
    ("🚣🏻\342\200\215\342\231\200\357\270\217" . ":woman_rowing_boat_tone1: :woman_rowing_boat_light_skin_tone: :woman_rowing_boat::skin-tone-1:")
    ("🚣🏼\342\200\215\342\231\200\357\270\217" . ":woman_rowing_boat_tone2: :woman_rowing_boat_medium_light_skin_tone: :woman_rowing_boat::skin-tone-2:")
    ("🚣🏽\342\200\215\342\231\200\357\270\217" . ":woman_rowing_boat_tone3: :woman_rowing_boat_medium_skin_tone: :woman_rowing_boat::skin-tone-3:")
    ("🚣🏾\342\200\215\342\231\200\357\270\217" . ":woman_rowing_boat_tone4: :woman_rowing_boat_medium_dark_skin_tone: :woman_rowing_boat::skin-tone-4:")
    ("🚣🏿\342\200\215\342\231\200\357\270\217" . ":woman_rowing_boat_tone5: :woman_rowing_boat_dark_skin_tone: :woman_rowing_boat::skin-tone-5:")
    ("🚣\342\200\215\342\231\202\357\270\217" . ":man_rowing_boat:")
    ("🚣🏻\342\200\215\342\231\202\357\270\217" . ":man_rowing_boat_tone1: :man_rowing_boat_light_skin_tone: :man_rowing_boat::skin-tone-1:")
    ("🚣🏼\342\200\215\342\231\202\357\270\217" . ":man_rowing_boat_tone2: :man_rowing_boat_medium_light_skin_tone: :man_rowing_boat::skin-tone-2:")
    ("🚣🏽\342\200\215\342\231\202\357\270\217" . ":man_rowing_boat_tone3: :man_rowing_boat_medium_skin_tone: :man_rowing_boat::skin-tone-3:")
    ("🚣🏾\342\200\215\342\231\202\357\270\217" . ":man_rowing_boat_tone4: :man_rowing_boat_medium_dark_skin_tone: :man_rowing_boat::skin-tone-4:")
    ("🚣🏿\342\200\215\342\231\202\357\270\217" . ":man_rowing_boat_tone5: :man_rowing_boat_dark_skin_tone: :man_rowing_boat::skin-tone-5:")
    ("🧗" . ":person_climbing:")
    ("🧗🏻" . ":person_climbing_tone1: :person_climbing_light_skin_tone: :person_climbing::skin-tone-1:")
    ("🧗🏼" . ":person_climbing_tone2: :person_climbing_medium_light_skin_tone: :person_climbing::skin-tone-2:")
    ("🧗🏽" . ":person_climbing_tone3: :person_climbing_medium_skin_tone: :person_climbing::skin-tone-3:")
    ("🧗🏾" . ":person_climbing_tone4: :person_climbing_medium_dark_skin_tone: :person_climbing::skin-tone-4:")
    ("🧗🏿" . ":person_climbing_tone5: :person_climbing_dark_skin_tone: :person_climbing::skin-tone-5:")
    ("🧗\342\200\215\342\231\200\357\270\217" . ":woman_climbing:")
    ("🧗🏻\342\200\215\342\231\200\357\270\217" . ":woman_climbing_tone1: :woman_climbing_light_skin_tone: :woman_climbing::skin-tone-1:")
    ("🧗🏼\342\200\215\342\231\200\357\270\217" . ":woman_climbing_tone2: :woman_climbing_medium_light_skin_tone: :woman_climbing::skin-tone-2:")
    ("🧗🏽\342\200\215\342\231\200\357\270\217" . ":woman_climbing_tone3: :woman_climbing_medium_skin_tone: :woman_climbing::skin-tone-3:")
    ("🧗🏾\342\200\215\342\231\200\357\270\217" . ":woman_climbing_tone4: :woman_climbing_medium_dark_skin_tone: :woman_climbing::skin-tone-4:")
    ("🧗🏿\342\200\215\342\231\200\357\270\217" . ":woman_climbing_tone5: :woman_climbing_dark_skin_tone: :woman_climbing::skin-tone-5:")
    ("🧗\342\200\215\342\231\202\357\270\217" . ":man_climbing:")
    ("🧗🏻\342\200\215\342\231\202\357\270\217" . ":man_climbing_tone1: :man_climbing_light_skin_tone: :man_climbing::skin-tone-1:")
    ("🧗🏼\342\200\215\342\231\202\357\270\217" . ":man_climbing_tone2: :man_climbing_medium_light_skin_tone: :man_climbing::skin-tone-2:")
    ("🧗🏽\342\200\215\342\231\202\357\270\217" . ":man_climbing_tone3: :man_climbing_medium_skin_tone: :man_climbing::skin-tone-3:")
    ("🧗🏾\342\200\215\342\231\202\357\270\217" . ":man_climbing_tone4: :man_climbing_medium_dark_skin_tone: :man_climbing::skin-tone-4:")
    ("🧗🏿\342\200\215\342\231\202\357\270\217" . ":man_climbing_tone5: :man_climbing_dark_skin_tone: :man_climbing::skin-tone-5:")
    ("🚵" . ":person_mountain_biking: :mountain_bicyclist:")
    ("🚵🏻" . ":person_mountain_biking_tone1: :mountain_bicyclist_tone1: :person_mountain_biking::skin-tone-1: :mountain_bicyclist::skin-tone-1:")
    ("🚵🏼" . ":person_mountain_biking_tone2: :mountain_bicyclist_tone2: :person_mountain_biking::skin-tone-2: :mountain_bicyclist::skin-tone-2:")
    ("🚵🏽" . ":person_mountain_biking_tone3: :mountain_bicyclist_tone3: :person_mountain_biking::skin-tone-3: :mountain_bicyclist::skin-tone-3:")
    ("🚵🏾" . ":person_mountain_biking_tone4: :mountain_bicyclist_tone4: :person_mountain_biking::skin-tone-4: :mountain_bicyclist::skin-tone-4:")
    ("🚵🏿" . ":person_mountain_biking_tone5: :mountain_bicyclist_tone5: :person_mountain_biking::skin-tone-5: :mountain_bicyclist::skin-tone-5:")
    ("🚵\342\200\215\342\231\200\357\270\217" . ":woman_mountain_biking:")
    ("🚵🏻\342\200\215\342\231\200\357\270\217" . ":woman_mountain_biking_tone1: :woman_mountain_biking_light_skin_tone: :woman_mountain_biking::skin-tone-1:")
    ("🚵🏼\342\200\215\342\231\200\357\270\217" . ":woman_mountain_biking_tone2: :woman_mountain_biking_medium_light_skin_tone: :woman_mountain_biking::skin-tone-2:")
    ("🚵🏽\342\200\215\342\231\200\357\270\217" . ":woman_mountain_biking_tone3: :woman_mountain_biking_medium_skin_tone: :woman_mountain_biking::skin-tone-3:")
    ("🚵🏾\342\200\215\342\231\200\357\270\217" . ":woman_mountain_biking_tone4: :woman_mountain_biking_medium_dark_skin_tone: :woman_mountain_biking::skin-tone-4:")
    ("🚵🏿\342\200\215\342\231\200\357\270\217" . ":woman_mountain_biking_tone5: :woman_mountain_biking_dark_skin_tone: :woman_mountain_biking::skin-tone-5:")
    ("🚵\342\200\215\342\231\202\357\270\217" . ":man_mountain_biking:")
    ("🚵🏻\342\200\215\342\231\202\357\270\217" . ":man_mountain_biking_tone1: :man_mountain_biking_light_skin_tone: :man_mountain_biking::skin-tone-1:")
    ("🚵🏼\342\200\215\342\231\202\357\270\217" . ":man_mountain_biking_tone2: :man_mountain_biking_medium_light_skin_tone: :man_mountain_biking::skin-tone-2:")
    ("🚵🏽\342\200\215\342\231\202\357\270\217" . ":man_mountain_biking_tone3: :man_mountain_biking_medium_skin_tone: :man_mountain_biking::skin-tone-3:")
    ("🚵🏾\342\200\215\342\231\202\357\270\217" . ":man_mountain_biking_tone4: :man_mountain_biking_medium_dark_skin_tone: :man_mountain_biking::skin-tone-4:")
    ("🚵🏿\342\200\215\342\231\202\357\270\217" . ":man_mountain_biking_tone5: :man_mountain_biking_dark_skin_tone: :man_mountain_biking::skin-tone-5:")
    ("🚴" . ":person_biking: :bicyclist:")
    ("🚴🏻" . ":person_biking_tone1: :bicyclist_tone1: :person_biking::skin-tone-1: :bicyclist::skin-tone-1:")
    ("🚴🏼" . ":person_biking_tone2: :bicyclist_tone2: :person_biking::skin-tone-2: :bicyclist::skin-tone-2:")
    ("🚴🏽" . ":person_biking_tone3: :bicyclist_tone3: :person_biking::skin-tone-3: :bicyclist::skin-tone-3:")
    ("🚴🏾" . ":person_biking_tone4: :bicyclist_tone4: :person_biking::skin-tone-4: :bicyclist::skin-tone-4:")
    ("🚴🏿" . ":person_biking_tone5: :bicyclist_tone5: :person_biking::skin-tone-5: :bicyclist::skin-tone-5:")
    ("🚴\342\200\215\342\231\200\357\270\217" . ":woman_biking:")
    ("🚴🏻\342\200\215\342\231\200\357\270\217" . ":woman_biking_tone1: :woman_biking_light_skin_tone: :woman_biking::skin-tone-1:")
    ("🚴🏼\342\200\215\342\231\200\357\270\217" . ":woman_biking_tone2: :woman_biking_medium_light_skin_tone: :woman_biking::skin-tone-2:")
    ("🚴🏽\342\200\215\342\231\200\357\270\217" . ":woman_biking_tone3: :woman_biking_medium_skin_tone: :woman_biking::skin-tone-3:")
    ("🚴🏾\342\200\215\342\231\200\357\270\217" . ":woman_biking_tone4: :woman_biking_medium_dark_skin_tone: :woman_biking::skin-tone-4:")
    ("🚴🏿\342\200\215\342\231\200\357\270\217" . ":woman_biking_tone5: :woman_biking_dark_skin_tone: :woman_biking::skin-tone-5:")
    ("🚴\342\200\215\342\231\202\357\270\217" . ":man_biking:")
    ("🚴🏻\342\200\215\342\231\202\357\270\217" . ":man_biking_tone1: :man_biking_light_skin_tone: :man_biking::skin-tone-1:")
    ("🚴🏼\342\200\215\342\231\202\357\270\217" . ":man_biking_tone2: :man_biking_medium_light_skin_tone: :man_biking::skin-tone-2:")
    ("🚴🏽\342\200\215\342\231\202\357\270\217" . ":man_biking_tone3: :man_biking_medium_skin_tone: :man_biking::skin-tone-3:")
    ("🚴🏾\342\200\215\342\231\202\357\270\217" . ":man_biking_tone4: :man_biking_medium_dark_skin_tone: :man_biking::skin-tone-4:")
    ("🚴🏿\342\200\215\342\231\202\357\270\217" . ":man_biking_tone5: :man_biking_dark_skin_tone: :man_biking::skin-tone-5:")
    ("🏆" . ":trophy:")
    ("🥇" . ":first_place_medal:")
    ("🥈" . ":second_place_medal:")
    ("🥉" . ":third_place_medal:")
    ("🏅" . ":sports_medal:")
    ("🎖\357\270\217" . ":military_medal:")
    ("🏵\357\270\217" . ":rosette:")
    ("🎗\357\270\217" . ":reminder_ribbon:")
    ("🎫" . ":ticket:")
    ("🎟\357\270\217" . ":admission_tickets:")
    ("🎪" . ":circus_tent:")
    ("🤹" . ":person_juggling: :juggler:")
    ("🤹🏻" . ":person_juggling_tone1: :juggler_tone1: :person_juggling::skin-tone-1: :juggler::skin-tone-1:")
    ("🤹🏼" . ":person_juggling_tone2: :juggler_tone2: :person_juggling::skin-tone-2: :juggler::skin-tone-2:")
    ("🤹🏽" . ":person_juggling_tone3: :juggler_tone3: :person_juggling::skin-tone-3: :juggler::skin-tone-3:")
    ("🤹🏾" . ":person_juggling_tone4: :juggler_tone4: :person_juggling::skin-tone-4: :juggler::skin-tone-4:")
    ("🤹🏿" . ":person_juggling_tone5: :juggler_tone5: :person_juggling::skin-tone-5: :juggler::skin-tone-5:")
    ("🤹\342\200\215\342\231\200\357\270\217" . ":woman_juggling:")
    ("🤹🏻\342\200\215\342\231\200\357\270\217" . ":woman_juggling_tone1: :woman_juggling_light_skin_tone: :woman_juggling::skin-tone-1:")
    ("🤹🏼\342\200\215\342\231\200\357\270\217" . ":woman_juggling_tone2: :woman_juggling_medium_light_skin_tone: :woman_juggling::skin-tone-2:")
    ("🤹🏽\342\200\215\342\231\200\357\270\217" . ":woman_juggling_tone3: :woman_juggling_medium_skin_tone: :woman_juggling::skin-tone-3:")
    ("🤹🏾\342\200\215\342\231\200\357\270\217" . ":woman_juggling_tone4: :woman_juggling_medium_dark_skin_tone: :woman_juggling::skin-tone-4:")
    ("🤹🏿\342\200\215\342\231\200\357\270\217" . ":woman_juggling_tone5: :woman_juggling_dark_skin_tone: :woman_juggling::skin-tone-5:")
    ("🤹\342\200\215\342\231\202\357\270\217" . ":man_juggling:")
    ("🤹🏻\342\200\215\342\231\202\357\270\217" . ":man_juggling_tone1: :man_juggling_light_skin_tone: :man_juggling::skin-tone-1:")
    ("🤹🏼\342\200\215\342\231\202\357\270\217" . ":man_juggling_tone2: :man_juggling_medium_light_skin_tone: :man_juggling::skin-tone-2:")
    ("🤹🏽\342\200\215\342\231\202\357\270\217" . ":man_juggling_tone3: :man_juggling_medium_skin_tone: :man_juggling::skin-tone-3:")
    ("🤹🏾\342\200\215\342\231\202\357\270\217" . ":man_juggling_tone4: :man_juggling_medium_dark_skin_tone: :man_juggling::skin-tone-4:")
    ("🤹🏿\342\200\215\342\231\202\357\270\217" . ":man_juggling_tone5: :man_juggling_dark_skin_tone: :man_juggling::skin-tone-5:")
    ("🎭" . ":performing_arts:")
    ("🩰" . ":ballet_shoes:")
    ("🎨" . ":art:")
    ("🎬" . ":clapper_board:")
    ("🎤" . ":microphone:")
    ("🎧" . ":headphones:")
    ("🎼" . ":musical_score:")
    ("🎹" . ":musical_keyboard:")
    ("🪇" . ":maracas:")
    ("🥁" . ":drum_with_drumsticks:")
    ("🪘" . ":long_drum:")
    ("🎷" . ":saxophone:")
    ("🎺" . ":trumpet:")
    ("🪗" . ":accordion:")
    ("🎸" . ":guitar:")
    ("🪕" . ":banjo:")
    ("🎻" . ":violin:")
    ("🪈" . ":flute:")
    ("🎲" . ":game_die:")
    ("\342\231\237\357\270\217" . ":chess_pawn:")
    ("🎯" . ":dart: :direct_hit:")
    ("🎳" . ":bowling:")
    ("🎮" . ":video_game:")
    ("🎰" . ":slot_machine:")
    ("🧩" . ":jigsaw: :puzzle_piece:")
    ("🚗" . ":red_car: :automobile:")
    ("🚕" . ":taxi:")
    ("🚙" . ":blue_car:")
    ("🛻" . ":pickup_truck:")
    ("🚐" . ":minibus:")
    ("🚌" . ":bus:")
    ("🚎" . ":trolleybus:")
    ("🏎\357\270\217" . ":race_car: :racing_car:")
    ("🚓" . ":police_car:")
    ("🚑" . ":ambulance:")
    ("🚒" . ":fire_engine:")
    ("🚚" . ":truck:")
    ("🚛" . ":articulated_lorry:")
    ("🚜" . ":tractor:")
    ("🦯" . ":probing_cane:")
    ("🦽" . ":manual_wheelchair:")
    ("🦼" . ":motorized_wheelchair:")
    ("🩼" . ":crutch:")
    ("🛴" . ":kick_scooter:")
    ("🚲" . ":bike: :bicycle:")
    ("🛵" . ":motor_scooter: :motorbike:")
    ("🏍\357\270\217" . ":racing_motorcycle:")
    ("🛺" . ":auto_rickshaw:")
    ("🛞" . ":wheel:")
    ("🚨" . ":rotating_light:")
    ("🚔" . ":oncoming_police_car:")
    ("🚍" . ":oncoming_bus:")
    ("🚘" . ":oncoming_automobile:")
    ("🚖" . ":oncoming_taxi:")
    ("🚡" . ":aerial_tramway:")
    ("🚠" . ":mountain_cableway:")
    ("🚟" . ":suspension_railway:")
    ("🚃" . ":railway_car:")
    ("🚋" . ":train: :tram_car:")
    ("🚞" . ":mountain_railway:")
    ("🚝" . ":monorail:")
    ("🚄" . ":bullettrain_side:")
    ("🚅" . ":bullettrain_front: :bullet_train:")
    ("🚈" . ":light_rail:")
    ("🚂" . ":steam_locomotive:")
    ("🚆" . ":train2:")
    ("🚇" . ":metro:")
    ("🚊" . ":tram:")
    ("🚉" . ":station:")
    ("\342\234\210\357\270\217" . ":airplane:")
    ("🛫" . ":airplane_departure:")
    ("🛬" . ":airplane_arriving:")
    ("🛩\357\270\217" . ":airplane_small: :small_airplane:")
    ("💺" . ":seat:")
    ("🛰\357\270\217" . ":satellite_orbital:")
    ("🚀" . ":rocket:")
    ("🛸" . ":flying_saucer:")
    ("🚁" . ":helicopter:")
    ("🛶" . ":canoe: :kayak:")
    ("\342\233\265" . ":sailboat:")
    ("🚤" . ":speedboat:")
    ("🛥\357\270\217" . ":motorboat: :motor_boat:")
    ("🛳\357\270\217" . ":cruise_ship: :passenger_ship:")
    ("\342\233\264\357\270\217" . ":ferry:")
    ("🚢" . ":ship:")
    ("🛟" . ":ring_buoy:")
    ("\342\232\223" . ":anchor:")
    ("🪝" . ":hook:")
    ("\342\233\275" . ":fuelpump: :fuel_pump:")
    ("🚧" . ":construction:")
    ("🚦" . ":vertical_traffic_light:")
    ("🚥" . ":traffic_light:")
    ("🚏" . ":busstop: :bus_stop:")
    ("🗺\357\270\217" . ":world_map:")
    ("🗿" . ":moyai: :moai:")
    ("🗽" . ":statue_of_liberty:")
    ("🗼" . ":tokyo_tower:")
    ("🏰" . ":european_castle:")
    ("🏯" . ":japanese_castle:")
    ("🏟\357\270\217" . ":stadium:")
    ("🎡" . ":ferris_wheel:")
    ("🎢" . ":roller_coaster:")
    ("🎠" . ":carousel_horse:")
    ("\342\233\262" . ":fountain:")
    ("\342\233\261\357\270\217" . ":beach_umbrella: :umbrella_on_ground:")
    ("🏖\357\270\217" . ":beach_with_umbrella:")
    ("🏝\357\270\217" . ":desert_island:")
    ("🏜\357\270\217" . ":desert:")
    ("🌋" . ":volcano:")
    ("\342\233\260\357\270\217" . ":mountain:")
    ("🏔\357\270\217" . ":mountain_snow: :snow_capped_mountain:")
    ("🗻" . ":mount_fuji:")
    ("🏕\357\270\217" . ":camping:")
    ("\342\233\272" . ":tent:")
    ("🏠" . ":house:")
    ("🏡" . ":house_with_garden:")
    ("🏘\357\270\217" . ":homes: :house_buildings: :houses:")
    ("🏚\357\270\217" . ":house_abandoned: :derelict_house_building:")
    ("🛖" . ":hut:")
    ("🏗\357\270\217" . ":construction_site: :building_construction:")
    ("🏭" . ":factory:")
    ("🏢" . ":office:")
    ("🏬" . ":department_store:")
    ("🏣" . ":post_office:")
    ("🏤" . ":european_post_office:")
    ("🏥" . ":hospital:")
    ("🏦" . ":bank:")
    ("🏨" . ":hotel:")
    ("🏪" . ":convenience_store:")
    ("🏫" . ":school:")
    ("🏩" . ":love_hotel:")
    ("💒" . ":wedding:")
    ("🏛\357\270\217" . ":classical_building:")
    ("\342\233\252" . ":church:")
    ("🕌" . ":mosque:")
    ("🕍" . ":synagogue:")
    ("🛕" . ":hindu_temple:")
    ("🕋" . ":kaaba:")
    ("\342\233\251\357\270\217" . ":shinto_shrine:")
    ("🛤\357\270\217" . ":railway_track: :railroad_track:")
    ("🛣\357\270\217" . ":motorway:")
    ("🗾" . ":map_of_japan:")
    ("🎑" . ":rice_scene:")
    ("🏞\357\270\217" . ":national_park:")
    ("🌅" . ":sunrise:")
    ("🌄" . ":sunrise_over_mountains:")
    ("🌠" . ":stars: :shooting_star:")
    ("🎇" . ":sparkler:")
    ("🎆" . ":fireworks:")
    ("🌇" . ":city_sunset: :city_sunrise:")
    ("🌆" . ":city_dusk:")
    ("🏙\357\270\217" . ":cityscape:")
    ("🌃" . ":night_with_stars:")
    ("🌌" . ":milky_way:")
    ("🌉" . ":bridge_at_night:")
    ("🌁" . ":foggy:")
    ("\342\214\232" . ":watch:")
    ("📱" . ":mobile_phone: :iphone:")
    ("📲" . ":calling:")
    ("💻" . ":computer:")
    ("\342\214\250\357\270\217" . ":keyboard:")
    ("🖥\357\270\217" . ":desktop_computer:")
    ("🖨\357\270\217" . ":printer:")
    ("🖱\357\270\217" . ":mouse_three_button: :three_button_mouse:")
    ("🖲\357\270\217" . ":trackball:")
    ("🕹\357\270\217" . ":joystick:")
    ("🗜\357\270\217" . ":compression: :clamp:")
    ("💽" . ":minidisc: :computer_disk:")
    ("💾" . ":floppy_disk:")
    ("💿" . ":cd: :optical_disk:")
    ("📀" . ":dvd:")
    ("📼" . ":vhs: :videocassette:")
    ("📷" . ":camera:")
    ("📸" . ":camera_with_flash:")
    ("📹" . ":video_camera:")
    ("🎥" . ":movie_camera:")
    ("📽\357\270\217" . ":film_projector:")
    ("🎞\357\270\217" . ":film_frames:")
    ("📞" . ":telephone_receiver:")
    ("\342\230\216\357\270\217" . ":telephone:")
    ("📟" . ":pager:")
    ("📠" . ":fax_machine:")
    ("📺" . ":tv: :television:")
    ("📻" . ":radio:")
    ("🎙\357\270\217" . ":microphone2: :studio_microphone:")
    ("🎚\357\270\217" . ":level_slider:")
    ("🎛\357\270\217" . ":control_knobs:")
    ("🧭" . ":compass:")
    ("\342\217\261\357\270\217" . ":stopwatch:")
    ("\342\217\262\357\270\217" . ":timer_clock:")
    ("\342\217\260" . ":alarm_clock:")
    ("🕰\357\270\217" . ":mantlepiece_clock:")
    ("\342\214\233" . ":hourglass:")
    ("\342\217\263" . ":hourglass_flowing_sand:")
    ("📡" . ":satellite:")
    ("🔋" . ":battery:")
    ("🪫" . ":low_battery:")
    ("🔌" . ":electric_plug:")
    ("💡" . ":light_bulb:")
    ("🔦" . ":flashlight:")
    ("🕯\357\270\217" . ":candle:")
    ("🪔" . ":diya_lamp:")
    ("🧯" . ":fire_extinguisher:")
    ("🛢\357\270\217" . ":oil_drum:")
    ("💸" . ":money_with_wings:")
    ("💵" . ":dollar:")
    ("💴" . ":yen_banknote:")
    ("💶" . ":euro_banknote:")
    ("💷" . ":pound:")
    ("🪙" . ":coin:")
    ("💰" . ":moneybag: :money_bag:")
    ("💳" . ":credit_card:")
    ("🪪" . ":identification_card:")
    ("💎" . ":gem_stone:")
    ("\342\232\226\357\270\217" . ":scales: :balance_scale:")
    ("🪜" . ":ladder:")
    ("🧰" . ":toolbox:")
    ("🪛" . ":screwdriver:")
    ("🔧" . ":wrench:")
    ("🔨" . ":hammer:")
    ("\342\232\222\357\270\217" . ":hammer_pick: :hammer_and_pick:")
    ("🛠\357\270\217" . ":tools: :hammer_and_wrench:")
    ("\342\233\217\357\270\217" . ":pick:")
    ("🪚" . ":carpentry_saw:")
    ("🔩" . ":nut_and_bolt:")
    ("\342\232\231\357\270\217" . ":gear:")
    ("🪤" . ":mouse_trap:")
    ("🧱" . ":bricks:")
    ("\342\233\223\357\270\217" . ":chains:")
    ("🔗" . ":link:")
    ("\342\233\223\357\270\217\342\200\215💥" . ":broken_chain:")
    ("🧲" . ":magnet:")
    ("🔫" . ":gun: :pistol:")
    ("💣" . ":bomb:")
    ("🧨" . ":firecracker:")
    ("🪓" . ":axe:")
    ("🔪" . ":kitchen_knife:")
    ("🗡\357\270\217" . ":dagger_knife:")
    ("\342\232\224\357\270\217" . ":crossed_swords:")
    ("🛡\357\270\217" . ":shield:")
    ("🚬" . ":smoking: :cigarette:")
    ("\342\232\260\357\270\217" . ":coffin:")
    ("🪦" . ":headstone:")
    ("\342\232\261\357\270\217" . ":funeral_urn:")
    ("🏺" . ":amphora:")
    ("🔮" . ":crystal_ball:")
    ("📿" . ":prayer_beads:")
    ("🧿" . ":nazar_amulet:")
    ("🪬" . ":hamsa:")
    ("💈" . ":barber_pole:")
    ("\342\232\227\357\270\217" . ":alembic:")
    ("🔭" . ":telescope:")
    ("🔬" . ":microscope:")
    ("🕳\357\270\217" . ":hole:")
    ("🩻" . ":x_ray:")
    ("🩹" . ":adhesive_bandage:")
    ("🩺" . ":stethoscope:")
    ("💊" . ":pill:")
    ("💉" . ":syringe:")
    ("🩸" . ":drop_of_blood:")
    ("🧬" . ":dna:")
    ("🦠" . ":microbe:")
    ("🧫" . ":petri_dish:")
    ("🧪" . ":test_tube:")
    ("🌡\357\270\217" . ":thermometer:")
    ("🧹" . ":broom:")
    ("🪠" . ":plunger:")
    ("🧺" . ":basket:")
    ("🧻" . ":roll_of_paper:")
    ("🚽" . ":toilet:")
    ("🚰" . ":potable_water:")
    ("🚿" . ":shower:")
    ("🛁" . ":bathtub:")
    ("🛀" . ":bath:")
    ("🛀🏻" . ":bath_tone1: :bath::skin-tone-1:")
    ("🛀🏼" . ":bath_tone2: :bath::skin-tone-2:")
    ("🛀🏽" . ":bath_tone3: :bath::skin-tone-3:")
    ("🛀🏾" . ":bath_tone4: :bath::skin-tone-4:")
    ("🛀🏿" . ":bath_tone5: :bath::skin-tone-5:")
    ("🧼" . ":soap:")
    ("🪥" . ":toothbrush:")
    ("🪒" . ":razor:")
    ("🪮" . ":hair_pick:")
    ("🧽" . ":sponge:")
    ("🪣" . ":bucket:")
    ("🧴" . ":squeeze_bottle: :lotion_bottle:")
    ("🛎\357\270\217" . ":bellhop_bell:")
    ("🔑" . ":key:")
    ("🗝\357\270\217" . ":key2: :old_key:")
    ("🚪" . ":door:")
    ("🪑" . ":chair:")
    ("🛋\357\270\217" . ":couch_and_lamp:")
    ("🛏\357\270\217" . ":bed:")
    ("🛌" . ":sleeping_accommodation: :person_in_bed:")
    ("🛌🏻" . ":person_in_bed_tone1: :person_in_bed_light_skin_tone: :sleeping_accommodation::skin-tone-1: :person_in_bed::skin-tone-1:")
    ("🛌🏼" . ":person_in_bed_tone2: :person_in_bed_medium_light_skin_tone: :sleeping_accommodation::skin-tone-2: :person_in_bed::skin-tone-2:")
    ("🛌🏽" . ":person_in_bed_tone3: :person_in_bed_medium_skin_tone: :sleeping_accommodation::skin-tone-3: :person_in_bed::skin-tone-3:")
    ("🛌🏾" . ":person_in_bed_tone4: :person_in_bed_medium_dark_skin_tone: :sleeping_accommodation::skin-tone-4: :person_in_bed::skin-tone-4:")
    ("🛌🏿" . ":person_in_bed_tone5: :person_in_bed_dark_skin_tone: :sleeping_accommodation::skin-tone-5: :person_in_bed::skin-tone-5:")
    ("🧸" . ":teddy_bear:")
    ("🪆" . ":nesting_dolls:")
    ("🖼\357\270\217" . ":frame_photo: :frame_with_picture:")
    ("🪞" . ":mirror:")
    ("🪟" . ":window:")
    ("🛍\357\270\217" . ":shopping_bags:")
    ("🛒" . ":shopping_cart: :shopping_trolley:")
    ("🎁" . ":wrapped_gift:")
    ("🎈" . ":balloon:")
    ("🎏" . ":flags: :carp_streamer:")
    ("🎀" . ":ribbon:")
    ("🪄" . ":magic_wand:")
    ("🪅" . ":pi\303\261ata:")
    ("🎊" . ":confetti_ball:")
    ("🎉" . ":tada: :party_popper:")
    ("🎎" . ":dolls:")
    ("🪭" . ":folding_hand_fan:")
    ("🏮" . ":izakaya_lantern:")
    ("🎐" . ":wind_chime:")
    ("🪩" . ":mirror_ball:")
    ("🧧" . ":red_envelope:")
    ("\342\234\211\357\270\217" . ":envelope:")
    ("📩" . ":envelope_with_arrow:")
    ("📨" . ":incoming_envelope:")
    ("📧" . ":e_mail: :email:")
    ("💌" . ":love_letter:")
    ("📥" . ":inbox_tray:")
    ("📤" . ":outbox_tray:")
    ("📦" . ":package:")
    ("🏷\357\270\217" . ":label:")
    ("🪧" . ":placard:")
    ("📪" . ":mailbox_closed:")
    ("📫" . ":mailbox:")
    ("📬" . ":mailbox_with_mail:")
    ("📭" . ":mailbox_with_no_mail:")
    ("📮" . ":postbox:")
    ("📯" . ":postal_horn:")
    ("📜" . ":scroll:")
    ("📃" . ":page_with_curl:")
    ("📄" . ":page_facing_up:")
    ("📑" . ":bookmark_tabs:")
    ("🧾" . ":receipt:")
    ("📊" . ":bar_chart:")
    ("📈" . ":chart_with_upwards_trend:")
    ("📉" . ":chart_with_downwards_trend:")
    ("🗒\357\270\217" . ":notepad_spiral: :spiral_note_pad:")
    ("🗓\357\270\217" . ":calendar_spiral: :spiral_calendar_pad:")
    ("📆" . ":calendar:")
    ("📅" . ":date:")
    ("🗑\357\270\217" . ":wastebasket:")
    ("📇" . ":card_index:")
    ("🗃\357\270\217" . ":card_box: :card_file_box:")
    ("🗳\357\270\217" . ":ballot_box_with_ballot:")
    ("🗄\357\270\217" . ":file_cabinet:")
    ("📋" . ":clipboard:")
    ("📁" . ":file_folder:")
    ("📂" . ":open_file_folder:")
    ("🗂\357\270\217" . ":card_index_dividers:")
    ("🗞\357\270\217" . ":newspaper2: :rolled_up_newspaper:")
    ("📰" . ":newspaper:")
    ("📓" . ":notebook:")
    ("📔" . ":notebook_with_decorative_cover:")
    ("📒" . ":ledger:")
    ("📕" . ":closed_book:")
    ("📗" . ":green_book:")
    ("📘" . ":blue_book:")
    ("📙" . ":orange_book:")
    ("📚" . ":books:")
    ("📖" . ":open_book:")
    ("🔖" . ":bookmark:")
    ("🧷" . ":safety_pin:")
    ("📎" . ":paperclip:")
    ("🖇\357\270\217" . ":linked_paperclips:")
    ("📐" . ":triangular_ruler:")
    ("📏" . ":straight_ruler:")
    ("🧮" . ":abacus:")
    ("📌" . ":pushpin:")
    ("📍" . ":round_pushpin:")
    ("\342\234\202\357\270\217" . ":scissors:")
    ("🖊\357\270\217" . ":pen_ballpoint: :lower_left_ballpoint_pen:")
    ("🖋\357\270\217" . ":pen_fountain: :lower_left_fountain_pen:")
    ("\342\234\222\357\270\217" . ":black_nib:")
    ("🖌\357\270\217" . ":lower_left_paintbrush:")
    ("🖍\357\270\217" . ":lower_left_crayon:")
    ("📝" . ":pencil: :memo:")
    ("\342\234\217\357\270\217" . ":pencil2:")
    ("🔍" . ":mag:")
    ("🔎" . ":mag_right:")
    ("🔏" . ":lock_with_ink_pen:")
    ("🔐" . ":closed_lock_with_key:")
    ("🔒" . ":locked:")
    ("🔓" . ":unlocked:")
    ("🩷" . ":pink_heart:")
    ("\342\235\244\357\270\217" . ":red_heart: <3 \342\231\241")
    ("🧡" . ":orange_heart:")
    ("💛" . ":yellow_heart:")
    ("💚" . ":green_heart:")
    ("🩵" . ":light_blue_heart:")
    ("💙" . ":blue_heart:")
    ("💜" . ":purple_heart:")
    ("🖤" . ":black_heart:")
    ("🩶" . ":grey_heart:")
    ("🤍" . ":white_heart:")
    ("🤎" . ":brown_heart:")
    ("💔" . ":broken_heart: </3 <\\3")
    ("\342\235\243\357\270\217" . ":heavy_heart_exclamation_mark_ornament:")
    ("💕" . ":two_hearts:")
    ("💞" . ":revolving_hearts:")
    ("💓" . ":heartbeat: :beating_heart:")
    ("💗" . ":heartpulse: :growing_heart:")
    ("💖" . ":sparkling_heart:")
    ("💘" . ":cupid:")
    ("💝" . ":gift_heart:")
    ("\342\235\244\357\270\217\342\200\215🩹" . ":mending_heart:")
    ("\342\235\244\357\270\217\342\200\215🔥" . ":heart_on_fire:")
    ("💟" . ":heart_decoration:")
    ("\342\230\256\357\270\217" . ":peace_symbol:")
    ("\342\234\235\357\270\217" . ":latin_cross:")
    ("\342\230\252\357\270\217" . ":star_and_crescent:")
    ("🕉\357\270\217" . ":om_symbol:")
    ("\342\230\270\357\270\217" . ":wheel_of_dharma:")
    ("🪯" . ":khanda:")
    ("\342\234\241\357\270\217" . ":star_of_david:")
    ("🔯" . ":six_pointed_star:")
    ("🕎" . ":menorah:")
    ("\342\230\257\357\270\217" . ":yin_yang:")
    ("\342\230\246\357\270\217" . ":orthodox_cross:")
    ("🛐" . ":place_of_worship: :worship_symbol:")
    ("\342\233\216" . ":ophiuchus:")
    ("\342\231\210" . ":aries:")
    ("\342\231\211" . ":taurus:")
    ("\342\231\212" . ":gemini:")
    ("\342\231\213" . ":cancer:")
    ("\342\231\214" . ":leo:")
    ("\342\231\215" . ":virgo:")
    ("\342\231\216" . ":libra:")
    ("\342\231\217" . ":scorpius: :scorpio:")
    ("\342\231\220" . ":sagittarius:")
    ("\342\231\221" . ":capricorn:")
    ("\342\231\222" . ":aquarius:")
    ("\342\231\223" . ":pisces:")
    ("🆔" . ":id:")
    ("\342\232\233\357\270\217" . ":atom_symbol:")
    ("🉑" . ":accept:")
    ("\342\230\242\357\270\217" . ":radioactive_sign:")
    ("\342\230\243\357\270\217" . ":biohazard_sign:")
    ("📴" . ":mobile_phone_off:")
    ("📳" . ":vibration_mode:")
    ("🈶" . ":u6709:")
    ("🈚" . ":u7121:")
    ("🈸" . ":u7533:")
    ("🈺" . ":u55b6:")
    ("🈷\357\270\217" . ":u6708:")
    ("\342\234\264\357\270\217" . ":eight_pointed_black_star:")
    ("🆚" . ":vs:")
    ("💮" . ":white_flower:")
    ("🉐" . ":ideograph_advantage:")
    ("\343\212\231\357\270\217" . ":secret:")
    ("\343\212\227\357\270\217" . ":congratulations:")
    ("🈴" . ":u5408:")
    ("🈵" . ":u6e80:")
    ("🈹" . ":u5272:")
    ("🈲" . ":u7981:")
    ("🅰\357\270\217" . ":a:")
    ("🅱\357\270\217" . ":b:")
    ("🆎" . ":ab:")
    ("🆑" . ":cl:")
    ("🅾\357\270\217" . ":o2:")
    ("🆘" . ":sos:")
    ("\342\235\214" . ":x: :cross_mark:")
    ("\342\255\225" . ":o:")
    ("🛑" . ":octagonal_sign: :stop_sign:")
    ("\342\233\224" . ":no_entry:")
    ("📛" . ":name_badge:")
    ("🚫" . ":no_entry_sign: :prohibited:")
    ("💯" . ":100:")
    ("💢" . ":anger:")
    ("\342\231\250\357\270\217" . ":hotsprings: :hot_springs:")
    ("🚷" . ":no_pedestrians:")
    ("🚯" . ":do_not_litter: :no_littering:")
    ("🚳" . ":no_bicycles:")
    ("🚱" . ":non_potable_water:")
    ("🔞" . ":underage:")
    ("📵" . ":no_mobile_phones:")
    ("🚭" . ":no_smoking:")
    ("\342\235\227" . ":exclamation:")
    ("\342\235\225" . ":grey_exclamation:")
    ("\342\235\223" . ":question_mark:")
    ("\342\235\224" . ":grey_question:")
    ("\342\200\274\357\270\217" . ":bangbang:")
    ("\342\201\211\357\270\217" . ":interrobang:")
    ("🔅" . ":low_brightness:")
    ("🔆" . ":high_brightness:")
    ("\343\200\275\357\270\217" . ":part_alternation_mark:")
    ("\342\232\240\357\270\217" . ":warning:")
    ("🚸" . ":children_crossing:")
    ("🔱" . ":trident:")
    ("\342\232\234\357\270\217" . ":fleur_de_lis:")
    ("🔰" . ":beginner:")
    ("\342\231\273\357\270\217" . ":recycle:")
    ("\342\234\205" . ":white_check_mark:")
    ("🈯" . ":u6307:")
    ("💹" . ":chart:")
    ("\342\235\207\357\270\217" . ":sparkle:")
    ("\342\234\263\357\270\217" . ":eight_spoked_asterisk:")
    ("\342\235\216" . ":negative_squared_cross_mark:")
    ("🌐" . ":globe_with_meridians:")
    ("💠" . ":diamond_shape_with_a_dot_inside:")
    ("\342\223\202\357\270\217" . ":circled_m:")
    ("🌀" . ":cyclone:")
    ("💤" . ":zzz:")
    ("🏧" . ":atm:")
    ("🚾" . ":wc: :water_closet:")
    ("\342\231\277" . ":wheelchair:")
    ("🅿\357\270\217" . ":parking:")
    ("🛗" . ":elevator:")
    ("🈳" . ":u7a7a:")
    ("🈂\357\270\217" . ":sa:")
    ("🛂" . ":passport_control:")
    ("🛃" . ":customs:")
    ("🛄" . ":baggage_claim:")
    ("🛅" . ":left_luggage:")
    ("🛜" . ":wireless:")
    ("🚹" . ":mens_room:")
    ("🚺" . ":womens_room:")
    ("🚼" . ":baby_symbol:")
    ("🚻" . ":restroom:")
    ("🚮" . ":put_litter_in_its_place:")
    ("🎦" . ":cinema:")
    ("📶" . ":signal_strength: :antenna_bars:")
    ("🈁" . ":koko:")
    ("🔣" . ":input_symbols:")
    ("\342\204\271\357\270\217" . ":information_source:")
    ("🔤" . ":abc:")
    ("🔡" . ":abcd:")
    ("🔠" . ":capital_abcd:")
    ("🆖" . ":ng:")
    ("🆗" . ":ok:")
    ("🆙" . ":up:")
    ("🆒" . ":cool:")
    ("🆕" . ":new:")
    ("🆓" . ":free:")
    ("0\357\270\217\342\203\243" . ":zero: :number_0:")
    ("1\357\270\217\342\203\243" . ":one: :number_1:")
    ("2\357\270\217\342\203\243" . ":two: :number_2:")
    ("3\357\270\217\342\203\243" . ":three: :number_3:")
    ("4\357\270\217\342\203\243" . ":four: :number_4:")
    ("5\357\270\217\342\203\243" . ":five: :number_5:")
    ("6\357\270\217\342\203\243" . ":six: :number_6:")
    ("7\357\270\217\342\203\243" . ":seven: :number_7:")
    ("8\357\270\217\342\203\243" . ":eight: :number_8:")
    ("9\357\270\217\342\203\243" . ":nine: :number_9:")
    ("🔟" . ":keycap_ten: :number_10:")
    ("🔢" . ":1234: :input_numbers:")
    ("#\357\270\217\342\203\243" . ":hash:")
    ("*\357\270\217\342\203\243" . ":keycap_asterisk:")
    ("\342\217\217\357\270\217" . ":eject_symbol:")
    ("\342\226\266\357\270\217" . ":arrow_forward:")
    ("\342\217\270\357\270\217" . ":pause_button: :double_vertical_bar:")
    ("\342\217\257\357\270\217" . ":play_pause:")
    ("\342\217\271\357\270\217" . ":stop_button:")
    ("\342\217\272\357\270\217" . ":record_button:")
    ("\342\217\255\357\270\217" . ":track_next: :next_track:")
    ("\342\217\256\357\270\217" . ":track_previous: :previous_track:")
    ("\342\217\251" . ":fast_forward:")
    ("\342\217\252" . ":rewind:")
    ("\342\217\253" . ":arrow_double_up:")
    ("\342\217\254" . ":arrow_double_down:")
    ("\342\227\200\357\270\217" . ":arrow_backward:")
    ("🔼" . ":arrow_up_small:")
    ("🔽" . ":arrow_down_small:")
    ("\342\236\241\357\270\217" . ":arrow_right: :right_arrow:")
    ("\342\254\205\357\270\217" . ":arrow_left: :left_arrow:")
    ("\342\254\206\357\270\217" . ":arrow_up: :up_arrow:")
    ("\342\254\207\357\270\217" . ":arrow_down: :down_arrow:")
    ("\342\206\227\357\270\217" . ":arrow_upper_right:")
    ("\342\206\230\357\270\217" . ":arrow_lower_right:")
    ("\342\206\231\357\270\217" . ":arrow_lower_left:")
    ("\342\206\226\357\270\217" . ":arrow_upper_left: :up_left_arrow:")
    ("\342\206\225\357\270\217" . ":arrow_up_down: :up_down_arrow:")
    ("\342\206\224\357\270\217" . ":left_right_arrow:")
    ("\342\206\252\357\270\217" . ":arrow_right_hook:")
    ("\342\206\251\357\270\217" . ":leftwards_arrow_with_hook:")
    ("\342\244\264\357\270\217" . ":arrow_heading_up:")
    ("\342\244\265\357\270\217" . ":arrow_heading_down:")
    ("🔀" . ":twisted_rightwards_arrows:")
    ("🔁" . ":repeat:")
    ("🔂" . ":repeat_one:")
    ("🔄" . ":arrows_counterclockwise:")
    ("🔃" . ":arrows_clockwise:")
    ("🎵" . ":musical_note:")
    ("🎶" . ":musical_notes:")
    ("\342\236\225" . ":heavy_plus_sign:")
    ("\342\236\226" . ":heavy_minus_sign:")
    ("\342\236\227" . ":heavy_division_sign:")
    ("\342\234\226\357\270\217" . ":heavy_multiplication_x:")
    ("🟰" . ":heavy_equals_sign:")
    ("\342\231\276\357\270\217" . ":infinity:")
    ("💲" . ":heavy_dollar_sign:")
    ("💱" . ":currency_exchange:")
    ("\342\204\242" . ":tm: :trade_mark:")
    ("\302\251" . ":copyright:")
    ("\302\256" . ":registered:")
    ("\343\200\260\357\270\217" . ":wavy_dash:")
    ("\342\236\260" . ":curly_loop:")
    ("\342\236\277" . ":loop:")
    ("🔚" . ":end_arrow:")
    ("🔙" . ":back_arrow:")
    ("🔛" . ":on_arrow:")
    ("🔝" . ":top_arrow:")
    ("🔜" . ":soon_arrow:")
    ("\342\234\224\357\270\217" . ":heavy_check_mark:")
    ("\342\230\221\357\270\217" . ":ballot_box_with_check:")
    ("🔘" . ":radio_button:")
    ("\342\232\252" . ":white_circle:")
    ("\342\232\253" . ":black_circle:")
    ("🔴" . ":red_circle:")
    ("🔵" . ":blue_circle:")
    ("🟤" . ":brown_circle:")
    ("🟣" . ":purple_circle:")
    ("🟢" . ":green_circle:")
    ("🟡" . ":yellow_circle:")
    ("🟠" . ":orange_circle:")
    ("🔺" . ":small_red_triangle:")
    ("🔻" . ":small_red_triangle_down:")
    ("🔸" . ":small_orange_diamond:")
    ("🔹" . ":small_blue_diamond:")
    ("🔶" . ":large_orange_diamond:")
    ("🔷" . ":large_blue_diamond:")
    ("🔳" . ":white_square_button:")
    ("🔲" . ":black_square_button:")
    ("\342\226\252\357\270\217" . ":black_small_square:")
    ("\342\226\253\357\270\217" . ":white_small_square:")
    ("\342\227\276" . ":black_medium_small_square:")
    ("\342\227\275" . ":white_medium_small_square:")
    ("\342\227\274\357\270\217" . ":black_medium_square:")
    ("\342\227\273\357\270\217" . ":white_medium_square:")
    ("\342\254\233" . ":black_large_square:")
    ("\342\254\234" . ":white_large_square:")
    ("🟧" . ":orange_square:")
    ("🟦" . ":blue_square:")
    ("🟥" . ":red_square:")
    ("🟫" . ":brown_square:")
    ("🟪" . ":purple_square:")
    ("🟩" . ":green_square:")
    ("🟨" . ":yellow_square:")
    ("🔈" . ":speaker:")
    ("🔇" . ":muted_speaker:")
    ("🔉" . ":sound:")
    ("🔊" . ":loud_sound:")
    ("🔔" . ":bell:")
    ("🔕" . ":no_bell:")
    ("📣" . ":megaphone:")
    ("📢" . ":loudspeaker:")
    ("🗨\357\270\217" . ":speech_left: :left_speech_bubble:")
    ("👁\342\200\215🗨" . ":eye_in_speech_bubble:")
    ("💬" . ":speech_balloon:")
    ("💭" . ":thought_balloon:")
    ("🗯\357\270\217" . ":anger_right: :right_anger_bubble:")
    ("\342\231\240\357\270\217" . ":spades: :spade_suit:")
    ("\342\231\243\357\270\217" . ":clubs: :club_suit:")
    ("\342\231\245\357\270\217" . ":hearts: :heart_suit:")
    ("\342\231\246\357\270\217" . ":diamonds: :diamond_suit:")
    ("🃏" . ":black_joker:")
    ("🎴" . ":flower_playing_cards:")
    ("🀄" . ":mahjong:")
    ("🕐" . ":clock1: :one_oclock:")
    ("🕑" . ":clock2: :two_oclock:")
    ("🕒" . ":clock3: :three_oclock:")
    ("🕓" . ":clock4: :four_oclock:")
    ("🕔" . ":clock5: :five_oclock:")
    ("🕕" . ":clock6: :six_oclock:")
    ("🕖" . ":clock7: :seven_oclock:")
    ("🕗" . ":clock8: :eight_oclock:")
    ("🕘" . ":clock9: :nine_oclock:")
    ("🕙" . ":clock10: :ten_oclock:")
    ("🕚" . ":clock11: :eleven_oclock:")
    ("🕛" . ":clock12: :twelve_oclock:")
    ("🕜" . ":clock130: :one_thirty:")
    ("🕝" . ":clock230: :two_thirty:")
    ("🕞" . ":clock330: :three_thirty:")
    ("🕟" . ":clock430: :four_thirty:")
    ("🕠" . ":clock530: :five_thirty:")
    ("🕡" . ":clock630: :six_thirty:")
    ("🕢" . ":clock730: :seven_thirty:")
    ("🕣" . ":clock830: :eight_thirty:")
    ("🕤" . ":clock930: :nine_thirty:")
    ("🕥" . ":clock1030: :ten_thirty:")
    ("🕦" . ":clock1130: :eleven_thirty:")
    ("🕧" . ":clock1230: :twelve_thirty:")
    ("\342\231\200\357\270\217" . ":female_sign:")
    ("\342\231\202\357\270\217" . ":male_sign:")
    ("\342\232\247" . ":transgender_symbol:")
    ("\342\232\225\357\270\217" . ":medical_symbol:")
    ("🇿" . ":regional_indicator_z:")
    ("🇾" . ":regional_indicator_y:")
    ("🇽" . ":regional_indicator_x:")
    ("🇼" . ":regional_indicator_w:")
    ("🇻" . ":regional_indicator_v:")
    ("🇺" . ":regional_indicator_u:")
    ("🇹" . ":regional_indicator_t:")
    ("🇸" . ":regional_indicator_s:")
    ("🇷" . ":regional_indicator_r:")
    ("🇶" . ":regional_indicator_q:")
    ("🇵" . ":regional_indicator_p:")
    ("🇴" . ":regional_indicator_o:")
    ("🇳" . ":regional_indicator_n:")
    ("🇲" . ":regional_indicator_m:")
    ("🇱" . ":regional_indicator_l:")
    ("🇰" . ":regional_indicator_k:")
    ("🇯" . ":regional_indicator_j:")
    ("🇮" . ":regional_indicator_i:")
    ("🇭" . ":regional_indicator_h:")
    ("🇬" . ":regional_indicator_g:")
    ("🇫" . ":regional_indicator_f:")
    ("🇪" . ":regional_indicator_e:")
    ("🇩" . ":regional_indicator_d:")
    ("🇨" . ":regional_indicator_c:")
    ("🇧" . ":regional_indicator_b:")
    ("🇦" . ":regional_indicator_a:")
    ("🏳\357\270\217" . ":flag_white:")
    ("🏴" . ":flag_black:")
    ("🏴\342\200\215\342\230\240\357\270\217" . ":pirate_flag:")
    ("🏁" . ":checkered_flag:")
    ("🚩" . ":triangular_flag_on_post:")
    ("🏳\357\270\217\342\200\215🌈" . ":rainbow_flag: :gay_pride_flag:")
    ("🏳\357\270\217\342\200\215\342\232\247\357\270\217" . ":transgender_flag:")
    ("🇺🇳" . ":united_nations:")
    ("🇦🇫" . ":flag_af:")
    ("🇦🇽" . ":flag_ax:")
    ("🇦🇱" . ":flag_al:")
    ("🇩🇿" . ":flag_dz:")
    ("🇦🇸" . ":flag_as:")
    ("🇦🇩" . ":flag_ad:")
    ("🇦🇴" . ":flag_ao:")
    ("🇦🇮" . ":flag_ai:")
    ("🇦🇶" . ":flag_aq:")
    ("🇦🇬" . ":flag_ag:")
    ("🇦🇷" . ":flag_ar:")
    ("🇦🇲" . ":flag_am:")
    ("🇦🇼" . ":flag_aw:")
    ("🇦🇺" . ":flag_au:")
    ("🇦🇹" . ":flag_at:")
    ("🇦🇿" . ":flag_az:")
    ("🇧🇸" . ":flag_bs:")
    ("🇧🇭" . ":flag_bh:")
    ("🇧🇩" . ":flag_bd:")
    ("🇧🇧" . ":flag_bb:")
    ("🇧🇾" . ":flag_by:")
    ("🇧🇪" . ":flag_be:")
    ("🇧🇿" . ":flag_bz:")
    ("🇧🇯" . ":flag_bj:")
    ("🇧🇲" . ":flag_bm:")
    ("🇧🇹" . ":flag_bt:")
    ("🇧🇴" . ":flag_bo:")
    ("🇧🇦" . ":flag_ba:")
    ("🇧🇼" . ":flag_bw:")
    ("🇧🇷" . ":flag_br:")
    ("🇮🇴" . ":flag_io:")
    ("🇻🇬" . ":flag_vg:")
    ("🇧🇳" . ":flag_bn:")
    ("🇧🇬" . ":flag_bg:")
    ("🇧🇫" . ":flag_bf:")
    ("🇧🇮" . ":flag_bi:")
    ("🇰🇭" . ":flag_kh:")
    ("🇨🇲" . ":flag_cm:")
    ("🇨🇦" . ":flag_ca:")
    ("🇮🇨" . ":flag_ic:")
    ("🇨🇻" . ":flag_cv:")
    ("🇧🇶" . ":flag_bq:")
    ("🇰🇾" . ":flag_ky:")
    ("🇨🇫" . ":flag_cf:")
    ("🇹🇩" . ":flag_td:")
    ("🇨🇱" . ":flag_cl:")
    ("🇨🇳" . ":flag_cn:")
    ("🇨🇽" . ":flag_cx:")
    ("🇨🇨" . ":flag_cc:")
    ("🇨🇴" . ":flag_co:")
    ("🇰🇲" . ":flag_km:")
    ("🇨🇬" . ":flag_cg:")
    ("🇨🇩" . ":flag_cd:")
    ("🇨🇰" . ":flag_ck:")
    ("🇨🇷" . ":flag_cr:")
    ("🇨🇮" . ":flag_ci:")
    ("🇭🇷" . ":flag_hr:")
    ("🇨🇺" . ":flag_cu:")
    ("🇨🇼" . ":flag_cw:")
    ("🇨🇾" . ":flag_cy:")
    ("🇨🇿" . ":flag_cz:")
    ("🇩🇰" . ":flag_dk:")
    ("🇩🇯" . ":flag_dj:")
    ("🇩🇲" . ":flag_dm:")
    ("🇩🇴" . ":flag_do:")
    ("🇪🇨" . ":flag_ec:")
    ("🇪🇬" . ":flag_eg:")
    ("🇸🇻" . ":flag_sv:")
    ("🇬🇶" . ":flag_gq:")
    ("🇪🇷" . ":flag_er:")
    ("🇪🇪" . ":flag_ee:")
    ("🇪🇹" . ":flag_et:")
    ("🇪🇺" . ":flag_eu:")
    ("🇫🇰" . ":flag_fk:")
    ("🇫🇴" . ":flag_fo:")
    ("🇫🇯" . ":flag_fj:")
    ("🇫🇮" . ":flag_fi:")
    ("🇫🇷" . ":flag_fr:")
    ("🇬🇫" . ":flag_gf:")
    ("🇵🇫" . ":flag_pf:")
    ("🇹🇫" . ":flag_tf:")
    ("🇬🇦" . ":flag_ga:")
    ("🇬🇲" . ":flag_gm:")
    ("🇬🇪" . ":flag_ge:")
    ("🇩🇪" . ":flag_de:")
    ("🇬🇭" . ":flag_gh:")
    ("🇬🇮" . ":flag_gi:")
    ("🇬🇷" . ":flag_gr:")
    ("🇬🇱" . ":flag_gl:")
    ("🇬🇩" . ":flag_gd:")
    ("🇬🇵" . ":flag_gp:")
    ("🇬🇺" . ":flag_gu:")
    ("🇬🇹" . ":flag_gt:")
    ("🇬🇬" . ":flag_gg:")
    ("🇬🇳" . ":flag_gn:")
    ("🇬🇼" . ":flag_gw:")
    ("🇬🇾" . ":flag_gy:")
    ("🇭🇹" . ":flag_ht:")
    ("🇭🇳" . ":flag_hn:")
    ("🇭🇰" . ":flag_hk:")
    ("🇭🇺" . ":flag_hu:")
    ("🇮🇸" . ":flag_is:")
    ("🇮🇳" . ":flag_in:")
    ("🇮🇩" . ":flag_id:")
    ("🇮🇷" . ":flag_ir:")
    ("🇮🇶" . ":flag_iq:")
    ("🇮🇪" . ":flag_ie:")
    ("🇮🇲" . ":flag_im:")
    ("🇮🇱" . ":flag_il:")
    ("🇮🇹" . ":flag_it:")
    ("🇯🇲" . ":flag_jm:")
    ("🇯🇵" . ":flag_jp:")
    ("🎌" . ":crossed_flags:")
    ("🇯🇪" . ":flag_je:")
    ("🇯🇴" . ":flag_jo:")
    ("🇰🇿" . ":flag_kz:")
    ("🇰🇪" . ":flag_ke:")
    ("🇰🇮" . ":flag_ki:")
    ("🇽🇰" . ":flag_xk:")
    ("🇰🇼" . ":flag_kw:")
    ("🇰🇬" . ":flag_kg:")
    ("🇱🇦" . ":flag_la:")
    ("🇱🇻" . ":flag_lv:")
    ("🇱🇧" . ":flag_lb:")
    ("🇱🇸" . ":flag_ls:")
    ("🇱🇷" . ":flag_lr:")
    ("🇱🇾" . ":flag_ly:")
    ("🇱🇮" . ":flag_li:")
    ("🇱🇹" . ":flag_lt:")
    ("🇱🇺" . ":flag_lu:")
    ("🇲🇴" . ":flag_mo:")
    ("🇲🇰" . ":flag_mk:")
    ("🇲🇬" . ":flag_mg:")
    ("🇲🇼" . ":flag_mw:")
    ("🇲🇾" . ":flag_my:")
    ("🇲🇻" . ":flag_mv:")
    ("🇲🇱" . ":flag_ml:")
    ("🇲🇹" . ":flag_mt:")
    ("🇲🇭" . ":flag_mh:")
    ("🇲🇶" . ":flag_mq:")
    ("🇲🇷" . ":flag_mr:")
    ("🇲🇺" . ":flag_mu:")
    ("🇾🇹" . ":flag_yt:")
    ("🇲🇽" . ":flag_mx:")
    ("🇫🇲" . ":flag_fm:")
    ("🇲🇩" . ":flag_md:")
    ("🇲🇨" . ":flag_mc:")
    ("🇲🇳" . ":flag_mn:")
    ("🇲🇪" . ":flag_me:")
    ("🇲🇸" . ":flag_ms:")
    ("🇲🇦" . ":flag_ma:")
    ("🇲🇿" . ":flag_mz:")
    ("🇲🇲" . ":flag_mm:")
    ("🇳🇦" . ":flag_na:")
    ("🇳🇷" . ":flag_nr:")
    ("🇳🇵" . ":flag_np:")
    ("🇳🇱" . ":flag_nl:")
    ("🇳🇨" . ":flag_nc:")
    ("🇳🇿" . ":flag_nz:")
    ("🇳🇮" . ":flag_ni:")
    ("🇳🇪" . ":flag_ne:")
    ("🇳🇬" . ":flag_ng:")
    ("🇳🇺" . ":flag_nu:")
    ("🇳🇫" . ":flag_nf:")
    ("🇰🇵" . ":flag_kp:")
    ("🇲🇵" . ":flag_mp:")
    ("🇳🇴" . ":flag_no:")
    ("🇴🇲" . ":flag_om:")
    ("🇵🇰" . ":flag_pk:")
    ("🇵🇼" . ":flag_pw:")
    ("🇵🇸" . ":flag_ps:")
    ("🇵🇦" . ":flag_pa:")
    ("🇵🇬" . ":flag_pg:")
    ("🇵🇾" . ":flag_py:")
    ("🇵🇪" . ":flag_pe:")
    ("🇵🇭" . ":flag_ph:")
    ("🇵🇳" . ":flag_pn:")
    ("🇵🇱" . ":flag_pl:")
    ("🇵🇹" . ":flag_pt:")
    ("🇵🇷" . ":flag_pr:")
    ("🇶🇦" . ":flag_qa:")
    ("🇷🇪" . ":flag_re:")
    ("🇷🇴" . ":flag_ro:")
    ("🇷🇺" . ":flag_ru:")
    ("🇷🇼" . ":flag_rw:")
    ("🇼🇸" . ":flag_ws:")
    ("🇸🇲" . ":flag_sm:")
    ("🇸🇹" . ":flag_st:")
    ("🇸🇦" . ":flag_sa:")
    ("🇸🇳" . ":flag_sn:")
    ("🇷🇸" . ":flag_rs:")
    ("🇸🇨" . ":flag_sc:")
    ("🇸🇱" . ":flag_sl:")
    ("🇸🇬" . ":flag_sg:")
    ("🇸🇽" . ":flag_sx:")
    ("🇸🇰" . ":flag_sk:")
    ("🇸🇮" . ":flag_si:")
    ("🇬🇸" . ":flag_gs:")
    ("🇸🇧" . ":flag_sb:")
    ("🇸🇴" . ":flag_so:")
    ("🇿🇦" . ":flag_za:")
    ("🇰🇷" . ":flag_kr:")
    ("🇸🇸" . ":flag_ss:")
    ("🇪🇸" . ":flag_es:")
    ("🇱🇰" . ":flag_lk:")
    ("🇧🇱" . ":flag_bl:")
    ("🇸🇭" . ":flag_sh:")
    ("🇰🇳" . ":flag_kn:")
    ("🇱🇨" . ":flag_lc:")
    ("🇵🇲" . ":flag_pm:")
    ("🇻🇨" . ":flag_vc:")
    ("🇸🇩" . ":flag_sd:")
    ("🇸🇷" . ":flag_sr:")
    ("🇸🇿" . ":flag_sz:")
    ("🇸🇪" . ":flag_se:")
    ("🇨🇭" . ":flag_ch:")
    ("🇸🇾" . ":flag_sy:")
    ("🇹🇼" . ":flag_tw:")
    ("🇹🇯" . ":flag_tj:")
    ("🇹🇿" . ":flag_tz:")
    ("🇹🇭" . ":flag_th:")
    ("🇹🇱" . ":flag_tl:")
    ("🇹🇬" . ":flag_tg:")
    ("🇹🇰" . ":flag_tk:")
    ("🇹🇴" . ":flag_to:")
    ("🇹🇹" . ":flag_tt:")
    ("🇹🇳" . ":flag_tn:")
    ("🇹🇷" . ":flag_tr:")
    ("🇹🇲" . ":flag_tm:")
    ("🇹🇨" . ":flag_tc:")
    ("🇻🇮" . ":flag_vi:")
    ("🇹🇻" . ":flag_tv:")
    ("🇺🇬" . ":flag_ug:")
    ("🇺🇦" . ":flag_ua:")
    ("🇦🇪" . ":flag_ae:")
    ("🇬🇧" . ":flag_gb:")
    ("🏴󠁧󠁢󠁥󠁮󠁧󠁿" . ":england:")
    ("🏴󠁧󠁢󠁳󠁣󠁴󠁿" . ":scotland:")
    ("🏴󠁧󠁢󠁷󠁬󠁳󠁿" . ":wales:")
    ("🇺🇸" . ":flag_us:")
    ("🇺🇾" . ":flag_uy:")
    ("🇺🇿" . ":flag_uz:")
    ("🇻🇺" . ":flag_vu:")
    ("🇻🇦" . ":flag_va:")
    ("🇻🇪" . ":flag_ve:")
    ("🇻🇳" . ":flag_vn:")
    ("🇼🇫" . ":flag_wf:")
    ("🇪🇭" . ":flag_eh:")
    ("🇾🇪" . ":flag_ye:")
    ("🇿🇲" . ":flag_zm:")
    ("🇿🇼" . ":flag_zw:")
    ("🇦🇨" . ":flag_ac:")
    ("🇧🇻" . ":flag_bv:")
    ("🇨🇵" . ":flag_cp:")
    ("🇪🇦" . ":flag_ea:")
    ("🇩🇬" . ":flag_dg:")
    ("🇭🇲" . ":flag_hm:")
    ("🇲🇫" . ":flag_mf:")
    ("🇸🇯" . ":flag_sj:")
    ("🇹🇦" . ":flag_ta:")
    ("🇺🇲" . ":flag_um:")))

(provide 'helm-emoji-data)

;; Local Variables:
;; indent-tabs-mode: nil
;; package-lint-main-file: "helm-emoji.el"
;; End:

;;; helm-emoji-data.el ends here
