;;; helm-emoji.el --- Select emojis with Helm  -*- lexical-binding: t -*-

;; Copyright (C) 2026 Timm Lichte

;; Author: Timm Lichte <timm.lichte@uni-tuebingen.de>
;; URL: https://codeberg.org/timmli/helm-emoji
;; Package-Version: 20260218.1141
;; Package-Revision: 4c125d3fc42f
;; Last modified: 2026-02-11 Wed 11:59:18
;; Package-Requires: ((emacs "29.1") (helm "3.9.6"))
;; Keywords: matching

;; This program is free software: you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation, either version 3 of the License, or
;; (at your option) any later version.

;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with this program.  If not, see <https://www.gnu.org/licenses/>.

;;; Commentary:

;; Select and insert emojis inside Emacs using Helm's completion mechanism.

;;; Code:

(require 'emoji)
(require 'helm)
(require 'helm-buffers)
(require 'helm-emoji-data)

(defvar helm-emoji-candidate-format
  "%s   %s   %s %s"
  "Visual format of a candidate.")

(defface helm-emoji-description-face
  '((t :inherit font-lock-comment-face))
  "Font face for the descrption part.")

(defvar helm-emoji--emoji-properties-alist nil
  "Alist of string-plist pairs representing emojis and their properties.

If nil, it will be populated when executing `helm-emoji--make-candidates'.")

(defvar helm-emoji--all-candidates nil
  "List of string-plist pairs representing the complete list of candidates.")

(defvar helm-emoji--abbreviation-alist (helm-emoji-data--return)
  "Alist of emoji symbols and their abbreviation.")

(defun helm-emoji--make-candidates (mode)
  "Make alist of candidates depending on MODE."
  (unless helm-emoji--emoji-properties-alist
    (message "helm-emoji: creating emoji database ...")
    (setq helm-emoji--all-candidates nil)
    (setq helm-emoji--emoji-properties-alist
          (let ((emoji-category-alist))
            ;; Process emoji--labels
            (dolist (category emoji--labels)
              (let ((label (car category)))
                (if (listp (cadr category))  ; Check if there is a subcategory
                    (dolist (subcategory (cdr category))
                      (dolist (emoji (cdr subcategory))
                        (push (cons emoji label) emoji-category-alist)))
                  (dolist (emoji (cdr category))  ; Handle categories without subcategories
                    (push (cons emoji label) emoji-category-alist)))))
            (cl-loop
             for key being the hash-keys of emoji--names
             for symbol = key
             for description = (propertize
                                (gethash key emoji--names)
                                'face 'helm-emoji-description-face)
             for abbreviation = (cdr (assoc symbol helm-emoji--abbreviation-alist))
             for category = (propertize
                             (concat "[" (cdr (assoc symbol emoji-category-alist)) "]")
                             'face 'helm-emoji-description-face)
             for candidate-format = (format helm-emoji-candidate-format
                                            symbol
                                            (or abbreviation "")
                                            category
                                            description)
             for candidate-plist = `(:symbol ,symbol
                                             :category ,category
                                             :description ,description
                                             :format ,candidate-format)
             if (and (stringp symbol)
                     (= (string-width symbol) 2))
             do (push `(,candidate-format . ,(list candidate-plist))
                      helm-emoji--all-candidates)
             collect `(,symbol . ,(list candidate-plist)))))
    (setq helm-emoji--all-candidates (reverse helm-emoji--all-candidates)))
  (if (and (eq mode 'recent)
           (multisession-value emoji--recent))
      (cl-loop
       for symbol in (multisession-value emoji--recent)
       for candidate-plist = (car (cdr (assoc symbol helm-emoji--emoji-properties-alist)))
       for candidate-format = (plist-get candidate-plist :format)
       collect `(,candidate-format . ,(list candidate-plist)))
    helm-emoji--all-candidates))

(defvar helm-emoji--actions
  (helm-make-actions
   "Insert emoji" #'helm-emoji-insert-action
   "Copy emoji to clipboard" #'helm-emoji-copy-to-clipboard-action)
  "List of pairs (STRING FUNCTIONSYMBOL).
They represent the actions used in `helm-emoji'.")

(defun helm-emoji-insert-action (candidate)
  "Insert selected emoji of CANDIDATE."
  (let ((symbol (plist-get (car candidate) :symbol)))
    (emoji--add-recent symbol)
    (insert symbol)))

(defun helm-emoji-copy-to-clipboard-action (candidate)
  "Copy selected emoji of CANDIDATE to clipboard."
  (let ((symbol (plist-get (car candidate) :symbol)))
    (emoji--add-recent symbol)
    (kill-new symbol)))

;;;###autoload
(defun helm-emoji (&optional input)
  "Access emojis through Helm and optionally keep the INPUT."
  (interactive)
  (emoji--init)
  (helm :sources (list
                  (helm-build-sync-source "Recently used emojis"
                    :candidates (lambda () (helm-emoji--make-candidates 'recent))
                    :display-to-real nil ; Transform the selected candidate when passing it to action.
                    :action helm-emoji--actions)
                  (helm-build-sync-source "Complete list of emojis"
                    :candidates (lambda () (helm-emoji--make-candidates 'all))
                    :display-to-real nil ; Transform the selected candidate when passing it to action.
                    :action helm-emoji--actions))
        :buffer "*helm-emoji*"
        :update (lambda () (setq helm-emoji--emoji-properties-alist nil))
        :truncate-lines helm-buffers-truncate-lines
        :input (or input "")))

(provide 'helm-emoji)

;; Local Variables:
;; indent-tabs-mode: nil
;; End:

;;; helm-emoji.el ends here
