;;; kaomel-utils.el --- Development utilities for kaomel -*- lexical-binding: t; -*-
;;
;; Author: Giovanni Crisalfi <giovanni.crisalfi@protonmail.com>
;; Maintainer: Giovanni Crisalfi <giovanni.crisalfi@protonmail.com>
;; Keywords: tools, development
;; Homepage: https://github.com/gicrisf/kaomel
;;
;; This file is not part of GNU Emacs.
;;
;; This program is free software: you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation, either version 3 of the License, or
;; (at your option) any later version.
;;
;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.
;;
;; You should have received a copy of the GNU General Public License
;; along with this program.  If not, see <https://www.gnu.org/licenses/>.
;;
;;; Commentary:
;;
;; This file contains development utilities for the Kaomel package.
;; It provides functions to regenerate kaomel-data.el from kaomel-data.json.
;; 
;; This is a development-only tool and is not shipped with the MELPA release.
;;
;; Usage:
;;   Called programmatically to regenerate kaomel-data.el from JSON
;;
;; Functions:
;;   kaomel-dev-retrieve-kaomojis-from-json - Load JSON data from file
;;
;;; Code:

(require 'json)

(defun kaomel-dev-retrieve-kaomojis-from-json (path)
  "Retrieve kaomojis from JSON file at PATH.
Returns a vector of hash tables representing the kaomoji data."
  (with-temp-buffer
    (set-buffer-multibyte t)
    (insert-file-contents path)
    (goto-char (point-min))
    (let ((json-object-type 'hash-table))
      (json-read))))

(defun kaomel-dev--generate-data-file (json-path output-path)
  "Generate kaomel-data.el from JSON-PATH, writing to OUTPUT-PATH."
  (let* ((kaomoji-data (kaomel-dev-retrieve-kaomojis-from-json json-path))
         (current-time (current-time-string)))
    
    (with-temp-buffer
      (insert ";;; kaomel-data.el --- Vectorized kaomoji data -*- lexical-binding: t; -*-\n")
      (insert ";;\n")
      (insert ";; (ɔ) Copyleft 2023 Giovanni Crisalfi (gicrisf)\n")
      (insert ";;\n")
      (insert ";; Author: Giovanni Crisalfi <giovanni.crisalfi@protonmail.com>\n")
      (insert ";; Maintainer: Giovanni Crisalfi <giovanni.crisalfi@protonmail.com>\n")
      (insert (format ";; Generated: %s\n" current-time))
      (insert ";; Keywords: convenience extensions faces tools\n")
      (insert ";; Homepage: https://github.com/gicrisf/kaomel\n")
      (insert ";;\n")
      (insert ";; This file is not part of GNU Emacs.\n")
      (insert ";;\n")
      (insert ";; This program is free software: you can redistribute it and/or modify\n")
      (insert ";; it under the terms of the GNU General Public License as published by\n")
      (insert ";; the Free Software Foundation, either version 3 of the License, or\n")
      (insert ";; (at your option) any later version.\n")
      (insert ";;\n")
      (insert ";; This program is distributed in the hope that it will be useful,\n")
      (insert ";; but WITHOUT ANY WARRANTY; without even the implied warranty of\n")
      (insert ";; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n")
      (insert ";; GNU General Public License for more details.\n")
      (insert ";;\n")
      (insert ";; You should have received a copy of the GNU General Public License\n")
      (insert ";; along with this program.  If not, see <https://www.gnu.org/licenses/>.\n")
      (insert ";;\n")
      (insert ";;; Commentary:\n")
      (insert ";;\n")
      (insert ";; This file contains the vectorized kaomoji data for the Kaomel package.\n")
      (insert ";; The data is stored as a Lisp vector for optimal performance.\n")
      (insert ";;\n")
      (insert ";; This file is automatically generated from kaomel-data.json.\n")
      (insert ";; Do not edit manually - use kaomel-dev-regenerate-data instead.\n")
      (insert ";;\n")
      (insert ";;; Code:\n\n")
      
      ;; Insert the data variable
      (insert "(defvar kaomel-data\n")
      (prin1 kaomoji-data (current-buffer))
      (insert "\n  \"Vector containing kaomoji data with multilingual tags.\")\n\n")
      
      (insert "(provide 'kaomel-data)\n")
      (insert ";;; kaomel-data.el ends here\n")
      
      ;; Write to file
      (write-file output-path))
    
    (message "Generated kaomel-data.el with %d kaomoji entries" (length kaomoji-data))))

(provide 'kaomel-utils)
;;; kaomel-utils.el ends here
