;;; magit-prime.el --- Prime cache before Magit refresh -*- lexical-binding: t; -*-

;; Author: Romain Ouabdelkader <romain.ouabdelkader@gmail.com>
;; URL: https://github.com/Azkae/magit-prime
;; Package-Version: 20250803.1914
;; Package-Revision: ebd58f95a564
;; Package-Requires: ((emacs "27.1") (magit "3.0.0"))

;; SPDX-License-Identifier: MIT

;;; Commentary:

;; A Magit extension that primes caches in parallel before refresh operations,
;; reducing Magit buffer refresh times.

;; Usage:
;; (add-hook 'magit-pre-refresh-hook 'magit-prime-refresh-cache)

;;; Code:

(require 'magit)
(require 'tramp-sh)

(defconst magit-prime--commands-phase-one
  '(("symbolic-ref" "--short" "HEAD")
    (t "describe" "--long" "--tags")
    ("describe" "--contains" "HEAD")
    ("rev-parse" "--git-dir")
    ("rev-parse" "--is-bare-repository")
    ("rev-parse" "--short" "HEAD")
    ("rev-parse" "--short" "HEAD~")
    ("rev-parse" "--verify" "--abbrev-ref" "master@{upstream}")
    ("rev-parse" "--verify" "--abbrev-ref" "main@{upstream}")
    (t "describe" "--contains" "HEAD")
    (t "rev-parse" "--verify" "HEAD")
    (t "rev-parse" "--verify" "refs/stash")
    (t "rev-parse" "--verify" "HEAD~10"))
  "Returns a list of git commands that will be used to prime magit's cache.

Commands prefixed with t are cached even on failure.")

(defun magit-prime--commands-phase-two ()
  "Return a list of git commands that will be used to prime magit's cache.
The commands depend on the current branch, remotes,
and upstream configurations fetched from `magit-prime--commands-phase-one''.

Commands prefixed with t are cached even on failure."
  (let* ((branch (magit-get-current-branch))
         (main (magit-main-branch))
         (push-main (magit-get-push-branch main))
         (push-branch (magit-get-push-branch branch))
         (upstream-main (magit-get-upstream-branch main))
         (push-remote (magit-get-push-remote branch))
         (primary-remote (magit-primary-remote)))
    (cl-remove-duplicates
     `(("rev-parse" "--verify" "--abbrev-ref" ,(concat main "@{upstream}"))
       ("rev-parse" "--verify" "--abbrev-ref" ,(concat branch "@{upstream}"))
       ("rev-parse" "--verify" "--abbrev-ref" ,(concat push-branch "^{commit}"))
       ("log" "--no-walk" "--format=%s" ,(concat upstream-main"^{commit}") "--")
       ("log" "--no-walk" "--format=%s" ,(concat push-main "^{commit}") "--")
       ("log" "--no-walk" "--format=%s" ,(concat push-branch "^{commit}") "--")
       ("log" "--no-walk" "--format=%h %s" "HEAD^{commit}" "--")
       (t "symbolic-ref" ,(format "refs/remotes/%s/HEAD" primary-remote))
       (t "symbolic-ref" ,(format "refs/remotes/%s/HEAD" push-remote))
       (t "rev-parse" "--verify" ,(concat "refs/tags/" branch))
       (t "rev-parse" "--verify" ,(concat "refs/tags/" main))
       (t "rev-parse" "--verify" ,(concat "refs/tags/" push-branch))
       (t "rev-parse" "--verify" ,(concat "refs/tags/" push-main))
       (t "rev-parse" "--verify" ,(format "refs/tags/%s/HEAD" primary-remote))
       (t "rev-parse" "--verify" ,push-branch))
     :test #'equal)))

(defun magit-prime-refresh-cache ()
  "Prime the refresh cache if possible."
  (when (and (or magit-refresh-status-buffer
                 (derived-mode-p 'magit-status-mode))
             magit--refresh-cache)
    (let ((elapsed
           (benchmark-elapse
             (magit-prime--refresh-cache magit-prime--commands-phase-one)
             (magit-prime--refresh-cache (magit-prime--commands-phase-two)))))
      (when magit-refresh-verbose
        (message "Refresh cached primed in %.3fs" elapsed)))))

(defun magit-prime--refresh-cache (commands)
  "Execute git COMMANDS to prime Magit's refresh cache."
  (if (file-remote-p default-directory)
      (magit-prime--refresh-cache-remote commands)
    (magit-prime--refresh-cache-local commands)))

(defun magit-prime--refresh-cache-local (commands)
  "Prime the refresh cache with the provided COMMANDS."
  (let* ((repo-path (magit-toplevel))
         (running 0)
         (buffers
          (mapcar
           (lambda (command)
             (let* ((buffer (generate-new-buffer " *magit-prime-refresh-cache*"))
                    (cachep (and (eq (car command) t) (pop command)))
                    (process-environment (magit-process-environment))
                    (default-process-coding-system (magit--process-coding-system)))
               (make-process
                :name (buffer-name buffer)
                :buffer buffer
                :noquery t
                :connection-type 'pipe
                :command (cons magit-git-executable
                               (magit-process-git-arguments command))
                :sentinel
                (lambda (proc _event)
                  (when (eq (process-status proc) 'exit)
                    (when-let* ((buf (process-buffer proc))
                                ((buffer-live-p buf))
                                ((or cachep
                                     (zerop (process-exit-status proc)))))
                      (push (cons (cons repo-path command)
                                  (with-current-buffer buf
                                    (and (zerop (process-exit-status proc))
                                         (not (bobp))
                                         (progn
                                           (goto-char (point-min))
                                           (buffer-substring-no-properties
                                            (point) (line-end-position))))))
                            (cdr magit--refresh-cache)))
                    (cl-decf running))))
               (cl-incf running)
               buffer))
           commands)))

    (with-timeout (1)
      (while (> running 0)
        (sit-for 0.01)
        (accept-process-output)))

    (mapc #'kill-buffer buffers)))

(defconst magit-prime--batch-commands-script "
COMMANDS=\"$1\"
REPO_DIR=\"$2\"

run_command() {
    local id=\"$1\"
    local command=\"$2\"

    local output
    local exit_code

    output=$(cd \"$REPO_DIR\" && eval \"$command\" 2>/dev/null)
    exit_code=$?

    output=$(echo \"$output\" | head -n1)

    echo \"($id $exit_code \\\"$output\\\")\"
}

echo \\(

echo \"$COMMANDS\" | {
    while IFS=':' read -r id command; do
        run_command \"$id\" \"$command\" &
    done

    wait
}

echo \\)
")

(defun magit-prime--refresh-cache-remote (commands)
  "Prime the refresh cache with the provided COMMANDS using tramp."
  (let ((vec (tramp-dissect-file-name default-directory))
        (str-commands (magit-prime--format-commands-for-bash commands))
        (repo-dir (file-remote-p default-directory 'localname))
        (repo-path (magit-toplevel)))
    (tramp-maybe-send-script vec magit-prime--batch-commands-script
                             "magit_prime__batch_commands")
    (let ((results
           (tramp-send-command-and-read
            vec (format "magit_prime__batch_commands '%s' '%s'" str-commands repo-dir))))
      (dolist (item results)
        (let* ((command (nth 0 item))
               (cachep (and (eq (car command) t) (pop command)))
               (status-code (nth 1 item))
               (output (nth 2 item)))
          (when (or cachep (zerop status-code))
            (push (cons (cons repo-path command) (and (zerop status-code) output))
                  (cdr magit--refresh-cache))))))))

(defun magit-prime--format-commands-for-bash (commands)
  "Convert COMMANDS list to bash script input format.
Each command becomes `LISP-FORM:git ARGS'
where LISP-FORM is the original command."
  (mapconcat
   (lambda (command)
     (let* ((original-command command)
            (_ (and (eq (car command) t) (pop command)))
            (clean-command (mapcar #'substring-no-properties command))
            (git-command (mapconcat #'shell-quote-argument clean-command " "))
            (line (format "%S:%s %s" original-command magit-remote-git-executable git-command)))
       line))
   commands
   "\n"))

;;;###autoload
(define-minor-mode magit-prime-mode
  "Global minor mode to enable magit-prime cache priming.
When enabled, automatically primes caches before Magit refresh operations."
  :global t
  :group 'magit-prime
  (if magit-prime-mode
      (add-hook 'magit-pre-refresh-hook #'magit-prime-refresh-cache)
    (remove-hook 'magit-pre-refresh-hook #'magit-prime-refresh-cache)))

(provide 'magit-prime)

;;; magit-prime.el ends here
