;;; magnus-coord.el --- Agent coordination for magnus -*- lexical-binding: t -*-

;; Copyright (C) 2026 Hrishikesh S
;; Author: Hrishikesh S <hrish2006@gmail.com>
;; URL: https://github.com/hrishikeshs/magnus
;; SPDX-License-Identifier: MIT

;;; Commentary:

;; This module provides inter-agent coordination through a shared
;; coordination file (.magnus-coord.md).  Agents use this file to:
;; - Announce what they're working on
;; - Communicate with other agents
;; - Record decisions and agreements
;; - Avoid stepping on each other's work

;;; Code:

(require 'magnus-instances)

(declare-function vterm-send-string "vterm")
(declare-function project-root "project")
(declare-function vterm-send-return "vterm")

(defvar magnus-claude-executable)
(declare-function magnus--headless-command "magnus")

;;; Vterm buffer activity tracking

(defvar magnus-coord--buffer-ticks (make-hash-table :test 'equal)
  "Hash: instance-id -> (TICK . FLOAT-TIME).
TICK is the `buffer-modified-tick' last observed, FLOAT-TIME is when it changed.")

(defcustom magnus-coord-quiescence-threshold 30
  "Seconds of vterm buffer inactivity before an agent is considered idle.
Only idle agents receive periodic nudges."
  :type 'integer
  :group 'magnus)

(defun magnus-coord--update-buffer-ticks ()
  "Update buffer activity timestamps for all running instances."
  (dolist (instance (magnus-instances-list))
    (when (eq (magnus-instance-status instance) 'running)
      (let* ((id (magnus-instance-id instance))
             (buffer (magnus-instance-buffer instance))
             (prev (gethash id magnus-coord--buffer-ticks))
             (prev-tick (car prev)))
        (when (and buffer (buffer-live-p buffer))
          (let ((tick (buffer-modified-tick buffer)))
            (if (and prev-tick (= tick prev-tick))
                ;; Tick unchanged — keep existing timestamp
                nil
              ;; Tick changed — record new tick + current time
              (puthash id (cons tick (float-time))
                       magnus-coord--buffer-ticks))))))))

(defun magnus-coord-agent-quiescent-p (instance)
  "Return non-nil if INSTANCE's vterm buffer has been quiet.
Quiet means no output for `magnus-coord-quiescence-threshold' seconds."
  (let* ((id (magnus-instance-id instance))
         (entry (gethash id magnus-coord--buffer-ticks))
         (last-change (cdr entry)))
    (or (null last-change)
        (> (- (float-time) last-change) magnus-coord-quiescence-threshold))))

;;; Customization

(defcustom magnus-coord-file ".magnus-coord.md"
  "Name of the coordination file in project directories."
  :type 'string
  :group 'magnus)

(defcustom magnus-coord-instructions-file ".claude/magnus-instructions.md"
  "Path to the instructions file for agents (relative to project)."
  :type 'string
  :group 'magnus)

(defcustom magnus-coord-mention-notify t
  "If non-nil, automatically notify agents when they are @mentioned."
  :type 'boolean
  :group 'magnus)

(defcustom magnus-coord-skill-file ".claude/skills/coordinate/SKILL.md"
  "Path to the coordination skill file (relative to project)."
  :type 'string
  :group 'magnus)

(defcustom magnus-coord-reminder-interval 600
  "Seconds between coordination file reminders to agents.
Set to nil to disable.  Default is 600 (10 minutes)."
  :type '(choice (integer :tag "Seconds")
                 (const :tag "Disabled" nil))
  :group 'magnus)

(defcustom magnus-coord-log-max-entries 25
  "Maximum number of log entries to keep in the coordination file.
Older entries are trimmed automatically.  Set to nil to disable."
  :type '(choice (integer :tag "Entries")
                 (const :tag "Unlimited" nil))
  :group 'magnus)

(defcustom magnus-coord-active-cooldown 120
  "Seconds after last chat message before nudges resume.
When the user sends a message via the chat buffer, all
coordination nudges are suppressed for this many seconds."
  :type 'integer
  :group 'magnus)

(defcustom magnus-coord-idle-threshold 300
  "Seconds of inactivity before telling agents to sleep.
When the user is idle for this long, a sleep message is sent to
all running agents and periodic nudges are suppressed.  When the
user returns, a wake-up message is sent.  Set to nil to disable."
  :type '(choice (integer :tag "Seconds")
                 (const :tag "Disabled" nil))
  :group 'magnus)

(defcustom magnus-coord-context-warn-threshold 0.80
  "Context utilization ratio at which to warn an agent.
When an agent's context usage exceeds this fraction of the
maximum window, a memory consolidation warning is sent.
Set to nil to disable.  Default is 0.80 (80%)."
  :type '(choice (float :tag "Ratio (0.0-1.0)")
                 (const :tag "Disabled" nil))
  :group 'magnus)

(defcustom magnus-coord-context-max-tokens 200000
  "Maximum context window size in tokens.
Used to calculate context utilization percentage."
  :type 'integer
  :group 'magnus)

;;; Atomic file writes

(defun magnus-coord--write-file-atomic (file)
  "Write the current buffer to FILE atomically.
Writes to a temporary file in the same directory, then renames.
This prevents partial reads when agents write concurrently."
  (let ((tmp (make-temp-file
              (expand-file-name ".magnus-coord-tmp" (file-name-directory file)))))
    (write-region (point-min) (point-max) tmp nil 'quiet)
    (rename-file tmp file t)))

;;; Sending messages to agents

(defun magnus-coord-nudge-agent (instance message)
  "Nudge INSTANCE by sending MESSAGE to its vterm buffer."
  (when-let ((buffer (magnus-instance-buffer instance)))
    (when (buffer-live-p buffer)
      (with-current-buffer buffer
        (vterm-send-string message)
        (vterm-send-return)))))

;;; Periodic reminders

(defvar magnus-coord--reminder-timer nil
  "Timer for periodic coordination reminders.")

(defun magnus-coord-start-reminders ()
  "Start periodic coordination reminders and AFK detection."
  (magnus-coord-stop-reminders)
  (when magnus-coord-reminder-interval
    (setq magnus-coord--reminder-timer
          (run-with-timer magnus-coord-reminder-interval
                         magnus-coord-reminder-interval
                         #'magnus-coord--send-reminders)))
  (magnus-coord--start-idle-watch))

(defun magnus-coord-stop-reminders ()
  "Stop periodic coordination reminders and AFK detection."
  (when magnus-coord--reminder-timer
    (cancel-timer magnus-coord--reminder-timer)
    (setq magnus-coord--reminder-timer nil))
  (magnus-coord--stop-idle-watch))

(defvar magnus-coord--reminder-templates
  '("Hey %s — take a quick look at .magnus-coord.md. Any messages for you? Anything you've learned that other agents should know? Drop it in the Discoveries section."
    "Coordination check, %s. Read .magnus-coord.md — has anyone posted something useful in Discoveries? If you've figured out something non-obvious, share it there."
    "%s, heads up: peek at .magnus-coord.md. Other agents may have left insights in Discoveries that save you time. And if you've hit a gotcha or found a pattern, pay it forward."
    "Quick sync, %s. Open .magnus-coord.md — update your status, check for messages, and if you've learned anything surprising about this codebase, add it to Discoveries.")
  "Rotating reminder messages. %s is replaced with the agent name.")

(defvar magnus-coord--reminder-index 0
  "Index into the rotating reminder templates.")

(defvar magnus-coord--last-contact (make-hash-table :test 'equal)
  "Hash table: instance-id -> `float-time' of last user message.
Agents contacted recently are skipped by periodic nudges.")

;;; Attention pattern learning

(defvar magnus-coord--attention-data (make-hash-table :test 'equal)
  "Hash: agent-name -> plist (:visits (FLOAT-TIME...) :messages INTEGER).
Visit timestamps are kept newest-first, max 50 entries.
Persists across sessions for adaptive nudge intervals.")

(defvar magnus-coord--attention-save-timer nil
  "Debounced timer for saving attention data.")

(defvar magnus-attention-data-file)

(defun magnus-coord--record-visit (agent-name)
  "Record a user visit to AGENT-NAME.
Debounced: only records if last visit was more than 30 seconds ago."
  (let* ((data (or (gethash agent-name magnus-coord--attention-data)
                   (list :visits nil :messages 0)))
         (visits (plist-get data :visits))
         (last-visit (car visits)))
    (when (or (null last-visit) (> (- (float-time) last-visit) 30))
      (puthash agent-name
               (plist-put data :visits
                          (seq-take (cons (float-time) visits) 50))
               magnus-coord--attention-data)
      (magnus-coord--schedule-attention-save))))

(defun magnus-coord--record-message (agent-name)
  "Record that a chat message was sent to AGENT-NAME."
  (let* ((data (or (gethash agent-name magnus-coord--attention-data)
                   (list :visits nil :messages 0)))
         (messages (or (plist-get data :messages) 0)))
    (puthash agent-name
             (plist-put data :messages (1+ messages))
             magnus-coord--attention-data)))

(defun magnus-coord--adaptive-interval (agent-name)
  "Return adaptive nudge interval for AGENT-NAME in seconds.
Matches the user's visiting rhythm.  Agents visited frequently
get shorter intervals (matching the pace).  Agents rarely visited
get longer intervals (up to the default)."
  (let* ((data (gethash agent-name magnus-coord--attention-data))
         (visits (plist-get data :visits))
         (default (or magnus-coord-reminder-interval 600)))
    (if (and visits (>= (length visits) 3))
        (let* ((recent (seq-take visits 10))
               (intervals (cl-mapcar (lambda (a b) (- a b))
                                     recent (cdr recent)))
               (avg (/ (apply #'+ intervals) (float (length intervals)))))
          (min default (max 300 (round (* avg 1.5)))))
      default)))

(defun magnus-coord--neglected-p (instance)
  "Return non-nil if INSTANCE is overdue for user attention.
Compares time since last visit against twice the adaptive interval."
  (when (eq (magnus-instance-status instance) 'running)
    (let* ((name (magnus-instance-name instance))
           (data (gethash name magnus-coord--attention-data))
           (visits (plist-get data :visits))
           (last-visit (car visits))
           (interval (magnus-coord--adaptive-interval name)))
      (and last-visit
           (> (- (float-time) last-visit) (* interval 2))))))

(defun magnus-coord--on-buffer-focus (_frame)
  "Record when the user switches to an agent's buffer."
  (let ((buf (window-buffer (selected-window))))
    (dolist (instance (magnus-instances-list))
      (when (eq buf (magnus-instance-buffer instance))
        (magnus-coord--record-visit (magnus-instance-name instance))))))

(defun magnus-coord-setup-attention-tracking ()
  "Start tracking user attention patterns."
  (add-hook 'window-buffer-change-functions
            #'magnus-coord--on-buffer-focus))

;;; Attention persistence

(defun magnus-coord--schedule-attention-save ()
  "Schedule a debounced save of attention data."
  (when magnus-coord--attention-save-timer
    (cancel-timer magnus-coord--attention-save-timer))
  (setq magnus-coord--attention-save-timer
        (run-with-idle-timer 5 nil #'magnus-coord-attention-save)))

(defun magnus-coord-attention-save ()
  "Save attention data to disk."
  (setq magnus-coord--attention-save-timer nil)
  (when (and (bound-and-true-p magnus-attention-data-file)
             (hash-table-count magnus-coord--attention-data))
    (let ((alist nil))
      (maphash (lambda (k v) (push (cons k v) alist))
               magnus-coord--attention-data)
      (with-temp-file magnus-attention-data-file
        (insert ";; Magnus attention data - do not edit manually\n")
        (pp alist (current-buffer))))))

(defun magnus-coord-attention-load ()
  "Load attention data from disk."
  (when (and (bound-and-true-p magnus-attention-data-file)
             (file-exists-p magnus-attention-data-file))
    (condition-case nil
        (let ((alist (with-temp-buffer
                       (insert-file-contents magnus-attention-data-file)
                       (goto-char (point-min))
                       (read (current-buffer)))))
          (clrhash magnus-coord--attention-data)
          (dolist (entry alist)
            (puthash (car entry) (cdr entry) magnus-coord--attention-data)))
      (error
       (message "Magnus: failed to load attention data")))))

;;; Contact recording

(defun magnus-coord-record-contact (instance-id)
  "Record that INSTANCE-ID was just contacted by the user.
Suppresses periodic nudges for this agent until the next interval.
Also updates attention pattern data."
  (puthash instance-id (float-time) magnus-coord--last-contact)
  (when-let ((instance (magnus-instances-get instance-id)))
    (let ((name (magnus-instance-name instance)))
      (magnus-coord--record-visit name)
      (magnus-coord--record-message name))))

(defun magnus-coord--recently-contacted-p (instance)
  "Return non-nil if INSTANCE was contacted within its adaptive interval."
  (let* ((id (magnus-instance-id instance))
         (name (magnus-instance-name instance))
         (last (gethash id magnus-coord--last-contact))
         (interval (magnus-coord--adaptive-interval name)))
    (and last (< (- (float-time) last) interval))))

(defvar magnus-coord--user-idle-p nil
  "Non-nil when the user has been detected as AFK.
Set by `magnus-coord--on-user-idle', cleared by `magnus-coord--on-user-return'.")

(defvar magnus-coord--user-active-until 0
  "Float-time until which the user is considered actively chatting.
Nudges are suppressed while `float-time' is less than this value.")

(defvar magnus-coord--do-not-disturb nil
  "Non-nil when do-not-disturb mode is active.
All periodic nudges are suppressed.  Toggle with `magnus-coord-toggle-dnd'.")

(defun magnus-coord-record-user-active ()
  "Record that the user is actively chatting.
Suppresses all nudges for `magnus-coord-active-cooldown' seconds."
  (setq magnus-coord--user-active-until
        (+ (float-time) magnus-coord-active-cooldown)))

(defun magnus-coord--user-active-p ()
  "Return non-nil if the user has been actively chatting recently."
  (> magnus-coord--user-active-until (float-time)))

(defun magnus-coord-toggle-dnd ()
  "Toggle do-not-disturb mode.
When active, all periodic coordination nudges are suppressed.
Agents keep working but are not poked."
  (interactive)
  (setq magnus-coord--do-not-disturb (not magnus-coord--do-not-disturb))
  (message "Magnus: Do Not Disturb %s"
           (if magnus-coord--do-not-disturb "ON" "OFF"))
  (run-hooks 'magnus-instances-changed-hook))

(defun magnus-coord-agent-busy-p (instance)
  "Return non-nil if INSTANCE has signaled it is busy.
Agents create a busy file to tell Magnus to stop nudging them."
  (file-exists-p (magnus-process--agent-busy-path instance)))

(defun magnus-coord--send-reminders ()
  "Send a coordination reminder to idle instances.
Skips agents that were recently contacted, are busy, or whose
vterm buffer is still active (not quiescent).
Suppressed entirely when AFK, actively chatting, or DND is on."
  ;; Always update buffer ticks so quiescence tracking stays current
  (magnus-coord--update-buffer-ticks)
  (unless (or magnus-coord--user-idle-p
              (magnus-coord--user-active-p)
              magnus-coord--do-not-disturb)
    (let ((template (nth (mod magnus-coord--reminder-index
                              (length magnus-coord--reminder-templates))
                         magnus-coord--reminder-templates)))
      (dolist (instance (magnus-instances-list))
        (when (and (eq (magnus-instance-status instance) 'running)
                   (not (magnus-coord--recently-contacted-p instance))
                   (not (magnus-coord-agent-busy-p instance))
                   (magnus-coord-agent-quiescent-p instance))
          (magnus-coord-nudge-agent
           instance
           (format template (magnus-instance-name instance)))))
      (setq magnus-coord--reminder-index
            (1+ magnus-coord--reminder-index))))
  ;; Trim logs while we're at it (even when idle)
  (magnus-coord-trim-all)
  ;; Check context utilization (runs even when idle/active)
  (magnus-coord-check-context-all))

;;; Log trimming

(defun magnus-coord-trim-log (directory)
  "Trim the Log section in DIRECTORY's coordination file.
Keeps only the last `magnus-coord-log-max-entries' entries."
  (when magnus-coord-log-max-entries
    (let ((file (magnus-coord-file-path directory)))
      (when (file-exists-p file)
        (with-temp-buffer
          (insert-file-contents file)
          (goto-char (point-min))
          (when (re-search-forward "^## Log" nil t)
            (let* ((log-start (match-beginning 0))
                   (section-end (save-excursion
                                  (if (re-search-forward "^## " nil t)
                                      (match-beginning 0)
                                    (point-max))))
                   (entries nil))
              ;; Collect log entries (timestamp lines)
              (goto-char log-start)
              (while (re-search-forward
                      "^\\[\\([0-9:]+\\)\\] .+$" section-end t)
                (push (match-beginning 0) entries))
              (setq entries (nreverse entries))
              ;; If over limit, delete everything before the Nth-from-last
              (when (> (length entries) magnus-coord-log-max-entries)
                (let ((cut-point (nth (- (length entries)
                                         magnus-coord-log-max-entries)
                                      entries)))
                  ;; Delete from after the Log header to the cut point
                  (goto-char log-start)
                  (forward-line 1)
                  ;; Skip blank lines and comments after header
                  (while (and (< (point) cut-point)
                              (or (looking-at "^$")
                                  (looking-at "^<!--")))
                    (if (looking-at "^<!--")
                        (if (re-search-forward "-->" nil t)
                            (forward-line 1)
                          (goto-char cut-point))
                      (forward-line 1)))
                  (when (< (point) cut-point)
                    (delete-region (point) cut-point))
                  (magnus-coord--write-file-atomic file))))))))))

(defun magnus-coord-trim-all ()
  "Trim coordination file logs for all active project directories."
  (let ((dirs (delete-dups
               (mapcar #'magnus-instance-directory
                       (magnus-instances-list)))))
    (dolist (dir dirs)
      (magnus-coord-trim-log dir))))

;;; AFK detection

(defvar magnus-coord--idle-timer nil
  "Idle timer that fires after `magnus-coord-idle-threshold' seconds.")

(defun magnus-coord--start-idle-watch ()
  "Start watching for user idleness."
  (magnus-coord--stop-idle-watch)
  (when magnus-coord-idle-threshold
    (setq magnus-coord--idle-timer
          (run-with-idle-timer magnus-coord-idle-threshold nil
                              #'magnus-coord--on-user-idle))))

(defun magnus-coord--stop-idle-watch ()
  "Stop watching for user idleness."
  (when magnus-coord--idle-timer
    (cancel-timer magnus-coord--idle-timer)
    (setq magnus-coord--idle-timer nil))
  (remove-hook 'pre-command-hook #'magnus-coord--on-user-return)
  (setq magnus-coord--user-idle-p nil))

(declare-function magnus-process--agent-memory-path "magnus-process")
(declare-function magnus-process--agent-busy-path "magnus-process")
(declare-function magnus-process--ensure-agent-dir "magnus-process")
(declare-function magnus-process--session-jsonl-path "magnus-process")
(declare-function magnus-process-create "magnus-process")

(defun magnus-coord--on-user-idle ()
  "Called when the user has been idle for `magnus-coord-idle-threshold'.
Tells agents to consolidate their memory and go to sleep."
  (setq magnus-coord--user-idle-p t)
  (dolist (instance (magnus-instances-list))
    (when (eq (magnus-instance-status instance) 'running)
      (let* ((name (magnus-instance-name instance))
             (memory-rel (format ".claude/agents/%s/memory.md" name)))
        (magnus-process--ensure-agent-dir instance)
        (magnus-coord-nudge-agent
         instance
         (format "The user is away. Before you sleep, update your memory file at %s — write down what matters from this session: key decisions, things you learned, gotchas, unfinished work, your relationships with other agents. This file persists across restarts — it's how future-you remembers. Then go to sleep and wait quietly."
                 memory-rel)))))
  (add-hook 'pre-command-hook #'magnus-coord--on-user-return))

(defun magnus-coord--on-user-return ()
  "Called when the user presses a key after being idle.
Sends a wake-up message to running agents and re-arms the idle timer."
  (remove-hook 'pre-command-hook #'magnus-coord--on-user-return)
  (setq magnus-coord--user-idle-p nil)
  (dolist (instance (magnus-instances-list))
    (when (eq (magnus-instance-status instance) 'running)
      (magnus-coord-nudge-agent
       instance
       "The user is back! Resume normal operation — check .magnus-coord.md for any updates.")))
  ;; Re-arm the idle timer for the next AFK period
  (magnus-coord--start-idle-watch))

;;; Context window monitoring

(defvar magnus-coord--context-warned (make-hash-table :test 'equal)
  "Hash table of instance-id to t for agents already warned about context.
Cleared when the agent's session changes (restart or compaction).")

(defun magnus-coord-check-context-all ()
  "Check context utilization for all running agents.
Warns agents approaching the context window limit."
  (when magnus-coord-context-warn-threshold
    (dolist (instance (magnus-instances-list))
      (when (eq (magnus-instance-status instance) 'running)
        (condition-case err
            (magnus-coord--check-context-one instance)
          (error (message "Magnus: context check error for %s: %s"
                          (magnus-instance-name instance)
                          (error-message-string err))))))))

(defun magnus-coord--check-context-one (instance)
  "Check context utilization for INSTANCE and warn if needed."
  (let ((id (magnus-instance-id instance)))
    (unless (gethash id magnus-coord--context-warned)
      (when-let ((usage (magnus-coord--read-context-usage instance)))
        (let ((ratio (/ (float usage) magnus-coord-context-max-tokens)))
          (when (>= ratio magnus-coord-context-warn-threshold)
            (puthash id t magnus-coord--context-warned)
            (let* ((name (magnus-instance-name instance))
                   (pct (round (* ratio 100)))
                   (memory-rel (format ".claude/agents/%s/memory.md" name)))
              (magnus-process--ensure-agent-dir instance)
              (magnus-coord-nudge-agent
               instance
               (format "Heads up: you're at %d%% context. Compaction is coming — write everything important to your memory file at %s NOW, while you still have full context. Key decisions, unfinished work, what you've learned, relationships with other agents. After compaction you'll lose detail."
                       pct memory-rel)))))))))

(defun magnus-coord--read-context-usage (instance)
  "Read the latest context token count from INSTANCE's session trace.
Returns total input tokens or nil if unavailable.  Only reads the
last 4KB of the file for efficiency."
  (when-let ((session-id (magnus-instance-session-id instance))
             (jsonl (magnus-process--session-jsonl-path
                     (magnus-instance-directory instance) session-id)))
    (let* ((attrs (file-attributes jsonl))
           (size (file-attribute-size attrs))
           (start (max 0 (- size 4096))))
      (with-temp-buffer
        (insert-file-contents jsonl nil start size)
        (magnus-coord--parse-last-usage)))))

(defun magnus-coord--parse-last-usage ()
  "Parse the last usage entry from the current buffer.
Looks for cache_read_input_tokens in the last complete JSON lines
and sums input_tokens + cache_creation_input_tokens +
cache_read_input_tokens."
  (goto-char (point-max))
  (let ((found nil))
    (while (and (not found)
                (re-search-backward "cache_read_input_tokens" nil t))
      (let ((line-start (line-beginning-position))
            (line-end (line-end-position)))
        ;; Extract the three token counts via regex (avoid full JSON parse)
        (let ((input (magnus-coord--extract-number
                      "\"input_tokens\":\\s*\\([0-9]+\\)" line-start line-end))
              (cache-create (magnus-coord--extract-number
                             "\"cache_creation_input_tokens\":\\s*\\([0-9]+\\)" line-start line-end))
              (cache-read (magnus-coord--extract-number
                           "\"cache_read_input_tokens\":\\s*\\([0-9]+\\)" line-start line-end)))
          (when (and input cache-read)
            (setq found (+ input (or cache-create 0) cache-read))))))
    found))

(defun magnus-coord--extract-number (pattern start end)
  "Extract a number matching PATTERN between START and END."
  (save-excursion
    (goto-char start)
    (when (re-search-forward pattern end t)
      (string-to-number (match-string 1)))))

;;; @mention watching

(defvar magnus-coord--watched-dirs nil
  "List of directories being polled for coordination file changes.")

(defvar magnus-coord--file-mtimes nil
  "Alist of (directory . modification-time) for polling dedup.")

(defvar magnus-coord--processed-mentions nil
  "Alist of (directory . list-of-processed-mention-hashes) to avoid duplicates.")

(defvar magnus-coord--processed-dms nil
  "Alist of (directory . list-of-processed-dm-hashes) to avoid duplicates.")

(defvar magnus-coord--processed-summons nil
  "Alist of (directory . list-of-processed-summon-hashes) to avoid duplicates.")

(defvar magnus-coord--poll-timer nil
  "Timer for polling coordination files for mentions, DMs, and summons.")

(defun magnus-coord-ensure-watchers ()
  "Start polling for all directories with active instances.
Call this on startup to ensure @mention and DM detection works for
instances restored from persistence."
  (let ((dirs (delete-dups
               (mapcar #'magnus-instance-directory (magnus-instances-list)))))
    (dolist (dir dirs)
      (when (file-exists-p (magnus-coord-file-path dir))
        (unless (member dir magnus-coord--watched-dirs)
          (magnus-coord-start-watching dir)))))
  (magnus-coord--start-poll-timer))

(defun magnus-coord-start-watching (directory)
  "Start polling the coordination file in DIRECTORY for mentions and DMs."
  (let ((file (magnus-coord-file-path directory)))
    (when (file-exists-p file)
      (cl-pushnew directory magnus-coord--watched-dirs :test #'equal)
      ;; Seed mtime so first poll doesn't re-process everything
      (setf (alist-get directory magnus-coord--file-mtimes nil nil #'equal)
            (file-attribute-modification-time (file-attributes file)))
      ;; Initialize processed state from current content
      (magnus-coord--init-processed-mentions directory)
      (magnus-coord--init-processed-dms directory)
      (magnus-coord--init-processed-summons directory))))

(defun magnus-coord-stop-watching (directory)
  "Stop polling the coordination file in DIRECTORY."
  (setq magnus-coord--watched-dirs (delete directory magnus-coord--watched-dirs))
  (setq magnus-coord--file-mtimes
        (assoc-delete-all directory magnus-coord--file-mtimes))
  (setq magnus-coord--processed-mentions
        (assoc-delete-all directory magnus-coord--processed-mentions))
  (setq magnus-coord--processed-dms
        (assoc-delete-all directory magnus-coord--processed-dms))
  (setq magnus-coord--processed-summons
        (assoc-delete-all directory magnus-coord--processed-summons)))

(defun magnus-coord--start-poll-timer ()
  "Start the coordination file poll timer."
  (unless magnus-coord--poll-timer
    (setq magnus-coord--poll-timer
          (run-with-timer 3 3 #'magnus-coord--poll-all))))

(defvar magnus-coord--last-trim-time 0
  "Timestamp of last log trim, for debouncing.")

(defun magnus-coord--poll-all ()
  "Poll all watched coordination files for new mentions, DMs, and summons."
  (dolist (directory magnus-coord--watched-dirs)
    (let* ((file (magnus-coord-file-path directory))
           (mtime (and (file-exists-p file)
                       (file-attribute-modification-time
                        (file-attributes file))))
           (last-mtime (alist-get directory magnus-coord--file-mtimes
                                  nil nil #'equal)))
      (when (and mtime (not (equal mtime last-mtime)))
        (setf (alist-get directory magnus-coord--file-mtimes nil nil #'equal)
              mtime)
        (condition-case err
            (progn
              (when magnus-coord-mention-notify
                (magnus-coord--check-new-mentions directory))
              (magnus-coord--check-new-dms directory)
              (magnus-coord--check-new-summons directory)
              ;; Trim logs every 3min when file is changing
              (let ((now (float-time)))
                (when (> (- now magnus-coord--last-trim-time) 180)
                  (setq magnus-coord--last-trim-time now)
                  (magnus-coord-trim-log directory))))
          (error
           (message "Magnus: coord poll error for %s: %s"
                    directory (error-message-string err))))))))

(defun magnus-coord-stop-all-watchers ()
  "Stop all coordination file polling."
  (when magnus-coord--poll-timer
    (cancel-timer magnus-coord--poll-timer)
    (setq magnus-coord--poll-timer nil))
  (setq magnus-coord--watched-dirs nil)
  (setq magnus-coord--file-mtimes nil)
  (setq magnus-coord--processed-mentions nil)
  (setq magnus-coord--processed-dms nil)
  (setq magnus-coord--processed-summons nil))

(defun magnus-coord--init-processed-mentions (directory)
  "Initialize processed mentions for DIRECTORY from current file content."
  (let ((file (magnus-coord-file-path directory)))
    (when (file-exists-p file)
      (let ((mentions (magnus-coord--extract-mentions
                       (with-temp-buffer
                         (insert-file-contents file)
                         (buffer-string)))))
        (setf (alist-get directory magnus-coord--processed-mentions nil nil #'equal)
              (mapcar #'magnus-coord--mention-hash mentions))))))

(defun magnus-coord--check-new-mentions (directory)
  "Check for new @mentions in DIRECTORY's coordination file."
  (let* ((file (magnus-coord-file-path directory))
         (content (when (file-exists-p file)
                    (with-temp-buffer
                      (insert-file-contents file)
                      (buffer-string))))
         (mentions (when content (magnus-coord--extract-mentions content)))
         (processed (alist-get directory magnus-coord--processed-mentions nil nil #'equal)))
    (dolist (mention mentions)
      (let ((hash (magnus-coord--mention-hash mention)))
        (unless (member hash processed)
          ;; New mention - notify the agent
          (magnus-coord--notify-mention directory mention)
          (push hash processed))))
    (setf (alist-get directory magnus-coord--processed-mentions nil nil #'equal)
          processed)))

(defun magnus-coord--extract-mentions (content)
  "Extract all @mentions from CONTENT.
Returns list of (agent-name . context-line) pairs."
  (let (mentions)
    (with-temp-buffer
      (insert content)
      (goto-char (point-min))
      (while (re-search-forward "@\\([a-zA-Z][-a-zA-Z0-9_]*\\)" nil t)
        (let ((agent (match-string 1))
              (line (buffer-substring-no-properties
                     (line-beginning-position)
                     (line-end-position))))
          (push (cons agent line) mentions))))
    (nreverse mentions)))

(defun magnus-coord--mention-hash (mention)
  "Create a hash for MENTION to track duplicates."
  (secure-hash 'md5 (format "%s:%s" (car mention) (cdr mention))))

(defun magnus-coord--notify-mention (directory mention)
  "Notify the agent named in MENTION within DIRECTORY."
  (let* ((agent-name (car mention))
         (context-line (cdr mention))
         (instance (magnus-coord--find-instance-by-name agent-name directory)))
    (when instance
      (magnus-coord--send-mention-notification instance context-line))))

(defun magnus-coord--find-instance-by-name (name directory)
  "Find an instance with NAME working in DIRECTORY."
  (cl-find-if (lambda (inst)
                (and (string= (magnus-instance-name inst) name)
                     (string= (magnus-instance-directory inst) directory)))
              (magnus-instances-list)))

(defun magnus-coord--extract-sender-and-message (context-line)
  "Extract sender name and message from CONTEXT-LINE.
Returns (sender . message) or nil."
  (when (string-match
         "\\[.*?\\] \\([^:]+\\): .*?@[^ ]+ \\(.*\\)"
         context-line)
    (let ((sender (match-string 1 context-line))
          (message (match-string 2 context-line)))
      (cons sender (string-trim message)))))

(defun magnus-coord--send-mention-notification (instance context-line)
  "Send a mention notification to INSTANCE with CONTEXT-LINE.
Delivers the message content without commanding the agent."
  (let* ((parsed (magnus-coord--extract-sender-and-message context-line))
         (msg (if parsed
                  (format "[From %s]: %s"
                          (car parsed) (cdr parsed))
                (format "[Mention in .magnus-coord.md]: %s"
                        context-line))))
    (magnus-coord-nudge-agent instance msg)))

;;; Agent-to-agent direct messages

(defun magnus-coord--init-processed-dms (directory)
  "Initialize processed DMs for DIRECTORY from current file content."
  (let ((file (magnus-coord-file-path directory)))
    (when (file-exists-p file)
      (let ((dms (magnus-coord--extract-dms
                  (with-temp-buffer
                    (insert-file-contents file)
                    (buffer-string)))))
        (setf (alist-get directory magnus-coord--processed-dms nil nil #'equal)
              (mapcar #'magnus-coord--dm-hash dms))))))

(defun magnus-coord--extract-dms (content)
  "Extract all [DM @name] patterns from CONTENT.
Returns list of (target sender message) tuples."
  (let (dms)
    (with-temp-buffer
      (insert content)
      (goto-char (point-min))
      (while (re-search-forward
              "\\[DM @\\([a-zA-Z][-a-zA-Z0-9_]*\\)\\][[:space:]]*\\(.*\\)"
              nil t)
        (let* ((target (match-string 1))
               (message (string-trim (match-string 2)))
               (line (buffer-substring-no-properties
                      (line-beginning-position)
                      (line-end-position)))
               (sender (when (string-match "\\] \\([^:]+\\):" line)
                         (match-string 1 line))))
          (push (list target (or sender "unknown") message) dms))))
    (nreverse dms)))

(defun magnus-coord--dm-hash (dm)
  "Create a hash for DM to track duplicates."
  (secure-hash 'md5 (format "%s:%s:%s" (nth 0 dm) (nth 1 dm) (nth 2 dm))))

(defun magnus-coord--check-new-dms (directory)
  "Check for new [DM @name] patterns in DIRECTORY's coordination file."
  (let* ((file (magnus-coord-file-path directory))
         (content (when (file-exists-p file)
                    (with-temp-buffer
                      (insert-file-contents file)
                      (buffer-string))))
         (dms (when content (magnus-coord--extract-dms content)))
         (processed (alist-get directory magnus-coord--processed-dms nil nil #'equal)))
    (dolist (dm dms)
      (let ((hash (magnus-coord--dm-hash dm)))
        (unless (member hash processed)
          (magnus-coord--deliver-dm directory dm)
          (push hash processed))))
    (setf (alist-get directory magnus-coord--processed-dms nil nil #'equal)
          processed)))

(defun magnus-coord--deliver-dm (directory dm)
  "Deliver DM to the target agent in DIRECTORY.
DM is (target sender message)."
  (let* ((target (nth 0 dm))
         (sender (nth 1 dm))
         (message (nth 2 dm))
         (instance (magnus-coord--find-instance-by-name target directory)))
    (when instance
      (magnus-coord-nudge-agent
       instance
       (format "[DM from %s]: %s" sender message)))))

;;; Agent-initiated summoning

(defvar magnus--summon-context)

(defun magnus-coord--init-processed-summons (directory)
  "Initialize processed summons for DIRECTORY from current file content."
  (let ((file (magnus-coord-file-path directory)))
    (when (file-exists-p file)
      (let ((summons (magnus-coord--extract-summons
                      (with-temp-buffer
                        (insert-file-contents file)
                        (buffer-string)))))
        (setf (alist-get directory magnus-coord--processed-summons nil nil #'equal)
              (mapcar #'magnus-coord--summon-hash summons))))))

(defun magnus-coord--extract-summons (content)
  "Extract all [SUMMON @name] patterns from CONTENT.
Returns list of (target sender reason) tuples."
  (let (summons)
    (with-temp-buffer
      (insert content)
      (goto-char (point-min))
      (while (re-search-forward
              "\\[SUMMON @\\([a-zA-Z][-a-zA-Z0-9_]*\\)\\][[:space:]]*\\(.*\\)"
              nil t)
        (let* ((target (match-string 1))
               (reason (string-trim (match-string 2)))
               (line (buffer-substring-no-properties
                      (line-beginning-position)
                      (line-end-position)))
               (sender (when (string-match "\\] \\([^:]+\\):" line)
                         (match-string 1 line))))
          (push (list target (or sender "unknown") reason) summons))))
    (nreverse summons)))

(defun magnus-coord--summon-hash (summon)
  "Create a hash for SUMMON to track duplicates."
  (secure-hash 'md5 (format "%s:%s:%s"
                             (nth 0 summon) (nth 1 summon) (nth 2 summon))))

(defun magnus-coord--check-new-summons (directory)
  "Check for new [SUMMON @name] patterns in DIRECTORY's coordination file."
  (let* ((file (magnus-coord-file-path directory))
         (content (when (file-exists-p file)
                    (with-temp-buffer
                      (insert-file-contents file)
                      (buffer-string))))
         (summons (when content (magnus-coord--extract-summons content)))
         (processed (alist-get directory magnus-coord--processed-summons
                               nil nil #'equal)))
    (dolist (summon summons)
      (let ((hash (magnus-coord--summon-hash summon)))
        (unless (member hash processed)
          ;; Wait for user to be idle before prompting
          (run-with-idle-timer 5 nil
                               #'magnus-coord--handle-summon directory summon)
          (push hash processed))))
    (setf (alist-get directory magnus-coord--processed-summons nil nil #'equal)
          processed)))

(defun magnus-coord--handle-summon (directory summon)
  "Handle a SUMMON request in DIRECTORY.
SUMMON is (target-name sender reason)."
  (let* ((target (nth 0 summon))
         (sender (nth 1 summon))
         (reason (nth 2 summon))
         (agents-dir (expand-file-name ".claude/agents/" directory))
         (memory (expand-file-name (concat target "/memory.md") agents-dir))
         (existing (mapcar #'magnus-instance-name (magnus-instances-list)))
         (is-active (member target existing))
         (is-dormant (and (file-exists-p memory) (not is-active))))
    (cond
     (is-dormant
      (when (y-or-n-p
             (format "Agent %s requests: summon %s (%s)? "
                     sender target
                     (if (string-empty-p reason) "no reason given" reason)))
        (magnus-coord--execute-summon directory target sender reason)))
     (is-active
      ;; Target is already running — notify the requesting agent
      (when-let ((requester (magnus-coord--find-instance-by-name sender directory)))
        (magnus-coord-nudge-agent
         requester
         (format "%s is already online — @mention them in .magnus-coord.md instead."
                 target)))))))

(defun magnus-coord--execute-summon (directory target sender reason)
  "Execute a summon of TARGET in DIRECTORY, requested by SENDER for REASON."
  (let ((magnus--summon-context (list :sender sender :reason reason)))
    (magnus-process-create directory target))
  (magnus-coord-add-log directory "Magnus"
                         (format "Summoned %s (requested by %s: %s)"
                                 target sender reason)))

;;; Session retrospectives

(defvar magnus-coord--session-start-times (make-hash-table :test 'equal)
  "Hash: directory -> float-time when first agent joined this session.")

(defvar magnus-coord--retro-generating nil
  "Non-nil while a retro is being generated.")

(defvar magnus-coord--retro-output ""
  "Accumulated output from the retro generator process.")

(defvar magnus-coord--latest-retro (make-hash-table :test 'equal)
  "Hash: directory -> path of the most recent retro file.")

(defun magnus-coord--git-log-since (directory since-time)
  "Get git log since SINCE-TIME in DIRECTORY."
  (let ((default-directory directory)
        (since (format-time-string "%Y-%m-%dT%H:%M:%S"
                                   (seconds-to-time since-time))))
    (condition-case nil
        (with-temp-buffer
          (call-process "git" nil t nil
                        "log" "--oneline" (format "--since=%s" since))
          (let ((output (string-trim (buffer-string))))
            (if (string-empty-p output) "No commits" output)))
      (error "No git data"))))

(defun magnus-coord--collect-retro-data (directory)
  "Collect session data for a retrospective in DIRECTORY.
Returns a plist with :log, :discoveries, :decisions, :git, :start, :end."
  (let* ((parsed (magnus-coord-parse directory))
         (log-entries (plist-get parsed :log))
         (decisions (plist-get parsed :decisions))
         (start-time (gethash directory magnus-coord--session-start-times))
         (git-log (when start-time
                    (magnus-coord--git-log-since directory start-time)))
         ;; Extract discoveries from the raw file
         (discoveries (magnus-coord--extract-discoveries directory)))
    (list :log log-entries
          :discoveries discoveries
          :decisions decisions
          :git (or git-log "No git data")
          :start start-time
          :end (float-time))))

(defun magnus-coord--extract-discoveries (directory)
  "Extract the Discoveries section text from DIRECTORY's coord file."
  (let ((file (magnus-coord-file-path directory)))
    (when (file-exists-p file)
      (with-temp-buffer
        (insert-file-contents file)
        (goto-char (point-min))
        (when (re-search-forward "^## Discoveries" nil t)
          (forward-line 1)
          (let ((start (point))
                (end (if (re-search-forward "^## " nil t)
                         (match-beginning 0)
                       (point-max))))
            (string-trim (buffer-substring-no-properties start end))))))))

(defun magnus-coord--format-log-for-retro (entries)
  "Format log ENTRIES for the retro prompt."
  (if entries
      (mapconcat (lambda (e)
                   (format "[%s] %s: %s"
                           (plist-get e :time)
                           (plist-get e :agent)
                           (plist-get e :message)))
                 (reverse entries) "\n")
    "No log entries"))

(defun magnus-coord--retro-prompt (data)
  "Build the Claude prompt for a session retro from DATA."
  (format "Summarize this multi-agent coding session. Be concise and useful.

## Coordination Log
%s

## Discoveries
%s

## Decisions
%s

## Git Commits This Session
%s

Write a session retrospective with these sections:
- **Accomplished**: What got done (2-4 bullets)
- **Decisions**: Key choices made and why
- **Discovered**: Important learnings or gotchas
- **Unfinished**: What's still pending
- **Next**: Suggested priorities for next session

Keep it under 250 words. No filler."
          (magnus-coord--format-log-for-retro (plist-get data :log))
          (or (plist-get data :discoveries) "None recorded")
          (if (plist-get data :decisions)
              (mapconcat #'identity (plist-get data :decisions) "\n")
            "None recorded")
          (plist-get data :git)))

(defun magnus-coord-generate-retro (directory)
  "Generate a session retrospective for DIRECTORY asynchronously."
  (when (and (not magnus-coord--retro-generating)
             (bound-and-true-p magnus-claude-executable)
             (executable-find magnus-claude-executable))
    (let ((data (magnus-coord--collect-retro-data directory)))
      (setq magnus-coord--retro-generating t
            magnus-coord--retro-output "")
      (condition-case err
          (let ((process-environment
                 (cl-remove-if
                  (lambda (e) (string-prefix-p "CLAUDECODE=" e))
                  process-environment)))
            (let ((proc (make-process
                         :name "magnus-retro-gen"
                         :command (magnus--headless-command
                                       (magnus-coord--retro-prompt data))
                         :connection-type 'pipe
                         :filter (lambda (_proc output)
                                   (setq magnus-coord--retro-output
                                         (concat magnus-coord--retro-output output)))
                         :sentinel (lambda (_proc event)
                                     (let ((status (string-trim event)))
                                       (if (string-prefix-p "finished" status)
                                           (magnus-coord--save-retro
                                            directory magnus-coord--retro-output data)
                                         (message "Magnus retro: generator %s" status))
                                       (setq magnus-coord--retro-generating nil))))))
              ;; Close stdin so claude doesn't block waiting for input
              (when (process-live-p proc)
                (process-send-eof proc))))
        (error
         (setq magnus-coord--retro-generating nil)
         (message "Magnus retro: %s" (error-message-string err)))))))

(defun magnus-coord--save-retro (directory content data)
  "Save retro CONTENT for DIRECTORY with session DATA metadata."
  (let* ((retros-dir (expand-file-name ".claude/retros/" directory))
         (timestamp (format-time-string "%Y-%m-%d-%H%M%S"))
         (file (expand-file-name (concat timestamp ".md") retros-dir))
         (start (plist-get data :start))
         (end-time (plist-get data :end)))
    (unless (file-directory-p retros-dir)
      (make-directory retros-dir t))
    (with-temp-file file
      (insert (format "# Session Retrospective — %s\n\n" timestamp))
      (when start
        (insert (format "**Session**: %s to %s\n\n"
                        (format-time-string "%H:%M" (seconds-to-time start))
                        (format-time-string "%H:%M" (seconds-to-time end-time)))))
      (insert content)
      (insert "\n"))
    (message "Magnus: session retro saved. Press F in magnus to view.")
    ;; Store the path for quick retrieval
    (puthash directory file magnus-coord--latest-retro)))

(defun magnus-coord--display-retro (file)
  "Display retro FILE in a buffer."
  (when (and file (file-exists-p file))
    (let ((buf (get-buffer-create "*magnus-retro*")))
      (with-current-buffer buf
        (let ((inhibit-read-only t))
          (erase-buffer)
          (insert-file-contents file)
          (goto-char (point-min)))
        (special-mode)
        (setq-local truncate-lines nil)
        (setq-local word-wrap t))
      (switch-to-buffer buf))))

(defun magnus-coord--find-latest-retro (directory)
  "Find the most recent retro file in DIRECTORY."
  (or (gethash directory magnus-coord--latest-retro)
      (let ((retros-dir (expand-file-name ".claude/retros/" directory)))
        (when (file-directory-p retros-dir)
          (let ((files (directory-files retros-dir t "\\.md$")))
            (car (last (sort files #'string<))))))))

(defun magnus-retro ()
  "Show the latest session retrospective, or generate one.
If agents are running, generates a mid-session retro.
If no agents are running, shows the most recent saved retro."
  (interactive)
  (let* ((dir (or (when-let ((inst (car (magnus-instances-list))))
                    (magnus-instance-directory inst))
                  (magnus-coord--get-directory)))
         (has-agents (cl-some
                      (lambda (inst)
                        (and (string= (magnus-instance-directory inst) dir)
                             (eq (magnus-instance-status inst) 'running)))
                      (magnus-instances-list))))
    (if has-agents
        (progn
          (message "Generating mid-session retro...")
          (magnus-coord-generate-retro dir))
      ;; No agents — show latest saved retro
      (let ((file (magnus-coord--find-latest-retro dir)))
        (if file
            (magnus-coord--display-retro file)
          (user-error "No retrospectives found for this project"))))))

;;; Coordination file management

(defun magnus-coord-file-path (directory)
  "Get the coordination file path for DIRECTORY."
  (expand-file-name magnus-coord-file directory))

(defun magnus-coord-instructions-path (directory)
  "Get the instructions file path for DIRECTORY."
  (expand-file-name magnus-coord-instructions-file directory))

(defun magnus-coord-ensure-file (directory)
  "Ensure coordination file exists in DIRECTORY."
  (let ((file (magnus-coord-file-path directory)))
    (unless (file-exists-p file)
      (magnus-coord--create-file file))
    file))

(defun magnus-coord--create-file (file)
  "Create a new coordination FILE with initial template."
  (with-temp-file file
    (insert "# Agent Coordination\n\n")
    (insert "This file is used by Claude Code agents to coordinate their work.\n")
    (insert "Agents should check this file before starting and announce their plans.\n\n")
    (insert "## Active Work\n\n")
    (insert "<!-- Agents: Update this section when you start/finish work -->\n\n")
    (insert "| Agent | Area | Status | Files |\n")
    (insert "|-------|------|--------|-------|\n")
    (insert "\n## Discoveries\n\n")
    (insert "<!-- Share things you learned that other agents should know -->\n\n")
    (insert "## Decisions\n\n")
    (insert "<!-- Record agreed-upon decisions here -->\n\n")
    (insert "## Log\n\n")
    (insert "<!-- Agents: Append messages here to communicate -->\n\n")))

(defun magnus-coord-ensure-instructions (directory)
  "Ensure agent instructions file exists in DIRECTORY."
  (let ((file (magnus-coord-instructions-path directory)))
    (unless (file-exists-p file)
      (magnus-coord--create-instructions file directory))
    file))

(defun magnus-coord--create-instructions (file directory)
  "Create instructions FILE for agents in DIRECTORY."
  (let ((dir (file-name-directory file)))
    (unless (file-exists-p dir)
      (make-directory dir t)))
  (with-temp-file file
    (insert (magnus-coord--instructions-content directory))))

(defun magnus-coord--instructions-content (_directory)
  "Generate instructions content."
  (format "# Magnus Coordination Protocol

You are one of multiple Claude Code agents working on this project.
To coordinate with other agents, follow this protocol:

## Before Starting Work

1. Read `%s` to see what other agents are working on
2. Check the Active Work table for potential conflicts
3. If your work might conflict, discuss in the Log section first

## Announcing Your Work

When you start a task, update the Active Work table:

```markdown
| your-name | area you're working on | in-progress | file1.ts, file2.ts |
```

## Communicating

To talk to other agents, append to the Log section:

```markdown
[HH:MM] your-name: Your message here. @other-agent if you need their attention.
```

**Note:** When you @mention another agent, they will automatically receive a
notification with your message. You don't need to wait for them to check the file.

## Sharing What You Learn

As you work, you will discover things — API quirks, gotchas, patterns, or
non-obvious behavior in the codebase. Add these to the **Discoveries** section
of the coordination file. Other agents read this, and your insight might save
them hours. Think of it as leaving breadcrumbs for your teammates.

## When You Finish

1. Update your row in Active Work (status: done, or remove it)
2. Log a message that you've finished
3. Add any non-obvious learnings to the Discoveries section
4. If you made decisions that affect others, add to the Decisions section
5. When you commit, include context in your commit message — what you learned,
   why you made the choices you did, any gotchas future developers should know.
   Commit messages are the permanent record; the coordination file is ephemeral.

## Conflict Resolution

If you need a file another agent is using:
1. Post in the Log asking if they can release it
2. Wait for their response before proceeding
3. If urgent, explain why in your message

## Requesting User Attention

When you need user input (permissions, confirmations, etc.):
1. BEFORE asking, announce in the Log: `[HH:MM] your-name: [ATTENTION] Need user input for <reason>`
2. Wait briefly for other agents to finish their current attention requests
3. Other agents should pause requests when they see another agent waiting
4. After receiving input, log: `[HH:MM] your-name: [ATTENTION] Done, input received`

This prevents multiple agents from asking for input simultaneously.

## Important Files

- Coordination: `%s`
- Shared context: `.magnus-context.md` (if exists)

Remember: Check the coordination file periodically, especially before major changes.
" magnus-coord-file magnus-coord-file))

;;; Coordination skill

(defun magnus-coord-skill-path (directory)
  "Get the coordination skill file path for DIRECTORY."
  (expand-file-name magnus-coord-skill-file directory))

(defun magnus-coord-ensure-skill (directory)
  "Ensure the coordinate skill file exists in DIRECTORY."
  (let ((file (magnus-coord-skill-path directory)))
    (unless (file-exists-p file)
      (magnus-coord--create-skill file))
    file))

(defun magnus-coord--create-skill (file)
  "Create the coordination skill FILE."
  (let ((dir (file-name-directory file)))
    (unless (file-exists-p dir)
      (make-directory dir t)))
  (with-temp-file file
    (insert (magnus-coord--skill-content))))

(defun magnus-coord--skill-content ()
  "Generate the coordination skill content."
  (format "# Coordination Check-in

When you run /coordinate, perform these steps in order:

## Steps

1. **Read the coordination file**: Open and read `%s` completely.
2. **Review active work**: Check the Active Work table. Note which agents are working on what files.
3. **Identify conflicts**: Compare your planned work against the table. Flag any file overlaps.
4. **Announce your claims**: Update the Active Work table with your row:
   - Your agent name
   - The area you are working on
   - Status: `in-progress`
   - Files you will touch (comma-separated)
5. **Log your check-in**: Append a message to the Log section:
   ```
   [HH:MM] your-name: Checked in. Working on <area>. Files: <list>.
   ```
6. **Resolve conflicts**: If you found conflicts in step 3, @mention the conflicting agent in the Log section and wait for acknowledgment before proceeding.

## After Completing a Task

1. Update your Active Work row: change status to `done` or remove it.
2. Log completion: `[HH:MM] your-name: Completed <task>. Files released: <list>.`
3. **Debrief**: Add anything you learned to the Discoveries section — gotchas, patterns, API quirks, things that surprised you. Your teammates will thank you.
4. If you made architectural decisions, add them to the Decisions section.
5. **Commit with context**: When committing, write a message that captures the *why* — what you learned, trade-offs you considered, gotchas you hit. This is the permanent record of your work.

## Important

- Always use the current time (HH:MM format) in log entries.
- Never modify another agent's Active Work row.
- If you are blocked by another agent, @mention them — they will be notified automatically.
- Read the Discoveries section when you check in — other agents may have learned something that helps you.
" magnus-coord-file))

;;; Parsing coordination file

(defun magnus-coord-parse (directory)
  "Parse the coordination file in DIRECTORY.
Returns a plist with :active, :log, and :decisions."
  (let ((file (magnus-coord-file-path directory)))
    (if (file-exists-p file)
        (with-temp-buffer
          (insert-file-contents file)
          (magnus-coord--parse-buffer))
      (list :active nil :log nil :decisions nil))))

(defun magnus-coord--parse-buffer ()
  "Parse the current buffer as a coordination file."
  (let ((active (magnus-coord--parse-active-table))
        (log (magnus-coord--parse-log))
        (decisions (magnus-coord--parse-decisions)))
    (list :active active :log log :decisions decisions)))

(defun magnus-coord--parse-active-table ()
  "Parse the Active Work table from current buffer."
  (save-excursion
    (goto-char (point-min))
    (let (entries)
      (when (re-search-forward "^## Active Work" nil t)
        ;; Skip to table content (past header rows)
        (when (re-search-forward "^|[-|]+" nil t)
          (forward-line 1)
          (while (looking-at "^| *\\([^|]+\\) *| *\\([^|]+\\) *| *\\([^|]+\\) *| *\\([^|]*\\) *|")
            (let ((agent (string-trim (match-string 1)))
                  (area (string-trim (match-string 2)))
                  (status (string-trim (match-string 3)))
                  (files (string-trim (match-string 4))))
              (unless (string-empty-p agent)
                (push (list :agent agent :area area :status status :files files)
                      entries)))
            (forward-line 1))))
      (nreverse entries))))

(defun magnus-coord--parse-log ()
  "Parse the Log section from current buffer."
  (save-excursion
    (goto-char (point-min))
    (let (entries)
      (when (re-search-forward "^## Log" nil t)
        (let ((section-end (save-excursion
                            (if (re-search-forward "^## " nil t)
                                (match-beginning 0)
                              (point-max)))))
          (while (re-search-forward "^\\[\\([0-9:]+\\)\\] \\([^:]+\\): \\(.+\\)$" section-end t)
            (push (list :time (match-string 1)
                       :agent (match-string 2)
                       :message (match-string 3))
                  entries))))
      (nreverse entries))))

(defun magnus-coord--parse-decisions ()
  "Parse the Decisions section from current buffer."
  (save-excursion
    (goto-char (point-min))
    (let (entries)
      (when (re-search-forward "^## Decisions" nil t)
        (while (re-search-forward "^- \\(.+\\)$" nil t)
          (push (match-string 1) entries)))
      (nreverse entries))))

;;; Writing to coordination file

(defun magnus-coord-add-log (directory agent message)
  "Add a log MESSAGE from AGENT to coordination file in DIRECTORY."
  (let ((file (magnus-coord-ensure-file directory))
        (time (format-time-string "%H:%M")))
    (with-temp-buffer
      (insert-file-contents file)
      (goto-char (point-min))
      (if (re-search-forward "^## Log\n+" nil t)
          (progn
            ;; Skip HTML comments (single or multi-line) and blank lines
            (while (and (not (eobp))
                        (or (looking-at "^$")
                            (looking-at "^<!--")))
              (if (looking-at "^<!--")
                  ;; Jump past closing -->, whether same line or later
                  (if (re-search-forward "-->" nil t)
                      (forward-line 1)
                    (goto-char (point-max)))
                (forward-line 1)))
            (insert (format "[%s] %s: %s\n\n" time agent message)))
        ;; No Log section, append at end
        (goto-char (point-max))
        (insert (format "\n[%s] %s: %s\n" time agent message)))
      (magnus-coord--write-file-atomic file))))

(defun magnus-coord-update-active (directory agent area status files)
  "Update AGENT's entry in the Active Work table in DIRECTORY.
AREA is what they're working on, STATUS is their status,
FILES is a list of files they're touching."
  (let ((file (magnus-coord-ensure-file directory))
        (files-str (if (listp files) (string-join files ", ") files)))
    (with-temp-buffer
      (insert-file-contents file)
      (goto-char (point-min))
      (if (re-search-forward "^## Active Work" nil t)
          (let ((table-start (save-excursion
                              (re-search-forward "^|[-|]+" nil t)
                              (forward-line 1)
                              (point)))
                (table-end (save-excursion
                            (if (re-search-forward "^## " nil t)
                                (match-beginning 0)
                              (point-max))))
                (found nil))
            ;; Look for existing entry
            (goto-char table-start)
            (while (and (< (point) table-end)
                        (not found)
                        (looking-at "^| *\\([^|]+\\) *|"))
              (if (string= (string-trim (match-string 1)) agent)
                  (progn
                    (setq found t)
                    (delete-region (line-beginning-position)
                                  (1+ (line-end-position)))
                    (unless (string= status "done")
                      (insert (format "| %s | %s | %s | %s |\n"
                                     agent area status files-str))))
                (forward-line 1)))
            ;; Add new entry if not found
            (unless (or found (string= status "done"))
              (goto-char table-start)
              (insert (format "| %s | %s | %s | %s |\n"
                             agent area status files-str))))
        ;; No Active Work section, create one
        (goto-char (point-min))
        (when (re-search-forward "^# " nil t)
          (end-of-line)
          (insert "\n\n## Active Work\n\n")
          (insert "| Agent | Area | Status | Files |\n")
          (insert "|-------|------|--------|-------|\n")
          (insert (format "| %s | %s | %s | %s |\n"
                         agent area status files-str))))
      (magnus-coord--write-file-atomic file))))

(defun magnus-coord-clear-agent (directory agent)
  "Remove AGENT from the Active Work table in DIRECTORY."
  (magnus-coord-update-active directory agent "" "done" ""))

(defun magnus-coord-mark-session-end (directory)
  "Mark the end of a session in DIRECTORY's coordination file.
Appends a log entry instead of clearing sections, preserving
Discoveries and Decisions for future agents."
  (magnus-coord-add-log directory "Magnus" "Session ended"))

(defun magnus-coord-reconcile (directory)
  "Reconcile the Active Work table in DIRECTORY with live instances.
Removes entries for agents not in the Magnus registry, and entries
with stale statuses like done, died, finished, completed, stopped."
  (let* ((file (magnus-coord-file-path directory))
         (live-names (mapcar #'magnus-instance-name
                             (cl-remove-if-not
                              (lambda (inst)
                                (string= (magnus-instance-directory inst) directory))
                              (magnus-instances-list)))))
    (when (file-exists-p file)
      (with-temp-buffer
        (insert-file-contents file)
        (goto-char (point-min))
        (when (re-search-forward "^## Active Work" nil t)
          (when (re-search-forward "^|[-|]+" nil t)
            (forward-line 1)
            (while (looking-at "^| *\\([^|]+\\) *| *\\([^|]+\\) *| *\\([^|]+\\) *|")
              (let ((agent (string-trim (match-string 1)))
                    (status (string-trim (match-string 3))))
                (if (or (not (member agent live-names))
                        (string-match-p "done\\|died\\|finished\\|completed\\|stopped"
                                        status))
                    (delete-region (line-beginning-position)
                                  (min (1+ (line-end-position)) (point-max)))
                  (forward-line 1))))))
        (magnus-coord--write-file-atomic file)))))

(defun magnus-coord-reconcile-all ()
  "Reconcile coordination files for all project directories."
  (let ((dirs (delete-dups
               (mapcar #'magnus-instance-directory (magnus-instances-list)))))
    (dolist (dir dirs)
      (magnus-coord-reconcile dir))))

;;; Agent registration

(defun magnus-coord-register-agent (directory instance)
  "Register INSTANCE as active in DIRECTORY's coordination file."
  (let ((name (magnus-instance-name instance)))
    (magnus-coord-ensure-file directory)
    (magnus-coord-ensure-instructions directory)
    (magnus-coord-ensure-skill directory)
    (magnus-coord-add-log directory name "Joined the session")
    ;; Start polling for @mentions and DMs if not already
    (unless (member directory magnus-coord--watched-dirs)
      (magnus-coord-start-watching directory))
    ;; Track session start time for retrospectives
    (unless (gethash directory magnus-coord--session-start-times)
      (puthash directory (float-time) magnus-coord--session-start-times))))

(defun magnus-coord-unregister-agent (directory instance)
  "Unregister INSTANCE from DIRECTORY's coordination file."
  (let ((name (magnus-instance-name instance)))
    (magnus-coord-clear-agent directory name)
    (magnus-coord-add-log directory name "Left the session")
    ;; If no agents remain, clean up for next session
    (let ((remaining (cl-count-if
                      (lambda (inst)
                        (and (not (eq inst instance))
                             (string= (magnus-instance-directory inst) directory)))
                      (magnus-instances-list))))
      (when (zerop remaining)
        (magnus-coord-generate-retro directory)
        (remhash directory magnus-coord--session-start-times)
        (magnus-coord-stop-watching directory)
        (magnus-coord-mark-session-end directory)))))

;;; Display

(defun magnus-coord-format-active (parsed)
  "Format the :active entries from PARSED for display."
  (let ((active (plist-get parsed :active)))
    (if active
        (mapconcat
         (lambda (entry)
           (format "  %s: %s [%s]"
                   (propertize (plist-get entry :agent) 'face 'magnus-status-instance-name)
                   (plist-get entry :area)
                   (propertize (plist-get entry :status)
                              'face (if (string= (plist-get entry :status) "in-progress")
                                       'magnus-status-running
                                     'magnus-status-instance-dir))))
         active
         "\n")
      (propertize "  No active work" 'face 'magnus-status-empty-hint))))

(defun magnus-coord-format-log (parsed &optional limit)
  "Format the :log entries from PARSED for display.
Show at most LIMIT entries (default 5)."
  (let* ((log (plist-get parsed :log))
         (entries (if limit (seq-take log limit) log)))
    (if entries
        (mapconcat
         (lambda (entry)
           (format "  [%s] %s: %s"
                   (propertize (plist-get entry :time) 'face 'magnus-status-instance-dir)
                   (propertize (plist-get entry :agent) 'face 'magnus-status-instance-name)
                   (plist-get entry :message)))
         entries
         "\n")
      (propertize "  No messages yet" 'face 'magnus-status-empty-hint))))

;;; Interactive commands

(defun magnus-coord-open (directory)
  "Open the coordination file for DIRECTORY."
  (interactive (list (magnus-coord--get-directory)))
  (find-file (magnus-coord-ensure-file directory)))

(defun magnus-coord-open-instructions (directory)
  "Open the instructions file for DIRECTORY."
  (interactive (list (magnus-coord--get-directory)))
  (find-file (magnus-coord-ensure-instructions directory)))

(defun magnus-coord--get-directory ()
  "Get directory for coordination, prompting if needed."
  (or (when (bound-and-true-p magnus-context--directory)
        magnus-context--directory)
      (when (fboundp 'project-current)
        (when-let ((project (project-current 'maybe)))
          (project-root project)))
      (read-directory-name "Project directory: ")))

(provide 'magnus-coord)
;;; magnus-coord.el ends here
