;;; romkan.el --- Romaji/Kana conversion library -*- lexical-binding: t; -*-
;;
;; Copyright (C) 2023, 2025 Giovanni Crisalfi
;; Copyright (C) 2012, 2013 Mort Yao
;; Copyright (C) 2010 Masato Hagiwara
;; Copyright (C) 2001 Satoru Takabayashi
;;
;; Author: gicrisf <giovanni.crisalfi@protonmail.com>
;; Maintainer: gicrisf <giovanni.crisalfi@protonmail.com>
;; Created: July 21, 2023
;; Modified: August 14, 2025
;; Package-Version: 20251019.1646
;; Package-Revision: 81cf9448450b
;; Keywords: i18n languages
;; Homepage: https://github.com/gicrisf/romkan.el
;; Package-Requires: ((emacs "24.3"))
;; SPDX-License-Identifier: BSD-3-Clause
;;
;; This file is not part of GNU Emacs.
;;
;; Redistribution and use in source and binary forms, with or without
;; modification, are permitted provided that the following conditions are met:
;;
;; 1. Redistributions of source code must retain the above copyright notice,
;;    this list of conditions and the following disclaimer.
;;
;; 2. Redistributions in binary form must reproduce the above copyright notice,
;;    this list of conditions and the following disclaimer in the documentation
;;    and/or other materials provided with the distribution.
;;
;; 3. Neither the name of the copyright holders nor the names of its
;;    contributors may be used to endorse or promote products derived from this
;;    software without specific prior written permission.
;;
;; THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
;; AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
;; IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
;; ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDERS OR CONTRIBUTORS BE
;; LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
;; CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
;; SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
;; INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
;; CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
;; ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
;; POSSIBILITY OF SUCH DAMAGE.
;;
;;; Commentary:
;;
;; romkan.el provides bidirectional conversion between Japanese Romaji
;; (ローマ字) and Kana (仮名) scripts. It supports both Hiragana (平仮名)
;; and Katakana (片仮名), as well as conversion between Hepburn
;; (ヘボン式) and Kunrei-shiki (訓令式) romanization systems.
;;
;; Usage:
;;
;;   (romkan-to-hiragana "konnichiwa")  ;; => "こんにちわ"
;;   (romkan-to-katakana "konnichiwa")  ;; => "コンニチワ"
;;   (romkan-to-roma "こんにちは")       ;; => "konnichiha"
;;   (romkan-to-hepburn "konnitiha")    ;; => "konnichiha"
;;   (romkan-to-kunrei "konnichiha")    ;; => "konnitiha"
;;
;; The package uses static conversion tables for efficiency and supports
;; common Romaji patterns including double consonants (っ/ッ), long vowels,
;; and special characters.
;;
;;; Code:

(require 'cl-lib)

(defconst romkan-kanrom '(("ァ" "xa")
                          ("ア" "a")
                          ("ィ" "xi")
                          ("イ" "i")
                          ("ゥ" "xu")
                          ("ウ" "u")
                          ("ヴ" "vu")
                          ("ヴァ" "va")
                          ("ヴィ" "vi")
                          ("ヴェ" "ve")
                          ("ヴォ" "vo")
                          ("ェ" "xe")
                          ("エ" "e")
                          ("ォ" "xo")
                          ("オ" "o")
                          ("カ" "ka")
                          ("ガ" "ga")
                          ("キ" "ki")
                          ("キャ" "kya")
                          ("キュ" "kyu")
                          ("キョ" "kyo")
                          ("ギ" "gi")
                          ("ギャ" "gya")
                          ("ギュ" "gyu")
                          ("ギョ" "gyo")
                          ("ク" "ku")
                          ("グ" "gu")
                          ("ケ" "ke")
                          ("ゲ" "ge")
                          ("コ" "ko")
                          ("ゴ" "go")
                          ("サ" "sa")
                          ("ザ" "za")
                          ("シ" "shi")
                          ("シャ" "sha")
                          ("シュ" "shu")
                          ("ショ" "sho")
                          ("シェ" "she")
                          ("ジ" "ji")
                          ("ジャ" "ja")
                          ("ジュ" "ju")
                          ("ジョ" "jo")
                          ("ス" "su")
                          ("ズ" "zu")
                          ("セ" "se")
                          ("ゼ" "ze")
                          ("ソ" "so")
                          ("ゾ" "zo")
                          ("タ" "ta")
                          ("ダ" "da")
                          ("チ" "chi")
                          ("チャ" "cha")
                          ("チュ" "chu")
                          ("チョ" "cho")
                          ("ヂ" "di")
                          ("ヂャ" "dya")
                          ("ヂュ" "dyu")
                          ("ヂョ" "dyo")
                          ("ティ" "ti")
                          ("ッ" "xtsu")
                          ("ッヴ" "vvu")
                          ("ッヴァ" "vva")
                          ("ッヴィ" "vvi")
                          ("ッヴェ" "vve")
                          ("ッヴォ" "vvo")
                          ("ッカ" "kka")
                          ("ッガ" "gga")
                          ("ッキ" "kki")
                          ("ッキャ" "kkya")
                          ("ッキュ" "kkyu")
                          ("ッキョ" "kkyo")
                          ("ッギ" "ggi")
                          ("ッギャ" "ggya")
                          ("ッギュ" "ggyu")
                          ("ッギョ" "ggyo")
                          ("ック" "kku")
                          ("ッグ" "ggu")
                          ("ッケ" "kke")
                          ("ッゲ" "gge")
                          ("ッコ" "kko")
                          ("ッゴ" "ggo")
                          ("ッサ" "ssa")
                          ("ッザ" "zza")
                          ("ッシ" "sshi")
                          ("ッシャ" "ssha")
                          ("ッシュ" "sshu")
                          ("ッショ" "ssho")
                          ("ッシェ" "sshe")
                          ("ッジ" "jji")
                          ("ッジャ" "jja")
                          ("ッジュ" "jju")
                          ("ッジョ" "jjo")
                          ("ッス" "ssu")
                          ("ッズ" "zzu")
                          ("ッセ" "sse")
                          ("ッゼ" "zze")
                          ("ッソ" "sso")
                          ("ッゾ" "zzo")
                          ("ッタ" "tta")
                          ("ッダ" "dda")
                          ("ッチ" "cchi")
                          ("ッティ" "tti")
                          ("ッチャ" "ccha")
                          ("ッチュ" "cchu")
                          ("ッチョ" "ccho")
                          ("ッヂ" "ddi")
                          ("ッヂャ" "ddya")
                          ("ッヂュ" "ddyu")
                          ("ッヂョ" "ddyo")
                          ("ッツ" "ttsu")
                          ("ッヅ" "ddu")
                          ("ッテ" "tte")
                          ("ッデ" "dde")
                          ("ット" "tto")
                          ("ッド" "ddo")
                          ("ッドゥ" "ddu")
                          ("ッハ" "hha")
                          ("ッバ" "bba")
                          ("ッパ" "ppa")
                          ("ッヒ" "hhi")
                          ("ッヒャ" "hhya")
                          ("ッヒュ" "hhyu")
                          ("ッヒョ" "hhyo")
                          ("ッビ" "bbi")
                          ("ッビャ" "bbya")
                          ("ッビュ" "bbyu")
                          ("ッビョ" "bbyo")
                          ("ッピ" "ppi")
                          ("ッピャ" "ppya")
                          ("ッピュ" "ppyu")
                          ("ッピョ" "ppyo")
                          ("ッフ" "ffu")
                          ("ッフュ" "ffu")
                          ("ッファ" "ffa")
                          ("ッフィ" "ffi")
                          ("ッフェ" "ffe")
                          ("ッフォ" "ffo")
                          ("ッブ" "bbu")
                          ("ップ" "ppu")
                          ("ッヘ" "hhe")
                          ("ッベ" "bbe")
                          ("ッペ" "ppe")
                          ("ッホ" "hho")
                          ("ッボ" "bbo")
                          ("ッポ" "ppo")
                          ("ッヤ" "yya")
                          ("ッユ" "yyu")
                          ("ッヨ" "yyo")
                          ("ッラ" "rra")
                          ("ッリ" "rri")
                          ("ッリャ" "rrya")
                          ("ッリュ" "rryu")
                          ("ッリョ" "rryo")
                          ("ッル" "rru")
                          ("ッレ" "rre")
                          ("ッロ" "rro")
                          ("ツ" "tsu")
                          ("ヅ" "du")
                          ("テ" "te")
                          ("デ" "de")
                          ("ト" "to")
                          ("ド" "do")
                          ("ドゥ" "du")
                          ("ナ" "na")
                          ("ニ" "ni")
                          ("ニャ" "nya")
                          ("ニュ" "nyu")
                          ("ニョ" "nyo")
                          ("ヌ" "nu")
                          ("ネ" "ne")
                          ("ノ" "no")
                          ("ハ" "ha")
                          ("バ" "ba")
                          ("パ" "pa")
                          ("ヒ" "hi")
                          ("ヒャ" "hya")
                          ("ヒュ" "hyu")
                          ("ヒョ" "hyo")
                          ("ビ" "bi")
                          ("ビャ" "bya")
                          ("ビュ" "byu")
                          ("ビョ" "byo")
                          ("ピ" "pi")
                          ("ピャ" "pya")
                          ("ピュ" "pyu")
                          ("ピョ" "pyo")
                          ("フ" "fu")
                          ("ファ" "fa")
                          ("フィ" "fi")
                          ("フェ" "fe")
                          ("フォ" "fo")
                          ("フュ" "fu")
                          ("ブ" "bu")
                          ("プ" "pu")
                          ("ヘ" "he")
                          ("ベ" "be")
                          ("ペ" "pe")
                          ("ホ" "ho")
                          ("ボ" "bo")
                          ("ポ" "po")
                          ("マ" "ma")
                          ("ミ" "mi")
                          ("ミャ" "mya")
                          ("ミュ" "myu")
                          ("ミョ" "myo")
                          ("ム" "mu")
                          ("メ" "me")
                          ("モ" "mo")
                          ("ャ" "xya")
                          ("ヤ" "ya")
                          ("ュ" "xyu")
                          ("ユ" "yu")
                          ("ョ" "xyo")
                          ("ヨ" "yo")
                          ("ラ" "ra")
                          ("リ" "ri")
                          ("リャ" "rya")
                          ("リュ" "ryu")
                          ("リョ" "ryo")
                          ("ル" "ru")
                          ("レ" "re")
                          ("ロ" "ro")
                          ("ヮ" "xwa")
                          ("ワ" "wa")
                          ("ウィ" "wi")
                          ("ヰ" "wi")
                          ("ヱ" "we")
                          ("ウェ" "we")
                          ("ヲ" "wo")
                          ("ウォ" "wo")
                          ("ン" "n'")
                          ("ディ" "di")
                          ("ー" "-")
                          ("チェ" "che")
                          ("ッチェ" "cche")
                          ("ジェ" "je")))

(defconst romkan-romkan '(("xa" "ァ")
                          ("a" "ア")
                          ("xi" "ィ")
                          ("i" "イ")
                          ("xu" "ゥ")
                          ("u" "ウ")
                          ("vu" "ヴ")
                          ("va" "ヴァ")
                          ("vi" "ヴィ")
                          ("ve" "ヴェ")
                          ("vo" "ヴォ")
                          ("xe" "ェ")
                          ("e" "エ")
                          ("xo" "ォ")
                          ("o" "オ")
                          ("ka" "カ")
                          ("ga" "ガ")
                          ("ki" "キ")
                          ("kya" "キャ")
                          ("kyu" "キュ")
                          ("kyo" "キョ")
                          ("gi" "ギ")
                          ("gya" "ギャ")
                          ("gyu" "ギュ")
                          ("gyo" "ギョ")
                          ("ku" "ク")
                          ("gu" "グ")
                          ("ke" "ケ")
                          ("ge" "ゲ")
                          ("ko" "コ")
                          ("go" "ゴ")
                          ("sa" "サ")
                          ("za" "ザ")
                          ("si" "シ")
                          ("sya" "シャ")
                          ("syu" "シュ")
                          ("syo" "ショ")
                          ("sye" "シェ")
                          ("zi" "ジ")
                          ("zya" "ジャ")
                          ("zyu" "ジュ")
                          ("zyo" "ジョ")
                          ("su" "ス")
                          ("zu" "ズ")
                          ("se" "セ")
                          ("ze" "ゼ")
                          ("so" "ソ")
                          ("zo" "ゾ")
                          ("ta" "タ")
                          ("da" "ダ")
                          ("ti" "チ")
                          ("tya" "チャ")
                          ("tyu" "チュ")
                          ("tyo" "チョ")
                          ("di" "ヂ")
                          ("dya" "ヂャ")
                          ("dyu" "ヂュ")
                          ("dyo" "ヂョ")
                          ("xtu" "ッ")
                          ("vvu" "ッヴ")
                          ("vva" "ッヴァ")
                          ("vvi" "ッヴィ")
                          ("vve" "ッヴェ")
                          ("vvo" "ッヴォ")
                          ("kka" "ッカ")
                          ("gga" "ッガ")
                          ("kki" "ッキ")
                          ("kkya" "ッキャ")
                          ("kkyu" "ッキュ")
                          ("kkyo" "ッキョ")
                          ("ggi" "ッギ")
                          ("ggya" "ッギャ")
                          ("ggyu" "ッギュ")
                          ("ggyo" "ッギョ")
                          ("kku" "ック")
                          ("ggu" "ッグ")
                          ("kke" "ッケ")
                          ("gge" "ッゲ")
                          ("kko" "ッコ")
                          ("ggo" "ッゴ")
                          ("ssa" "ッサ")
                          ("zza" "ッザ")
                          ("ssi" "ッシ")
                          ("ssya" "ッシャ")
                          ("ssyu" "ッシュ")
                          ("ssyo" "ッショ")
                          ("ssye" "ッシェ")
                          ("zzi" "ッジ")
                          ("zzya" "ッジャ")
                          ("zzyu" "ッジュ")
                          ("zzyo" "ッジョ")
                          ("ssu" "ッス")
                          ("zzu" "ッズ")
                          ("sse" "ッセ")
                          ("zze" "ッゼ")
                          ("sso" "ッソ")
                          ("zzo" "ッゾ")
                          ("tta" "ッタ")
                          ("dda" "ッダ")
                          ("tti" "ッティ")
                          ("ttya" "ッチャ")
                          ("ttyu" "ッチュ")
                          ("ttyo" "ッチョ")
                          ("ddi" "ッヂ")
                          ("ddya" "ッヂャ")
                          ("ddyu" "ッヂュ")
                          ("ddyo" "ッヂョ")
                          ("ttu" "ッツ")
                          ("ddu" "ッドゥ")
                          ("tte" "ッテ")
                          ("dde" "ッデ")
                          ("tto" "ット")
                          ("ddo" "ッド")
                          ("hha" "ッハ")
                          ("bba" "ッバ")
                          ("ppa" "ッパ")
                          ("hhi" "ッヒ")
                          ("hhya" "ッヒャ")
                          ("hhyu" "ッヒュ")
                          ("hhyo" "ッヒョ")
                          ("bbi" "ッビ")
                          ("bbya" "ッビャ")
                          ("bbyu" "ッビュ")
                          ("bbyo" "ッビョ")
                          ("ppi" "ッピ")
                          ("ppya" "ッピャ")
                          ("ppyu" "ッピュ")
                          ("ppyo" "ッピョ")
                          ("hhu" "ッフ")
                          ("ffu" "ッフュ")
                          ("ffa" "ッファ")
                          ("ffi" "ッフィ")
                          ("ffe" "ッフェ")
                          ("ffo" "ッフォ")
                          ("bbu" "ッブ")
                          ("ppu" "ップ")
                          ("hhe" "ッヘ")
                          ("bbe" "ッベ")
                          ("ppe" "ッペ")
                          ("hho" "ッホ")
                          ("bbo" "ッボ")
                          ("ppo" "ッポ")
                          ("yya" "ッヤ")
                          ("yyu" "ッユ")
                          ("yyo" "ッヨ")
                          ("rra" "ッラ")
                          ("rri" "ッリ")
                          ("rrya" "ッリャ")
                          ("rryu" "ッリュ")
                          ("rryo" "ッリョ")
                          ("rru" "ッル")
                          ("rre" "ッレ")
                          ("rro" "ッロ")
                          ("tu" "ツ")
                          ("du" "ヅ")
                          ("te" "テ")
                          ("de" "デ")
                          ("to" "ト")
                          ("do" "ド")
                          ("na" "ナ")
                          ("ni" "ニ")
                          ("nya" "ニャ")
                          ("nyu" "ニュ")
                          ("nyo" "ニョ")
                          ("nu" "ヌ")
                          ("ne" "ネ")
                          ("no" "ノ")
                          ("ha" "ハ")
                          ("ba" "バ")
                          ("pa" "パ")
                          ("hi" "ヒ")
                          ("hya" "ヒャ")
                          ("hyu" "ヒュ")
                          ("hyo" "ヒョ")
                          ("bi" "ビ")
                          ("bya" "ビャ")
                          ("byu" "ビュ")
                          ("byo" "ビョ")
                          ("pi" "ピ")
                          ("pya" "ピャ")
                          ("pyu" "ピュ")
                          ("pyo" "ピョ")
                          ("hu" "フ")
                          ("fa" "ファ")
                          ("fi" "フィ")
                          ("fe" "フェ")
                          ("fo" "フォ")
                          ("fu" "フ")
                          ("bu" "ブ")
                          ("pu" "プ")
                          ("he" "ヘ")
                          ("be" "ベ")
                          ("pe" "ペ")
                          ("ho" "ホ")
                          ("bo" "ボ")
                          ("po" "ポ")
                          ("ma" "マ")
                          ("mi" "ミ")
                          ("mya" "ミャ")
                          ("myu" "ミュ")
                          ("myo" "ミョ")
                          ("mu" "ム")
                          ("me" "メ")
                          ("mo" "モ")
                          ("xya" "ャ")
                          ("ya" "ヤ")
                          ("xyu" "ュ")
                          ("yu" "ユ")
                          ("xyo" "ョ")
                          ("yo" "ヨ")
                          ("ra" "ラ")
                          ("ri" "リ")
                          ("rya" "リャ")
                          ("ryu" "リュ")
                          ("ryo" "リョ")
                          ("ru" "ル")
                          ("re" "レ")
                          ("ro" "ロ")
                          ("xwa" "ヮ")
                          ("wa" "ワ")
                          ("wi" "ウィ")
                          ("we" "ウェ")
                          ("wo" "ヲ")
                          ("n" "ン")
                          ("n'" "ン")
                          ("dyi" "ディ")
                          ("-" "ー")
                          ("tye" "チェ")
                          ("ttye" "ッチェ")
                          ("zye" "ジェ")
                          ("shi" "シ")
                          ("sha" "シャ")
                          ("shu" "シュ")
                          ("sho" "ショ")
                          ("she" "シェ")
                          ("ji" "ジ")
                          ("ja" "ジャ")
                          ("ju" "ジュ")
                          ("jo" "ジョ")
                          ("chi" "チ")
                          ("cha" "チャ")
                          ("chu" "チュ")
                          ("cho" "チョ")
                          ("xtsu" "ッ")
                          ("sshi" "ッシ")
                          ("ssha" "ッシャ")
                          ("sshu" "ッシュ")
                          ("ssho" "ッショ")
                          ("sshe" "ッシェ")
                          ("jji" "ッジ")
                          ("jja" "ッジャ")
                          ("jju" "ッジュ")
                          ("jjo" "ッジョ")
                          ("cchi" "ッチ")
                          ("ccha" "ッチャ")
                          ("cchu" "ッチュ")
                          ("ccho" "ッチョ")
                          ("ttsu" "ッツ")
                          ("tsu" "ツ")
                          ("che" "チェ")
                          ("cche" "ッチェ")
                          ("je" "ジェ")))

(defconst romkan-kanrom-h '(("ぁ" "xa")
                            ("あ" "a")
                            ("ぃ" "xi")
                            ("い" "i")
                            ("ぅ" "xu")
                            ("う" "u")
                            ("う゛" "vu")
                            ("う゛ぁ" "va")
                            ("う゛ぃ" "vi")
                            ("う゛ぇ" "ve")
                            ("う゛ぉ" "vo")
                            ("ぇ" "xe")
                            ("え" "e")
                            ("ぉ" "xo")
                            ("お" "o")
                            ("か" "ka")
                            ("が" "ga")
                            ("き" "ki")
                            ("きゃ" "kya")
                            ("きゅ" "kyu")
                            ("きょ" "kyo")
                            ("ぎ" "gi")
                            ("ぎゃ" "gya")
                            ("ぎゅ" "gyu")
                            ("ぎょ" "gyo")
                            ("く" "ku")
                            ("ぐ" "gu")
                            ("け" "ke")
                            ("げ" "ge")
                            ("こ" "ko")
                            ("ご" "go")
                            ("さ" "sa")
                            ("ざ" "za")
                            ("し" "shi")
                            ("しゃ" "sha")
                            ("しゅ" "shu")
                            ("しょ" "sho")
                            ("じ" "ji")
                            ("じゃ" "ja")
                            ("じゅ" "ju")
                            ("じょ" "jo")
                            ("す" "su")
                            ("ず" "zu")
                            ("せ" "se")
                            ("ぜ" "ze")
                            ("そ" "so")
                            ("ぞ" "zo")
                            ("た" "ta")
                            ("だ" "da")
                            ("ち" "chi")
                            ("ちゃ" "cha")
                            ("ちゅ" "chu")
                            ("ちょ" "cho")
                            ("ぢ" "di")
                            ("ぢゃ" "dya")
                            ("ぢゅ" "dyu")
                            ("ぢょ" "dyo")
                            ("っ" "xtsu")
                            ("っう゛" "vvu")
                            ("っう゛ぁ" "vva")
                            ("っう゛ぃ" "vvi")
                            ("っう゛ぇ" "vve")
                            ("っう゛ぉ" "vvo")
                            ("っか" "kka")
                            ("っが" "gga")
                            ("っき" "kki")
                            ("っきゃ" "kkya")
                            ("っきゅ" "kkyu")
                            ("っきょ" "kkyo")
                            ("っぎ" "ggi")
                            ("っぎゃ" "ggya")
                            ("っぎゅ" "ggyu")
                            ("っぎょ" "ggyo")
                            ("っく" "kku")
                            ("っぐ" "ggu")
                            ("っけ" "kke")
                            ("っげ" "gge")
                            ("っこ" "kko")
                            ("っご" "ggo")
                            ("っさ" "ssa")
                            ("っざ" "zza")
                            ("っし" "sshi")
                            ("っしゃ" "ssha")
                            ("っしゅ" "sshu")
                            ("っしょ" "ssho")
                            ("っじ" "jji")
                            ("っじゃ" "jja")
                            ("っじゅ" "jju")
                            ("っじょ" "jjo")
                            ("っす" "ssu")
                            ("っず" "zzu")
                            ("っせ" "sse")
                            ("っぜ" "zze")
                            ("っそ" "sso")
                            ("っぞ" "zzo")
                            ("った" "tta")
                            ("っだ" "dda")
                            ("っち" "cchi")
                            ("っちゃ" "ccha")
                            ("っちゅ" "cchu")
                            ("っちょ" "ccho")
                            ("っぢ" "ddi")
                            ("っぢゃ" "ddya")
                            ("っぢゅ" "ddyu")
                            ("っぢょ" "ddyo")
                            ("っつ" "ttsu")
                            ("っづ" "ddu")
                            ("って" "tte")
                            ("っで" "dde")
                            ("っと" "tto")
                            ("っど" "ddo")
                            ("っは" "hha")
                            ("っば" "bba")
                            ("っぱ" "ppa")
                            ("っひ" "hhi")
                            ("っひゃ" "hhya")
                            ("っひゅ" "hhyu")
                            ("っひょ" "hhyo")
                            ("っび" "bbi")
                            ("っびゃ" "bbya")
                            ("っびゅ" "bbyu")
                            ("っびょ" "bbyo")
                            ("っぴ" "ppi")
                            ("っぴゃ" "ppya")
                            ("っぴゅ" "ppyu")
                            ("っぴょ" "ppyo")
                            ("っふ" "ffu")
                            ("っふぁ" "ffa")
                            ("っふぃ" "ffi")
                            ("っふぇ" "ffe")
                            ("っふぉ" "ffo")
                            ("っぶ" "bbu")
                            ("っぷ" "ppu")
                            ("っへ" "hhe")
                            ("っべ" "bbe")
                            ("っぺ" "ppe")
                            ("っほ" "hho")
                            ("っぼ" "bbo")
                            ("っぽ" "ppo")
                            ("っや" "yya")
                            ("っゆ" "yyu")
                            ("っよ" "yyo")
                            ("っら" "rra")
                            ("っり" "rri")
                            ("っりゃ" "rrya")
                            ("っりゅ" "rryu")
                            ("っりょ" "rryo")
                            ("っる" "rru")
                            ("っれ" "rre")
                            ("っろ" "rro")
                            ("つ" "tsu")
                            ("づ" "du")
                            ("て" "te")
                            ("で" "de")
                            ("と" "to")
                            ("ど" "do")
                            ("な" "na")
                            ("に" "ni")
                            ("にゃ" "nya")
                            ("にゅ" "nyu")
                            ("にょ" "nyo")
                            ("ぬ" "nu")
                            ("ね" "ne")
                            ("の" "no")
                            ("は" "ha")
                            ("ば" "ba")
                            ("ぱ" "pa")
                            ("ひ" "hi")
                            ("ひゃ" "hya")
                            ("ひゅ" "hyu")
                            ("ひょ" "hyo")
                            ("び" "bi")
                            ("びゃ" "bya")
                            ("びゅ" "byu")
                            ("びょ" "byo")
                            ("ぴ" "pi")
                            ("ぴゃ" "pya")
                            ("ぴゅ" "pyu")
                            ("ぴょ" "pyo")
                            ("ふ" "fu")
                            ("ふぁ" "fa")
                            ("ふぃ" "fi")
                            ("ふぇ" "fe")
                            ("ふぉ" "fo")
                            ("ぶ" "bu")
                            ("ぷ" "pu")
                            ("へ" "he")
                            ("べ" "be")
                            ("ぺ" "pe")
                            ("ほ" "ho")
                            ("ぼ" "bo")
                            ("ぽ" "po")
                            ("ま" "ma")
                            ("み" "mi")
                            ("みゃ" "mya")
                            ("みゅ" "myu")
                            ("みょ" "myo")
                            ("む" "mu")
                            ("め" "me")
                            ("も" "mo")
                            ("ゃ" "xya")
                            ("や" "ya")
                            ("ゅ" "xyu")
                            ("ゆ" "yu")
                            ("ょ" "xyo")
                            ("よ" "yo")
                            ("ら" "ra")
                            ("り" "ri")
                            ("りゃ" "rya")
                            ("りゅ" "ryu")
                            ("りょ" "ryo")
                            ("る" "ru")
                            ("れ" "re")
                            ("ろ" "ro")
                            ("ゎ" "xwa")
                            ("わ" "wa")
                            ("ゐ" "wi")
                            ("ゑ" "we")
                            ("を" "wo")
                            ("ん" "n'")
                            ("でぃ" "dyi")
                            ("ー" "-")
                            ("ちぇ" "che")
                            ("っちぇ" "cche")
                            ("じぇ" "je")))

(defconst romkan-romkan-h '(("xa" "ぁ")
                            ("a" "あ")
                            ("xi" "ぃ")
                            ("i" "い")
                            ("xu" "ぅ")
                            ("u" "う")
                            ("vu" "う゛")
                            ("va" "う゛ぁ")
                            ("vi" "う゛ぃ")
                            ("ve" "う゛ぇ")
                            ("vo" "う゛ぉ")
                            ("xe" "ぇ")
                            ("e" "え")
                            ("xo" "ぉ")
                            ("o" "お")
                            ("ka" "か")
                            ("ga" "が")
                            ("ki" "き")
                            ("kya" "きゃ")
                            ("kyu" "きゅ")
                            ("kyo" "きょ")
                            ("gi" "ぎ")
                            ("gya" "ぎゃ")
                            ("gyu" "ぎゅ")
                            ("gyo" "ぎょ")
                            ("ku" "く")
                            ("gu" "ぐ")
                            ("ke" "け")
                            ("ge" "げ")
                            ("ko" "こ")
                            ("go" "ご")
                            ("sa" "さ")
                            ("za" "ざ")
                            ("si" "し")
                            ("sya" "しゃ")
                            ("syu" "しゅ")
                            ("syo" "しょ")
                            ("zi" "じ")
                            ("zya" "じゃ")
                            ("zyu" "じゅ")
                            ("zyo" "じょ")
                            ("su" "す")
                            ("zu" "ず")
                            ("se" "せ")
                            ("ze" "ぜ")
                            ("so" "そ")
                            ("zo" "ぞ")
                            ("ta" "た")
                            ("da" "だ")
                            ("ti" "ち")
                            ("tya" "ちゃ")
                            ("tyu" "ちゅ")
                            ("tyo" "ちょ")
                            ("di" "ぢ")
                            ("dya" "ぢゃ")
                            ("dyu" "ぢゅ")
                            ("dyo" "ぢょ")
                            ("xtu" "っ")
                            ("vvu" "っう゛")
                            ("vva" "っう゛ぁ")
                            ("vvi" "っう゛ぃ")
                            ("vve" "っう゛ぇ")
                            ("vvo" "っう゛ぉ")
                            ("kka" "っか")
                            ("gga" "っが")
                            ("kki" "っき")
                            ("kkya" "っきゃ")
                            ("kkyu" "っきゅ")
                            ("kkyo" "っきょ")
                            ("ggi" "っぎ")
                            ("ggya" "っぎゃ")
                            ("ggyu" "っぎゅ")
                            ("ggyo" "っぎょ")
                            ("kku" "っく")
                            ("ggu" "っぐ")
                            ("kke" "っけ")
                            ("gge" "っげ")
                            ("kko" "っこ")
                            ("ggo" "っご")
                            ("ssa" "っさ")
                            ("zza" "っざ")
                            ("ssi" "っし")
                            ("ssya" "っしゃ")
                            ("ssyu" "っしゅ")
                            ("ssyo" "っしょ")
                            ("zzi" "っじ")
                            ("zzya" "っじゃ")
                            ("zzyu" "っじゅ")
                            ("zzyo" "っじょ")
                            ("ssu" "っす")
                            ("zzu" "っず")
                            ("sse" "っせ")
                            ("zze" "っぜ")
                            ("sso" "っそ")
                            ("zzo" "っぞ")
                            ("tta" "った")
                            ("dda" "っだ")
                            ("tti" "っち")
                            ("ttya" "っちゃ")
                            ("ttyu" "っちゅ")
                            ("ttyo" "っちょ")
                            ("ddi" "っぢ")
                            ("ddya" "っぢゃ")
                            ("ddyu" "っぢゅ")
                            ("ddyo" "っぢょ")
                            ("ttu" "っつ")
                            ("ddu" "っづ")
                            ("tte" "って")
                            ("dde" "っで")
                            ("tto" "っと")
                            ("ddo" "っど")
                            ("hha" "っは")
                            ("bba" "っば")
                            ("ppa" "っぱ")
                            ("hhi" "っひ")
                            ("hhya" "っひゃ")
                            ("hhyu" "っひゅ")
                            ("hhyo" "っひょ")
                            ("bbi" "っび")
                            ("bbya" "っびゃ")
                            ("bbyu" "っびゅ")
                            ("bbyo" "っびょ")
                            ("ppi" "っぴ")
                            ("ppya" "っぴゃ")
                            ("ppyu" "っぴゅ")
                            ("ppyo" "っぴょ")
                            ("hhu" "っふ")
                            ("ffa" "っふぁ")
                            ("ffi" "っふぃ")
                            ("ffe" "っふぇ")
                            ("ffo" "っふぉ")
                            ("bbu" "っぶ")
                            ("ppu" "っぷ")
                            ("hhe" "っへ")
                            ("bbe" "っべ")
                            ("ppe" "っぺ")
                            ("hho" "っほ")
                            ("bbo" "っぼ")
                            ("ppo" "っぽ")
                            ("yya" "っや")
                            ("yyu" "っゆ")
                            ("yyo" "っよ")
                            ("rra" "っら")
                            ("rri" "っり")
                            ("rrya" "っりゃ")
                            ("rryu" "っりゅ")
                            ("rryo" "っりょ")
                            ("rru" "っる")
                            ("rre" "っれ")
                            ("rro" "っろ")
                            ("tu" "つ")
                            ("du" "づ")
                            ("te" "て")
                            ("de" "で")
                            ("to" "と")
                            ("do" "ど")
                            ("na" "な")
                            ("ni" "に")
                            ("nya" "にゃ")
                            ("nyu" "にゅ")
                            ("nyo" "にょ")
                            ("nu" "ぬ")
                            ("ne" "ね")
                            ("no" "の")
                            ("ha" "は")
                            ("ba" "ば")
                            ("pa" "ぱ")
                            ("hi" "ひ")
                            ("hya" "ひゃ")
                            ("hyu" "ひゅ")
                            ("hyo" "ひょ")
                            ("bi" "び")
                            ("bya" "びゃ")
                            ("byu" "びゅ")
                            ("byo" "びょ")
                            ("pi" "ぴ")
                            ("pya" "ぴゃ")
                            ("pyu" "ぴゅ")
                            ("pyo" "ぴょ")
                            ("hu" "ふ")
                            ("fa" "ふぁ")
                            ("fi" "ふぃ")
                            ("fe" "ふぇ")
                            ("fo" "ふぉ")
                            ("bu" "ぶ")
                            ("pu" "ぷ")
                            ("he" "へ")
                            ("be" "べ")
                            ("pe" "ぺ")
                            ("ho" "ほ")
                            ("bo" "ぼ")
                            ("po" "ぽ")
                            ("ma" "ま")
                            ("mi" "み")
                            ("mya" "みゃ")
                            ("myu" "みゅ")
                            ("myo" "みょ")
                            ("mu" "む")
                            ("me" "め")
                            ("mo" "も")
                            ("xya" "ゃ")
                            ("ya" "や")
                            ("xyu" "ゅ")
                            ("yu" "ゆ")
                            ("xyo" "ょ")
                            ("yo" "よ")
                            ("ra" "ら")
                            ("ri" "り")
                            ("rya" "りゃ")
                            ("ryu" "りゅ")
                            ("ryo" "りょ")
                            ("ru" "る")
                            ("re" "れ")
                            ("ro" "ろ")
                            ("xwa" "ゎ")
                            ("wa" "わ")
                            ("wi" "うぃ")
                            ("we" "うぇ")
                            ("wo" "を")
                            ("n" "ん")
                            ("n'" "ん")
                            ("dyi" "でぃ")
                            ("-" "ー")
                            ("tye" "ちぇ")
                            ("ttye" "っちぇ")
                            ("zye" "じぇ")
                            ("shi" "し")
                            ("sha" "しゃ")
                            ("shu" "しゅ")
                            ("sho" "しょ")
                            ("ji" "じ")
                            ("ja" "じゃ")
                            ("ju" "じゅ")
                            ("jo" "じょ")
                            ("chi" "ち")
                            ("cha" "ちゃ")
                            ("chu" "ちゅ")
                            ("cho" "ちょ")
                            ("xtsu" "っ")
                            ("sshi" "っし")
                            ("ssha" "っしゃ")
                            ("sshu" "っしゅ")
                            ("ssho" "っしょ")
                            ("jji" "っじ")
                            ("jja" "っじゃ")
                            ("jju" "っじゅ")
                            ("jjo" "っじょ")
                            ("cchi" "っち")
                            ("ccha" "っちゃ")
                            ("cchu" "っちゅ")
                            ("ccho" "っちょ")
                            ("ttsu" "っつ")
                            ("ffu" "っふ")
                            ("tsu" "つ")
                            ("fu" "ふ")
                            ("che" "ちぇ")
                            ("cche" "っちぇ")
                            ("je" "じぇ")))

(defconst romkan-to-hepburn '(("xa" "xa")
                              ("a" "a")
                              ("xi" "xi")
                              ("i" "i")
                              ("xu" "xu")
                              ("u" "u")
                              ("vu" "vu")
                              ("va" "va")
                              ("vi" "vi")
                              ("ve" "ve")
                              ("vo" "vo")
                              ("xe" "xe")
                              ("e" "e")
                              ("xo" "xo")
                              ("o" "o")
                              ("ka" "ka")
                              ("ga" "ga")
                              ("ki" "ki")
                              ("kya" "kya")
                              ("kyu" "kyu")
                              ("kyo" "kyo")
                              ("gi" "gi")
                              ("gya" "gya")
                              ("gyu" "gyu")
                              ("gyo" "gyo")
                              ("ku" "ku")
                              ("gu" "gu")
                              ("ke" "ke")
                              ("ge" "ge")
                              ("ko" "ko")
                              ("go" "go")
                              ("sa" "sa")
                              ("za" "za")
                              ("si" "shi")
                              ("sya" "sha")
                              ("syu" "shu")
                              ("syo" "sho")
                              ("sye" "she")
                              ("zi" "ji")
                              ("zya" "ja")
                              ("zyu" "ju")
                              ("zyo" "jo")
                              ("su" "su")
                              ("zu" "zu")
                              ("se" "se")
                              ("ze" "ze")
                              ("so" "so")
                              ("zo" "zo")
                              ("ta" "ta")
                              ("da" "da")
                              ("ti" "chi")
                              ("tya" "cha")
                              ("tyu" "chu")
                              ("tyo" "cho")
                              ("di" "di")
                              ("dya" "dya")
                              ("dyu" "dyu")
                              ("dyo" "dyo")
                              ("xtu" "xtsu")
                              ("vvu" "vvu")
                              ("vva" "vva")
                              ("vvi" "vvi")
                              ("vve" "vve")
                              ("vvo" "vvo")
                              ("kka" "kka")
                              ("gga" "gga")
                              ("kki" "kki")
                              ("kkya" "kkya")
                              ("kkyu" "kkyu")
                              ("kkyo" "kkyo")
                              ("ggi" "ggi")
                              ("ggya" "ggya")
                              ("ggyu" "ggyu")
                              ("ggyo" "ggyo")
                              ("kku" "kku")
                              ("ggu" "ggu")
                              ("kke" "kke")
                              ("gge" "gge")
                              ("kko" "kko")
                              ("ggo" "ggo")
                              ("ssa" "ssa")
                              ("zza" "zza")
                              ("ssi" "sshi")
                              ("ssya" "ssha")
                              ("ssyu" "sshu")
                              ("ssyo" "ssho")
                              ("ssye" "sshe")
                              ("zzi" "jji")
                              ("zzya" "jja")
                              ("zzyu" "jju")
                              ("zzyo" "jjo")
                              ("ssu" "ssu")
                              ("zzu" "zzu")
                              ("sse" "sse")
                              ("zze" "zze")
                              ("sso" "sso")
                              ("zzo" "zzo")
                              ("tta" "tta")
                              ("dda" "dda")
                              ("tti" "tti")
                              ("ttya" "ccha")
                              ("ttyu" "cchu")
                              ("ttyo" "ccho")
                              ("ddi" "ddi")
                              ("ddya" "ddya")
                              ("ddyu" "ddyu")
                              ("ddyo" "ddyo")
                              ("ttu" "ttsu")
                              ("ddu" "ddu")
                              ("tte" "tte")
                              ("dde" "dde")
                              ("tto" "tto")
                              ("ddo" "ddo")
                              ("hha" "hha")
                              ("bba" "bba")
                              ("ppa" "ppa")
                              ("hhi" "hhi")
                              ("hhya" "hhya")
                              ("hhyu" "hhyu")
                              ("hhyo" "hhyo")
                              ("bbi" "bbi")
                              ("bbya" "bbya")
                              ("bbyu" "bbyu")
                              ("bbyo" "bbyo")
                              ("ppi" "ppi")
                              ("ppya" "ppya")
                              ("ppyu" "ppyu")
                              ("ppyo" "ppyo")
                              ("hhu" "ffu")
                              ("ffu" "ffu")
                              ("ffa" "ffa")
                              ("ffi" "ffi")
                              ("ffe" "ffe")
                              ("ffo" "ffo")
                              ("bbu" "bbu")
                              ("ppu" "ppu")
                              ("hhe" "hhe")
                              ("bbe" "bbe")
                              ("ppe" "ppe")
                              ("hho" "hho")
                              ("bbo" "bbo")
                              ("ppo" "ppo")
                              ("yya" "yya")
                              ("yyu" "yyu")
                              ("yyo" "yyo")
                              ("rra" "rra")
                              ("rri" "rri")
                              ("rrya" "rrya")
                              ("rryu" "rryu")
                              ("rryo" "rryo")
                              ("rru" "rru")
                              ("rre" "rre")
                              ("rro" "rro")
                              ("tu" "tsu")
                              ("du" "du")
                              ("te" "te")
                              ("de" "de")
                              ("to" "to")
                              ("do" "do")
                              ("na" "na")
                              ("ni" "ni")
                              ("nya" "nya")
                              ("nyu" "nyu")
                              ("nyo" "nyo")
                              ("nu" "nu")
                              ("ne" "ne")
                              ("no" "no")
                              ("ha" "ha")
                              ("ba" "ba")
                              ("pa" "pa")
                              ("hi" "hi")
                              ("hya" "hya")
                              ("hyu" "hyu")
                              ("hyo" "hyo")
                              ("bi" "bi")
                              ("bya" "bya")
                              ("byu" "byu")
                              ("byo" "byo")
                              ("pi" "pi")
                              ("pya" "pya")
                              ("pyu" "pyu")
                              ("pyo" "pyo")
                              ("hu" "fu")
                              ("fa" "fa")
                              ("fi" "fi")
                              ("fe" "fe")
                              ("fo" "fo")
                              ("fu" "fu")
                              ("bu" "bu")
                              ("pu" "pu")
                              ("he" "he")
                              ("be" "be")
                              ("pe" "pe")
                              ("ho" "ho")
                              ("bo" "bo")
                              ("po" "po")
                              ("ma" "ma")
                              ("mi" "mi")
                              ("mya" "mya")
                              ("myu" "myu")
                              ("myo" "myo")
                              ("mu" "mu")
                              ("me" "me")
                              ("mo" "mo")
                              ("xya" "xya")
                              ("ya" "ya")
                              ("xyu" "xyu")
                              ("yu" "yu")
                              ("xyo" "xyo")
                              ("yo" "yo")
                              ("ra" "ra")
                              ("ri" "ri")
                              ("rya" "rya")
                              ("ryu" "ryu")
                              ("ryo" "ryo")
                              ("ru" "ru")
                              ("re" "re")
                              ("ro" "ro")
                              ("xwa" "xwa")
                              ("wa" "wa")
                              ("wi" "wi")
                              ("we" "we")
                              ("wo" "wo")
                              ("n" "n")
                              ("n'" "n'")
                              ("dyi" "di")
                              ("-" "-")
                              ("tye" "che")
                              ("ttye" "cche")
                              ("zye" "je")))

(defconst romkan-to-kunrei '(("xa" "xa")
                             ("a" "a")
                             ("xi" "xi")
                             ("i" "i")
                             ("xu" "xu")
                             ("u" "u")
                             ("vu" "vu")
                             ("va" "va")
                             ("vi" "vi")
                             ("ve" "ve")
                             ("vo" "vo")
                             ("xe" "xe")
                             ("e" "e")
                             ("xo" "xo")
                             ("o" "o")
                             ("ka" "ka")
                             ("ga" "ga")
                             ("ki" "ki")
                             ("kya" "kya")
                             ("kyu" "kyu")
                             ("kyo" "kyo")
                             ("gi" "gi")
                             ("gya" "gya")
                             ("gyu" "gyu")
                             ("gyo" "gyo")
                             ("ku" "ku")
                             ("gu" "gu")
                             ("ke" "ke")
                             ("ge" "ge")
                             ("ko" "ko")
                             ("go" "go")
                             ("sa" "sa")
                             ("za" "za")
                             ("shi" "si")
                             ("sha" "sya")
                             ("shu" "syu")
                             ("sho" "syo")
                             ("she" "sye")
                             ("ji" "zi")
                             ("ja" "zya")
                             ("ju" "zyu")
                             ("jo" "zyo")
                             ("su" "su")
                             ("zu" "zu")
                             ("se" "se")
                             ("ze" "ze")
                             ("so" "so")
                             ("zo" "zo")
                             ("ta" "ta")
                             ("da" "da")
                             ("chi" "ti")
                             ("cha" "tya")
                             ("chu" "tyu")
                             ("cho" "tyo")
                             ("di" "dyi")
                             ("dya" "dya")
                             ("dyu" "dyu")
                             ("dyo" "dyo")
                             ("ti" "ti")
                             ("xtsu" "xtu")
                             ("vvu" "vvu")
                             ("vva" "vva")
                             ("vvi" "vvi")
                             ("vve" "vve")
                             ("vvo" "vvo")
                             ("kka" "kka")
                             ("gga" "gga")
                             ("kki" "kki")
                             ("kkya" "kkya")
                             ("kkyu" "kkyu")
                             ("kkyo" "kkyo")
                             ("ggi" "ggi")
                             ("ggya" "ggya")
                             ("ggyu" "ggyu")
                             ("ggyo" "ggyo")
                             ("kku" "kku")
                             ("ggu" "ggu")
                             ("kke" "kke")
                             ("gge" "gge")
                             ("kko" "kko")
                             ("ggo" "ggo")
                             ("ssa" "ssa")
                             ("zza" "zza")
                             ("sshi" "ssi")
                             ("ssha" "ssya")
                             ("sshu" "ssyu")
                             ("ssho" "ssyo")
                             ("sshe" "ssye")
                             ("jji" "zzi")
                             ("jja" "zzya")
                             ("jju" "zzyu")
                             ("jjo" "zzyo")
                             ("ssu" "ssu")
                             ("zzu" "zzu")
                             ("sse" "sse")
                             ("zze" "zze")
                             ("sso" "sso")
                             ("zzo" "zzo")
                             ("tta" "tta")
                             ("dda" "dda")
                             ("cchi" "tti")
                             ("tti" "tti")
                             ("ccha" "ttya")
                             ("cchu" "ttyu")
                             ("ccho" "ttyo")
                             ("ddi" "ddi")
                             ("ddya" "ddya")
                             ("ddyu" "ddyu")
                             ("ddyo" "ddyo")
                             ("ttsu" "ttu")
                             ("ddu" "ddu")
                             ("tte" "tte")
                             ("dde" "dde")
                             ("tto" "tto")
                             ("ddo" "ddo")
                             ("hha" "hha")
                             ("bba" "bba")
                             ("ppa" "ppa")
                             ("hhi" "hhi")
                             ("hhya" "hhya")
                             ("hhyu" "hhyu")
                             ("hhyo" "hhyo")
                             ("bbi" "bbi")
                             ("bbya" "bbya")
                             ("bbyu" "bbyu")
                             ("bbyo" "bbyo")
                             ("ppi" "ppi")
                             ("ppya" "ppya")
                             ("ppyu" "ppyu")
                             ("ppyo" "ppyo")
                             ("ffu" "ffu")
                             ("ffa" "ffa")
                             ("ffi" "ffi")
                             ("ffe" "ffe")
                             ("ffo" "ffo")
                             ("bbu" "bbu")
                             ("ppu" "ppu")
                             ("hhe" "hhe")
                             ("bbe" "bbe")
                             ("ppe" "ppe")
                             ("hho" "hho")
                             ("bbo" "bbo")
                             ("ppo" "ppo")
                             ("yya" "yya")
                             ("yyu" "yyu")
                             ("yyo" "yyo")
                             ("rra" "rra")
                             ("rri" "rri")
                             ("rrya" "rrya")
                             ("rryu" "rryu")
                             ("rryo" "rryo")
                             ("rru" "rru")
                             ("rre" "rre")
                             ("rro" "rro")
                             ("tsu" "tu")
                             ("du" "du")
                             ("te" "te")
                             ("de" "de")
                             ("to" "to")
                             ("do" "do")
                             ("na" "na")
                             ("ni" "ni")
                             ("nya" "nya")
                             ("nyu" "nyu")
                             ("nyo" "nyo")
                             ("nu" "nu")
                             ("ne" "ne")
                             ("no" "no")
                             ("ha" "ha")
                             ("ba" "ba")
                             ("pa" "pa")
                             ("hi" "hi")
                             ("hya" "hya")
                             ("hyu" "hyu")
                             ("hyo" "hyo")
                             ("bi" "bi")
                             ("bya" "bya")
                             ("byu" "byu")
                             ("byo" "byo")
                             ("pi" "pi")
                             ("pya" "pya")
                             ("pyu" "pyu")
                             ("pyo" "pyo")
                             ("fu" "fu")
                             ("fa" "fa")
                             ("fi" "fi")
                             ("fe" "fe")
                             ("fo" "fo")
                             ("bu" "bu")
                             ("pu" "pu")
                             ("he" "he")
                             ("be" "be")
                             ("pe" "pe")
                             ("ho" "ho")
                             ("bo" "bo")
                             ("po" "po")
                             ("ma" "ma")
                             ("mi" "mi")
                             ("mya" "mya")
                             ("myu" "myu")
                             ("myo" "myo")
                             ("mu" "mu")
                             ("me" "me")
                             ("mo" "mo")
                             ("xya" "xya")
                             ("ya" "ya")
                             ("xyu" "xyu")
                             ("yu" "yu")
                             ("xyo" "xyo")
                             ("yo" "yo")
                             ("ra" "ra")
                             ("ri" "ri")
                             ("rya" "rya")
                             ("ryu" "ryu")
                             ("ryo" "ryo")
                             ("ru" "ru")
                             ("re" "re")
                             ("ro" "ro")
                             ("xwa" "xwa")
                             ("wa" "wa")
                             ("wi" "wi")
                             ("we" "we")
                             ("wo" "wo")
                             ("n" "n")
                             ("n'" "n'")
                             ("-" "-")
                             ("che" "tye")
                             ("cche" "ttye")
                             ("je" "zye")))

(defconst romkan-to-hepburn-h '(("xa" "xa")
                                ("a" "a")
                                ("xi" "xi")
                                ("i" "i")
                                ("xu" "xu")
                                ("u" "u")
                                ("vu" "vu")
                                ("va" "va")
                                ("vi" "vi")
                                ("ve" "ve")
                                ("vo" "vo")
                                ("xe" "xe")
                                ("e" "e")
                                ("xo" "xo")
                                ("o" "o")
                                ("ka" "ka")
                                ("ga" "ga")
                                ("ki" "ki")
                                ("kya" "kya")
                                ("kyu" "kyu")
                                ("kyo" "kyo")
                                ("gi" "gi")
                                ("gya" "gya")
                                ("gyu" "gyu")
                                ("gyo" "gyo")
                                ("ku" "ku")
                                ("gu" "gu")
                                ("ke" "ke")
                                ("ge" "ge")
                                ("ko" "ko")
                                ("go" "go")
                                ("sa" "sa")
                                ("za" "za")
                                ("si" "shi")
                                ("sya" "sha")
                                ("syu" "shu")
                                ("syo" "sho")
                                ("zi" "ji")
                                ("zya" "ja")
                                ("zyu" "ju")
                                ("zyo" "jo")
                                ("su" "su")
                                ("zu" "zu")
                                ("se" "se")
                                ("ze" "ze")
                                ("so" "so")
                                ("zo" "zo")
                                ("ta" "ta")
                                ("da" "da")
                                ("ti" "chi")
                                ("tya" "cha")
                                ("tyu" "chu")
                                ("tyo" "cho")
                                ("di" "di")
                                ("dya" "dya")
                                ("dyu" "dyu")
                                ("dyo" "dyo")
                                ("xtu" "xtsu")
                                ("vvu" "vvu")
                                ("vva" "vva")
                                ("vvi" "vvi")
                                ("vve" "vve")
                                ("vvo" "vvo")
                                ("kka" "kka")
                                ("gga" "gga")
                                ("kki" "kki")
                                ("kkya" "kkya")
                                ("kkyu" "kkyu")
                                ("kkyo" "kkyo")
                                ("ggi" "ggi")
                                ("ggya" "ggya")
                                ("ggyu" "ggyu")
                                ("ggyo" "ggyo")
                                ("kku" "kku")
                                ("ggu" "ggu")
                                ("kke" "kke")
                                ("gge" "gge")
                                ("kko" "kko")
                                ("ggo" "ggo")
                                ("ssa" "ssa")
                                ("zza" "zza")
                                ("ssi" "sshi")
                                ("ssya" "ssha")
                                ("ssyu" "sshu")
                                ("ssyo" "ssho")
                                ("zzi" "jji")
                                ("zzya" "jja")
                                ("zzyu" "jju")
                                ("zzyo" "jjo")
                                ("ssu" "ssu")
                                ("zzu" "zzu")
                                ("sse" "sse")
                                ("zze" "zze")
                                ("sso" "sso")
                                ("zzo" "zzo")
                                ("tta" "tta")
                                ("dda" "dda")
                                ("tti" "cchi")
                                ("ttya" "ccha")
                                ("ttyu" "cchu")
                                ("ttyo" "ccho")
                                ("ddi" "ddi")
                                ("ddya" "ddya")
                                ("ddyu" "ddyu")
                                ("ddyo" "ddyo")
                                ("ttu" "ttsu")
                                ("ddu" "ddu")
                                ("tte" "tte")
                                ("dde" "dde")
                                ("tto" "tto")
                                ("ddo" "ddo")
                                ("hha" "hha")
                                ("bba" "bba")
                                ("ppa" "ppa")
                                ("hhi" "hhi")
                                ("hhya" "hhya")
                                ("hhyu" "hhyu")
                                ("hhyo" "hhyo")
                                ("bbi" "bbi")
                                ("bbya" "bbya")
                                ("bbyu" "bbyu")
                                ("bbyo" "bbyo")
                                ("ppi" "ppi")
                                ("ppya" "ppya")
                                ("ppyu" "ppyu")
                                ("ppyo" "ppyo")
                                ("hhu" "ffu")
                                ("ffa" "ffa")
                                ("ffi" "ffi")
                                ("ffe" "ffe")
                                ("ffo" "ffo")
                                ("bbu" "bbu")
                                ("ppu" "ppu")
                                ("hhe" "hhe")
                                ("bbe" "bbe")
                                ("ppe" "ppe")
                                ("hho" "hho")
                                ("bbo" "bbo")
                                ("ppo" "ppo")
                                ("yya" "yya")
                                ("yyu" "yyu")
                                ("yyo" "yyo")
                                ("rra" "rra")
                                ("rri" "rri")
                                ("rrya" "rrya")
                                ("rryu" "rryu")
                                ("rryo" "rryo")
                                ("rru" "rru")
                                ("rre" "rre")
                                ("rro" "rro")
                                ("tu" "tsu")
                                ("du" "du")
                                ("te" "te")
                                ("de" "de")
                                ("to" "to")
                                ("do" "do")
                                ("na" "na")
                                ("ni" "ni")
                                ("nya" "nya")
                                ("nyu" "nyu")
                                ("nyo" "nyo")
                                ("nu" "nu")
                                ("ne" "ne")
                                ("no" "no")
                                ("ha" "ha")
                                ("ba" "ba")
                                ("pa" "pa")
                                ("hi" "hi")
                                ("hya" "hya")
                                ("hyu" "hyu")
                                ("hyo" "hyo")
                                ("bi" "bi")
                                ("bya" "bya")
                                ("byu" "byu")
                                ("byo" "byo")
                                ("pi" "pi")
                                ("pya" "pya")
                                ("pyu" "pyu")
                                ("pyo" "pyo")
                                ("hu" "fu")
                                ("fa" "fa")
                                ("fi" "fi")
                                ("fe" "fe")
                                ("fo" "fo")
                                ("bu" "bu")
                                ("pu" "pu")
                                ("he" "he")
                                ("be" "be")
                                ("pe" "pe")
                                ("ho" "ho")
                                ("bo" "bo")
                                ("po" "po")
                                ("ma" "ma")
                                ("mi" "mi")
                                ("mya" "mya")
                                ("myu" "myu")
                                ("myo" "myo")
                                ("mu" "mu")
                                ("me" "me")
                                ("mo" "mo")
                                ("xya" "xya")
                                ("ya" "ya")
                                ("xyu" "xyu")
                                ("yu" "yu")
                                ("xyo" "xyo")
                                ("yo" "yo")
                                ("ra" "ra")
                                ("ri" "ri")
                                ("rya" "rya")
                                ("ryu" "ryu")
                                ("ryo" "ryo")
                                ("ru" "ru")
                                ("re" "re")
                                ("ro" "ro")
                                ("xwa" "xwa")
                                ("wa" "wa")
                                ("wi" "wi")
                                ("we" "we")
                                ("wo" "wo")
                                ("n" "n")
                                ("n'" "n'")
                                ("dyi" "dyi")
                                ("-" "-")
                                ("tye" "che")
                                ("ttye" "cche")
                                ("zye" "je")))

(defconst romkan-to-kunrei-h '(("xa" "xa")
                               ("a" "a")
                               ("xi" "xi")
                               ("i" "i")
                               ("xu" "xu")
                               ("u" "u")
                               ("vu" "vu")
                               ("va" "va")
                               ("vi" "vi")
                               ("ve" "ve")
                               ("vo" "vo")
                               ("xe" "xe")
                               ("e" "e")
                               ("xo" "xo")
                               ("o" "o")
                               ("ka" "ka")
                               ("ga" "ga")
                               ("ki" "ki")
                               ("kya" "kya")
                               ("kyu" "kyu")
                               ("kyo" "kyo")
                               ("gi" "gi")
                               ("gya" "gya")
                               ("gyu" "gyu")
                               ("gyo" "gyo")
                               ("ku" "ku")
                               ("gu" "gu")
                               ("ke" "ke")
                               ("ge" "ge")
                               ("ko" "ko")
                               ("go" "go")
                               ("sa" "sa")
                               ("za" "za")
                               ("shi" "si")
                               ("sha" "sya")
                               ("shu" "syu")
                               ("sho" "syo")
                               ("ji" "zi")
                               ("ja" "zya")
                               ("ju" "zyu")
                               ("jo" "zyo")
                               ("su" "su")
                               ("zu" "zu")
                               ("se" "se")
                               ("ze" "ze")
                               ("so" "so")
                               ("zo" "zo")
                               ("ta" "ta")
                               ("da" "da")
                               ("chi" "ti")
                               ("cha" "tya")
                               ("chu" "tyu")
                               ("cho" "tyo")
                               ("di" "di")
                               ("dya" "dya")
                               ("dyu" "dyu")
                               ("dyo" "dyo")
                               ("xtsu" "xtu")
                               ("vvu" "vvu")
                               ("vva" "vva")
                               ("vvi" "vvi")
                               ("vve" "vve")
                               ("vvo" "vvo")
                               ("kka" "kka")
                               ("gga" "gga")
                               ("kki" "kki")
                               ("kkya" "kkya")
                               ("kkyu" "kkyu")
                               ("kkyo" "kkyo")
                               ("ggi" "ggi")
                               ("ggya" "ggya")
                               ("ggyu" "ggyu")
                               ("ggyo" "ggyo")
                               ("kku" "kku")
                               ("ggu" "ggu")
                               ("kke" "kke")
                               ("gge" "gge")
                               ("kko" "kko")
                               ("ggo" "ggo")
                               ("ssa" "ssa")
                               ("zza" "zza")
                               ("sshi" "ssi")
                               ("ssha" "ssya")
                               ("sshu" "ssyu")
                               ("ssho" "ssyo")
                               ("jji" "zzi")
                               ("jja" "zzya")
                               ("jju" "zzyu")
                               ("jjo" "zzyo")
                               ("ssu" "ssu")
                               ("zzu" "zzu")
                               ("sse" "sse")
                               ("zze" "zze")
                               ("sso" "sso")
                               ("zzo" "zzo")
                               ("tta" "tta")
                               ("dda" "dda")
                               ("cchi" "tti")
                               ("ccha" "ttya")
                               ("cchu" "ttyu")
                               ("ccho" "ttyo")
                               ("ddi" "ddi")
                               ("ddya" "ddya")
                               ("ddyu" "ddyu")
                               ("ddyo" "ddyo")
                               ("ttsu" "ttu")
                               ("ddu" "ddu")
                               ("tte" "tte")
                               ("dde" "dde")
                               ("tto" "tto")
                               ("ddo" "ddo")
                               ("hha" "hha")
                               ("bba" "bba")
                               ("ppa" "ppa")
                               ("hhi" "hhi")
                               ("hhya" "hhya")
                               ("hhyu" "hhyu")
                               ("hhyo" "hhyo")
                               ("bbi" "bbi")
                               ("bbya" "bbya")
                               ("bbyu" "bbyu")
                               ("bbyo" "bbyo")
                               ("ppi" "ppi")
                               ("ppya" "ppya")
                               ("ppyu" "ppyu")
                               ("ppyo" "ppyo")
                               ("ffu" "hhu")
                               ("ffa" "ffa")
                               ("ffi" "ffi")
                               ("ffe" "ffe")
                               ("ffo" "ffo")
                               ("bbu" "bbu")
                               ("ppu" "ppu")
                               ("hhe" "hhe")
                               ("bbe" "bbe")
                               ("ppe" "ppe")
                               ("hho" "hho")
                               ("bbo" "bbo")
                               ("ppo" "ppo")
                               ("yya" "yya")
                               ("yyu" "yyu")
                               ("yyo" "yyo")
                               ("rra" "rra")
                               ("rri" "rri")
                               ("rrya" "rrya")
                               ("rryu" "rryu")
                               ("rryo" "rryo")
                               ("rru" "rru")
                               ("rre" "rre")
                               ("rro" "rro")
                               ("tsu" "tu")
                               ("du" "du")
                               ("te" "te")
                               ("de" "de")
                               ("to" "to")
                               ("do" "do")
                               ("na" "na")
                               ("ni" "ni")
                               ("nya" "nya")
                               ("nyu" "nyu")
                               ("nyo" "nyo")
                               ("nu" "nu")
                               ("ne" "ne")
                               ("no" "no")
                               ("ha" "ha")
                               ("ba" "ba")
                               ("pa" "pa")
                               ("hi" "hi")
                               ("hya" "hya")
                               ("hyu" "hyu")
                               ("hyo" "hyo")
                               ("bi" "bi")
                               ("bya" "bya")
                               ("byu" "byu")
                               ("byo" "byo")
                               ("pi" "pi")
                               ("pya" "pya")
                               ("pyu" "pyu")
                               ("pyo" "pyo")
                               ("fu" "hu")
                               ("fa" "fa")
                               ("fi" "fi")
                               ("fe" "fe")
                               ("fo" "fo")
                               ("bu" "bu")
                               ("pu" "pu")
                               ("he" "he")
                               ("be" "be")
                               ("pe" "pe")
                               ("ho" "ho")
                               ("bo" "bo")
                               ("po" "po")
                               ("ma" "ma")
                               ("mi" "mi")
                               ("mya" "mya")
                               ("myu" "myu")
                               ("myo" "myo")
                               ("mu" "mu")
                               ("me" "me")
                               ("mo" "mo")
                               ("xya" "xya")
                               ("ya" "ya")
                               ("xyu" "xyu")
                               ("yu" "yu")
                               ("xyo" "xyo")
                               ("yo" "yo")
                               ("ra" "ra")
                               ("ri" "ri")
                               ("rya" "rya")
                               ("ryu" "ryu")
                               ("ryo" "ryo")
                               ("ru" "ru")
                               ("re" "re")
                               ("ro" "ro")
                               ("xwa" "xwa")
                               ("wa" "wa")
                               ("wi" "wi")
                               ("we" "we")
                               ("wo" "wo")
                               ("n" "n")
                               ("n'" "n'")
                               ("dyi" "dyi")
                               ("-" "-")
                               ("che" "tye")
                               ("cche" "ttye")
                               ("je" "zye")))

(defun romkan--get-regex-pattern-from-keys (table)
  "Create a regex pattern from the keys in the TABLE."
  (let* ((keys (mapcar (lambda (el) (car el)) table))
         (keys (sort keys (lambda (a b) (> (length a) (length b))))))
    (rx-to-string `(or ,@keys))))

(defconst romkan-kanpat (romkan--get-regex-pattern-from-keys romkan-kanrom))
(defconst romkan-kanpat-h (romkan--get-regex-pattern-from-keys romkan-kanrom-h))
(defconst romkan-rompat (romkan--get-regex-pattern-from-keys romkan-romkan))
(defconst romkan-rompat-h (romkan--get-regex-pattern-from-keys romkan-romkan-h))

(defconst romkan-heppat (romkan--get-regex-pattern-from-keys romkan-to-kunrei))
(defconst romkan-heppat-h (romkan--get-regex-pattern-from-keys romkan-to-kunrei-h))
(defconst romkan-kunpat (romkan--get-regex-pattern-from-keys romkan-to-hepburn))
(defconst romkan-kunpat-h (romkan--get-regex-pattern-from-keys romkan-to-hepburn-h))

(defun romkan--remove-apostrophes (str)
  "Remove unnecessary apostrophes in STR."
  (replace-regexp-in-string
   (rx "n'" (or (not (or "a" "i" "u" "e" "o" "y" "n")) eol))
   (lambda (substr)
     (replace-regexp-in-string
      "'.*"
      (lambda (subsubstr) (substring subsubstr 1))
      substr))
   str))

(defun romkan--normalize-double-n (str)
  "Normalize double n in STR."
  (let ((str (replace-regexp-in-string "nn" "n'" str)))
    (romkan--remove-apostrophes str)))

(defun romkan-to-katakana (str)
  "Convert a Romaji (ローマ字) to a Katakana (片仮名) in STR."
  (let* ((str (downcase str))
         (str (replace-regexp-in-string "nn" "n'" str))
         (str (replace-regexp-in-string "n'(?=[^aiueoyn]\\|\\$)" "n" str)))
    (romkan--transliterate romkan-rompat romkan-romkan str)))

(defun romkan--transliterate (regex-matcher alist-dict str)
  "Run REGEX-MATCHER on STR and replace units following the given ALIST-DICT."
  (replace-regexp-in-string
   regex-matcher
   (lambda (substr)
     (cadr (assoc substr alist-dict)))
   str))

(defun romkan-to-hiragana (str)
  "Convert a Romaji (ローマ字) to a Hiragana (平仮名) in STR."
  (let* ((str (downcase str))
         (str (romkan--normalize-double-n str)))
    (romkan--transliterate romkan-rompat-h romkan-romkan-h str)))

(defalias 'romkan-to-kana #'romkan-to-katakana)

(defun romkan-to-roma (str)
  "Convert a Kana (仮名) to a Hepburn Romaji (ヘボン式ローマ字) in STR."
  (let* ((str (romkan--transliterate romkan-kanpat romkan-kanrom str))
         (str (romkan--transliterate romkan-kanpat-h romkan-kanrom-h str))
         (str (romkan--remove-apostrophes str)))
    str))

(defun romkan-to-kunrei (str)
  "Convert a Kana (仮名) or a Kunrei-shiki Romaji (訓令式ローマ字)
to a Hepburn Romaji (ヘボン式ローマ字) in STR."
  (let* ((tmp (romkan-to-roma str))
         ;; If unmodified, it's a Kunrei-shiki Romaji
         (tmp (if (equal tmp str)
                  ;; -- convert it to a Hepburn Romaji
                  (let ((tmp (downcase tmp)))
                    (romkan--normalize-double-n tmp))
                tmp)))
    (romkan--transliterate romkan-heppat romkan-to-kunrei tmp)))

(defun romkan-to-hepburn (str)
  "Convert a Kana (仮名) or a Kunrei-shiki Romaji (訓令式ローマ字)
to a Hepburn Romaji (ヘボン式ローマ字) in STR."
  (let* ((tmp (romkan-to-roma str))
         ;; If unmodified, it's a Kunrei-shiki Romaji
         (tmp (if (equal tmp str)
                  ;; -- convert it to a Hepburn Romaji
                  (let ((tmp (downcase tmp)))
                    (romkan--normalize-double-n tmp))
                tmp)))
    (romkan--transliterate romkan-kunpat romkan-to-hepburn tmp)))

(provide 'romkan)
;;; romkan.el ends here
