;;; sail.el --- NOAA tide and wind reports for sailors -*- lexical-binding: t; -*-

;; Copyright (C) 2026 Bruce Rannala

;; Author: Bruce Rannala <brannala@ucdavis.edu>
;; Maintainer: Bruce Rannala <brannala@ucdavis.edu>
;; Package-Version: 20260209.411
;; Package-Revision: 1cc38e174139
;; Package-Requires: ((emacs "25.1"))
;; Keywords: comm
;; URL: https://github.com/brannala/esail

;; This file is not part of GNU Emacs.

;; This program is free software: you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation, either version 3 of the License, or
;; (at your option) any later version.

;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with this program.  If not, see <https://www.gnu.org/licenses/>.

;;; Commentary:

;; Sail provides interactive Emacs commands for fetching real-time tide
;; predictions and wind conditions from the NOAA Tides and Currents API.
;;
;; Usage:
;;   M-x sail-get-tides   Fetch the next 24 hours of high/low tides.
;;   M-x sail-get-wind    Fetch the latest wind conditions.
;;
;; Customize the station IDs via `customize-group' sail, or set
;; `sail-tide-station' and `sail-wind-station' directly.  Station IDs
;; can be found at <https://tidesandcurrents.noaa.gov/stations.html>.

;;; Code:

(require 'url)

(defgroup sail nil
  "NOAA tide and wind reports for sailors."
  :group 'comm
  :prefix "sail-")

(defcustom sail-tide-station '("Benicia" "9415111")
  "Tide station as a list of (NAME ID) for NOAA API queries.
NAME is a human-readable label and ID is the NOAA station number."
  :type '(list string string)
  :group 'sail)

(defcustom sail-wind-station '("Martinez" "9415102")
  "Wind station as a list of (NAME ID) for NOAA API queries.
NAME is a human-readable label and ID is the NOAA station number."
  :type '(list string string)
  :group 'sail)

(defconst sail--api-base-url
  "https://api.tidesandcurrents.noaa.gov/api/prod/datagetter"
  "Base URL for the NOAA Tides and Currents API.")

(defun sail--build-url (params)
  "Build a NOAA API URL from PARAMS, an alist of query parameters."
  (concat sail--api-base-url
          "?"
          (mapconcat (lambda (pair)
                       (concat (url-hexify-string (car pair))
                               "="
                               (url-hexify-string (cdr pair))))
                     params
                     "&")))

(defun sail--handle-response (status callback)
  "Check STATUS for errors, then call CALLBACK in the response buffer.
HTTP headers are stripped before CALLBACK is called.  The response
buffer is killed after CALLBACK returns or if an error occurs."
  (if (plist-get status :error)
      (message "NOAA fetch failed: %s" (plist-get status :error))
    (let ((buf (current-buffer)))
      (unwind-protect
          (progn
            (goto-char (point-min))
            (re-search-forward "\n\n" nil t)
            (delete-region (point-min) (point))
            (funcall callback))
        (kill-buffer buf)))))

;;;###autoload
(defun sail-get-tides ()
  "Fetch and display the next 24 hours of high/low tide predictions."
  (interactive)
  (let* ((now (decode-time))
         (begin-date (format "%d%02d%02d %02d:00"
                             (nth 5 now) (nth 4 now)
                             (nth 3 now) (nth 2 now)))
         (url (sail--build-url
               `(("begin_date" . ,begin-date)
                 ("range" . "24")
                 ("station" . ,(nth 1 sail-tide-station))
                 ("product" . "predictions")
                 ("units" . "english")
                 ("datum" . "mllw")
                 ("interval" . "hilo")
                 ("time_zone" . "lst_ldt")
                 ("application" . "ports_screen")
                 ("format" . "csv")))))
    (url-retrieve
     url
     (lambda (status)
       (sail--handle-response
        status
        (lambda ()
          (goto-char (point-min))
          (let* ((datastring (buffer-string))
                 (start (string-match "[0-9]\\{4\\}-.*" datastring))
                 (datalist (butlast (split-string
                                     (substring datastring start) "\n")))
                 (tide-report
                  (concat (nth 0 sail-tide-station) " [24H+] Tides"
                          (mapconcat (lambda (cv) (concat " => " cv))
                                     datalist ""))))
            (message "%s" tide-report))))))))

;;;###autoload
(defun sail-get-wind ()
  "Fetch and display the latest wind conditions."
  (interactive)
  (let ((url (sail--build-url
              `(("date" . "latest")
                ("station" . ,(nth 1 sail-wind-station))
                ("product" . "wind")
                ("units" . "english")
                ("time_zone" . "lst_ldt")
                ("application" . "ports_screen")
                ("format" . "csv")))))
    (url-retrieve
     url
     (lambda (status)
       (sail--handle-response
        status
        (lambda ()
          (goto-char (point-min))
          (re-search-forward "[0-9]\\{4\\}-.*")
          (let* ((noaa-weather (buffer-substring-no-properties
                                (line-beginning-position)
                                (line-end-position)))
                 (weather-list (split-string noaa-weather ","))
                 (wind-time (nth 0 weather-list))
                 (wind-speed (nth 1 weather-list))
                 (wind-degrees (nth 2 weather-list))
                 (wind-direction (nth 3 weather-list))
                 (wind-gusts (nth 4 weather-list)))
            (message "%s [%s] Wind => Speed: %skn Gusts: %skn Direction: %s (%3dM)"
                     (nth 0 sail-wind-station)
                     wind-time wind-speed wind-gusts
                     wind-direction
                     (string-to-number wind-degrees)))))))))

(provide 'sail)
;;; sail.el ends here
