;;; selected-window-contrast.el --- Highlight by brightness of text and background   -*- lexical-binding: t -*-

;; Copyright (c) 2025 Anoncheg
;;
;; Maintainer: Anoncheg <vitalij@gmx.com>
;; Author: Anoncheg <vitalij@gmx.com>
;;
;; Keywords:  color, windows, faces, buffer, background
;; URL: https://codeberg.org/Anoncheg/selected-window-contrast
;; Package-Version: 20260227.1846
;; Package-Revision: 5d461544c0e1
;; Created: 11 dec 2024
;; Package-Requires: ((emacs "25.1"))
;; SPDX-License-Identifier: AGPL-3.0-or-later

;;; License

;; This file is not part of GNU Emacs.

;; This program is free software: you can redistribute it and/or modify
;; it under the terms of the GNU Affero General Public License as published by
;; the Free Software Foundation, either version 3 of the License, or
;; (at your option) any later version.

;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU Affero General Public License for more details.

;; You should have received a copy of the GNU Affero General Public License
;; along with this program.  If not,
;; see <https://www.gnu.org/licenses/agpl-3.0.en.html>.

;;; Commentary:

;; Highlight selected window by adjusting contrast of text
;;  "foreground" and background.
;; Working good if you switch themes frequently, contrast will be kept.
;; Also this works for modeline.
;; We also highligh cursor position, this may be disabled with
;;  (setopt selected-window-contrast-region-flag nil) in .emacs
;;  or
;;  M-x customize-variable RET selected-window-contrast-region-flag

;;;; Usage:

;; (add-to-list 'load-path "path_to/selected-window-contrast") ; optional
;; (when (require 'selected-window-contrast nil 'noerror)
;;   (setopt selected-window-contrast-bg-selected 0.95)
;;   (setopt selected-window-contrast-bg-others 0.75)
;;   (setopt selected-window-contrast-text-selected 0.9)
;;   (setopt selected-window-contrast-text-others 0.6)
;;   (add-hook 'buffer-list-update-hook
;;             #'selected-window-contrast-highlight-selected-window))

;;;;  How this works:

;;  1) We get color with `face-attribute' `selected-frame' for
;;  foreground and backgraound.
;;  2) Convert color to HSL
;;  3) adjust brightness in direction of foreground-background average
;;  4) convert color to RGB, then to HEX
;;  5) apply color

;; Customize: M-x customize-group RET selected-window-contrast

;; Note, if you use C-o to switch windows, this may conflict with
;;  rectangle-mark-mode-map
;; Rebind keys then:
;; (keymap-set rectangle-mark-mode-map "C-c o" #'open-rectangle)
;; (keymap-set rectangle-mark-mode-map "C-o" #'other-window)

;;;; Donate:

;; - BTC (Bitcoin) address: 1CcDWSQ2vgqv5LxZuWaHGW52B9fkT5io25
;; - USDT (Tether TRX-TRON) address: TVoXfYMkVYLnQZV3mGZ6GvmumuBfGsZzsN
;; - TON (Telegram) address: UQC8rjJFCHQkfdp7KmCkTZCb5dGzLFYe2TzsiZpfsnyTFt9D

;;;; Other packages:

;; - Modern navigation in major modes https://github.com/Anoncheg1/firstly-search
;; - Search with Chinese	https://github.com/Anoncheg1/pinyin-isearch
;; - Ediff no 3-th window	https://github.com/Anoncheg1/ediffnw
;; - Dired history		https://github.com/Anoncheg1/dired-hist
;; - Copy link to clipboard	https://github.com/Anoncheg1/emacs-org-links
;; - Solution for "callback hell"	https://github.com/Anoncheg1/emacs-async1
;; - Restore buffer state	https://github.com/Anoncheg1/emacs-unmodified-buffer1
;; - outline.el usage		https://github.com/Anoncheg1/emacs-outline-it
;; - Call LLMs & AIfrom Org-mode block.  https://github.com/Anoncheg1/emacs-oai

;;; Code:

;; Touch: Global variables bound deep is not good, it is a type of the inversion of control.
;; I am the best that is why I am the winner.
(require 'color)
(require 'rect)

(defgroup selected-window-contrast nil
 "Highlight by brightness of text and background."
 :group 'faces)

;; - configurable:
(defcustom selected-window-contrast-bg-selected nil
  "Non-nil used to set selected window background contrast in [0-1] range.
Higher value increase contrast between text and background.
This value change contrast of text regarding to background."
  :type '(choice (restricted-sexp :match-alternatives
                                  ((lambda (x) (and (numberp x) (<= 0 x 1))))
                                  :message "Contrast value must be between 0 and 1")
                 (const :tag "Don't change default contrast of theme." nil)))

(defcustom selected-window-contrast-bg-others 0.8
  "Non-nil used to set not selected windows background contrast.
in [0-1] range."
  :type '(choice
          (const :tag "Don't change default contrast of theme." nil)
          (restricted-sexp :match-alternatives
                           ((lambda (x) (and (numberp x) (<= 0 x 1))))
                           :message "Contrast value must be between 0 and 1")))

(defcustom selected-window-contrast-text-selected nil
  "Non-nil used to set not selected windows text contrast in [0-1] range."
  :type '(choice (restricted-sexp :match-alternatives
                                  ((lambda (x) (and (numberp x) (<= 0 x 1))))
                                  :message "Contrast value must be between 0 and 1")
                 (const :tag "Don't change default contrast of theme." nil)))

(defcustom selected-window-contrast-text-others nil
  "Non-nil used to set not selected windows text contrast in [0-1] range."
  :type '(choice (restricted-sexp :match-alternatives
                           ((lambda (x) (and (numberp x) (<= 0 x 1))))
                           :message "Contrast value must be between 0 and 1")
                 (const :tag "Don't change default contrast of theme." nil)))

(defcustom selected-window-contrast-region-flag t
  "Non-nil means enable highlighting cursor by region around it."
  :type 'boolean)

(defcustom selected-window-contrast-flag t
  "Non-nil means enable highlighting by contrast."
  :type 'boolean)

(defcustom selected-window-contrast-region-timeout 0.5
  "Highlighting curson, second for which to show rectangle around."
  :type 'float)

(defcustom selected-window-contrast-region-width 8
  "Highlighting curson, width in chars of rectangle."
  :type 'number)

(defcustom selected-window-contrast-region-lines 2
  "Highlighting curson, height in lines of rectangle."
  :type 'number)

(defun selected-window-contrast--get-current-colors ()
  "Get current text and background color of default face.
Returns list: (foreground background), both strings."
  (list (face-attribute 'default :foreground)
        (face-attribute 'default :background)))

(defun selected-window-contrast-adjust-contrast (text-color background-color bg-mag text-mag)
  "Maximize visual contrast between TEXT-COLOR and BACKGROUND-COLOR.
This function remaps the lightness component of both input colors so that:
- Colors above the midpoint (0.5) are pushed towards full white.
- Colors below the midpoint are pushed towards full black.
Arguments:
 TEXT-COLOR        String, name or hex.
 BACKGROUND-COLOR  String, name or hex.
 BG-MAG (float in [0,1]) controls stretching of contrast for background.
 TEXT-MAG (float in [0,1], optional): stretching of contrast for text.
Returns:
 List: (NEW-TEXT-RGB NEW-BACKGROUND-RGB), each as (R G B) floats in [0,1]."
  (let* ((text-hsl (apply #'color-rgb-to-hsl (color-name-to-rgb text-color)))
         (bg-hsl   (apply #'color-rgb-to-hsl (color-name-to-rgb background-color)))
         (mid 0.5))
    (list
     (if (not text-mag)
         (apply #'color-hsl-to-rgb text-hsl)
       ;; else
       (let* ((t-l (nth 2 text-hsl))
              (new-t-l (if (> t-l mid)
                           (+ mid (* text-mag (- 1.0 mid)))
                         (- mid (* text-mag (- mid 0))))))
         (apply #'color-hsl-to-rgb (list (nth 0 text-hsl) (nth 1 text-hsl) new-t-l))))
     (if (not bg-mag)
         (apply #'color-hsl-to-rgb bg-hsl)
       ;; else
       (let* ((b-l (nth 2 bg-hsl))
              (new-b-l (if (> b-l mid)
                           (+ mid (* bg-mag (- 1.0 mid)))
                         (- mid (* bg-mag (- mid 0))))))
         (apply #'color-hsl-to-rgb (list (nth 0 bg-hsl) (nth 1 bg-hsl) new-b-l)))))))

(defun selected-window-contrast--rgb-to-hex (rgb &optional digits)
  "Convert normalized RGB list to hex string.
RGB: list of 3 floats in [0,1].  DIGITS: 2 or 4 digits/component."
  (apply #'color-rgb-to-hex (append rgb (list (or digits 2)))))

(defun selected-window-contrast-change-window (contrast-background contrast-text)
  "Increase contrast between text and background in buffer.
CONTRAST-BACKGROUND, CONTRAST-TEXT: float in [0,1]; of contrast 1 - is
full contrast, 0 - no contrast.
Works on both dark (light text/dark bg) and light (dark text/light bg) themes."
  (unless (or (when (and contrast-background
                         (or (not (numberp contrast-background))
                             (not (<= 0 contrast-background 1))))
                (message "Contrast-background must be floats in [0,1]"))
              (when (and contrast-text
                         (or (not (numberp contrast-text))
                             (not (<= 0 contrast-text 1))))
                (message "contrast-text must be floats in [0,1]")))
    (let* ((current-colors (selected-window-contrast--get-current-colors))
           (new-colors (selected-window-contrast-adjust-contrast (nth 0 current-colors)
                                                                 (nth 1 current-colors)
                                                                 contrast-background
                                                                 contrast-text)))
      (let ((background-rgb (nth 1 new-colors))
            (text-rgb (nth 0 new-colors)))
        (if (and contrast-background contrast-text)
            ;; :foreground (selected-window-contrast--rgb-to-hex text-rgb)
            (buffer-face-set (list :foreground (selected-window-contrast--rgb-to-hex text-rgb)
                                   :background (selected-window-contrast--rgb-to-hex background-rgb)))
          ;; else
          (when contrast-text
            (buffer-face-set (list :foreground (selected-window-contrast--rgb-to-hex text-rgb)
                                   :background (face-attribute 'default :background))))
          (when contrast-background
            (buffer-face-set (list :foreground (face-attribute 'default :foreground)
                                   :background (selected-window-contrast--rgb-to-hex background-rgb)))))))))

(defun selected-window-contrast-change-modeline (contrast-background contrast-text)
  "Adjust modeline brightness of text and background.
Arguments CONTRAST-BACKGROUND, CONTRAST-TEXT is float value to increase
or decrease contrast."
  (let* ((back (face-attribute 'mode-line-active :background))
         (fore (face-attribute 'mode-line-active :foreground)))
    (when (or (eq back 'unspecified) (eq back 'unspecified-bg))
      (setq back (face-attribute 'default :background)))
    (when (eq fore 'unspecified)
      (setq fore (face-attribute 'default :foreground)))

    (if (or (eq back 'unspecified)
            (eq back 'unspecified-bg)
            (eq fore 'unspecified))
        (message "backgound or foreground color is unspecified in active mode line.")
      ;; else
      (let* ((new-colors (selected-window-contrast-adjust-contrast fore
                                               back
                                               contrast-background
                                               contrast-text))
             (new-fore (apply #'selected-window-contrast--rgb-to-hex (nth 0 new-colors)))
             (new-back (apply #'selected-window-contrast--rgb-to-hex (nth 1 new-colors))))
        (set-face-attribute 'mode-line-active nil
                            :foreground new-fore
                            :background new-back)
        t))))

(defvar selected-window-contrast-prev-window nil
  "Saved current window, because `previous-window' is not working.
Used in `selected-window-contrast-mark-small-rectangle-temporary'.")

(defun selected-window-contrast-mark-small-rectangle-temporary (window)
  "Mark a 2x2 rectangle around point for 1 sec, to hightlight WINDOW.
Use `rectangle-mark-mode'.  Deactivate rectangle after 1 second or less."
  (interactive)
  ;; (condition-case err
      (when (and window
                 (eq window (selected-window))
                 (not (window-minibuffer-p window))
                 (not (when selected-window-contrast-prev-window (window-minibuffer-p selected-window-contrast-prev-window))))
        (unless (region-active-p)
          ;; Enable rectangle selection.
          (rectangle-mark-mode 1)
          (rectangle-next-line selected-window-contrast-region-lines) ; 2
          (rectangle-forward-char selected-window-contrast-region-width) ; 8
          (rectangle-exchange-point-and-mark)
          ;; Start timer to deactivate mark and rectangle mode.
          (run-with-timer selected-window-contrast-region-timeout
                          nil (lambda (buf)
                                (with-current-buffer buf
                                  (when (region-active-p)
                                    ;; (exchange-point-and-mark)
                                    (deactivate-mark))))
                          (current-buffer))))
    ;; save current window, because `previous-window' is not working.
    (setq selected-window-contrast-prev-window (selected-window))
    ;; (message "selected-window-contrast debug: %s" err))
    )

(defun selected-window-contrast-highlight-selected-window-with-timeout ()
  "Highlight not selected windows with a different background color.
Timeout 0.1 sec.
For case of opening new frame with new buffer by call:
$ emacsclient -c ~/file"
  (run-with-idle-timer 0.4 nil #'selected-window-contrast-highlight-selected-window))

(defun selected-window-contrast-highlight-selected-window ()
  "Highlight not selected windows with a different background color."
  (let ((cbn (buffer-name (current-buffer)))
        (sw (selected-window)))
    (when (/= (aref cbn 0) ?\s) ; ignore system buffers
      ;; - not selected:
      (when selected-window-contrast-flag
        (walk-windows (lambda (w)
                        (unless (or (eq sw w)
                                    (eq cbn (buffer-name (window-buffer w))))
                          (with-selected-window w
                            (buffer-face-set 'default)
                            (selected-window-contrast-change-window
                             selected-window-contrast-bg-others
                             selected-window-contrast-text-others))))
                      -1)) ; -1 means to not include minibuffer

      ;; - selected:
      (when selected-window-contrast-flag
        (selected-window-contrast-change-window selected-window-contrast-bg-selected
                                                selected-window-contrast-text-selected))
      (if selected-window-contrast-region-flag
        (add-hook 'window-selection-change-functions
                  #'selected-window-contrast-mark-small-rectangle-temporary nil t)
        ;; else
        (remove-hook 'window-selection-change-functions
                     #'selected-window-contrast-mark-small-rectangle-temporary t)))))

(provide 'selected-window-contrast)
;;; selected-window-contrast.el ends here
