;;; timu-symbol-extract.el --- Extract symbols to org table -*- lexical-binding: t; -*-

;; Author: Aimé Bertrand <aime.bertrand@macowners.club>
;; Package-Version: 20260111.1243
;; Package-Revision: 5f3b778eb744
;; Package-Requires: ((emacs "28.1"))
;; Created: 2025-11-25
;; Keywords: defaults tools helper elisp
;; Homepage: https://gitlab.com/aimebertrand/timu-symbol-extract
;; This file is NOT part of GNU Emacs.

;; The MIT License (MIT)
;;
;; Copyright (C) 2023-2025 Aimé Bertrand
;;
;; Permission is hereby granted, free of charge, to any person obtaining
;; a copy of this software and associated documentation files (the
;; "Software"), to deal in the Software without restriction, including
;; without limitation the rights to use, copy, modify, merge, publish,
;; distribute, sublicense, and/or sell copies of the Software, and to
;; permit persons to whom the Software is furnished to do so, subject to
;; the following conditions:
;;
;; The above copyright notice and this permission notice shall be
;; included in all copies or substantial portions of the Software.
;;
;; THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
;; EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
;; MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
;; IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
;; CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
;; TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
;; SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

;;; Commentary:
;; Functions to scan directory for function and variable definitions
;; in elisp files and output to org-mode table

;;; Code:

(require 'lisp-mnt)
(require 'org)


;;;; Customization
(defgroup timu-symbol-extract nil
  "Extract symbols from Emacs Lisp files to org tables."
  :group 'tools
  :prefix "timu-symbol-extract-")

(defcustom timu-symbol-extract-file-pattern "\\.el\\'"
  "Default file pattern for directory scans.
Only files matching this regexp will be processed."
  :type 'regexp
  :group 'timu-symbol-extract)

(defcustom timu-symbol-extract-recursive t
  "Whether to scan directories recursively.
When non-nil, subdirectories are included in the scan."
  :type 'boolean
  :group 'timu-symbol-extract)


;;;; Private helper functions
(defun timu-symbol-extract--from-file (file extractor)
  "Extract definitions from FILE using EXTRACTOR function."
  (with-temp-buffer
    (insert-file-contents file)
    (emacs-lisp-mode)
    (funcall extractor)))

(defun timu-symbol-extract--from-directory (dir extractor &optional pattern)
  "Extract definitions from DIR using EXTRACTOR function.
Optional PATTERN specifies file pattern.
Uses `timu-symbol-extract-file-pattern' if PATTERN is nil.
Uses `timu-symbol-extract-recursive' to determine scan depth."
  (let* ((pattern (or pattern timu-symbol-extract-file-pattern))
         (files (if timu-symbol-extract-recursive
                    (directory-files-recursively dir pattern)
                  (directory-files dir t pattern)))
         (all-definitions '()))
    (dolist (file files)
      (when (file-regular-p file)
        (message "Processing %s..." file)
        (let ((definitions (funcall extractor file)))
          (dolist (def definitions)
            (push (plist-put def :file (file-relative-name file dir))
                  all-definitions)))))
    (nreverse all-definitions)))

(defun timu-symbol-extract--to-org-table (definitions buffer-name label)
  "Create org table from DEFINITIONS in buffer BUFFER-NAME.
LABEL is used in the message (e.g. \"definitions\", \"functions\")."
  (let ((buffer (get-buffer-create buffer-name)))
    (with-current-buffer buffer
      (erase-buffer)
      (org-mode)
      (insert "| Kind | File | Name | Docstring |\n")
      (insert "|-\n")
      (dolist (def definitions)
        (let ((name (plist-get def :name))
              (kind (plist-get def :kind))
              (docstring (or (plist-get def :docstring-first-line) ""))
              (file (or (plist-get def :file) "")))
          (setq docstring (replace-regexp-in-string "|" "\\\\vert{}" docstring))
          (insert (format "| %s | %s | %s | %s |\n" kind file name docstring))))
      (goto-char (point-min))
      (org-table-align))
    (switch-to-buffer buffer)
    (message "Extracted %d %s" (length definitions) label)))


;;;; Core buffer extraction functions
(defun timu-symbol-extract-functions-from-buffer ()
  "Extract all function definitions from current buffer.
Returns a list of plists with :name, :kind, and :docstring-first-line."
  (save-excursion
    (goto-char (point-min))
    (let ((functions '()))
      (while (re-search-forward "^(defun\\s-+\\(\\(?:\\sw\\|\\s_\\)+\\)" nil t)
        (let* ((func-name (match-string 1))
               (func-symbol (intern func-name))
               (func-start (match-beginning 0))
               (func-form nil)
               (kind nil)
               (docstring nil)
               (docstring-first-line ""))
          (save-excursion
            (goto-char func-start)
            (condition-case nil
                (progn
                  (setq func-form (read (current-buffer)))
                  (eval func-form))
              (error nil)))
          (when func-form
            (setq kind (if (commandp func-symbol) "command" "function"))
            (when (and (listp func-form)
                       (>= (length func-form) 4)
                       (stringp (nth 3 func-form)))
              (setq docstring (nth 3 func-form)))
            (when (and (not docstring)
                       (listp func-form)
                       (>= (length func-form) 5)
                       (stringp (nth 4 func-form)))
              (setq docstring (nth 4 func-form)))
            (when docstring
              (setq docstring-first-line
                    (car (split-string docstring "\n" t "[ \t]+"))))
            (push (list :name func-name
                        :kind kind
                        :docstring-first-line docstring-first-line)
                  functions))))
      (nreverse functions))))

(defun timu-symbol-extract-variables-from-buffer ()
  "Extract all variable definitions (defvar and defcustom) from current buffer.
Returns a list of plists with :name, :kind, and :docstring-first-line."
  (save-excursion
    (goto-char (point-min))
    (let ((variables '()))
      (while (re-search-forward
              "^(\\(defvar\\|defcustom\\)\\s-+\\(\\(?:\\sw\\|\\s_\\)+\\)" nil t)
        (let* ((kind (if (equal (match-string 1) "defvar")
                         "variable"
                       "custom variable"))
               (var-name (match-string 2))
               (var-start (match-beginning 0))
               (var-form nil)
               (docstring nil)
               (docstring-first-line ""))
          (save-excursion
            (goto-char var-start)
            (condition-case nil
                (setq var-form (read (current-buffer)))
              (error nil)))
          (when var-form
            (when (and (listp var-form)
                       (>= (length var-form) 4)
                       (stringp (nth 3 var-form)))
              (setq docstring (nth 3 var-form)))
            (when docstring
              (setq docstring-first-line
                    (car (split-string docstring "\n" t "[ \t]+"))))
            (push (list :name var-name
                        :kind kind
                        :docstring-first-line docstring-first-line)
                  variables))))
      (nreverse variables))))

(defun timu-symbol-extract-definitions-from-buffer ()
  "Extract all definitions (functions and variables) from current buffer.
Returns a list of plists with :name, :kind, and :docstring-first-line."
  (append (timu-symbol-extract-functions-from-buffer)
          (timu-symbol-extract-variables-from-buffer)))


;;;; File extraction functions
(defun timu-symbol-extract-definitions-from-file (file)
  "Extract function and variable definitions from FILE."
  (timu-symbol-extract--from-file
   file
   #'timu-symbol-extract-definitions-from-buffer))

(defun timu-symbol-extract-functions-from-file (file)
  "Extract function definitions from FILE."
  (timu-symbol-extract--from-file
   file
   #'timu-symbol-extract-functions-from-buffer))

(defun timu-symbol-extract-variables-from-file (file)
  "Extract variable definitions from FILE."
  (timu-symbol-extract--from-file
   file
   #'timu-symbol-extract-variables-from-buffer))


;;;; Directory extraction functions
(defun timu-symbol-extract-definitions-from-directory (dir &optional pattern)
  "Extract all definitions from elisp files in DIR.
Optional PATTERN specifies file pattern.
Uses `timu-symbol-extract-file-pattern' if PATTERN is nil."
  (timu-symbol-extract--from-directory
   dir #'timu-symbol-extract-definitions-from-file pattern))

(defun timu-symbol-extract-functions-from-directory (dir &optional pattern)
  "Extract all function definitions from elisp files in DIR.
Optional PATTERN specifies file pattern.
Uses `timu-symbol-extract-file-pattern' if PATTERN is nil."
  (timu-symbol-extract--from-directory
   dir #'timu-symbol-extract-functions-from-file pattern))

(defun timu-symbol-extract-variables-from-directory (dir &optional pattern)
  "Extract all variable definitions from elisp files in DIR.
Optional PATTERN specifies file pattern.
Uses `timu-symbol-extract-file-pattern' if PATTERN is nil."
  (timu-symbol-extract--from-directory
   dir #'timu-symbol-extract-variables-from-file pattern))


;;;; Org table functions (directory)
;;;###autoload
(defun timu-symbol-extract-definitions-to-org-table (dir &optional pattern)
  "Create an `org-mode' table of all definitions from DIR.
Optional PATTERN specifies file pattern.
Uses `timu-symbol-extract-file-pattern' if PATTERN is nil."
  (interactive "DDirectory: ")
  (timu-symbol-extract--to-org-table
   (timu-symbol-extract-definitions-from-directory dir pattern)
   "*Definition Extraction*" "definitions"))

;;;###autoload
(defun timu-symbol-extract-functions-to-org-table (dir &optional pattern)
  "Create an `org-mode' table of functions from DIR.
Optional PATTERN specifies file pattern.
Uses `timu-symbol-extract-file-pattern' if PATTERN is nil."
  (interactive "DDirectory: ")
  (timu-symbol-extract--to-org-table
   (timu-symbol-extract-functions-from-directory dir pattern)
   "*Function Extraction*" "functions"))

;;;###autoload
(defun timu-symbol-extract-variables-to-org-table (dir &optional pattern)
  "Create an `org-mode' table of variables from DIR.
Optional PATTERN specifies file pattern.
Uses `timu-symbol-extract-file-pattern' if PATTERN is nil."
  (interactive "DDirectory: ")
  (timu-symbol-extract--to-org-table
   (timu-symbol-extract-variables-from-directory dir pattern)
   "*Variable Extraction*" "variables"))


;;;; Org table functions (current file)
;;;###autoload
(defun timu-symbol-extract-definitions-to-org-table-current-file ()
  "Create an `org-mode' table of all definitions from current buffer."
  (interactive)
  (let ((current-file (or (buffer-file-name) (buffer-name)))
        (definitions (timu-symbol-extract-definitions-from-buffer)))
    (dolist (def definitions)
      (plist-put def :file current-file))
    (timu-symbol-extract--to-org-table
     definitions "*Definition Extraction*" "definitions")))

;;;###autoload
(defun timu-symbol-extract-functions-to-org-table-current-file ()
  "Create an `org-mode' table of functions from current buffer."
  (interactive)
  (let ((current-file (or (buffer-file-name) (buffer-name)))
        (functions (timu-symbol-extract-functions-from-buffer)))
    (dolist (func functions)
      (plist-put func :file current-file))
    (timu-symbol-extract--to-org-table
     functions "*Function Extraction*" "functions")))

;;;###autoload
(defun timu-symbol-extract-variables-to-org-table-current-file ()
  "Create an `org-mode' table of variables from current buffer."
  (interactive)
  (let ((current-file (or (buffer-file-name) (buffer-name)))
        (variables (timu-symbol-extract-variables-from-buffer)))
    (dolist (var variables)
      (plist-put var :file current-file))
    (timu-symbol-extract--to-org-table
     variables "*Variable Extraction*" "variables")))


(provide 'timu-symbol-extract)

;;; timu-symbol-extract.el ends here
