;;; tomorrow-night-deepblue-theme.el --- The Tomorrow Night Deepblue color theme -*- lexical-binding: t -*-

;; Copyright (C) 2024-2025 James Cherti

;; Author: James Cherti
;; URL: https://github.com/jamescherti/tomorrow-night-deepblue-theme.el
;; Package-Requires: ((emacs "26.1"))
;; Keywords: faces themes
;; Package-Version: 20251015.1359
;; Package-Revision: 85084ded2c4f
;; SPDX-License-Identifier:  GPL-3.0-or-later

;; This file is NOT part of GNU Emacs.

;; This file is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; This file is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs.  If not, see <https://www.gnu.org/licenses/>.

;;; Commentary:
;; The Emacs theme Tomorrow Night Deepblue is a beautiful deep blue variant of
;; the Tomorrow Night theme, which is renowned for its elegant color
;; palette that is pleasing to the eyes.
;;
;; The Tomorrow Night Deepblue features a deep blue background color that
;; creates a calming atmosphere. The contrasting colors make it easy to
;; distinguish between different elements of your code. The
;; tomorrow-night-deepblue theme is also a great choice for programmer who miss
;; the blue themes that were trendy a few years ago.
;;
;; The theme was inspired by classic text editors such as QuickBASIC, RHIDE, and
;; Turbo Pascal, as well as tools such as Midnight Commander which featured blue
;; backgrounds by default. There's something special about the early days of
;; programming and the tools we used that brings back fond memories.
;;
;; Installation from MELPA:
;; ------------------------
;; (use-package tomorrow-night-deepblue-theme
;;   :ensure t
;;   :config
;;   ;; Disable all themes and load the Tomorrow Night Deep Blue theme
;;   (mapc #'disable-theme custom-enabled-themes)
;;   ;; Load the tomorrow-night-deepblue theme
;;   (load-theme 'tomorrow-night-deepblue t))
;;
;; Links:
;; ------
;; - GitHub: https://github.com/jamescherti/tomorrow-night-deepblue-theme.el
;;

;;; Code:

(defconst tomorrow-night-deepblue-theme
  '((night-deepblue . ((background . "#00006f")
                       (current-line . "#00346e")
                       (selection . "#003f8e")
                       (code . "#082572")
                       (code2 . "#17416f")
                       (hl-line . "#1f5189")
                       (foreground . "#ffffff")
                       (comment . "#7285b7")
                       (red . "#ff9da4")
                       (orange . "#ffc58f")
                       (yellow . "#ffeead")
                       (green . "#d1f1a9")
                       (aqua . "#99ffff")
                       (blue . "#bbdaff")
                       (purple . "#ebbbff")
                       (fg-removed . "#ffbfbf")
                       (fg-changed . "#efef80")
                       (fg-added . "#a0e0a0")
                       (bg-removed . "#450f1f")
                       (bg-changed . "#335533")
                       (bg-added . "#335533") ;; mod
                       (bg-added-refine . "#3f6234")
                       (bg-changed-refine . "#615f1f")
                       (bg-removed-refine . "#7f3325")
                       (darkblue . "#00005f")))))

(defmacro tomorrow-night-deepblue-theme--with-colors (mode &rest body)
  "Execute BODY in a scope with variables bound to various tomorrow colors.

MODE should be set to either day, night, night-eighties, night-blue or
night-bright."
  `(let* ((colors (or (cdr (assoc ,mode tomorrow-night-deepblue-theme))
                      (error "No such theme flavor")))
          (background   (cdr (assoc 'background colors)))
          (current-line (cdr (assoc 'current-line colors)))
          (selection    (cdr (assoc 'selection colors)))
          (fg-removed   (cdr (assoc 'fg-removed colors)))
          (fg-changed   (cdr (assoc 'fg-changed colors)))
          (fg-added     (cdr (assoc 'fg-added colors)))
          (bg-removed   (cdr (assoc 'bg-removed colors)))
          (bg-added     (cdr (assoc 'bg-added colors)))
          (bg-changed   (cdr (assoc 'bg-changed colors)))
          (bg-added-refine (cdr (assoc 'bg-added-refine colors)))
          (bg-changed-refine (cdr (assoc 'bg-changed-refine colors)))
          (bg-removed-refine (cdr (assoc 'bg-removed-refine colors)))
          (code         (cdr (assoc 'code colors)))
          (code2        (cdr (assoc 'code2 colors)))
          (hl-line      (cdr (assoc 'hl-line colors)))
          (foreground   (cdr (assoc 'foreground colors)))
          (comment      (cdr (assoc 'comment colors)))
          (red          (cdr (assoc 'red colors)))
          (orange       (cdr (assoc 'orange colors)))
          (yellow       (cdr (assoc 'yellow colors)))
          (green        (cdr (assoc 'green colors)))
          (aqua         (cdr (assoc 'aqua colors)))
          (blue         (cdr (assoc 'blue colors)))
          (purple       (cdr (assoc 'purple colors)))
          (darkblue     (cdr (assoc 'darkblue colors)))
          (contrast-bg  (cdr (assoc 'current-line colors)))
          (low-contrast-bg (cdr (assoc 'current-line colors)))
          (highlight    (cdr (assoc 'current-line colors)))
          (term-white   (if (eq ,mode 'day) contrast-bg comment))
          (term-black   (if (eq ,mode 'day) comment contrast-bg))
          (class '((class color) (min-colors 89))))
     ,@body))

(defmacro tomorrow-night-deepblue-theme--face-specs ()
  "Define a list of face specs and return a backquote.

It expects to be evaluated in a scope in which the various color
names to which it refers are bound."
  (quote
   `(;; Standard font lock faces
     (default ((,class (:foreground ,foreground :background ,background))))
     (bold ((,class (:weight bold))))
     (bold-italic ((,class (:slant italic :weight bold))))
     (underline ((,class (:underline t))))
     (italic ((,class (:slant italic))))
     (shadow ((,class (:foreground ,comment))))
     (success ((,class (:foreground ,green))))  ;; Change to ,comment?
     (error ((,class (:foreground ,red))))
     (warning ((,class (:foreground ,red))))
     (border ((,class (:background ,contrast-bg :foreground ,highlight))))
     (border-glyph ((,class (nil))))
     (font-lock-bracket-face ((,class (:foreground ,blue))))
     (font-lock-builtin-face ((,class (:foreground ,blue))))
     (font-lock-comment-delimiter-face ((,class (:foreground ,comment))))
     (font-lock-comment-face ((,class (:foreground ,comment))))
     (font-lock-constant-face ((,class (:foreground ,blue))))
     (font-lock-doc-face ((,class (:foreground ,green))))
     (font-lock-doc-string-face ((,class (:foreground ,yellow))))
     (font-lock-escape-face ((,class (:foreground ,yellow))))
     (font-lock-function-name-face ((,class (:foreground ,blue))))
     (font-lock-function-call-face ((,class (:foreground ,blue))))
     (highlight-defined-face-name-face ((,class (:foreground ,foreground))))
     (font-lock-keyword-face ((,class (:foreground ,purple))))
     (font-lock-misc-punctuation-face ((,class (:inherit font-lock-punctuation-face))))
     (font-lock-negation-char-face ((,class (:foreground ,orange))))
     (font-lock-number-face ((,class (:foreground ,blue))))
     (font-lock-operator-face ((,class (:foreground ,blue))))
     (font-lock-preprocessor-face ((,class (:foreground ,purple))))
     ;; (font-lock-variable-name-face ((,class (:foreground ,yellow))))
     (font-lock-variable-name-face ((,class (:foreground ,orange))))
     (font-lock-property-use-face ((,class (:inherit font-lock-variable-name-face))))
     (font-lock-property-name-face ((,class (:inherit font-lock-variable-name-face))))
     (font-lock-variable-use-face ((,class (:inherit font-lock-variable-name-face))))
     (font-lock-string-face ((,class (:foreground ,green))))
     (font-lock-regexp-face ((,class (:inherit font-lock-string-face))))
     (font-lock-regexp-grouping-backslash ((,class (:foreground ,orange))))
     (font-lock-regexp-grouping-construct ((,class (:foreground ,purple))))
     (font-lock-type-face ((,class (:foreground ,blue))))
     (mode-line ((,class (:foreground ,foreground :background ,contrast-bg :weight normal :box (:line-width 1 :color ,contrast-bg)))))
     (mode-line-buffer-id ((,class (:foreground ,purple :background unspecified))))
     (mode-line-emphasis ((,class (:foreground ,foreground :slant italic))))
     (mode-line-highlight ((,class (:foreground ,purple :box nil :weight bold))))
     (mode-line-inactive ((,class (:inherit mode-line :foreground ,comment :background ,highlight :weight normal))))
     (header-line ((,class (:inherit mode-line-inactive :foreground ,blue :background unspecified))))
     (match ((,class (:foreground ,blue :background ,background :inverse-video t))))
     (highlight ((,class (:inverse-video nil :background ,highlight))))
     (highlight-80+ ((,class (:background ,contrast-bg))))
     (highlight-symbol-face ((,class (:inherit highlight))))
     (link ((,class (:foreground ,blue :underline t))))
     (link-visited ((,class (:foreground ,purple))))
     (font-lock-warning-face ((,class (:weight bold :foreground ,red))))
     (lazy-highlight ((,class (:foreground ,yellow :background ,background :inverse-video t))))
     (isearch-lazy-highlight-face ((,class (:inherit lazy-highlight))))
     (ansi-color-black ((,class (:foreground ,term-black :background ,term-black))))
     (ansi-color-blue ((,class (:foreground ,blue :background ,blue))))
     (ansi-color-cyan ((,class (:foreground ,aqua :background ,aqua))))
     (ansi-color-green ((,class (:foreground ,green :background ,green))))
     (ansi-color-magenta ((,class (:foreground ,purple :background ,purple))))
     (ansi-color-red ((,class (:foreground ,red :background ,red))))
     (ansi-color-white ((,class (:foreground ,term-white :background ,term-white))))
     (ansi-color-yellow ((,class (:foreground ,yellow :background ,yellow))))
     (ansi-color-bright-black ((,class (:inherit ansi-color-black :weight bold))))
     (ansi-color-bright-blue ((,class (:inherit ansi-color-blue :weight bold))))
     (ansi-color-bright-cyan ((,class (:inherit ansi-color-cyan :weight bold))))
     (ansi-color-bright-green ((,class (:inherit ansi-color-green :weight bold))))
     (ansi-color-bright-magenta ((,class (:inherit ansi-color-magenta :weight bold))))
     (ansi-color-bright-red ((,class (:inherit ansi-color-red :weight bold))))
     (ansi-color-bright-white ((,class (:inherit ansi-color-white :weight bold))))
     (ansi-color-bright-yellow ((,class (:inherit ansi-color-yellow :weight bold))))
     (flymake-end-of-line-diagnostics-face ((,class (:inherit italic :height 0.85 :box (:line-width 1)))))
     (flymake-errline ((,class (:underline ,red))))
     (flymake-error ((,class (:underline (:style wave :color ,red)))))
     (flymake-error-echo ((,class (:inherit error))))
     (flymake-error-echo-at-eol ((,class (:inherit flymake-end-of-line-diagnostics-face :foreground ,red))))
     (flymake-note ((,class (:underline (:style wave :color ,aqua)))))
     (flymake-note-echo ((,class (:foreground ,aqua))))
     (flymake-note-echo-at-eol ((,class (:inherit flymake-end-of-line-diagnostics-face :foreground ,aqua))))
     (flymake-warning ((,class (:underline (:style wave :color ,orange)))))
     (flymake-warning-echo ((,class (:foreground ,orange))))
     (flymake-warning-echo-at-eol ((,class (:inherit flymake-end-of-line-diagnostics-face :foreground ,orange))))
     (flymake-warnline ((,class (:underline ,orange))))
     (show-paren-match ((,class (:inverse-video t))))
     (show-paren-mismatch ((,class (:background ,red))))
     (Info-quoted ((,class (:inherit font-lock-constant-face))))
     (TeX-fold-folded-face ((,class (:foreground ,purple))))
     (TeX-fold-unfolded-face ((,class (:inherit highlight))))
     (antlr-keyword ((,class (:inherit font-lock-keyword-face))))
     (antlr-literal ((,class (:inherit font-lock-constant-face))))
     (antlr-ruledef ((,class (:inherit font-lock-function-name-face))))
     (antlr-ruleref ((,class (:inherit font-lock-type-face))))
     (antlr-syntax ((,class (:inherit font-lock-constant-face))))
     (antlr-tokendef ((,class (:inherit font-lock-function-name-face))))
     (antlr-tokenref ((,class (:inherit font-lock-type-face))))
     (anzu-match-1 ((,class (:foreground ,yellow ))))
     (anzu-match-2 ((,class (:foreground ,orange))))
     (anzu-match-3 ((,class (:foreground ,blue))))
     (anzu-mode-line ((,class (:foreground ,orange))))
     (anzu-mode-line-no-match ((,class (:foreground ,red))))
     (anzu-replace-highlight ((,class (:inherit lazy-highlight))))
     (anzu-replace-to ((,class (:inherit isearch))))
     (avy-background-face ((,class (:foreground ,contrast-bg))))
     (avy-lead-face ((,class (:foreground ,background :background ,yellow))))
     (avy-lead-face-0 ((,class (:foreground ,background :background ,blue))))
     (avy-lead-face-1 ((,class (:foreground ,background :background ,aqua))))
     (avy-lead-face-2 ((,class (:foreground ,background :background ,orange))))
     (aw-background-face ((,class (:foreground ,contrast-bg))))
     (aw-leading-char-face ((,class (:foreground ,yellow))))
     (bm-face ((,class (:background ,contrast-bg :foreground ,foreground :extend t))))
     (bm-persistent-face ((,class (:background ,blue :foreground ,background :extend t))))
     (bmkp-*-mark ((,class (:foreground ,background :background ,yellow))))
     (bmkp->-mark ((,class (:foreground ,yellow))))
     (bmkp-D-mark ((,class (:foreground ,background :background ,red))))
     (bmkp-X-mark ((,class (:foreground ,red))))
     (bmkp-a-mark ((,class (:background ,red))))
     (bmkp-bad-bookmark ((,class (:foreground ,background :background ,yellow))))
     (bmkp-bookmark-file ((,class (:foreground ,purple :background ,contrast-bg))))
     (bmkp-bookmark-list ((,class (:background ,contrast-bg))))
     (bmkp-buffer ((,class (:foreground ,blue))))
     (bmkp-desktop ((,class (:foreground ,background :background ,purple))))
     (bmkp-file-handler ((,class (:background ,red))))
     (bmkp-function ((,class (:foreground ,green))))
     (bmkp-gnus ((,class (:foreground ,orange))))
     (bmkp-heading ((,class (:foreground ,yellow))))
     (bmkp-info ((,class (:foreground ,aqua))))
     (bmkp-light-autonamed ((,class (:foreground ,aqua :background ,highlight))))
     (bmkp-light-autonamed-region ((,class (:foreground ,red :background ,highlight))))
     (bmkp-light-fringe-autonamed ((,class (:foreground ,contrast-bg :background ,purple))))
     (bmkp-light-fringe-non-autonamed ((,class (:foreground ,contrast-bg :background ,green))))
     (bmkp-light-mark ((,class (:foreground ,background :background ,aqua))))
     (bmkp-light-non-autonamed ((,class (:foreground ,purple :background ,highlight))))
     (bmkp-light-non-autonamed-region ((,class (:foreground ,orange :background ,highlight))))
     (bmkp-local-directory ((,class (:foreground ,background :background ,purple))))
     (bmkp-local-file-with-region ((,class (:foreground ,yellow))))
     (bmkp-local-file-without-region ((,class (:foreground ,comment))))
     (bmkp-man ((,class (:foreground ,purple))))
     (bmkp-no-jump ((,class (:foreground ,comment))))
     (bmkp-no-local ((,class (:foreground ,yellow))))
     (bmkp-non-file ((,class (:foreground ,green))))
     (bmkp-remote-file ((,class (:foreground ,orange))))
     (bmkp-sequence ((,class (:foreground ,blue))))
     (bmkp-su-or-sudo ((,class (:foreground ,red))))
     (bmkp-t-mark ((,class (:foreground ,purple))))
     (bmkp-url ((,class (:foreground ,blue :underline t))))
     (bmkp-variable-list ((,class (:foreground ,green))))
     (bookmark-face ((,class (:foreground ,background :background ,yellow))))
     (button ((,class (:underline t :foreground ,aqua))))
     (caml-types-def-face ((,class (:inherit highlight :box (:color ,blue :line-width -1)))))
     (caml-types-expr-face ((,class (:inherit highlight :box (:color ,aqua :line-width -1)))))
     (caml-types-occ-face ((,class (:inherit highlight :box (:color ,yellow :line-width -1)))))
     (caml-types-scope-face ((,class (:inherit highlight :box (:color ,green :line-width -1)))))
     (caml-types-typed-face ((,class (:inherit highlight :box (:color ,purple :line-width -1)))))
     (cider-debug-code-overlay-face ((,class (:background ,contrast-bg))))
     (cider-deprecated-face ((,class (:foreground ,contrast-bg :background ,yellow))))
     (cider-result-overlay-face ((,class (:background ,contrast-bg :box (:line-width -1 :color ,yellow)))))
     (cider-enlightened-face ((,class (:inherit cider-result-overlay-face :box (:color ,orange :line-width -1)))))
     (cider-enlightened-local-face ((,class (:weight bold :foreground ,orange))))
     (cider-error-highlight-face ((,class (:underline (:style wave :color ,red) :inherit unspecified))))
     (cider-fringe-good-face ((,class (:foreground ,green))))
     (cider-instrumented-face ((,class (:box (:color ,red :line-width -1)))))
     (cider-test-error-face ((,class (:foreground ,contrast-bg :background ,orange))))
     (cider-test-failure-face ((,class (:foreground ,contrast-bg :background ,red))))
     (cider-test-success-face ((,class (:foreground ,contrast-bg :background ,green))))
     (cider-traced-face ((,class (:box ,aqua :line-width -1))))
     (cider-warning-highlight-face ((,class (:underline (:style wave :color ,yellow) :inherit unspecified))))
     (circe-fool-face ((,class (:foreground ,comment))))
     (circe-highlight-nick-face ((,class (:foreground ,orange))))
     (circe-my-message-face ((,class (:foreground ,green))))
     (circe-prompt-face ((,class (:foreground ,blue))))
     (circe-server-face ((,class (:foreground ,green))))
     (circe-topic-diff-new-face ((,class (:foreground ,blue))))
     (circe-topic-diff-removed-face ((,class (:foreground ,red))))
     (clojure-braces ((,class (:foreground ,green))))
     (clojure-brackets ((,class (:foreground ,yellow))))
     (clojure-double-quote ((,class (:foreground ,aqua :background unspecified))))
     (clojure-java-call ((,class (:foreground ,purple))))
     (clojure-keyword ((,class (:foreground ,yellow))))
     (clojure-parens ((,class (:foreground ,foreground))))
     (clojure-special ((,class (:foreground ,blue))))
     (clojure-test-error-face ((,class (:background unspecified :inherit flymake-errline))))
     (clojure-test-failure-face ((,class (:background unspecified :inherit flymake-warnline))))
     (clojure-test-success-face ((,class (:background unspecified :foreground unspecified :underline ,green))))
     (coffee-mode-class-name ((,class (:foreground ,orange :weight bold))))
     (coffee-mode-function-param ((,class (:foreground ,purple))))
     (company-echo-common ((,class (:inherit company-echo :foreground ,red))))
     (company-preview ((,class (:foreground ,comment :background ,contrast-bg))))
     (company-preview-common ((,class (:inherit company-preview :foreground ,red))))
     (company-preview-search ((,class (:inherit company-preview :foreground ,blue))))
     (company-scrollbar-bg ((,class (:inherit 'company-tooltip :background ,highlight))))
     (company-scrollbar-fg ((,class (:background ,contrast-bg))))
     (company-tooltip ((,class (:background ,contrast-bg))))
     (company-tooltip-annotation ((,class (:inherit company-tooltip :foreground ,green))))
     (company-tooltip-annotation-selection ((,class (:inherit company-tooltip-selection :foreground ,green))))
     (company-tooltip-common ((,class (:inherit company-tooltip :foreground ,red))))
     (company-tooltip-common-selection ((,class (:inherit company-tooltip-selection :foreground ,red))))
     (company-tooltip-search ((,class (:inherit company-tooltip :foreground ,blue))))
     (company-tooltip-selection ((,class (:foreground ,comment :inverse-video t))))
     (compilation-column-number ((,class (:foreground ,yellow))))
     (compilation-line-number ((,class (:foreground ,yellow))))
     (compilation-message-face ((,class (:foreground ,blue))))
     (compilation-mode-line-exit ((,class (:foreground ,green))))
     (compilation-mode-line-fail ((,class (:foreground ,red))))
     (compilation-mode-line-run ((,class (:foreground ,blue))))
     (completions-annotations ((,class (:foreground ,comment :slant italic))))
     (completions-common-part ((,class (:foreground ,blue))))
     (completions-first-difference ((,class (:foreground ,orange :weight bold))))
     (counsel-css-selector-depth-face-1 ((,class (:foreground ,yellow))))
     (counsel-css-selector-depth-face-2 ((,class (:foreground ,orange))))
     (counsel-css-selector-depth-face-3 ((,class (:foreground ,green))))
     (counsel-css-selector-depth-face-4 ((,class (:foreground ,aqua))))
     (counsel-css-selector-depth-face-5 ((,class (:foreground ,blue))))
     (counsel-css-selector-depth-face-6 ((,class (:foreground ,purple))))
     (cperl-array-face ((,class (:foreground ,blue :weight bold))))
     (cperl-hash-face ((,class (:foreground ,red :slant italic))))
     (cperl-nonoverridable-face ((,class (:foreground ,purple))))
     (cscope-file-face ((,class (:foreground ,green))))
     (cscope-function-face ((,class (:foreground ,blue))))
     (cscope-line-number-face ((,class (:foreground ,red))))
     (cscope-separator-face ((,class (:bold t :overline t :underline t :foreground ,purple))))
     (csv-separator-face ((,class (:foreground ,orange))))
     (cursor ((,class (:background ,red))))
     (custom-group-tag ((,class (:foreground ,blue))))
     (custom-state ((,class (:foreground ,green))))
     (custom-variable-tag ((,class (:foreground ,blue))))
     (debbugs-gnu-done ((,class (:foreground ,comment))))
     (debbugs-gnu-forwarded ((,class (:foreground ,yellow))))
     (debbugs-gnu-handled ((,class (:foreground ,green))))
     (debbugs-gnu-new ((,class (:foreground ,red))))
     (debbugs-gnu-pending ((,class (:foreground ,blue))))
     (debbugs-gnu-stale-1 ((,class (:foreground ,orange))))
     (debbugs-gnu-stale-2 ((,class (:foreground ,green))))
     (debbugs-gnu-stale-3 ((,class (:foreground ,purple))))
     (debbugs-gnu-stale-4 ((,class (:foreground ,aqua))))
     (debbugs-gnu-stale-5 ((,class (:foreground ,foreground))))
     (debbugs-gnu-tagged ((,class (:foreground ,red))))
     (define-it-headline-face ((,class (:foreground ,blue :bold t))))
     (define-it-pop-tip-color ((,class (:foreground ,comment :background ,contrast-bg))))
     (define-it-sense-number-face ((,class (:foreground ,purple :bold t))))
     (define-it-type-face ((,class (:foreground ,aqua))))
     (define-it-var-face ((,class (:foreground ,orange :bold t))))
     (diary ((,class (:foreground ,yellow))))
     (vdiff-subtraction-face ((,class (:foreground ,background :background ,red))))
     (vdiff-closed-fold-face ((,class (:foreground ,blue :background ,darkblue))))
     (vdiff-refine-changed ((,class (:foreground ,background :background ,blue))))
     (diff-context ((,class ())))
     (yas-field-highlight-face ((,class (:background ,contrast-bg))))
     (diff-added ((,class (:background ,bg-added :foreground ,fg-added))))
     (diff-changed ((,class (:background ,bg-changed :foreground ,fg-changed :extend t))))
     (diff-removed ((,class (:background ,bg-removed :foreground ,fg-removed))))
     ;; (diff-indicator-added ((,c :inherit diff-added :foreground ,fg-added)))
     (diff-indicator-added ((,class (:inherit diff-added :foreground ,fg-added))))
     ;; (diff-indicator-changed ((,c :inherit diff-changed :foreground ,fg-changed)))
     (diff-indicator-changed ((,class (:inherit diff-changed :foreground ,fg-changed))))
     ;; (diff-indicator-removed ((,c :inherit diff-removed :foreground ,fg-removed)))
     (diff-indicator-removed ((,class (:inherit diff-removed :foreground ,fg-removed))))
     ;; (diff-file-header ((,c :inherit bold)))
     (diff-file-header ((,class (:background ,selection))))
     (diff-header ((,class (:background ,current-line))))
     (diff-hl-change ((,class (:background ,bg-changed-refine))))
     ;; (diff-hl-delete ((,c :background ,bg-removed-refine)))
     (diff-hl-delete ((,class (:foreground ,background :background ,red))))
     (diff-hl-insert ((,class (:background ,bg-added-refine))))
     ;; (diff-hunk-header ((,c :inherit bold :background ,bg-alt)))
     (diff-hunk-header ((,class (:background ,current-line :foreground ,purple))))
     (diff-refine-added ((,class (:background ,bg-added-refine :foreground ,fg-added))))
     (diff-refine-removed ((,class (:background ,bg-removed-refine :foreground ,fg-removed))))
     ;; (diff-hl-reverted-hunk-highlight ((,c :background ,fg-main :foreground ,bg-main)))
     ;; (diff-changed-unspecified ((,c :inherit diff-changed)))
     ;; (diff-refine-changed ((,c :background ,bg-changed-refine :foreground ,fg-changed)))
     ;; (diff-error ((,c :inherit error)))
     ;; (diff-function ((,c :background ,bg-alt)))
     ;; (diff-index ((,c :inherit italic)))
     ;; (diff-nonexistent ((,c :inherit bold)))
     (dired-async-failures ((,class (:inherit error))))
     (dired-async-message ((,class (:inherit success))))
     (dired-async-mode-message ((,class (:inherit warning))))
     (dired-mark ((,class (:foreground ,blue))))
     (dired-marked ((,class (:foreground ,green))))
     (dired-narrow-blink ((,class (:foreground ,background :background ,yellow))))
     (diredfl-compressed-file-name ((,class (:foreground ,blue))))
     (diredfl-compressed-file-suffix ((,class (:foreground ,blue))))
     (diredfl-date-time ((,class (:foreground ,blue))))
     (diredfl-deletion ((,class (:inherit error :inverse-video t))))
     (diredfl-deletion-file-name ((,class (:inherit error))))
     (diredfl-dir-heading ((,class (:foreground ,green :weight bold))))
     (diredfl-dir-name ((,class (:foreground ,aqua))))
     (diredfl-dir-priv ((,class (:foreground ,aqua :background unspecified))))
     (diredfl-exec-priv ((,class (:foreground ,orange :background unspecified))))
     (diredfl-executable-tag ((,class (:foreground ,red :background unspecified))))
     (diredfl-file-name ((,class (:foreground ,yellow))))
     (diredfl-file-suffix ((,class (:foreground ,green))))
     (diredfl-flag-mark ((,class (:foreground ,green :inverse-video t))))
     (diredfl-flag-mark-line ((,class (:background unspecified :inherit highlight))))
     (diredfl-ignored-file-name ((,class (:foreground ,comment))))
     (diredfl-link-priv ((,class (:background unspecified :foreground ,purple))))
     (diredfl-mode-line-flagged ((,class (:foreground ,red))))
     (diredfl-mode-line-marked ((,class (:foreground ,green))))
     (diredfl-no-priv ((,class (:background unspecified))))
     (diredfl-number ((,class (:foreground ,yellow))))
     (diredfl-other-priv ((,class (:background unspecified :foreground ,purple))))
     (diredfl-rare-priv ((,class (:foreground ,red :background unspecified))))
     (diredfl-read-priv ((,class (:foreground ,green :background unspecified))))
     (diredfl-symlink ((,class (:foreground ,purple))))
     (diredfl-write-priv ((,class (:foreground ,yellow :background unspecified))))
     (diredp-compressed-file-name ((,class (:inherit diredfl-compressed-file-name))))
     (diredp-compressed-file-suffix ((,class (:foreground ,blue))))
     (diredp-date-time ((,class (:inherit diredfl-date-time))))
     (diredp-deletion ((,class (:inherit diredfl-deletion))))
     (diredp-deletion-file-name ((,class (:inherit diredfl-deletion-file-name))))
     (diredp-dir-heading ((,class (:inherit diredfl-dir-heading))))
     (diredp-dir-name ((,class (:inherit diredfl-dir-name))))
     (diredp-dir-priv ((,class (:inherit diredfl-dir-priv))))
     (diredp-exec-priv ((,class (:inherit diredfl-exec-priv))))
     (diredp-executable-tag ((,class (:inherit diredfl-executable-tag))))
     (diredp-file-name ((,class (:inherit diredfl-file-name))))
     (diredp-file-suffix ((,class (:inherit diredfl-file-suffix))))
     (diredp-flag-mark ((,class (:inherit diredfl-flag-mark))))
     (diredp-flag-mark-line ((,class (:inherit diredfl-flag-mark-line))))
     (diredp-ignored-file-name ((,class (:inherit diredfl-ignored-file-name))))
     (diredp-link-priv ((,class (:inherit diredfl-link-priv))))
     (diredp-mode-line-flagged ((,class (:inherit diredfl-mode-line-flagged))))
     (diredp-mode-line-marked ((,class (:inherit diredfl-mode-line-marked))))
     (diredp-no-priv ((,class (:inherit diredfl-no-priv))))
     (diredp-number ((,class (:inherit diredfl-number))))
     (diredp-other-priv ((,class (:inherit diredfl-other-priv))))
     (diredp-rare-priv ((,class (:inherit diredfl-rare-priv))))
     (diredp-read-priv ((,class (:inherit diredfl-read-priv))))
     (diredp-symlink ((,class (:inherit diredfl-symlink))))
     (diredp-write-priv ((,class (:inherit diredfl-write-priv))))
     (e2wm:face-history-list-normal ((,class (:foreground ,foreground))))
     (e2wm:face-history-list-select1 ((,class (:foreground ,aqua))))
     (e2wm:face-history-list-select2 ((,class (:foreground ,yellow))))
     (ediff-current-diff-A ((,class (:background ,bg-removed :foreground ,fg-removed))))
     ;; TODO `(ediff-current-diff-Ancestor ((,c :background ,bg-region)))
     (ediff-current-diff-Ancestor ((,class (:foreground ,aqua :background ,highlight)))) ;; TODO
     (ediff-current-diff-B ((,class (:background ,bg-added :foreground ,fg-added))))
     (ediff-current-diff-C ((,class (:background ,bg-changed :foreground ,fg-changed))))
     ;; `(ediff-even-diff-A ((,c :background ,bg-dim)))
     (ediff-even-diff-A ((,class (:foreground ,blue :background ,contrast-bg :extend t))))
     ;; `(ediff-even-diff-Ancestor ((,c :background ,bg-dim)))
     (ediff-even-diff-Ancestor ((,class (:foreground ,purple :background ,highlight))))
     ;; `(ediff-even-diff-B ((,c :background ,bg-dim)))
     ;; `(ediff-even-diff-C ((,c :background ,bg-dim)))
     ;; TODO `(ediff-fine-diff-A ((,c )))
     (ediff-even-diff-B ((,class (:foreground ,blue :background ,darkblue :extend t))))
     (ediff-even-diff-C ((,class (:foreground ,blue :background ,darkblue :extend t))))
     (ediff-fine-diff-A ((,class (:background ,bg-removed-refine :foreground ,fg-removed))))
     ;; `(ediff-fine-diff-Ancestor ((,c :background ,bg-active :foreground ,fg-intense)))
     (ediff-fine-diff-Ancestor ((,class (:foreground ,yellow :background ,highlight))))
     (ediff-fine-diff-B ((,class (:background ,bg-added-refine :foreground ,fg-added))))
     (ediff-fine-diff-C ((,class (:background ,bg-changed-refine :foreground ,fg-changed))))
     (ediff-odd-diff-A  ((,class (:inherit ediff-even-diff-A))))
     (ediff-odd-diff-Ancestor ((,class (:inherit ediff-even-diff-Ancestor))))
     (ediff-odd-diff-B  ((,class (:inherit ediff-even-diff-B))))
     (ediff-odd-diff-C ((,class (:inherit ediff-even-diff-C))))
     (edts-face-error-line ((,class (:background unspecified :inherit flymake-errline))))
     (edts-face-error-mode-line ((,class (:background unspecified :foreground ,red :weight bold))))
     (edts-face-warning-line ((,class (:background unspecified :inherit flymake-warnline))))
     (edts-face-warning-mode-line ((,class (:background unspecified :foreground ,orange :weight bold))))
     (eglot-diagnostic-tag-unnecessary-face ((,class (:underline ,red))))
     (elfeed-log-debug-level-face ((,class (:foreground ,comment))))
     (elfeed-log-error-level-face ((,class (:inherit error))))
     (elfeed-log-info-level-face ((,class (:inherit success))))
     (elfeed-log-warn-level-face ((,class (:inherit warning))))
     (elfeed-search-date-face ((,class (:foreground ,blue))))
     (elfeed-search-feed-face ((,class (:foreground ,yellow))))
     (elfeed-search-tag-face ((,class (:foreground ,comment))))
     (elfeed-search-title-face ((,class (:foreground ,comment))))
     (elfeed-search-unread-count-face ((,class (:foreground ,yellow))))
     (elfeed-search-unread-title-face ((,class (:foreground ,foreground :weight bold))))
     (emms-browser-album-face ((,class (:foreground ,aqua))))
     (emms-browser-artist-face ((,class (:foreground ,purple))))
     (emms-browser-track-face ((,class (:foreground ,yellow))))
     (emms-browser-year/genre-face ((,class (:foreground ,blue))))
     (emms-playlist-selected-face ((,class (:inverse-video t))))
     (emms-playlist-track-face ((,class (:foreground ,yellow))))
     (erb-comment-delim-face ((,class (:background ,contrast-bg))))
     (erb-comment-face ((,class (:background ,contrast-bg :weight bold))))
     (erb-delim-face ((,class (:background ,contrast-bg))))
     (erb-exec-delim-face ((,class (:background ,contrast-bg))))
     (erb-exec-face ((,class (:background ,contrast-bg :weight bold))))
     (erb-out-delim-face ((,class (:background ,contrast-bg))))
     (erb-out-face ((,class (:background ,contrast-bg :weight bold))))
     (erc-current-nick-face ((,class (:foreground ,green))))
     (erc-direct-msg-face ((,class (:foreground ,orange))))
     (erc-error-face ((,class (:foreground ,red))))
     (erc-header-face ((,class (:foreground ,foreground :background ,highlight))))
     (erc-input-face ((,class (:foreground ,green))))
     (erc-keyword-face ((,class (:foreground ,yellow))))
     (erc-my-nick-face ((,class (:foreground ,green))))
     (erc-nick-default-face ((,class (:weight normal :foreground ,purple))))
     (erc-nick-msg-face ((,class (:weight normal :foreground ,yellow))))
     (erc-notice-face ((,class (:foreground ,comment))))
     (erc-pal-face ((,class (:foreground ,orange))))
     (erc-prompt-face ((,class (:foreground ,blue))))
     (erc-timestamp-face ((,class (:foreground ,aqua))))
     (ert-test-result-expected ((,class (:inherit success))))
     (ert-test-result-unexpected ((,class (:inherit error))))
     (eshell-ls-archive ((,class (:foreground ,blue))))
     (eshell-ls-backup ((,class (:foreground ,comment))))
     (eshell-ls-clutter ((,class (:foreground ,orange :weight bold))))
     (eshell-ls-directory ((,class :foreground ,blue :weight bold)))
     (eshell-ls-executable ((,class (:foreground ,yellow :weight bold))))
     (eshell-ls-missing ((,class (:foreground ,red :weight bold))))
     (eshell-ls-product ((,class (:foreground ,green))))
     (eshell-ls-readonly ((,class (:foreground ,red))))
     (eshell-ls-special ((,class (:foreground ,purple :weight bold))))
     (eshell-ls-symlink ((,class (:foreground ,aqua :weight bold))))
     (eshell-ls-unreadable ((,class (:foreground ,comment))))
     (eshell-prompt ((,class (:foreground ,yellow :weight bold))))
     (eyebrowse-mode-line-active ((,class (:foreground ,orange :weight bold))))
     (eyebrowse-mode-line-delimiters ((,class (:foreground ,purple))))
     (eyebrowse-mode-line-inactive ((,class (:foreground ,comment))))
     (eyebrowse-mode-line-separator ((,class (:foreground ,purple))))
     (fill-column-indicator ((,class (:foreground ,current-line :box nil :inverse-video nil :overline nil :slant normal :strike-through nil :underline nil :weight normal :stipple nil))))
     (flx-highlight-face ((,class (:inherit nil :foreground ,yellow :weight bold :underline nil))))
     (flycheck-color-mode-line-error-face ((,class (:foreground ,red))))
     (flycheck-color-mode-line-info-face ((,class (:foreground ,aqua))))
     (flycheck-color-mode-line-running-face ((,class (:foreground ,comment))))
     (flycheck-color-mode-line-success-face ((,class (:foreground ,green))))
     (flycheck-color-mode-line-warning-face ((,class (:foreground ,orange))))
     (flycheck-error ((,class (:underline (:style wave :color ,red)))))
     (flycheck-fringe-error ((,class (:foreground ,red))))
     (flycheck-fringe-info ((,class (:foreground ,aqua))))
     (flycheck-fringe-warning ((,class (:foreground ,orange))))
     (flycheck-info ((,class (:underline (:style wave :color ,aqua)))))
     (flycheck-warning ((,class (:underline (:style wave :color ,orange)))))
     (flyspell-incorrect ((,class (:underline (:style wave :color ,red)))))
     (fold-this-overlay ((,class (:foreground ,green))))
     (font-latex-bold-face ((,class (:foreground ,green :weight bold))))
     (font-latex-doctex-documentation-face ((,class (:inherit highlight))))
     (font-latex-italic-face ((,class (:foreground ,green :slant italic))))
     (font-latex-math-face ((,class (:foreground ,purple))))
     (font-latex-script-char-face ((,class (:foreground ,red))))
     (font-latex-sedate-face ((,class (:foreground ,orange))))
     (font-latex-sectioning-0-face ((,class (:foreground ,yellow))))
     (font-latex-sectioning-1-face ((,class (:foreground ,yellow))))
     (font-latex-sectioning-2-face ((,class (:foreground ,yellow))))
     (font-latex-sectioning-3-face ((,class (:foreground ,yellow))))
     (font-latex-sectioning-4-face ((,class (:foreground ,yellow))))
     (font-latex-sectioning-5-face ((,class (:foreground ,yellow))))
     (font-latex-slide-title-face ((,class (:foreground ,blue :weight bold :height 1.2))))
     (font-latex-string-face ((,class (:foreground ,yellow))))
     (font-latex-verbatim-face ((,class (:foreground ,orange))))
     (font-latex-warning-face ((,class (:inherit warning))))
     (fringe ((,class (:inherit font-lock-comment-face))))
     (git-gutter+-added ((,class (:foreground ,green :weight bold))))
     (git-gutter+-deleted ((,class (:foreground ,red :weight bold))))
     (git-gutter+-modified ((,class (:foreground ,purple :weight bold))))
     (git-gutter+-separator ((,class (:foreground ,aqua :weight bold))))
     (git-gutter+-unchanged ((,class (:background ,yellow))))
     (git-gutter:added ((,class (:foreground ,green :weight bold))))
     (git-gutter:deleted ((,class (:foreground ,red :weight bold))))
     (git-gutter:modified ((,class (:foreground ,purple :weight bold))))
     (git-gutter:separator ((,class (:foreground ,aqua :weight bold))))
     (git-gutter:unchanged ((,class (:background ,yellow))))
     (git-timemachine-minibuffer-author-face ((,class (:foreground ,purple))))
     (git-timemachine-minibuffer-detail-face ((,class (:foreground ,green))))
     (message-header-name ((,class (:foreground ,blue :background unspecified))))
     (message-header-subject ((,class (:inherit message-header-other :weight bold :foreground ,yellow))))
     (grep-context-face ((,class (:foreground ,comment))))
     (grep-error-face ((,class (:foreground ,red :weight bold :underline t))))
     (grep-hit-face ((,class (:foreground ,blue))))
     (grep-match-face ((,class (:foreground unspecified :background unspecified :inherit match))))
     (gui-element ((,class (:background ,current-line :foreground ,foreground))))
     (guide-key/highlight-command-face ((,class (:foreground ,green))))
     (guide-key/key-face ((,class (:foreground ,comment))))
     (guide-key/prefix-command-face ((,class (:foreground ,blue))))
     (helm-M-x-key ((,class (:foreground ,orange :underline t))))
     (helm-bookmark-addressbook ((,class (:foreground ,red))))
     (helm-bookmark-file ((,class (:foreground ,aqua))))
     (helm-bookmark-file-not-found ((,class (:foreground ,background))))
     (helm-bookmark-gnus ((,class (:foreground ,purple))))
     (helm-bookmark-info ((,class (:foreground ,green))))
     (helm-bookmark-man ((,class (:foreground ,orange))))
     (helm-bookmark-w3m ((,class (:foreground ,yellow))))
     (helm-buffer-archive ((,class (:foreground ,yellow))))
     (helm-buffer-directory ((,class (:foreground ,blue))))
     (helm-buffer-not-saved ((,class (:foreground ,orange))))
     (helm-buffer-process ((,class (:foreground ,aqua))))
     (helm-buffer-saved-out ((,class (:inherit warning))))
     (helm-buffer-size ((,class (:foreground ,yellow))))
     (helm-candidate-number ((,class (:foreground ,green))))
     (helm-comint-prompts-buffer-name ((,class (:foreground ,green))))
     (helm-comint-prompts-promptidx ((,class (:foreground ,aqua))))
     (helm-delete-async-message ((,class (:foreground ,yellow))))
     (helm-eshell-prompts-buffer-name ((,class (:foreground ,green))))
     (helm-eshell-prompts-promptidx ((,class (:foreground ,aqua))))
     (helm-etags-file ((,class (:foreground ,yellow :underline t))))
     (helm-ff-denied ((,class (:foreground ,background :background ,red))))
     (helm-ff-directory ((,class (:foreground ,aqua))))
     (helm-ff-dotted-directory ((,class (:foreground ,comment))))
     (helm-ff-dotted-symlink-directory ((,class (:foreground ,comment))))
     (helm-ff-executable ((,class (:foreground ,green))))
     (helm-ff-invalid-symlink ((,class (:foreground ,background :background ,red))))
     (helm-ff-pipe ((,class (:foreground ,yellow))))
     (helm-ff-prefix ((,class (:foreground ,background :background ,yellow))))
     (helm-ff-socket ((,class (:foreground ,purple))))
     (helm-ff-suid ((,class (:foreground ,background :background ,red))))
     (helm-grep-file ((,class (:foreground ,purple :underline t))))
     (helm-grep-finish ((,class (:foreground ,green))))
     (helm-grep-lineno ((,class (:foreground ,orange))))
     (helm-grep-match ((,class (:inherit match))))
     (helm-header-line-left-margin ((,class (:foreground ,background :background ,yellow))))
     (helm-lisp-completion-info ((,class (:foreground ,red))))
     (helm-lisp-show-completion ((,class (:background ,contrast-bg))))
     (helm-locate-finish ((,class (:foreground ,green))))
     (helm-ls-git-added-copied-face ((,class (:foreground ,green))))
     (helm-ls-git-added-modified-face ((,class (:foreground ,yellow))))
     (helm-ls-git-conflict-face ((,class (:foreground ,red))))
     (helm-ls-git-deleted-and-staged-face ((,class (:foreground ,purple))))
     (helm-ls-git-deleted-not-staged-face ((,class (:foreground ,comment))))
     (helm-ls-git-modified-and-staged-face ((,class (:foreground ,green))))
     (helm-ls-git-modified-not-staged-face ((,class (:foreground ,yellow))))
     (helm-ls-git-renamed-modified-face ((,class (:foreground ,blue))))
     (helm-ls-git-untracked-face ((,class (:foreground ,aqua))))
     (helm-match ((,class (:inherit match))))
     (helm-moccur-buffer ((,class (:foreground ,aqua :underline t))))
     (helm-mode-prefix ((,class (:foreground ,background :background ,red))))
     (helm-prefarg ((,class (:foreground ,red))))
     (helm-resume-need-update ((,class (:background ,red))))
     (helm-rg-active-arg-face ((,class (:foreground ,green))))
     (helm-rg-base-rg-cmd-face ((,class (:foreground ,foreground))))
     (helm-rg-colon-separator-ripgrep-output-face ((,class (:foreground ,foreground))))
     (helm-rg-directory-cmd-face ((,class (:foreground ,orange))))
     (helm-rg-directory-header-face ((,class (:foreground ,foreground :extend t))))
     (helm-rg-error-message ((,class (:foreground ,red))))
     (helm-rg-extra-arg-face ((,class (:foreground ,yellow))))
     (helm-rg-file-match-face ((,class (:foreground ,aqua :underline t))))
     (helm-rg-inactive-arg-face ((,class (:foreground ,comment))))
     (helm-rg-line-number-match-face ((,class (:foreground ,orange :underline t))))
     (helm-rg-match-text-face ((,class (:foreground ,background :background ,blue))))
     (helm-rg-preview-line-highlight ((,class (:foreground ,green))))
     (helm-rg-title-face ((,class (:foreground ,purple))))
     (helm-selection ((,class (:inherit highlight :extend t))))
     (helm-selection-line ((,class (:inherit highlight :extend t))))
     (helm-separator ((,class (:foreground ,purple))))
     (helm-source-header ((,class (:weight bold :foreground ,orange))))
     (helm-switch-shell-new-shell-face ((,class (:foreground ,background :background ,purple))))
     (helm-time-zone-current ((,class (:foreground ,green))))
     (helm-time-zone-home ((,class (:foreground ,red))))
     (helm-ucs-char ((,class (:foreground ,yellow))))
     (helm-visible-mark ((,class (:foreground ,blue))))

     (hi-yellow ((,class (:foreground ,background :background ,yellow))))
     (hi-pink ((,class (:foreground ,background :background ,purple))))
     (hi-green ((,class (:foreground ,background :background ,green))))
     (hi-blue ((,class (:foreground ,background :background ,blue))))
     (hi-salmon ((,class (:foreground ,background :background "light salmon"))))
     (hi-aquamarine ((,class (:foreground ,background :background "aquamarine"))))

     (hi-black-b ((,class (:weight bold :foreground ,background :background "grey"))))
     (hi-blue-b ((,class (:weight bold :foreground ,background :background ,aqua))))
     (hi-green-b ((,class (:weight bold :foreground ,background :background "green3"))))
     (hi-red-b ((,class (:weight bold :foreground ,background :background ,red))))
     (hi-black-hb ((,class (:weight bold :foreground ,background :background "grey70"))))

     (hl-line ((,class (:background ,hl-line))))
     (highlight-indentation-face ((,class (:background ,hl-line))))
     (highlight-indentation-current-column-mode ((,class (:background ,contrast-bg))))
     (hl-sexp-face ((,class (:background ,contrast-bg))))
     (holiday ((,class (:foreground ,background :background ,orange))))
     (hydra-face-amaranth ((,class (:foreground ,orange))))
     (hydra-face-blue ((,class (:foreground ,blue))))
     (hydra-face-pink ((,class (:foreground ,purple))))
     (hydra-face-red ((,class (:foreground ,red))))
     (hydra-face-teal ((,class (:foreground ,aqua))))
     (icomplete-first-match ((,class (:foreground ,green :weight bold))))
     (ido-first-match ((,class (:foreground ,orange))))
     (ido-indicator ((,class (:foreground ,red :background ,background))))
     (ido-only-match ((,class (:foreground ,red :weight bold))))
     (ido-subdir ((,class (:foreground ,comment))))
     (ido-virtual ((,class (:foreground ,comment))))
     (info-command-ref-item ((,class (:foreground ,green :background ,highlight))))
     (info-constant-ref-item ((,class (:foreground ,purple :background ,highlight))))
     (info-double-quoted-name ((,class (:inherit font-lock-comment-face))))
     (info-file ((,class (:foreground ,yellow :background ,highlight))))
     (info-function-ref-item ((,class (:inherit font-lock-function-name-face :background ,highlight))))
     (info-index-match ((,class (:inherit isearch))))
     (info-macro-ref-item ((,class (:foreground ,orange :background ,highlight))))
     (info-menu ((,class (:foreground ,green))))
     (info-menu-header ((,class (:foreground ,green :weight bold :height 1.4))))
     (info-menu-star ((,class (:foreground ,yellow))))
     (info-node ((,class (:foreground ,green :weight bold :slant italic))))
     (info-quoted-name ((,class (:inherit font-lock-constant-face))))
     (info-reference-item ((,class (:background ,highlight))))
     (info-single-quote ((,class (:inherit font-lock-keyword-face))))
     (info-special-form-ref-item ((,class (:foreground ,orange :background ,highlight))))
     (info-string ((,class (:inherit font-lock-string-face))))
     (info-syntax-class-item ((,class (:foreground ,blue :background ,highlight))))
     (info-title-1 ((,class (:weight bold :height 1.4))))
     (info-title-2 ((,class (:weight bold :height 1.2))))
     (info-title-3 ((,class (:weight bold :foreground ,orange))))
     (info-title-4 ((,class (:weight bold :foreground ,purple))))
     (info-user-option-ref-item ((,class (:foreground ,red :background ,highlight))))
     (info-variable-ref-item ((,class (:inherit font-lock-variable-name-face :background ,highlight))))
     (info-xref-bookmarked ((,class (:foreground ,purple))))
     (info-xref-visited ((,class (:foreground ,comment :underline t))))
     (isearch ((,class (:foreground ,yellow :background ,background :inverse-video t))))
     (isearch-fail ((,class (:background ,background :inherit font-lock-warning-face :inverse-video t))))
     (ivy-action ((,class (:foreground ,purple))))
     (ivy-confirm-face ((,class (:foreground ,green))))
     (ivy-current-match ((,class (:background ,contrast-bg))))
     (ivy-cursor ((,class (:background ,contrast-bg))))
     (ivy-match-required-face ((,class (:foreground ,red :background ,background))))
     (ivy-minibuffer-match-face-1 ((,class (:foreground ,aqua))))
     (ivy-minibuffer-match-face-2 ((,class (:foreground ,yellow))))
     (ivy-minibuffer-match-face-3 ((,class (:foreground ,blue))))
     (ivy-minibuffer-match-face-4 ((,class (:foreground ,orange))))
     (ivy-remote ((,class (:foreground ,orange))))
     (ivy-subdir ((,class (:foreground ,purple))))
     (ivy-virtual ((,class (:foreground ,comment))))
     (j-adverb-face ((,class (:inherit font-lock-function-name-face))))
     (j-conjunction-face ((,class (:inherit font-lock-keyword-face))))
     (j-other-face ((,class (:inherit font-lock-preprocessor-face))))
     (j-verb-face ((,class (:inherit font-lock-constant-face))))
     (jabber-activity-face ((,class (:foreground ,purple))))
     (jabber-activity-personal-face ((,class (:foreground ,aqua))))
     (jabber-chat-prompt-foreign ((,class (:foreground ,orange))))
     (jabber-chat-prompt-local ((,class (:foreground ,yellow))))
     (jabber-chat-prompt-system ((,class (:foreground ,yellow :weight bold))))
     (jabber-chat-text-error ((,class (:foreground ,red))))
     (jabber-chat-text-foreign ((,class (:foreground ,orange))))
     (jabber-chat-text-local ((,class (:foreground ,yellow))))
     (jabber-rare-time-face ((,class (:foreground ,comment))))
     (jabber-roster-user-away ((,class (:foreground ,orange))))
     (jabber-roster-user-chatty ((,class (:foreground ,purple))))
     (jabber-roster-user-dnd ((,class :foreground ,yellow)))
     (jabber-roster-user-error ((,class (:foreground ,red))))
     (jabber-roster-user-offline ((,class (:foreground ,comment))))
     (jabber-roster-user-online ((,class (:foreground ,green))))
     (jabber-roster-user-xa ((,class :foreground ,comment)))
     (js2-error ((,class (:foreground unspecified :underline ,red))))
     (js2-external-variable ((,class (:foreground ,purple))))
     (js2-external-variable-face ((,class (:foreground ,purple))))
     (js2-function-call ((,class (:foreground ,green))))
     (js2-function-param ((,class (:foreground ,blue))))
     (js2-function-param-face ((,class (:foreground ,blue))))
     (js2-instance-member ((,class (:foreground ,blue))))
     (js2-instance-member-face ((,class (:foreground ,blue))))
     (js2-jsdoc-html-tag-delimiter ((,class (:foreground ,orange))))
     (js2-jsdoc-html-tag-name ((,class (:foreground ,orange))))
     (js2-jsdoc-tag ((,class (:foreground ,aqua))))
     (js2-jsdoc-type ((,class (:foreground ,orange))))
     (js2-jsdoc-value ((,class (:foreground ,orange))))
     (js2-object-property ((,class (:foreground ,orange))))
     (js2-private-function-call ((,class (:foreground ,red))))
     (js2-private-function-call-face ((,class (:foreground ,red))))
     (js2-private-member ((,class (:foreground ,purple))))
     (js2-warning ((,class (:underline ,orange))))
     (js3-error-face ((,class (:foreground nil :underline ,red))))
     (js3-error-face ((,class (:foreground unspecified :underline ,red))))
     (js3-external-variable-face ((,class (:foreground ,purple))))
     (js3-function-param-face ((,class (:foreground ,blue))))
     (js3-instance-member-face ((,class (:foreground ,blue))))
     (js3-jsdoc-html-tag-delimiter-face ((,class (:foreground ,green))))
     (js3-jsdoc-html-tag-name-face ((,class (:foreground ,blue))))
     (js3-jsdoc-tag-face ((,class (:foreground ,orange))))
     (js3-jsdoc-type-face ((,class (:foreground ,aqua))))
     (js3-jsdoc-value-face ((,class (:foreground ,yellow))))
     (js3-private-function-call-face ((,class (:foreground ,red))))
     (js3-warning-face ((,class (:underline ,orange))))
     (kaocha-runner-error-face ((,class (:foreground ,red))))
     (kaocha-runner-success-face ((,class (:foreground ,green))))
     (kaocha-runner-warning-face ((,class (:foreground ,yellow))))
     (ledger-font-comment-face ((,class (:inherit font-lock-comment-face))))
     (ledger-font-occur-narrowed-face ((,class (:inherit font-lock-comment-face :invisible t))))
     (ledger-font-occur-xact-face ((,class (:inherit highlight))))
     (ledger-font-payee-cleared-face ((,class (:foreground ,green))))
     (ledger-font-payee-uncleared-face ((,class (:foreground ,aqua))))
     (ledger-font-posting-account-cleared-face ((,class (:foreground ,blue))))
     (ledger-font-posting-account-face ((,class (:foreground ,purple))))
     (ledger-font-posting-account-pending-face ((,class (:foreground ,yellow))))
     (ledger-font-posting-amount-face ((,class (:foreground ,foreground))))
     (ledger-font-posting-date-face ((,class (:foreground ,orange))))
     (ledger-font-xact-highlight-face ((,class (:inherit highlight))))
     (ledger-occur-narrowed-face ((,class (:inherit font-lock-comment-face :invisible t))))
     (ledger-occur-xact-face ((,class (:inherit highlight))))
     (line-number ((,class (:inherit default :foreground ,comment))))
     (line-number-current-line ((,class (:inherit line-number :weight bold))))
     (linum ((,class (:background ,low-contrast-bg :foreground ,comment :italic nil :underline nil))))
     (lispy-command-name-face ((,class (:inherit font-lock-function-name-face :background ,highlight))))
     (lispy-cursor-face ((,class (:foreground ,background :background ,foreground))))
     (lispy-face-hint ((,class (:foreground ,green :background ,contrast-bg))))
     (lsp-face-highlight-read ((,class (:background ,selection))))
     (macrostep-expansion-highlight-face ((,class (:inherit highlight :foreground unspecified))))
     (magit-bisect-bad ((,class (:foreground ,red))))
     (magit-bisect-good ((,class (:foreground ,green))))
     (magit-bisect-skip ((,class (:foreground ,orange))))
     (magit-blame-date ((,class (:foreground ,red))))
     (magit-blame-heading ((,class (:foreground ,orange :background ,highlight :extend t))))
     (magit-branch ((,class (:foreground ,green))))
     (magit-branch-current ((,class (:foreground ,blue))))
     (magit-branch-local ((,class (:foreground ,aqua))))
     (magit-branch-remote ((,class (:foreground ,green))))
     (magit-cherry-equivalent ((,class (:foreground ,purple))))
     (magit-cherry-unmatched ((,class (:foreground ,aqua))))
     (magit-diff-added ((,class (:foreground ,green :extend t))))
     (magit-diff-added-highlight ((,class (:foreground ,green :background ,highlight :extend t))))
     (magit-diff-base ((,class (:foreground ,background :background ,orange :extend t))))
     (magit-diff-base-highlight ((,class (:foreground ,orange :background ,highlight :extend t))))
     (magit-diff-context ((,class (:foreground ,comment :extend t))))
     (magit-diff-context-highlight ((,class (:foreground ,comment :background ,highlight :extend t))))
     (magit-diff-file-heading ((,class (:foreground ,foreground :extend t))))
     (magit-diff-file-heading-highlight ((,class (:background ,contrast-bg :weight bold :extend t))))
     (magit-diff-file-heading-selection ((,class (:foreground ,orange :background ,highlight :extend t))))
     (magit-diff-hunk-heading ((,class (:foreground ,foreground :background ,contrast-bg :extend t))))
     (magit-diff-hunk-heading-highlight ((,class (:background ,contrast-bg :extend t))))
     (magit-diff-lines-heading ((,class (:foreground ,yellow :background ,red :extend t))))
     (magit-diff-removed ((,class (:foreground ,red :extend t))))
     (magit-diff-removed-highlight ((,class (:foreground ,red :background ,highlight :extend t))))
     (magit-diffstat-added ((,class (:foreground ,green))))
     (magit-diffstat-removed ((,class (:foreground ,red))))
     (magit-dimmed ((,class (:foreground ,comment))))
     (magit-filename ((,class (:foreground ,purple))))
     (magit-hash ((,class (:foreground ,comment))))
     (magit-header ((,class (:inherit nil :weight bold))))
     (magit-header-line ((,class (:inherit nil :weight bold))))
     (magit-item-highlight ((,class (:inherit highlight :background unspecified))))
     (magit-log-author ((,class (:foreground ,orange))))
     (magit-log-date ((,class (:foreground ,blue))))
     (magit-log-graph ((,class (:foreground ,comment))))
     (magit-log-head-label-bisect-bad ((,class (:foreground ,red))))
     (magit-log-head-label-bisect-good ((,class (:foreground ,green))))
     (magit-log-head-label-default ((,class (:foreground ,yellow :box nil :weight bold))))
     (magit-log-head-label-local ((,class (:foreground ,blue))))
     (magit-log-head-label-remote ((,class (:foreground ,green))))
     (magit-log-head-label-tags ((,class (:foreground ,aqua :box nil :weight bold))))
     (magit-log-sha1 ((,class (:foreground ,purple))))
     (magit-mode-line-process ((,class (:foreground ,orange))))
     (magit-mode-line-process-error ((,class (:foreground ,red))))
     (magit-process-ng ((,class (:inherit error))))
     (magit-process-ok ((,class (:inherit success))))
     (magit-reflog-amend ((,class (:foreground ,purple))))
     (magit-reflog-checkout ((,class (:foreground ,blue))))
     (magit-reflog-cherry-pick ((,class (:foreground ,green))))
     (magit-reflog-commit ((,class (:foreground ,green))))
     (magit-reflog-merge ((,class (:foreground ,green))))
     (magit-reflog-other ((,class (:foreground ,aqua))))
     (magit-reflog-rebase ((,class (:foreground ,purple))))
     (magit-reflog-remote ((,class (:foreground ,aqua))))
     (magit-reflog-reset ((,class (:inherit error))))
     (magit-refname ((,class (:foreground ,comment))))
     (magit-section-heading ((,class (:foreground ,yellow :weight bold :extend t))))
     (magit-section-heading-selection ((,class (:foreground ,orange :weight bold :extend t))))
     (magit-section-highlight ((,class (:background ,highlight :weight bold :extend t))))
     (magit-section-title ((,class (:inherit diff-hunk-header))))
     (magit-sequence-drop ((,class (:foreground ,red))))
     (magit-sequence-head ((,class (:foreground ,blue))))
     (magit-sequence-part ((,class (:foreground ,orange))))
     (magit-sequence-stop ((,class (:foreground ,green))))
     (magit-signature-bad ((,class (:inherit error))))
     (magit-signature-error ((,class (:inherit error))))
     (magit-signature-expired ((,class (:foreground ,orange))))
     (magit-signature-good ((,class (:inherit success))))
     (magit-signature-revoked ((,class (:foreground ,purple))))
     (magit-signature-untrusted ((,class (:foreground ,aqua))))
     (magit-tag ((,class (:foreground ,yellow))))
     (markdown-code-face ((,class (:foreground ,foreground :background ,code :extend t))))
     (markdown-inline-code-face ((,class (:foreground ,foreground :background ,code))))
     (markdown-link-face ((,class (:foreground ,blue :underline t))))
     (markdown-url-face ((,class (:inherit link))))
     (markup-code-face ((,class (:foreground ,purple))))
     (markup-complex-replacement-face ((,class (:background ,background))))
     (markup-error-face ((,class (:foreground ,red :background ,background :weight bold))))
     (markup-gen-face ((,class (:foreground ,blue))))
     (markup-list-face ((,class (:foreground unspecified :background unspecified))))
     (markup-meta-face ((,class (:foreground ,comment))))
     (markup-meta-hide-face ((,class (:foreground ,comment))))
     (markup-reference-face ((,class (:inherit link))))
     (markup-secondary-text-face ((,class (:foreground ,comment))))
     (markup-title-0-face ((,class (:foreground ,blue :weight bold :height 1.4))))
     (markup-title-1-face ((,class (:foreground ,purple :weight bold :height 1.2))))
     (markup-title-2-face ((,class (:foreground ,orange :weight bold))))
     (markup-title-3-face ((,class (:foreground ,green :weight bold))))
     (markup-title-4-face ((,class (:foreground ,blue :weight bold))))
     (markup-title-5-face ((,class (:foreground ,purple :weight bold))))
     (markup-typewriter-face ((,class (:inherit shadow))))
     (markup-verbatim-face ((,class (:inherit shadow :background ,background))))
     (meow-beacon-fake-cursor ((,class (:foreground ,orange :inverse-video t))))
     (meow-search-highlight ((,class (:inherit lazy-highlight))))
     (merlin-compilation-error-face ((,class (:inherit flycheck-error))))
     (merlin-compilation-warning-face ((,class (:inherit flycheck-warning))))
     (message-header-cc ((,class (:inherit message-header-to :foreground unspecified))))
     (message-header-newsgroups ((,class (:foreground ,aqua :background unspecified :slant normal))))
     (message-header-other ((,class (:foreground unspecified :background unspecified :weight normal))))
     (message-header-to ((,class (:inherit message-header-other :weight bold :foreground ,orange))))
     (message-separator ((,class (:foreground ,purple))))
     (mini-modeline-mode-line ((,class (:inherit mode-line :height 0.1 :box nil))))
     (minibuffer-prompt ((,class (:foreground ,blue))))
     (help-key-binding ((,class (:background ,low-contrast-bg :inherit minibuffer-prompt))))  ; Used by `y-or-n-p'
     (mm-uu-extract ((,class (:foreground ,green :background ,highlight))))
     (mm/master-face ((,class (:inherit region :foreground unspecified :background unspecified))))
     (mm/mirror-face ((,class (:inherit region :foreground nil :background unspecified))))
     (mm/mirror-face ((,class (:inherit region :foreground unspecified :background unspecified))))
     (mmm-code-submode-face ((,class (:background ,contrast-bg))))
     (mmm-comment-submode-face ((,class (:inherit font-lock-comment-face))))
     (mmm-output-submode-face ((,class (:background ,contrast-bg))))
     (mu4e-cited-1-face ((,class (:foreground ,blue :slant normal))))
     (mu4e-cited-2-face ((,class (:foreground ,purple :slant normal))))
     (mu4e-cited-3-face ((,class (:foreground ,aqua :slant normal))))
     (mu4e-cited-4-face ((,class (:foreground ,yellow :slant normal))))
     (mu4e-cited-5-face ((,class (:foreground ,orange :slant normal))))
     (mu4e-cited-6-face ((,class (:foreground ,blue :slant normal))))
     (mu4e-cited-7-face ((,class (:foreground ,purple :slant normal))))
     (mu4e-flagged-face ((,class (:foreground ,orange :inherit nil))))
     (mu4e-header-highlight-face ((,class (:underline nil :inherit region))))
     (mu4e-header-marks-face ((,class (:underline nil :foreground ,yellow))))
     (mu4e-highlight-face ((,class (:inherit highlight))))
     (mu4e-ok-face ((,class (:foreground ,green))))
     (mu4e-replied-face ((,class (:foreground ,blue :inherit nil))))
     (mu4e-title-face ((,class (:inherit nil :foreground ,green))))
     (mu4e-unread-face ((,class (:foreground ,yellow :inherit nil))))
     (mu4e-view-attach-number-face ((,class (:inherit nil :foreground ,orange))))
     (mu4e-view-contact-face ((,class (:inherit nil :foreground ,yellow))))
     (mu4e-view-link-face ((,class (:inherit link :foreground ,blue))))
     (mu4e-view-url-number-face ((,class (:inherit nil :foreground ,aqua))))
     (neo-banner-face ((,class (:foreground ,blue :weight bold))))
     (neo-button-face ((,class (:underline t))))
     (neo-dir-link-face ((,class (:foreground ,orange))))
     (neo-expand-btn-face ((,class (:foreground ,comment))))
     (neo-file-link-face ((,class (:foreground ,foreground))))
     (neo-header-face ((,class (:foreground ,foreground :background ,highlight))))
     (neo-root-dir-face ((,class (:foreground ,blue :weight bold))))
     (neo-vc-added-face ((,class (:foreground ,green))))
     (neo-vc-conflict-face ((,class (:foreground ,red))))
     (neo-vc-default-face ((,class (:foreground ,foreground))))
     (neo-vc-edited-face ((,class (:foreground ,purple))))
     (neo-vc-ignored-face ((,class (:foreground ,contrast-bg))))
     (neo-vc-missing-face ((,class (:foreground ,red))))
     (neo-vc-needs-merge-face ((,class (:foreground ,red))))
     (neo-vc-unlocked-changes-face ((,class (:foreground ,blue :slant italic))))
     (neo-vc-user-face ((,class (:foreground ,red :slant italic))))
     (nim-font-lock-export-face ((,class (:inherit font-lock-function-name-face))))
     (nim-font-lock-number-face ((,class (:inherit default))))
     (nim-font-lock-pragma-face ((,class (:inherit font-lock-preprocessor-face))))
     (nim-non-overloadable-face ((,class (:inherit font-lock-builtin-face))))
     (nswbuff-current-buffer-face ((,class (:foreground ,blue :weight bold :underline nil))))
     (nswbuff-default-face ((,class (:foreground ,foreground :background ,background))))
     (nswbuff-separator-face ((,class (:foreground ,comment))))
     (nswbuff-special-buffers-face ((,class (:foreground ,purple :bold nil :underline nil))))
     (nxml-attribute-local-name-face ((,class (:foreground unspecified :inherit font-lock-variable-name-face))))
     (nxml-delimited-data-face ((,class (:foreground unspecified :inherit font-lock-string-face))))
     (nxml-delimiter-face ((,class (:foreground unspecified :inherit font-lock-keyword-face))))
     (nxml-name-face ((,class (:foreground unspecified :inherit font-lock-constant-face))))
     (nxml-ref-face ((,class (:foreground unspecified :inherit font-lock-preprocessor-face))))
     (orderless-match-face-0 ((,class (:foreground ,aqua))))
     (orderless-match-face-1 ((,class (:foreground ,yellow))))
     (orderless-match-face-2 ((,class (:foreground ,orange))))
     (orderless-match-face-3 ((,class (:foreground ,blue))))
     (org-agenda-current-time ((,class (:foreground ,yellow))))
     (org-agenda-date ((,class (:foreground ,blue :underline nil))))
     (org-agenda-dimmed-todo-face ((,class (:foreground ,comment))))
     (org-agenda-done ((,class (:foreground ,green))))
     (org-agenda-structure ((,class (:foreground ,purple))))
     (org-block ((,class (:background ,code))))
     (org-block-begin-line ((,class (:foreground ,blue :background ,code2 :extend t))))
     (org-block-end-line ((,class (:foreground ,blue :background ,code2 :extend t))))
     (org-verbatim ((,class (:foreground ,orange))))
     (org-code ((,class (:foreground ,yellow :background ,code))))
     (org-column ((,class (:background ,current-line))))
     (org-column-title ((,class (:inherit org-column :weight bold :underline t))))
     (org-date ((,class (:foreground ,purple :underline t))))
     (org-date-selected ((,class (:foreground ,aqua :inverse-video t))))
     (org-document-info ((,class (:foreground ,aqua))))
     (org-document-info-keyword ((,class (:foreground ,green))))
     (org-document-title ((,class (:weight bold :foreground ,orange :height 1.4))))
     (org-done ((,class (:foreground ,green))))
     (org-ellipsis ((,class (:foreground ,comment))))
     (org-footnote ((,class (:foreground ,aqua))))
     (org-formula ((,class (:foreground ,red))))
     (org-habit-alert-face ((,class (:foreground ,background :background ,yellow))))
     (org-habit-alert-future-face ((,class (:foreground ,background :background ,orange))))
     (org-habit-clear-face ((,class (:foreground ,background :background ,comment))))
     (org-habit-clear-future-face ((,class (:foreground ,background :background ,purple))))
     (org-habit-overdue-face ((,class (:foreground ,background :background ,blue))))
     (org-habit-overdue-future-face ((,class (:foreground ,background :background ,red))))
     (org-habit-ready-face ((,class (:foreground ,background :background ,aqua))))
     (org-habit-ready-future-face ((,class (:foreground ,background :background ,green))))
     (org-headline-done ((,class (:foreground ,green))))
     (org-headline-todo ((,class (:foreground ,red))))
     (org-hide ((,class (:foreground ,background))))
     (org-link ((,class (:foreground ,blue :underline t))))
     (org-mode-line-clock-overrun ((,class (:inherit mode-line :background ,red))))
     (org-scheduled ((,class (:foreground ,green))))
     (org-scheduled-previously ((,class (:foreground ,orange))))
     (org-scheduled-today ((,class (:foreground ,green))))
     (org-special-keyword ((,class (:foreground ,orange))))
     (org-src ((,class (:background ,current-line))))
     (org-table ((,class (:foreground ,purple))))
     (org-time-grid ((,class (:foreground ,yellow))))
     (org-todo ((,class (:foreground ,red))))
     (org-upcoming-deadline ((,class (:foreground ,orange))))
     (org-warning ((,class (:weight bold :foreground ,red))))
     (outline-1 ((,class (:inherit nil :foreground ,blue))))
     (outline-2 ((,class (:inherit nil :foreground ,purple))))
     (outline-3 ((,class (:inherit nil :foreground ,orange))))
     (outline-4 ((,class (:inherit nil :foreground ,yellow))))
     (outline-5 ((,class (:inherit nil :foreground ,red))))
     (outline-6 ((,class (:inherit nil :foreground ,green))))
     (outline-7 ((,class (:inherit nil :foreground ,aqua))))
     (outline-8 ((,class (:inherit nil :foreground ,blue))))
     (outline-9 ((,class (:inherit nil :foreground ,purple))))
     (outline-minor-0 ((,class (:weight bold :background ,low-contrast-bg))))
     (outline-minor-1 ((,class (:inherit (outline-minor-0 outline-1)))))
     (gnus-button ((,class (:inherit link :foreground unspecified))))
     (gnus-cite-1 ((,class (:inherit outline-1 :foreground nil))))
     (gnus-cite-10 ((,class (:foreground ,comment))))
     (gnus-cite-11 ((,class (:foreground ,contrast-bg))))
     (gnus-cite-2 ((,class (:inherit outline-2 :foreground unspecified))))
     (gnus-cite-3 ((,class (:inherit outline-3 :foreground unspecified))))
     (gnus-cite-4 ((,class (:inherit outline-4 :foreground unspecified))))
     (gnus-cite-5 ((,class (:inherit outline-5 :foreground unspecified))))
     (gnus-cite-6 ((,class (:inherit outline-6 :foreground unspecified))))
     (gnus-cite-7 ((,class (:inherit outline-7 :foreground unspecified))))
     (gnus-cite-8 ((,class (:inherit outline-8 :foreground unspecified))))
     (gnus-cite-9 ((,class (:foreground ,red))))
     (gnus-emphasis-highlight-words ((,class (:foreground ,yellow :background ,highlight))))
     (gnus-group-mail-1 ((,class (:foreground ,blue :weight normal))))
     (gnus-group-mail-1-empty ((,class (:inherit gnus-group-mail-1 :foreground ,comment))))
     (gnus-group-mail-2 ((,class (:foreground ,purple :weight normal))))
     (gnus-group-mail-2-empty ((,class (:inherit gnus-group-mail-2 :foreground ,comment))))
     (gnus-group-mail-3 ((,class (:foreground ,aqua :weight normal))))
     (gnus-group-mail-3-empty ((,class (:inherit gnus-group-mail-3 :foreground ,comment))))
     (gnus-group-mail-4 ((,class (:foreground ,yellow :weight normal))))
     (gnus-group-mail-4-empty ((,class (:inherit gnus-group-mail-4 :foreground ,comment))))
     (gnus-group-mail-5 ((,class (:foreground ,orange :weight normal))))
     (gnus-group-mail-5-empty ((,class (:inherit gnus-group-mail-5 :foreground ,comment))))
     (gnus-group-mail-6 ((,class (:foreground ,blue :weight normal))))
     (gnus-group-mail-6-empty ((,class (:inherit gnus-group-mail-6 :foreground ,comment))))
     (gnus-group-mail-low ((,class (:foreground ,comment))))
     (gnus-group-mail-low-empty ((,class (:foreground ,comment))))
     (gnus-group-news-1 ((,class (:foreground ,blue :weight normal))))
     (gnus-group-news-1-empty ((,class (:inherit gnus-group-news-1 :foreground ,comment))))
     (gnus-group-news-2 ((,class (:foreground ,purple :weight normal))))
     (gnus-group-news-2-empty ((,class (:inherit gnus-group-news-2 :foreground ,comment))))
     (gnus-group-news-3 ((,class (:foreground ,aqua :weight normal))))
     (gnus-group-news-3-empty ((,class (:inherit gnus-group-news-3 :foreground ,comment))))
     (gnus-group-news-4 ((,class (:foreground ,yellow :weight normal))))
     (gnus-group-news-4-empty ((,class (:inherit gnus-group-news-4 :foreground ,comment))))
     (gnus-group-news-5 ((,class (:foreground ,orange :weight normal))))
     (gnus-group-news-5-empty ((,class (:inherit gnus-group-news-5 :foreground ,comment))))
     (gnus-group-news-6 ((,class (:foreground ,blue :weight normal))))
     (gnus-group-news-6-empty ((,class (:inherit gnus-group-news-6 :foreground ,comment))))
     (gnus-header-content ((,class (:inherit message-header-other))))
     (gnus-header-from ((,class (:inherit message-header-other-face :weight bold :foreground ,orange))))
     (gnus-header-name ((,class (:inherit message-header-name))))
     (gnus-header-newsgroups ((,class (:foreground ,yellow :slant italic))))
     (gnus-header-subject ((,class (:inherit message-header-subject))))
     (gnus-server-agent ((,class (:foreground ,aqua :weight bold))))
     (gnus-server-closed ((,class (:foreground ,comment :slant italic))))
     (gnus-server-cloud ((,class (:foreground ,orange :weight bold))))
     (gnus-server-denied ((,class (:foreground ,red :weight bold))))
     (gnus-server-offline ((,class (:foreground ,blue :weight bold))))
     (gnus-server-opened ((,class (:foreground ,green :weight bold))))
     (gnus-signature ((,class (:inherit font-lock-comment-face))))
     (gnus-splash ((,class (:foreground ,aqua))))
     (gnus-summary-cancelled ((,class (:foreground ,red :background unspecified :weight normal))))
     (gnus-summary-high-ancient ((,class (:foreground ,green :weight normal))))
     (gnus-summary-high-read ((,class (:foreground ,green :weight normal))))
     (gnus-summary-high-ticked ((,class (:foreground ,orange :weight normal))))
     (gnus-summary-high-undownloaded ((,class (:foreground ,foreground :weight bold))))
     (gnus-summary-high-unread ((,class (:foreground ,yellow :weight normal))))
     (gnus-summary-low-ancient ((,class (:foreground ,comment :weight normal))))
     (gnus-summary-low-read ((,class (:foreground ,comment :weight normal))))
     (gnus-summary-low-ticked ((,class (:foreground ,comment :slant italic))))
     (gnus-summary-low-undownloaded ((,class (:foreground ,foreground :slant italic))))
     (gnus-summary-low-unread ((,class (:foreground ,comment :weight normal))))
     (gnus-summary-normal-ancient ((,class (:foreground ,aqua :weight normal))))
     (gnus-summary-normal-read ((,class (:foreground ,foreground :weight normal))))
     (gnus-summary-normal-ticked ((,class (:foreground ,orange :weight normal))))
     (gnus-summary-normal-undownloaded ((,class (:foreground ,foreground))))
     (gnus-summary-normal-unread ((,class (:foreground ,blue :weight normal))))
     (gnus-x-face ((,class (:foreground ,background :background ,foreground))))
     (package-name ((,class (:foreground ,yellow))))
     (package-status-available ((,class (:foreground ,blue))))
     (package-status-dependency ((,class (:foreground ,aqua))))
     (package-status-installed ((,class (:foreground ,green))))
     (paren-face ((,class (:foreground ,comment :background unspecified))))
     (paren-face-match ((,class (:foreground unspecified :background unspecified :inherit show-paren-match))))
     (paren-face-mismatch ((,class (:foreground unspecified :background unspecified :inherit show-paren-mismatch))))
     (paren-face-no-match ((,class (:foreground unspecified :background unspecified :inherit show-paren-mismatch))))
     (persp-selected-face ((,class (:foreground ,yellow :weight bold))))
     (powerline-active1 ((,class (:foreground ,foreground :background ,highlight))))
     (powerline-active2 ((,class (:foreground ,foreground :background ,contrast-bg))))
     (powerline-evil-base-face ((,class (:inherit mode-line :foreground ,background))))
     (powerline-evil-emacs-face ((,class (:inherit powerline-evil-base-face :background ,purple))))
     (powerline-evil-insert-face ((,class (:inherit powerline-evil-base-face :background ,blue))))
     (powerline-evil-motion-face ((,class (:inherit powerline-evil-base-face :background ,orange))))
     (powerline-evil-normal-face ((,class (:inherit powerline-evil-base-face :background ,green))))
     (powerline-evil-operator-face ((,class (:inherit powerline-evil-base-face :background ,aqua))))
     (powerline-evil-replace-face ((,class (:inherit powerline-evil-base-face :background ,red))))
     (powerline-evil-visual-face ((,class (:inherit powerline-evil-base-face :background ,yellow))))
     (pulse-highlight-start-face ((,class (:background ,red))))
     (py-builtins-face ((,class (:foreground ,orange :weight normal))))
     (rainbow-delimiters-depth-1-face ((,class (:foreground ,purple))))
     (rainbow-delimiters-depth-2-face ((,class (:foreground ,blue))))
     (rainbow-delimiters-depth-3-face ((,class (:foreground ,aqua))))
     (rainbow-delimiters-depth-4-face ((,class (:foreground ,green))))
     (rainbow-delimiters-depth-5-face ((,class (:foreground ,yellow))))
     (rainbow-delimiters-depth-6-face ((,class (:foreground ,orange))))
     (rainbow-delimiters-depth-7-face ((,class (:foreground ,red))))
     (rainbow-delimiters-depth-8-face ((,class (:foreground ,comment))))
     (rainbow-delimiters-depth-9-face ((,class (:foreground ,foreground))))
     (rainbow-delimiters-unmatched-face ((,class (:foreground ,background :background ,red))))
     (rcirc-bright-nick ((,class (:foreground ,yellow))))
     (rcirc-dim-nick ((,class (:foreground ,comment))))
     (rcirc-keyword ((,class (:foreground ,green))))
     (rcirc-my-nick ((,class (:foreground ,green))))
     (rcirc-nick-in-message ((,class (:foreground ,yellow))))
     (rcirc-nick-in-message-full-line ((,class (:foreground ,orange))))
     (rcirc-other-nick ((,class (:foreground ,purple))))
     (rcirc-prompt ((,class (:foreground ,blue))))
     (rcirc-server ((,class (:foreground ,green))))
     (rcirc-timestamp ((,class (:foreground ,aqua))))
     (reb-match-0 ((,class (:foreground ,background :background ,aqua))))
     (reb-match-1 ((,class (:foreground ,background :background ,yellow))))
     (reb-match-2 ((,class (:foreground ,background :background ,orange))))
     (reb-match-3 ((,class (:foreground ,background :background ,blue))))
     (regex-tool-matched-face ((,class (:foreground unspecified :background unspecified :inherit match))))
     (region ((,class (:background ,contrast-bg :inverse-video nil :extend t))))
     (rng-error-face ((,class (:underline ,red))))
     (rpm-spec-dir-face ((,class (:foreground ,green))))
     (rpm-spec-doc-face ((,class (:foreground ,green))))
     (rpm-spec-ghost-face ((,class (:foreground ,red))))
     (rpm-spec-macro-face ((,class (:foreground ,yellow))))
     (rpm-spec-obsolete-tag-face ((,class (:foreground ,red))))
     (rpm-spec-package-face ((,class (:foreground ,red))))
     (rpm-spec-section-face ((,class (:foreground ,yellow))))
     (rpm-spec-tag-face ((,class (:foreground ,blue))))
     (rpm-spec-var-face ((,class (:foreground ,red))))
     (ruler-mode-column-number ((,class (:foreground ,foreground :background ,highlight))))
     (ruler-mode-comment-column ((,class (:foreground ,comment :background ,contrast-bg))))
     (ruler-mode-current-column ((,class (:foreground ,yellow :background ,contrast-bg :weight bold))))
     (ruler-mode-default ((,class (:foreground ,comment :background ,highlight))))
     (ruler-mode-fill-column ((,class (:foreground ,red :background ,contrast-bg))))
     (ruler-mode-fringes ((,class (:foreground ,green :background ,contrast-bg))))
     (ruler-mode-goal-column ((,class (:foreground ,red :background ,contrast-bg))))
     (ruler-mode-margins ((,class (:foreground ,orange :background ,contrast-bg))))
     (ruler-mode-pad ((,class (:foreground ,background :background ,comment))))
     (ruler-mode-tab-stop ((,class (:foreground ,blue :background ,contrast-bg))))
     (secondary-selection ((,class (:background ,highlight :extend t))))
     (selectrum-completion-docsig ((,class (:inherit completions-annotation :underline t))))
     (selectrum-current-candidate ((,class (:background ,contrast-bg))))
     (selectrum-primary-highlight ((,class (:foreground ,aqua))))
     (selectrum-secondary-highlight ((,class (:foreground ,yellow))))
     (sh-heredoc ((,class (:foreground unspecified :inherit font-lock-string-face :weight normal))))
     (sh-quoted-exec ((,class (:foreground unspecified :inherit font-lock-preprocessor-face))))
     (slime-highlight-edits-face ((,class (:weight bold))))
     (slime-repl-input-face ((,class (:weight normal :underline nil))))
     (slime-repl-inputed-output-face ((,class (:foreground ,comment))))
     (slime-repl-output-face ((,class (:foreground ,blue :background ,background))))
     (slime-repl-prompt-face ((,class (:underline nil :weight bold :foreground ,purple))))
     (slime-repl-result-face ((,class (:foreground ,green))))
     (sly-error-face ((,class (:underline (:style wave :color ,red)))))
     (sly-mrepl-output-face ((,class (:foreground ,purple :background ,background))))
     (sly-note-face ((,class (:underline (:style wave :color ,green)))))
     (sly-stickers-armed-face ((,class (:foreground ,background :background ,blue))))
     (sly-stickers-empty-face ((,class (:foreground ,background :background ,comment))))
     (sly-stickers-placed-face ((,class (:foreground ,background :background ,foreground))))
     (sly-stickers-recordings-face ((,class (:foreground ,background :background ,green))))
     (sly-style-warning-face ((,class (:underline (:style wave :color ,yellow)))))
     (sly-warning-face ((,class (:underline (:style wave :color ,orange)))))
     (sp-show-pair-match-face ((,class (:foreground unspecified :background unspecified :inherit show-paren-match))))
     (sp-show-pair-mismatch-face ((,class (:foreground unspecified :background unspecified :inherit show-paren-mismatch))))
     (speedbar-button-face ((,class (:foreground ,green))))
     (speedbar-directory-face ((,class (:foreground ,orange))))
     (speedbar-file-face ((,class (:foreground ,aqua))))
     (speedbar-highlight-face ((,class (:inherit highlight))))
     (speedbar-selected-face ((,class (:foreground ,red :underline t))))
     (speedbar-separator-face ((,class (:foreground ,background :background ,blue :overline ,background))))
     (speedbar-tag-face ((,class (:foreground ,yellow))))
     (stripe-highlight ((,class (:inherit highlight))))
     (swiper-line-face ((,class (:underline t))))
     (sx-question-list-answers ((,class (:inherit sx-question-list-parent :foreground ,green))))
     (sx-question-mode-accepted ((,class (:inherit sx-question-mode-title :foreground ,green))))
     (sx-question-mode-content-face ((,class (:background ,highlight))))
     (sx-question-mode-kbd-tag ((,class (:weight semi-bold :box (:line-width 3 :style released-button :color ,contrast-bg)))))
     (symbol-overlay-default-face ((,class (:inherit highlight :underline t))))
     (syslog-debug ((,class (:weight bold :foreground ,green))))
     (syslog-error ((,class (:weight bold :foreground ,red))))
     (syslog-hide ((,class (:foregound ,comment))))
     (syslog-info ((,class (:weight bold :foreground ,blue))))
     (syslog-su ((,class (:weight bold :foreground ,purple))))
     (syslog-warn ((,class (:weight bold :foreground ,orange))))
     (tab-bar ((,class (:foreground ,comment :background ,highlight :box (:line-width 3 :color ,highlight :style nil)))))
     (tab-bar-tab-inactive ((,class (:foreground ,comment :background ,highlight :box (:line-width 3 :color ,highlight :style nil)))))
     (tab-bar-tab ((,class (:background ,background :foreground ,foreground :box (:line-width 3 :color ,background :style nil)))))
     (tab-bar-tab-ungrouped ((,class (:inherit tab-bar-tab-inactive))))
     (tab-bar-tab-group-current ((,class (:inherit tab-bar-tab :foregound ,purple))))
     (tab-bar-tab-group-inactive ((,class (:inherit tab-bar-tab-inactive))))
     (tab-line ((,class (:inherit tab-bar))))
     (tab-line-tab-inactive ((,class (:inherit tab-bar-tab-inactive))))
     (tab-line-tab-current ((,class (:inherit tab-bar-tab))))
     (term ((,class (:foreground unspecified :background unspecified :inherit default))))
     (term-color-black ((,class (:foreground ,term-black :background ,term-black))))
     (term-color-blue ((,class (:foreground ,blue :background ,blue))))
     (term-color-cyan ((,class (:foreground ,aqua :background ,aqua))))
     (term-color-green ((,class (:foreground ,green :background ,green))))
     (term-color-magenta ((,class (:foreground ,purple :background ,purple))))
     (term-color-red ((,class (:foreground ,red :background ,red))))
     (term-color-white ((,class (:foreground ,term-white :background ,term-white))))
     (term-color-yellow ((,class (:foreground ,yellow :background ,yellow))))
     (tooltip ((,class (:foreground ,yellow :background ,background :inverse-video t))))
     (whitespace-trailing ((,class (:background ,orange :foreground ,contrast-bg))))
     (trailing-whitespace ((,class (:inherit whitespace-trailing))))
     (transient-disabled-suffix ((,class (:foreground ,foreground :background ,red :weight bold))))
     (transient-enabled-suffix ((,class (:foreground ,low-contrast-bg :background ,green :weight bold))))
     (tuareg-font-double-semicolon-face ((,class (:inherit warning :slant italic))))
     (tuareg-font-lock-constructor-face ((,class (:inherit default :weight bold))))
     (tuareg-font-lock-error-face ((,class (:inherit error :slant italic))))
     (tuareg-font-lock-governing-face ((,class (:inherit font-lock-keyword-face :weight bold))))
     (tuareg-font-lock-interactive-directive-face)
     (tuareg-font-lock-interactive-error-face ((,class (:inherit error))))
     (tuareg-font-lock-interactive-output-face)
     (tuareg-font-lock-line-number-face ((,class (:foreground ,comment))))
     (tuareg-font-lock-multistage-face ((,class (:inherit font-lock-preprocessor-face))))
     (tuareg-font-lock-operator-face ((,class (:inherit font-lock-preprocessor-face))))
     (twittering-timeline-footer-face ((,class (:inherit twittering-timeline-header-face))))
     (twittering-timeline-header-face ((,class (:foreground ,green :weight bold))))
     (twittering-uri-face ((,class (:foreground ,blue :inherit link))))
     (twittering-username-face ((,class (:inherit erc-pal-face))))
     (uiua-dyadic-function ((,class (:foreground ,blue))))
     (uiua-dyadic-modifier ((,class (:foreground ,purple))))
     (uiua-monadic-function ((,class (:foreground ,green))))
     (uiua-monadic-modifier ((,class (:foreground ,yellow))))
     (uiua-noadic-or-constant ((,class (:foreground ,red))))
     (uiua-number ((,class (:foreground ,orange))))
     (undo-tree-visualizer-active-branch-face ((,class (:foreground ,red))))
     (undo-tree-visualizer-current-face ((,class (:foreground ,green :weight bold))))
     (undo-tree-visualizer-default-face ((,class (:foreground ,foreground))))
     (undo-tree-visualizer-register-face ((,class (:foreground ,yellow))))
     (vertical-border ((,class (:foreground ,contrast-bg))))
     (vertico-current ((,class (:background ,contrast-bg :extend t))))
     (vhdl-speedbar-architecture-face ((,class (:foreground ,blue))))
     (vhdl-speedbar-architecture-selected-face ((,class (:foreground ,blue :underline t))))
     (vhdl-speedbar-configuration-face ((,class (:foreground ,green))))
     (vhdl-speedbar-configuration-selected-face ((,class (:foreground ,green :underline t))))
     (vhdl-speedbar-entity-face ((,class (:foreground ,orange))))
     (vhdl-speedbar-entity-selected-face ((,class (:foreground ,orange :underline t))))
     (vhdl-speedbar-instantiation-face ((,class (:foreground ,yellow))))
     (vhdl-speedbar-instantiation-selected-face ((,class (:foreground ,yellow :underline t))))
     (vhdl-speedbar-library-face ((,class (:foreground ,purple))))
     (vhdl-speedbar-package-face ((,class (:foreground ,aqua))))
     (vhdl-speedbar-package-selected-face ((,class (:foreground ,aqua :underline t))))
     (vhdl-speedbar-subprogram-face ((,class (:foreground ,green))))
     (vr/group-0 ((,class (:foreground ,purple :background ,background :inverse-video t))))
     (vr/group-1 ((,class (:foreground ,green :background ,background :inverse-video t))))
     (vr/group-2 ((,class (:foreground ,orange :background ,background :inverse-video t))))
     (vr/match-0 ((,class (:foreground ,yellow :background ,background :inverse-video t))))
     (vr/match-1 ((,class (:foreground ,aqua :background ,background :inverse-video t))))
     (vterm-color-black ((,class (:background ,term-black :foreground ,term-black))))
     (vterm-color-blue ((,class (:background ,blue :foreground ,blue))))
     (vterm-color-cyan ((,class (:background ,aqua :foreground ,aqua))))
     (vterm-color-default ((,class (:foreground unspecified :background unspecified :inherit default))))
     (vterm-color-green ((,class (:background ,green :foreground ,green))))
     (vterm-color-inverse-video ((,class (:background ,background :inverse-video t))))
     (vterm-color-magenta ((,class (:background ,purple :foreground ,purple))))
     (vterm-color-red ((,class (:background ,red :foreground ,red))))
     (vterm-color-underline ((,class (:underline t))))
     (vterm-color-white ((,class (:background ,term-white :foreground ,term-white))))
     (vterm-color-yellow ((,class (:background ,yellow :foreground ,yellow))))
     (web-mode-doctype-face ((,class (:inherit font-lock-string-face))))
     (web-mode-html-attr-equal-face ((,class (:foreground unspecified :background unspecified :inherit default))))
     (web-mode-html-attr-name-face ((,class (:inherit font-lock-variable-name-face))))
     (web-mode-html-tag-bracket-face ((,class (:inherit font-lock-function-name-face))))
     (web-mode-html-tag-face ((,class (:inherit font-lock-function-name-face))))
     (web-mode-symbol-face ((,class (:inherit font-lock-constant-face))))
     (weechat-highlight-face ((,class (:foreground ,orange))))
     (weechat-nick-self-face ((,class (:foreground ,green))))
     (weechat-time-face ((,class (:foreground ,aqua))))
     (wgrep-delete-face ((,class (:foreground ,red))))
     (wgrep-done-face ((,class (:foreground ,blue))))
     (wgrep-face ((,class (:foreground ,green :background ,contrast-bg))))
     (wgrep-file-face ((,class (:foreground ,comment :background ,contrast-bg))))
     (wgrep-reject-face ((,class (:foreground ,orange :weight bold))))
     (which-func ((,class (:foreground ,blue :background unspecified :weight bold))))
     (whitespace-big-indent ((,class (:background ,red :foreground ,contrast-bg))))
     (whitespace-empty ((,class (:background ,yellow :foreground ,orange))))
     (whitespace-hspace ((,class (:background ,contrast-bg :foreground ,comment))))
     (whitespace-indentation ((,class (:background ,contrast-bg :foreground ,comment))))
     (whitespace-line ((,class (:background ,contrast-bg :foreground ,orange))))
     (whitespace-newline ((,class (:background ,contrast-bg :foreground ,comment))))
     (whitespace-space ((,class (:background ,contrast-bg :foreground ,comment))))
     (whitespace-space-after-tab ((,class (:background ,contrast-bg :foreground ,yellow))))
     (whitespace-space-before-tab ((,class (:background ,contrast-bg :foreground ,orange))))
     (whitespace-tab ((,class (:background ,contrast-bg :foreground ,comment))))
     (widget-button ((,class (:underline t))))
     (widget-field ((,class (:background ,contrast-bg :box (:line-width 1 :color ,foreground)))))
     (window-divider ((,class (:foreground ,comment))))
     (window-divider-first-pixel ((,class (:foreground ,contrast-bg))))
     (window-divider-last-pixel ((,class (:foreground ,contrast-bg))))
     (ztreep-arrow-face ((,class (:foreground ,highlight))))
     (ztreep-diff-header-face ((,class (:foreground ,yellow :weight bold))))
     (ztreep-diff-header-small-face ((,class (:foregorund ,yellow))))
     (ztreep-diff-model-add-face ((,class (:foreground ,green))))
     (ztreep-diff-model-diff-face ((,class (:foreground ,red))))
     (ztreep-diff-model-ignored-face ((,class (:foreground ,orange))))
     (ztreep-diff-model-normal-face ((,class (:foreground ,foreground))))
     (ztreep-expand-sign-face ((,class (:foreground ,foreground))))
     (ztreep-header-face ((,class (:forground ,yellow :weight bold))))
     (ztreep-leaf-face ((,class (:foreground ,aqua))))
     (ztreep-node-face ((,class (:foreground ,foreground)))))))

(defmacro tomorrow-night-deepblue-theme--frame-parameter-specs ()
  "Define a list of frame parameter and return a backquote which .

These are required by color-theme's `color-theme-install', but
not by the new `deftheme' mechanism. It expects to be evaluated
in a scope in which the various color names to which it refers
are bound."
  (quote
   `(((background-color . ,background)
      (background-mode . light)
      (border-color . ,foreground)
      (cursor-color . ,purple)
      (foreground-color . ,foreground)
      (mouse-color . ,aqua)))))

(defmacro tomorrow-night-deepblue-theme--define-theme (mode)
  "Define a theme for the tomorrow variant MODE."
  (let ((name (intern (format "tomorrow-%s" (symbol-name mode))))
        (doc (format "tomorrow-%s" mode)))
    `(progn
       (deftheme ,name ,doc)
       (tomorrow-night-deepblue-theme--with-colors
        ',mode
        (apply #'custom-theme-set-faces ',name
               (tomorrow-night-deepblue-theme--face-specs))
        (custom-theme-set-variables
         ',name
         `(fci-rule-color ,current-line)
         `(ansi-color-names-vector (vector ,foreground ,red ,green ,yellow ,blue ,purple ,aqua ,background))
         '(ansi-color-faces-vector [default bold shadow italic underline bold bold-italic bold])))
       (provide-theme ',name))))

(tomorrow-night-deepblue-theme--define-theme night-deepblue)

;;;###autoload
(when (and (boundp 'custom-theme-load-path) load-file-name)
  (add-to-list 'custom-theme-load-path
               (file-name-as-directory (file-name-directory load-file-name))))

(provide 'tomorrow-night-deepblue-theme)

;; provide-theme not required because it is already in
;; `tomorrow-night-deepblue-theme--define-theme'
;; (provide-theme 'tomorrow-night-deepblue)

;; Local Variables:
;; byte-compile-warnings: (not cl-functions)
;; End:

;;; tomorrow-night-deepblue-theme.el ends here
