;;; wttrin.el --- Emacs Frontend for Service wttr.in -*- lexical-binding: t; coding: utf-8; -*-
;;
;; Copyright (C) 2024 Craig Jennings
;; Maintainer: Craig Jennings <c@cjennings.net>
;;
;; Original Authors: Carl X. Su <bcbcarl@gmail.com>
;;                   ono hiroko (kuanyui) <azazabc123@gmail.com>
;; Package-Version: 20260221.1311
;; Package-Revision: b74b98f177d9
;; Package-Requires: ((emacs "24.4") (xterm-color "1.0"))
;; Keywords: weather, wttrin, games
;; URL: https://github.com/cjennings/emacs-wttrin

;; SPDX-License-Identifier: GPL-3.0-or-later

;; This program is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation, either version 3 of the License, or
;; (at your option) any later version.

;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with this program.  If not, see <https://www.gnu.org/licenses/>.

;; This file is NOT part of GNU Emacs.

;;; Commentary:
;; Displays the weather information from the wttr.in service for your submitted
;; location.

;;; Code:

(require 'face-remap)
(require 'url)

;; Declare xterm-color functions (loaded on-demand)
(declare-function xterm-color-filter "xterm-color" (string))

;; Declare functions from wttrin-debug.el (loaded conditionally)
(declare-function wttrin--debug-mode-line-info "wttrin-debug")
(declare-function wttrin--debug-log "wttrin-debug")

(defgroup wttrin nil
  "Emacs frontend for the weather web service wttr.in."
  :prefix "wttrin-"
  :group 'comm)

(defcustom wttrin-font-name "Liberation Mono"
  "Preferred monospaced font name for weather display."
  :group 'wttrin
  :type 'string)

(defcustom wttrin-font-height 130
  "Preferred font height for weather display."
  :group 'wttrin
  :type 'integer)

(defcustom wttrin-default-locations '("Honolulu, HI"
									  "Berkeley, CA"
									  "New Orleans, LA"
									  "New York, NY"
                                      "London, GB"
                                      "Paris, FR"
                                      "Berlin, DE"
                                      "Naples, IT"
                                      "Athens, GR"
									  "Kyiv, UA"
									  "Tokyo, JP"
                                      "Taipei, TW")
  "Specify default locations list for quick completion."
  :group 'wttrin
  :type '(repeat string))

(defcustom wttrin-default-languages
  '("Accept-Language" . "en-US,en;q=0.8,zh-CN;q=0.6,zh;q=0.4")
  "Specify default HTTP request Header for Accept-Language."
  :group 'wttrin
  :type '(cons (string :tag "Header") (string :tag "Language codes")))

(defcustom wttrin-unit-system nil
  "Specify units of measurement.
Use \='m\=' for \='metric\=', \='u\=' for \='USCS\=', or nil for location based
units (default)."
  :group 'wttrin
  :type 'string)


(defcustom wttrin-refresh-interval 3600  ; 1 hour
  "Interval in seconds between proactive weather data refreshes.
Controls how often the background timer refreshes cached weather data
for `wttrin-favorite-location'.  Data older than 2x this interval
is considered stale."
  :group 'wttrin
  :type 'integer)

(define-obsolete-variable-alias 'wttrin-cache-ttl 'wttrin-refresh-interval "0.3.0")

(defcustom wttrin-cache-max-entries 50
  "Maximum number of entries to keep in cache."
  :group 'wttrin
  :type 'integer)

(defconst wttrin--cache-cleanup-percentage 0.20
  "Percentage of cache entries to remove when max size is exceeded.
When cache reaches `wttrin-cache-max-entries', remove the oldest 20%
to avoid frequent cleanup cycles.  This value (0.20) means remove 1/5
of entries, providing a reasonable buffer before the next cleanup.")

(defcustom wttrin-favorite-location nil
  "Favorite location to display weather for.
When nil, favorite location features are disabled.
Set to a location string (e.g., \"New Orleans, LA\") to enable mode-line
weather display and other location-based features.
The weather icon and tooltip will update automatically in the background."
  :group 'wttrin
  :type '(choice (const :tag "Disabled" nil)
                 (string :tag "Location")))

(defcustom wttrin-mode-line-refresh-interval 3600
  "Interval in seconds to refresh mode-line weather data.
Default is 3600 seconds (1 hour)."
  :group 'wttrin
  :type 'integer)

(defcustom wttrin-mode-line-startup-delay 3
  "Seconds to delay initial mode-line weather fetch after Emacs starts.
This allows network stack and daemon initialization to complete before
fetching weather data.  Must be between 1 and 10 seconds."
  :group 'wttrin
  :type '(restricted-sexp :match-alternatives
                          ((lambda (val)
                             (and (integerp val)
                                  (>= val 1)
                                  (<= val 10))))))

(defcustom wttrin-mode-line-emoji-font "Noto Color Emoji"
  "Font family to use for mode-line weather emoji.
Common color emoji fonts include:
- \"Noto Color Emoji\" (Linux)
- \"Apple Color Emoji\" (macOS)
- \"Segoe UI Emoji\" (Windows)
- \"Twitter Color Emoji\"
Set to nil to use default font (may render as monochrome)."
  :group 'wttrin
  :type '(choice (const :tag "Use default font" nil)
                 (string :tag "Font family name")))

(defcustom wttrin-mode-line-auto-enable nil
  "If non-nil, automatically enable mode-line weather display when loading wttrin.
When enabled, weather for `wttrin-favorite-location' will appear
in the mode-line automatically.  You can also manually toggle the mode-line
display with `wttrin-mode-line-mode'."
  :group 'wttrin
  :type 'boolean)

(defcustom wttrin-debug nil
  "Enable debug functions for troubleshooting wttrin behavior.
When non-nil, loads wttrin-debug.el which provides:
- Automatic mode-line diagnostic logging when wttrin runs
- Raw weather data saved to timestamped files in variable
  `temporary-file-directory'
- Interactive debug commands for troubleshooting

Set this to t BEFORE loading wttrin, typically in your init file:
  (setq wttrin-debug t)
  (require \\='wttrin)"
  :group 'wttrin
  :type 'boolean)

;; Load debug functions if enabled
(when wttrin-debug
  (require 'wttrin-debug
           (expand-file-name "wttrin-debug.el"
                             (file-name-directory (or load-file-name buffer-file-name)))
           t))

(defvar wttrin--cache (make-hash-table :test 'equal)
  "Cache for weather data: cache-key -> (timestamp . data).")

(defvar wttrin--force-refresh nil
  "When non-nil, bypass cache on next fetch.")

(defvar wttrin-mode-line-string nil
  "Mode-line string showing weather for favorite location.")
;;;###autoload(put 'wttrin-mode-line-string 'risky-local-variable t)
(put 'wttrin-mode-line-string 'risky-local-variable t)

(defvar wttrin--mode-line-timer nil
  "Timer object for mode-line weather refresh.")

(defvar wttrin--mode-line-cache nil
  "Cached mode-line weather data as (timestamp . data) cons cell.
When non-nil, car is the `float-time' when data was fetched,
and cdr is the weather string from the API.")

(defvar wttrin--mode-line-map
  (let ((map (make-sparse-keymap)))
    (define-key map [mode-line mouse-1] 'wttrin-mode-line-click)
    (define-key map [mode-line mouse-3] 'wttrin-mode-line-force-refresh)
    map)
  "Keymap for mode-line weather widget interactions.
Left-click: refresh weather and open buffer.
Right-click: force-refresh cache and update tooltip.")

(defun wttrin--format-age (seconds)
  "Format SECONDS as a human-readable age string.
Returns \"just now\" for <60s, \"X minutes ago\", \"X hours ago\", or \"X days ago\"."
  (cond
   ((< seconds 60) "just now")
   ((< seconds 3600)
    (let ((minutes (floor (/ seconds 60))))
      (format "%d %s ago" minutes (if (= minutes 1) "minute" "minutes"))))
   ((< seconds 86400)
    (let ((hours (floor (/ seconds 3600))))
      (format "%d %s ago" hours (if (= hours 1) "hour" "hours"))))
   (t
    (let ((days (floor (/ seconds 86400))))
      (format "%d %s ago" days (if (= days 1) "day" "days"))))))

(defun wttrin-additional-url-params ()
  "Concatenates extra information into the URL."
  (if wttrin-unit-system
      (concat "?" wttrin-unit-system)
    "?"))

(defun wttrin--build-url (query)
  "Build wttr.in URL for QUERY with configured parameters.
This is a pure function with no side effects, suitable for testing."
  (when (null query)
    (error "Query cannot be nil"))
  (concat "https://wttr.in/"
          (url-hexify-string query)
          (wttrin-additional-url-params)
          "A"))

(defun wttrin--extract-response-body ()
  "Extract and decode HTTP response body from current buffer.
Skips headers and returns UTF-8 decoded body.
Returns nil on error.  Kills buffer when done."
  (condition-case err
      (unwind-protect
          (progn
            (goto-char (point-min))
            (re-search-forward "\r?\n\r?\n" nil t)
            (let ((body (decode-coding-string
                         (buffer-substring-no-properties (point) (point-max))
                         'utf-8)))
              (when (featurep 'wttrin-debug)
                (wttrin--debug-log "wttrin--extract-response-body: Successfully fetched %d bytes"
                                   (length body)))
              body))
        (ignore-errors (kill-buffer (current-buffer))))
    (error
     (when (featurep 'wttrin-debug)
       (wttrin--debug-log "wttrin--extract-response-body: Error - %s"
                          (error-message-string err)))
     (ignore-errors (kill-buffer (current-buffer)))
     nil)))

(defun wttrin--handle-fetch-callback (status callback)
  "Handle `url-retrieve' callback STATUS and invoke CALLBACK with result.
Extracts response body or handles errors, then calls CALLBACK with data or nil."
  (when (featurep 'wttrin-debug)
    (wttrin--debug-log "wttrin--handle-fetch-callback: Invoked with status = %S" status))
  (let ((data nil))
    (if (plist-get status :error)
        (when (featurep 'wttrin-debug)
          (wttrin--debug-log "wttrin--handle-fetch-callback: Network error - %s"
                             (cdr (plist-get status :error))))
      (setq data (wttrin--extract-response-body)))
    (condition-case err
        (progn
          (when (featurep 'wttrin-debug)
            (wttrin--debug-log "wttrin--handle-fetch-callback: Calling user callback with %s"
                               (if data (format "%d bytes" (length data)) "nil")))
          (funcall callback data))
      (error
       (when (featurep 'wttrin-debug)
         (wttrin--debug-log "wttrin--handle-fetch-callback: Error in user callback - %s"
                            (error-message-string err)))
       (message "wttrin: Error in callback - %s" (error-message-string err))))))

(defun wttrin--fetch-url (url callback)
  "Asynchronously fetch URL and call CALLBACK with decoded response.
CALLBACK is called with the weather data string when ready, or nil on error.
Handles header skipping, UTF-8 decoding, and error handling automatically."
  (when (featurep 'wttrin-debug)
    (wttrin--debug-log "wttrin--fetch-url: Starting fetch for URL: %s" url))
  (let ((url-request-extra-headers (list wttrin-default-languages))
        (url-user-agent "curl"))
    (url-retrieve url
                  (lambda (status)
                    (wttrin--handle-fetch-callback status callback)))))

(defun wttrin-fetch-raw-string (query callback)
  "Asynchronously fetch weather information for QUERY.
CALLBACK is called with the weather data string when ready, or nil on error."
  (wttrin--fetch-url (wttrin--build-url query) callback))

(defun wttrin-exit ()
  "Exit the wttrin buffer."
  (interactive)
  (quit-window t))

(defun wttrin-requery ()
  "Kill buffer and requery wttrin."
  (interactive)
  (let ((new-location (completing-read
                       "Location Name: " wttrin-default-locations nil nil
                       (when (= (length wttrin-default-locations) 1)
                         (car wttrin-default-locations)))))
    (when (get-buffer "*wttr.in*")
      (kill-buffer "*wttr.in*"))
    (wttrin-query new-location)))

(defvar wttrin-mode-map
  (let ((map (make-sparse-keymap)))
    (define-key map (kbd "a") 'wttrin-requery)
    (define-key map (kbd "g") 'wttrin-requery-force)
    ;; Note: 'q' is bound to quit-window by special-mode
    map)
  "Keymap for wttrin-mode.")

(define-derived-mode wttrin-mode special-mode "Wttrin"
  "Major mode for displaying wttr.in weather information.

Weather data is displayed in a read-only buffer with the following keybindings:

\\{wttrin-mode-map}"
  (buffer-disable-undo)
  ;; Use face-remap instead of buffer-face-mode to preserve xterm-color faces
  (face-remap-add-relative 'default
                           :family wttrin-font-name
                           :height wttrin-font-height))

(defun wttrin--save-debug-data (location-name raw-string)
  "Save RAW-STRING to a timestamped debug file for LOCATION-NAME.
Returns the path to the saved file."
  (let* ((timestamp (format-time-string "%Y%m%d-%H%M%S"))
         (filename (format "wttrin-debug-%s.txt" timestamp))
         (filepath (expand-file-name filename temporary-file-directory)))
    (with-temp-file filepath
      (insert (format "Location: %s\n" location-name))
      (insert (format "Timestamp: %s\n" (format-time-string "%Y-%m-%d %H:%M:%S")))
      (insert (format "wttrin-unit-system: %s\n" wttrin-unit-system))
      (insert "\n--- Raw Response ---\n\n")
      (insert raw-string))
    (when (featurep 'wttrin-debug)
      (wttrin--debug-log "Debug data saved to: %s" filepath))
    filepath))

(defun wttrin--validate-weather-data (raw-string)
  "Check if RAW-STRING has valid weather data.
Return t if valid, nil if missing or contains errors."
  (not (or (null raw-string) (string-match "ERROR" raw-string))))

(defun wttrin--process-weather-content (raw-string)
  "Process RAW-STRING: apply ANSI filtering and remove verbose lines.
Returns processed string ready for display."
  (require 'xterm-color)
  (let ((processed (xterm-color-filter raw-string)))
    ;; Remove verbose Location: coordinate line
    (with-temp-buffer
      (insert processed)
      (goto-char (point-min))
      (while (re-search-forward "^\\s-*Location:.*\\[.*\\].*$" nil t)
        (delete-region (line-beginning-position) (1+ (line-end-position))))
      (buffer-string))))

(defun wttrin--add-buffer-instructions ()
  "Add user instructions at bottom of current buffer."
  (goto-char (point-max))
  (insert "\n\nPress: [a] for another location [g] to refresh [q] to quit"))

(defun wttrin--format-staleness-header (location)
  "Return a staleness header string for LOCATION, or nil if no cache entry.
Looks up the cache timestamp for LOCATION and formats a line like
\"Last updated: 2:30 PM (5 minutes ago)\"."
  (let* ((cache-key (wttrin--make-cache-key location))
         (cached (gethash cache-key wttrin--cache)))
    (when cached
      (let* ((timestamp (car cached))
             (age (- (float-time) timestamp))
             (time-str (format-time-string "%l:%M %p" (seconds-to-time timestamp)))
             (age-str (wttrin--format-age age)))
        (format "Last updated: %s (%s)" (string-trim time-str) age-str)))))

(defun wttrin--display-weather (location-name raw-string)
  "Display weather data RAW-STRING for LOCATION-NAME in weather buffer."
  ;; Save debug data if enabled
  (when wttrin-debug
    (wttrin--save-debug-data location-name raw-string))

  (if (not (wttrin--validate-weather-data raw-string))
      (message "Cannot retrieve weather data. Perhaps the location was misspelled?")
    (let ((buffer (get-buffer-create (format "*wttr.in*"))))
      (switch-to-buffer buffer)

      ;; Enable wttrin-mode first (calls kill-all-local-variables)
      ;; This must be done before setting any buffer-local variables
      (wttrin-mode)

      ;; Temporarily allow editing
      (let ((inhibit-read-only t))
        (erase-buffer)
        ;; Initialize xterm-color state AFTER wttrin-mode to prevent it being wiped
        (require 'xterm-color)
        (setq-local xterm-color--state :char)
        (insert (wttrin--process-weather-content raw-string))
        (let ((staleness (wttrin--format-staleness-header location-name)))
          (when staleness
            (insert "\n" staleness)))
        (wttrin--add-buffer-instructions)
        ;; align buffer to top
        (goto-char (point-min)))

      ;; Set location after mode initialization (mode calls kill-all-local-variables)
      (setq-local wttrin--current-location location-name)

      ;; Auto-generate debug diagnostics if debug mode is enabled
      (when (featurep 'wttrin-debug)
        (wttrin--debug-mode-line-info)))))

(defun wttrin-query (location-name)
  "Asynchronously query weather of LOCATION-NAME, display result when ready."
  (let ((buffer (get-buffer-create (format "*wttr.in*"))))
    (switch-to-buffer buffer)
    (setq buffer-read-only nil)
    (erase-buffer)
    (insert "Loading weather for " location-name "...")
    (setq buffer-read-only t)
    (wttrin--get-cached-or-fetch
     location-name
     (lambda (raw-string)
       (when (buffer-live-p buffer)
         (with-current-buffer buffer
           (wttrin--display-weather location-name raw-string)))))))

(defun wttrin--make-cache-key (location)
  "Create cache key from LOCATION and current settings."
  (concat location "|" (or wttrin-unit-system "default")))

(defun wttrin--get-cached-or-fetch (location callback)
  "Get cached weather for LOCATION or fetch if not cached.
If cache has data and not force-refreshing, serves it immediately
regardless of age.  The background refresh timer keeps data fresh.
CALLBACK is called with the weather data string when ready, or nil on error."
  (let* ((cache-key (wttrin--make-cache-key location))
         (cached (gethash cache-key wttrin--cache))
         (data (cdr cached))
         (now (float-time)))
    (if (and cached (not wttrin--force-refresh))
        ;; Return cached data immediately regardless of age
        (funcall callback data)
      ;; Fetch fresh data asynchronously
      (wttrin-fetch-raw-string
       location
       (lambda (fresh-data)
         (if fresh-data
             (progn
               (wttrin--cleanup-cache-if-needed)
               (puthash cache-key (cons now fresh-data) wttrin--cache)
               (funcall callback fresh-data))
           ;; On error, return stale cache if available
           (if cached
               (progn
                 (message "Failed to fetch new data, using cached version")
                 (funcall callback data))
             (funcall callback nil))))))))

(defun wttrin--get-cache-entries-by-age ()
  "Return list of (key . timestamp) pairs sorted oldest-first.
Extracts all cache entries and sorts them by timestamp in ascending order.
Returns a list where each element is a cons cell (key . timestamp)."
  (let ((entries nil))
    (maphash (lambda (key value)
               (push (cons key (car value)) entries))  ; car value = timestamp
             wttrin--cache)
    (sort entries (lambda (a b) (< (cdr a) (cdr b))))))

(defun wttrin--cleanup-cache-if-needed ()
  "Remove oldest entries if cache exceeds max size.
Removes oldest entries based on `wttrin--cache-cleanup-percentage'
when cache count exceeds `wttrin-cache-max-entries'.
This creates headroom to avoid frequent cleanups."
  (when (> (hash-table-count wttrin--cache) wttrin-cache-max-entries)
    (let* ((entries-by-age (wttrin--get-cache-entries-by-age))
           (num-to-remove (floor (* (length entries-by-age)
                                    wttrin--cache-cleanup-percentage))))
      (dotimes (i num-to-remove)
        (remhash (car (nth i entries-by-age)) wttrin--cache)))))

(defun wttrin-clear-cache ()
  "Clear the weather cache."
  (interactive)
  (clrhash wttrin--cache)
  (message "Weather cache cleared"))

(defvar-local wttrin--current-location nil
  "Current location displayed in this weather buffer.")

(defun wttrin-requery-force ()
  "Force refresh weather data for current location, bypassing cache."
  (interactive)
  (if wttrin--current-location
      (let ((wttrin--force-refresh t))
        (message "Refreshing weather data...")
        (wttrin-query wttrin--current-location))
    (message "No location to refresh")))

;;; Mode-line weather display

(defun wttrin--mode-line-valid-response-p (weather-string)
  "Return non-nil if WEATHER-STRING looks like a valid mode-line response.
Expected format: \"Location: emoji temp conditions\",
e.g., \"Paris: ☀️ +61°F Clear\"."
  (and (stringp weather-string)
       (not (string-empty-p weather-string))
       (string-match-p ":" weather-string)))

(defun wttrin--mode-line-update-placeholder-error ()
  "Update placeholder to show fetch error state.
Keeps the hourglass icon but updates tooltip to explain the failure
and indicate when retry will occur."
  (let* ((icon (if wttrin-mode-line-emoji-font
                   (propertize "⏳"
                               'face (list :family wttrin-mode-line-emoji-font
                                           :height 1.0))
                 "⏳"))
         (retry-minutes (ceiling (/ wttrin-mode-line-refresh-interval 60.0))))
    (setq wttrin-mode-line-string
          (propertize (concat " " icon)
                      'help-echo (format "Weather fetch failed for %s — will retry in %d minutes"
                                         wttrin-favorite-location retry-minutes)
                      'mouse-face 'mode-line-highlight
                      'local-map wttrin--mode-line-map)))
  (force-mode-line-update t))

(defun wttrin--mode-line-fetch-weather ()
  "Fetch weather for favorite location and update mode-line display.
Uses wttr.in custom format for concise weather with emoji.
On success, writes to `wttrin--mode-line-cache' and updates display.
On failure with existing cache, shows stale data.
On failure with no cache, shows error placeholder."
  (when (featurep 'wttrin-debug)
    (wttrin--debug-log "mode-line-fetch: Starting fetch for %s" wttrin-favorite-location))
  (if (not wttrin-favorite-location)
      (when (featurep 'wttrin-debug)
        (wttrin--debug-log "mode-line-fetch: No favorite location set, skipping"))
    (let* ((location wttrin-favorite-location)
           (format-params (if wttrin-unit-system
                              (concat "?" wttrin-unit-system "&format=%l:+%c+%t+%C")
                            "?format=%l:+%c+%t+%C"))
           (url (concat "https://wttr.in/"
                       (url-hexify-string location)
                       format-params)))
      (when (featurep 'wttrin-debug)
        (wttrin--debug-log "mode-line-fetch: URL = %s" url))
      (wttrin--fetch-url
       url
       (lambda (data)
         (if data
             (let ((trimmed-data (string-trim data)))
               (when (featurep 'wttrin-debug)
                 (wttrin--debug-log "mode-line-fetch: Received data = %S" trimmed-data))
               (if (wttrin--mode-line-valid-response-p trimmed-data)
                   (progn
                     (setq wttrin--mode-line-cache (cons (float-time) trimmed-data))
                     (wttrin--mode-line-update-display))
                 (when (featurep 'wttrin-debug)
                   (wttrin--debug-log "mode-line-fetch: Invalid response, keeping previous display"))))
           ;; Network error / nil data
           (when (featurep 'wttrin-debug)
             (wttrin--debug-log "mode-line-fetch: No data received (network error)"))
           (if wttrin--mode-line-cache
               ;; Have stale cache — update display to show staleness
               (wttrin--mode-line-update-display)
             ;; No cache at all — show error placeholder
             (wttrin--mode-line-update-placeholder-error))))))))

(defun wttrin--mode-line-update-display ()
  "Update mode-line display from `wttrin--mode-line-cache'.
Reads cached weather data, computes age, and sets the mode-line string.
If data is stale (age > 2x refresh interval), dims the emoji and
shows staleness info in tooltip."
  (when wttrin--mode-line-cache
    (let* ((timestamp (car wttrin--mode-line-cache))
           (weather-string (cdr wttrin--mode-line-cache))
           (age (- (float-time) timestamp))
           (stale-p (> age (* 2 wttrin-mode-line-refresh-interval)))
           (age-str (wttrin--format-age age)))
      (when (featurep 'wttrin-debug)
        (wttrin--debug-log "mode-line-display: Updating from cache, age=%s, stale=%s"
                           age-str stale-p))
      ;; Extract just the emoji for mode-line display
      (let* ((emoji (if (string-match ":\\s-*\\(.\\)" weather-string)
                        (match-string 1 weather-string)
                      "?"))
             (emoji-with-font
              (if wttrin-mode-line-emoji-font
                  (propertize emoji
                              'face (list :family wttrin-mode-line-emoji-font
                                          :height 1.0
                                          :foreground (when stale-p "gray60")))
                (if stale-p
                    (propertize emoji 'face '(:foreground "gray60"))
                  emoji)))
             (tooltip (if stale-p
                          (format "%s\nStale: updated %s — fetch failed, will retry"
                                  weather-string age-str)
                        (format "%s\nUpdated %s" weather-string age-str))))
        (when (featurep 'wttrin-debug)
          (wttrin--debug-log "mode-line-display: Extracted emoji = %S, stale = %s"
                             emoji stale-p))
        (setq wttrin-mode-line-string
              (propertize (concat " " emoji-with-font)
                          'help-echo tooltip
                          'mouse-face 'mode-line-highlight
                          'local-map wttrin--mode-line-map)))))
  (force-mode-line-update t))

(defun wttrin-mode-line-click ()
  "Handle left-click on mode-line weather widget.
Check cache, refresh if needed, then open weather buffer."
  (interactive)
  (when wttrin-favorite-location
    (wttrin wttrin-favorite-location)))

(defun wttrin-mode-line-force-refresh ()
  "Handle right-click on mode-line weather widget.
Force-refresh cache and update tooltip without opening buffer."
  (interactive)
  (when wttrin-favorite-location
    (let ((wttrin--force-refresh t))
      (wttrin--mode-line-fetch-weather))))

(defun wttrin--mode-line-set-placeholder ()
  "Set a placeholder icon in the mode-line while waiting for weather data."
  (let ((icon (if wttrin-mode-line-emoji-font
                  (propertize "⏳"
                              'face (list :family wttrin-mode-line-emoji-font
                                          :height 1.0))
                "⏳")))
    (setq wttrin-mode-line-string
          (propertize (concat " " icon)
                      'help-echo (format "Fetching weather for %s..."
                                         wttrin-favorite-location)
                      'mouse-face 'mode-line-highlight
                      'local-map wttrin--mode-line-map)))
  (force-mode-line-update t))

(defvar wttrin--buffer-refresh-timer nil
  "Timer object for proactive buffer cache refresh.")

(defun wttrin--buffer-cache-refresh ()
  "Proactively refresh the buffer cache for `wttrin-favorite-location'.
Fetches fresh weather data and updates the buffer cache entry without
displaying anything.  This keeps buffer data fresh for when the user
opens the weather buffer."
  (when wttrin-favorite-location
    (let* ((location wttrin-favorite-location)
           (cache-key (wttrin--make-cache-key location)))
      (wttrin-fetch-raw-string
       location
       (lambda (fresh-data)
         (when fresh-data
           (wttrin--cleanup-cache-if-needed)
           (puthash cache-key (cons (float-time) fresh-data) wttrin--cache)))))))

(defun wttrin--mode-line-start ()
  "Start mode-line weather display and refresh timer."
  (when (featurep 'wttrin-debug)
    (wttrin--debug-log "wttrin mode-line: Starting mode-line display (location=%s, interval=%s)"
                       wttrin-favorite-location
                       wttrin-mode-line-refresh-interval))
  (when wttrin-favorite-location
    ;; Show placeholder immediately so user knows wttrin is active
    (wttrin--mode-line-set-placeholder)
    ;; Delay initial fetch to allow network to initialize during startup
    (run-at-time wttrin-mode-line-startup-delay nil #'wttrin--mode-line-fetch-weather)
    ;; Set up refresh timer (starts after the interval from now)
    (when wttrin--mode-line-timer
      (cancel-timer wttrin--mode-line-timer))
    (setq wttrin--mode-line-timer
          (run-at-time wttrin-mode-line-refresh-interval
                      wttrin-mode-line-refresh-interval
                      #'wttrin--mode-line-fetch-weather))
    ;; Start buffer cache refresh timer
    (when wttrin--buffer-refresh-timer
      (cancel-timer wttrin--buffer-refresh-timer))
    (setq wttrin--buffer-refresh-timer
          (run-at-time wttrin-refresh-interval
                      wttrin-refresh-interval
                      #'wttrin--buffer-cache-refresh))
    (when (featurep 'wttrin-debug)
      (wttrin--debug-log "wttrin mode-line: Initial fetch scheduled in %s seconds, then every %s seconds"
                         wttrin-mode-line-startup-delay
                         wttrin-mode-line-refresh-interval))))

(defun wttrin--mode-line-stop ()
  "Stop mode-line weather display and cancel timers."
  (when (featurep 'wttrin-debug)
    (wttrin--debug-log "wttrin mode-line: Stopping mode-line display"))
  (when wttrin--mode-line-timer
    (cancel-timer wttrin--mode-line-timer)
    (setq wttrin--mode-line-timer nil))
  (when wttrin--buffer-refresh-timer
    (cancel-timer wttrin--buffer-refresh-timer)
    (setq wttrin--buffer-refresh-timer nil))
  (setq wttrin-mode-line-string nil)
  (setq wttrin--mode-line-cache nil)
  (force-mode-line-update t))

;;;###autoload
(define-minor-mode wttrin-mode-line-mode
  "Toggle weather display in mode-line.
When enabled, shows weather for `wttrin-favorite-location'."
  :global t
  :lighter (:eval wttrin-mode-line-string)
  (if wttrin-mode-line-mode
      (progn
        (when (featurep 'wttrin-debug)
          (wttrin--debug-log "wttrin mode-line: Mode enabled"))
        ;; Delay network activity until Emacs is fully initialized
        (if (and (not after-init-time) (not noninteractive))
            (progn
              (when (featurep 'wttrin-debug)
                (wttrin--debug-log "wttrin mode-line: Deferring start until after-init-hook"))
              (add-hook 'after-init-hook #'wttrin--mode-line-start))
          (wttrin--mode-line-start))
        ;; Add modeline string to global-mode-string for custom modelines
        (if global-mode-string
            (add-to-list 'global-mode-string 'wttrin-mode-line-string 'append)
          (setq global-mode-string '("" wttrin-mode-line-string)))
        (when (featurep 'wttrin-debug)
          (wttrin--debug-log "wttrin mode-line: Added to global-mode-string = %S" global-mode-string)))
    (when (featurep 'wttrin-debug)
      (wttrin--debug-log "wttrin mode-line: Mode disabled"))
    (wttrin--mode-line-stop)
    ;; Remove from global-mode-string
    (setq global-mode-string
          (delq 'wttrin-mode-line-string global-mode-string))))

;;;###autoload
(defun wttrin (location)
  "Display weather information for LOCATION.
Weather data is fetched asynchronously to avoid blocking Emacs."
  (interactive
   (list
    (completing-read "Location Name: " wttrin-default-locations nil nil
                     (when (= (length wttrin-default-locations) 1)
                       (car wttrin-default-locations)))))
  (wttrin-query location))

;; Auto-enable mode-line display if requested
(when wttrin-mode-line-auto-enable
  (wttrin-mode-line-mode 1))

(provide 'wttrin)
;;; wttrin.el ends here
